/* File "matrix.c":
 * TCL/TK package that implements feature/value matrixes for canvas widgets. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <setjmp.h>
#include <ctype.h>
#include <assert.h>
#include <math.h>
#include <X11/Xlib.h>
#include <tcl.h>
#include <tk.h>
#include "basic.h"
#include "scanner.h"

#undef GLOBAL
#define GLOBAL

/* error management =========================================================*/

#define VALUE_ERROR_STRING_LENGTH 256

static char value_error_string[VALUE_ERROR_STRING_LENGTH];
/* buffer for parse errors */

LOCAL bool_t parse_value_error;
LOCAL jmp_buf *parse_value_error_jump_point;
/* the current point of return in case of an error */

/*---------------------------------------------------------------------------*/

void error (string_t format, ...)
/* Manage syntax errors during parsing of Malaga value structures. */
{
  va_list arg;
  
  va_start (arg, format);

  if (parse_value_error_jump_point != NULL) 
  {
    vsprintf (value_error_string, format, arg);
    va_end (arg);
    
    parse_value_error = 1;
    longjmp (*parse_value_error_jump_point, 1);
  }
  else 
  {
    fflush (stdout);
    fprintf (stderr, "error: ");
    
    vfprintf (stderr, format, arg);
    fputc ('\n', stderr);
    
    va_end (arg);
    exit (1);
  }
}

/* parsing Malaga values ====================================================*/

typedef enum /* type of a node in "matrix_value_t" */
{ 
  STRING_TYPE, 
  RECORD_TYPE, 
  LIST_TYPE 
} 
value_type_t;

typedef struct MATRIX_VALUE_T   /* a Malaga value in TK */
{
  value_type_t value_type;      /* type of this node */
  struct MATRIX_VALUE_T *first; /* first element if record or list */
  struct MATRIX_VALUE_T *next;  /* next element for record and list elements */
  string_t attribute;           /* attribute name (if record element) */
  string_t string;              /* string (for STRING_TYPE) */
  long_t attribute_width;       /* width of attribute */
  long_t width;                 /* width of value box */
  long_t height;                /* height of value box */
} matrix_value_t;

FORWARD matrix_value_t *parse_value_local (void);

/*---------------------------------------------------------------------------*/

LOCAL void free_value (matrix_value_t *value_ptr)
/* Free memory used by the tree. */
{
  matrix_value_t *next_value;

  while (value_ptr != NULL)
  {
    if (value_ptr->first != NULL)
      free_value (value_ptr->first);
    
    if (value_ptr->attribute != NULL)
      free (value_ptr->attribute);
    
    if (value_ptr->string != NULL)
      free (value_ptr->string);
    
    next_value = value_ptr->next;
    free (value_ptr);
    value_ptr = next_value;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL string_t parse_symbol (void)
/* Parse a symbol and return it. */
{
  string_t symbol;
   
  test_token (TOK_IDENT);
  symbol = new_string (token_name);
  read_next_token ();
  return symbol;
}

/*---------------------------------------------------------------------------*/

LOCAL matrix_value_t *parse_attribute_value_pair (void)
/* Parse an attribute-value pair and return a pointer to it. */
{
  string_t attribute;
  matrix_value_t *new_value;
  
  if (next_token == '(') /* Read a hidden attribute. */
  {
    read_next_token ();
    attribute = parse_symbol ();
    parse_token (')');
    new_value = (matrix_value_t *) new_mem (sizeof (matrix_value_t));
    new_value->value_type = STRING_TYPE;
    new_value->attribute = concat_strings (attribute, ":", NULL);
    new_value->string = new_string ("...");
    free (attribute);
  }
  else
  {
    attribute = parse_symbol ();
    parse_token (':');
    new_value = parse_value_local ();
    new_value->attribute = concat_strings (attribute, ":", NULL);
    free (attribute);
  }
  return new_value;
}

/*---------------------------------------------------------------------------*/

LOCAL matrix_value_t *parse_value_local (void)
/* Parse a value and return it as a "matrix_value". */
{
  matrix_value_t *new_value_ptr;

  /* pointer to last list or record element */
  matrix_value_t *element_ptr;

  new_value_ptr = (matrix_value_t *) new_mem (sizeof (matrix_value_t));

  switch (next_token)
  {
  case '<':
    new_value_ptr->value_type = LIST_TYPE;
    read_next_token ();
    if (next_token != '>') /* Insert "new_value" as first list element. */
    {
      new_value_ptr->first = parse_value_local ();
      element_ptr = new_value_ptr->first;
      
      while (next_token == ',')
      {
	read_next_token ();
	       
	/* Insert "new_value" as successor element. */
	element_ptr->next = parse_value_local ();
	element_ptr = element_ptr->next;
      }
    }
    parse_token ('>');
    break;
    
  case '[':
    new_value_ptr->value_type = RECORD_TYPE;
    read_next_token ();
    if (next_token != ']')
    {
      new_value_ptr->first = parse_attribute_value_pair ();
      element_ptr = new_value_ptr->first;
      while (next_token == ',')
      {
	read_next_token ();
	element_ptr->next = parse_attribute_value_pair ();
	element_ptr = element_ptr->next;
      }
    }
    parse_token (']');
    break;
    
  case TOK_IDENT:
    new_value_ptr->value_type = STRING_TYPE;
    new_value_ptr->string = parse_symbol ();
    break;
    
  case TOK_STRING:
    delete_escapes (token_name);
    new_value_ptr->value_type = STRING_TYPE;
    new_value_ptr->string = concat_strings ("\"", token_name, "\"", NULL);
    read_next_token ();
    break;
    
  case TOK_NUMBER:
  {
    char string_buffer[25];
    
    sprintf (string_buffer, "%.11G", token_number);
    new_value_ptr->value_type = STRING_TYPE;
    new_value_ptr->string = new_string (string_buffer);
    read_next_token ();
    break;
  }
  
  default:
    error ("value expected, not `%s'", token_as_text (next_token));
  }
  return new_value_ptr;
}

/*---------------------------------------------------------------------------*/

LOCAL matrix_value_t *build_value_tree (string_t string_value)
/* Parse Malaga value "*string_value" and return it as a "matrix_value". */
{ 
  /* where to jump after an error */
  static jmp_buf local_parse_error_jump_point;
  matrix_value_t * volatile first_value;
  
  first_value = NULL;
  setjmp (local_parse_error_jump_point);
  parse_value_error_jump_point = &local_parse_error_jump_point;
   
  if (parse_value_error) 
  {
    free_value (first_value);
    parse_value_error_jump_point = NULL;
    parse_value_error = FALSE;
    return NULL;
  }
  
  set_scanner_input ((string_t) string_value);
  if (next_token == '!') /* Enter a string without quotes. */
  {
    read_next_token ();
    test_token (TOK_STRING);
    first_value = (matrix_value_t *) new_mem (sizeof (matrix_value_t));
    first_value->value_type = STRING_TYPE;
    first_value->string = new_string (token_name);
    read_next_token ();
  }
  else
    first_value = parse_value_local ();
  
  /* Allow an additional ';'. */
  if (next_token == ';')
    read_next_token ();
  
  test_token (EOF);
  parse_value_error_jump_point = NULL;

  return first_value;
}

/* displaying Malaga values =================================================*/

#define HANGUL_FONT "-*-mincho-medium-r-normal--*-*-*-*-c-*-ksc5601.1987-0"

#define RECORD_TOP_BORDER 3
#define RECORD_BOTTOM_BORDER 3
#define RECORD_LEFT_BORDER 6
#define RECORD_RIGHT_BORDER 6
#define LIST_TOP_BORDER 3
#define LIST_BOTTOM_BORDER 3
#define ANGLE_BRACKET_RATIO 6 /* height:width ratio of angle brackets */

typedef struct MATRIX_ITEM /* The structure defines a canvas item. */
{
  Tk_Item header;             /* generic stuff - MUST BE FIRST IN STRUCTURE! */
  double x, y;                /* positioning point for matrix: upper left */
  GC gc;                      /* graphics context for lines and latin text */
  short_t line_width;         /* width of a brace line */
  XColor *color;              /* color for line and text */
  string_t char_set;          /* latin1 and hangul (KSC5601) are supported. */
#if TK_MAJOR_VERSION > 4
  Tk_Font font; /* font for drawing text */
#else
  XFontStruct *font; /* font for drawing text */
#endif
  short ascent; /* ascent of <font> */  
  short line_height;          /* height of a line in latin1 font */
  short space_width;          /* with of a space in latin1 font */
  short comma_width;          /* with of a comma in latin1 font */
  matrix_value_t *root_value; /* value tree for matrix object */
  XFontStruct *hangul_font; 
} matrix_t;

LOCAL XFontStruct *hangul_font; 
LOCAL long_t hangul_font_refs; /* number of references to <hangul_font> */

/*---------------------------------------------------------------------------*/

static Tk_CustomOption tagsOption = 
{Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, (ClientData) NULL};

static Tk_ConfigSpec config_specs[] = 
/* Information used for parsing configuration specs. If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateMatrix. */
{
  {TK_CONFIG_COLOR, "-fill", NULL, NULL,
   "black", Tk_Offset (matrix_t, color), TK_CONFIG_NULL_OK},
  {TK_CONFIG_PIXELS, "-line_width", NULL, NULL,
   "1", Tk_Offset (matrix_t, line_width), TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_STRING, "-char_set", NULL, NULL,
   "latin1", Tk_Offset (matrix_t, char_set), 0},
  {TK_CONFIG_FONT, "-font", NULL, NULL,
   "-*-helvetica-medium-r-normal--14-*-75-75-p-*-iso8859-1",
   Tk_Offset (matrix_t, font), 0},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL,
   NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
  {TK_CONFIG_END, NULL, NULL, NULL,
   NULL, 0, 0}
};

/*===========================================================================*/

#define BUFFER_SIZE 500
static XChar2b string16[BUFFER_SIZE];

/*---------------------------------------------------------------------------*/

static long_t string_to_string16 (string_t string)
/* Convert <string> to <string16> and return its length. */
{
  long_t n;

  for (n = 0; n < BUFFER_SIZE; n++)
  {
    if (ORD (string[0]) < 0xa0 || ORD (string[1]) < 0xa0)
      break;
    
      /* Convert KSC5601 character. */
      string16[n].byte1 = ORD (string[0]) & 0x7f;
      string16[n].byte2 = ORD (string[1]) & 0x7f;
      string += 2;
  }
  return n;
}

/*---------------------------------------------------------------------------*/

static long_t string_width (matrix_t *matrix, string_t string)
/* Return width of <string> in pixels when displayed as part of <matrix>. */
{
  if (matrix->hangul_font != NULL)
  {
    long_t n, width;
    
    width = 0;
    while (*string != EOS)
    { /* Add width of a Hangul segment. */
      
      n = string_to_string16 (string);
      string += 2 * n;
      if (n > 0)
	width += XTextWidth16 (hangul_font, string16, n);
      
      /* Add width of an ASCII segment. */
      n = 0;
      while (*string != EOS 
	     && (ORD (string[0]) < 0xa0 || ORD (string[1]) < 0xa0))
      {
	string++;
	n++;
      }
      if (n > 0)
      {
#if TK_MAJOR_VERSION > 4
	width += Tk_TextWidth (matrix->font, string - n, n);
#else
	width += XTextWidth (matrix->font, string - n, n);
#endif
      }
    }
    return width;
  }
  
#if TK_MAJOR_VERSION > 4
  return Tk_TextWidth (matrix->font, string, strlen (string));
#else
  return XTextWidth (matrix->font, string, strlen (string));
#endif
}

/*---------------------------------------------------------------------------*/

static void draw_string (Display *display,
			 Drawable drawable,
			 matrix_t *matrix,
			 long_t origin_x,
			 long_t origin_y,
			 string_t string)
/* Draw a string as part of a value matrix. */
{
  origin_y += matrix->ascent;
  
  if (matrix->hangul_font != NULL)
  {
    /* Hangul KSC5601 section - show hangul characters in Hangul font.
     * All other characters are shown in normal font. */
    
    long_t n;
    
    while (*string != '\0')
    {
      /* Display a Hangul segment. */
      n = string_to_string16 (string);
      string += 2 * n;
      if (n > 0)
      {
	XSetFont (display, matrix->gc, hangul_font->fid);
	XDrawString16 (display, drawable, matrix->gc, 
		       origin_x, origin_y, string16, n);
	origin_x += XTextWidth16 (hangul_font, string16, n);
      }
      
      /* Print an ASCII segment. */
      n = 0;
      while (*string != EOS 
	     && (ORD (string[0]) < 0xa0 || ORD (string[1]) < 0xa0))
      {
	string++;
	n++;
      }
      if (n > 0)
      {
#if TK_MAJOR_VERSION > 4
	XSetFont (display, matrix->gc, Tk_FontId (matrix->font));
	Tk_DrawChars (display, drawable, matrix->gc, matrix->font, 
		      string - n, n, origin_x, origin_y);
	origin_x += Tk_TextWidth (matrix->font, string - n, n);
#else
	XSetFont (display, matrix->gc, matrix->font->fid);
	XDrawString (display, drawable, matrix->gc, origin_x, origin_y, 
		     string - n, n);
	origin_x += XTextWidth (matrix->font, string - n, n);
#endif
      }
    }
  }
  else 
  { 
#if TK_MAJOR_VERSION > 4
    Tk_DrawChars (display, drawable, matrix->gc, matrix->font, 
		  string, strlen (string), origin_x, origin_y);
#else
    XDrawString (display, drawable, matrix->gc, origin_x, origin_y, 
		 string, strlen (string));
#endif
  }
}

/*===========================================================================*/

static long_t max_attribute_width (matrix_value_t *value)
/* Compute maximum attribute width of a record. */
{
  matrix_value_t *element;
  long_t max_length = 0;
  
  for (element = value->first; element != NULL; element = element->next)
    max_length = MAX (max_length, element->attribute_width);
  return max_length;
}

/*---------------------------------------------------------------------------*/

static void compute_value_bounding_box (matrix_t *matrix, 
					matrix_value_t *value)
/* Compute the coordinates of matrix value <value> in <matrix> */
{
  matrix_value_t *element;

  if (value->attribute != NULL)
    value->attribute_width = string_width (matrix, value->attribute);

  switch (value->value_type) 
  {
  case RECORD_TYPE:
  {
    long_t max_attrib_width, max_element_width;
    
    value->height = 0;
    max_attrib_width = 0;
    max_element_width = 0;
    for (element = value->first; element != NULL; element = element->next)
    {
      compute_value_bounding_box (matrix, element);
      max_attrib_width = MAX (max_attrib_width, element->attribute_width);
      max_element_width = MAX (max_element_width, element->width);
      value->height += element->height;
    }
	
    value->height = (RECORD_TOP_BORDER
		     + MAX (value->height, matrix->line_height)
		     + RECORD_BOTTOM_BORDER);
    value->width = (RECORD_LEFT_BORDER + max_attrib_width 
		    + matrix->space_width + max_element_width
		    + RECORD_RIGHT_BORDER);
    break;
  }

  case LIST_TYPE:
  {
    long_t max_element_height;
    
    value->width = 2;
    max_element_height = matrix->line_height;
    for (element = value->first; element != NULL; element = element->next)
    {
      compute_value_bounding_box (matrix, element);

      max_element_height = MAX (max_element_height, element->height);
      value->width += element->width;
      if (element->next != NULL) 
	value->width += (matrix->comma_width + matrix->space_width);
    }
    value->width += 2;
    
    value->height = (LIST_TOP_BORDER + max_element_height 
		     + LIST_BOTTOM_BORDER);
    value->width += 2 * (value->height / ANGLE_BRACKET_RATIO);
    break;
  }
      
  case STRING_TYPE:
    value->width = string_width (matrix, value->string);
    value->height = matrix->line_height;
    break;

  default:
    error ("unexpected node type");
  }
}

/*---------------------------------------------------------------------------*/

static void compute_matrix_bounding_box (matrix_t *matrix)
/* Compute the bounding box of all the pixels that may be
 * drawn as part of matrix <matrix>. */
{
  compute_value_bounding_box (matrix, matrix->root_value);
  matrix->header.x1 = (int) matrix->x ;
  matrix->header.y1 = (int) matrix->y;
  matrix->header.x2 = (int) (matrix->x + matrix->root_value->width);
  matrix->header.y2 = (int) (matrix->y + matrix->root_value->height);
}

/*---------------------------------------------------------------------------*/

static int configure_matrix (Tcl_Interp *interp, 
			     Tk_Canvas canvas, 
			     Tk_Item *item, 
			     int argc, 
			     string_t argv[], 
			     int flags)
/* Configure various aspects of matrix <item>. */
{
  matrix_t *matrix = (matrix_t *) item; 
  XGCValues gc_values;
  GC new_gc;
  u_long_t mask;
  Tk_Window tk_window = Tk_CanvasTkwin (canvas);
  
  if (Tk_ConfigureWidget (interp, tk_window, config_specs, 
			  argc, argv, (char *) matrix, flags) != TCL_OK) 
    return TCL_ERROR;
  
  /* A few of the options require additional processing. */

  if ((strcmp (matrix->char_set, "latin1") != 0) &&
      (strcmp (matrix->char_set, "hangul") != 0)) 
  {
    Tcl_AppendResult (interp, "wrong # args: should be \"",
		      Tk_PathName (tk_window), "\" create ",
		      item->typePtr->name, " x1 y1 value ?options?",
		      NULL);
    return TCL_ERROR;
  }
  
  if (strcmp (matrix->char_set, "hangul") == 0 
      && matrix->hangul_font == NULL)
  {
    if (hangul_font == NULL)
      hangul_font = XLoadQueryFont (Tk_Display (tk_window), HANGUL_FONT);
    if (hangul_font == NULL)
      return TCL_ERROR;

    matrix->hangul_font = hangul_font;
    hangul_font_refs++;
  }
  else if (strcmp (matrix->char_set, "hangul") != 0
	   && matrix->hangul_font != NULL)
  {
    matrix->hangul_font = NULL;
    hangul_font_refs--;
  }

  new_gc = None;
  if (matrix->color != NULL && matrix->font != NULL) 
  {
    gc_values.foreground = matrix->color->pixel;
#if TK_MAJOR_VERSION > 4
    gc_values.font = Tk_FontId (matrix->font);
#else
    gc_values.font = matrix->font->fid;
#endif    
    mask = GCForeground|GCFont;
    new_gc = Tk_GetGC (tk_window, mask, &gc_values);
    
#if TK_MAJOR_VERSION > 4
    {
      Tk_FontMetrics font_metrics;
      
      Tk_GetFontMetrics (matrix->font, &font_metrics);
      matrix->ascent = font_metrics.ascent;
      matrix->line_height = (font_metrics.ascent + font_metrics.descent);
      matrix->space_width = Tk_TextWidth (matrix->font, " ", 1);
      matrix->comma_width = Tk_TextWidth (matrix->font, ",", 1);
    }
#else
    matrix->ascent = matrix->font->ascent;
    matrix->line_height = (matrix->font->ascent + matrix->font->descent);
    matrix->space_width = XTextWidth (matrix->font, " ", 1);
    matrix->comma_width = XTextWidth (matrix->font, ",", 1);
#endif
  }

  if (matrix->gc != None)
    Tk_FreeGC (Tk_Display (tk_window), matrix->gc);
  
  matrix->gc = new_gc;
  
  compute_matrix_bounding_box (matrix);
  
  return TCL_OK;
}

/*---------------------------------------------------------------------------*/

static void delete_matrix (Tk_Canvas canvas, Tk_Item *item, Display *display)
/* Delete Matrix <item>. */
{
  matrix_t *matrix = (matrix_t *) item;

  if (matrix->root_value != NULL) 
    free_value (matrix->root_value);

  if (matrix->color != NULL)
    Tk_FreeColor (matrix->color);
  
  if (matrix->gc != None)
    Tk_FreeGC (display, matrix->gc);

  if (matrix->font != NULL)
  {
#if TK_MAJOR_VERSION > 4
    Tk_FreeFont (matrix->font);
#else
    Tk_FreeFontStruct (matrix->font);
#endif
  }

  if (matrix->hangul_font != NULL)
  {
    matrix->hangul_font = NULL;
    hangul_font_refs--;
  }

  if (hangul_font != NULL && hangul_font_refs == 0)
  {
    XFreeFont (display, hangul_font);
    hangul_font = NULL;
  }
}

/*---------------------------------------------------------------------------*/

static int create_matrix (Tcl_Interp *interp, 
			  Tk_Canvas canvas, 
			  Tk_Item *item, 
			  int argc, 
			  string_t argv[])
/* Create a new matrix item in a canvas. */
{
  matrix_t *matrix = (matrix_t *) item;
  
  if (argc < 3) 
  {
    Tcl_AppendResult (interp, "wrong # args: should be \"",
		      Tk_PathName (Tk_CanvasTkwin (canvas)), "\" create ",
		      item->typePtr->name, " x1 y1 value ?options?",
		      NULL);
    return TCL_ERROR;
  }
  
  /* Carry out initialisation that is needed to set defaults and to
   * allow proper cleanup after errors during the the remainder of
   * this procedure. */
  
  /* Process the arguments to fill in the item record. */
  if (Tk_CanvasGetCoord (interp, canvas, argv[0], &matrix->x) != TCL_OK
      || Tk_CanvasGetCoord (interp, canvas, argv[1], &matrix->y) != TCL_OK)
    return TCL_ERROR;
  
  matrix->gc = None;
  matrix->color = None;
  matrix->line_width = 1;
  matrix->char_set = NULL;
  matrix->font = NULL;
  matrix->line_height = 0;
  matrix->space_width = 0;
  matrix->comma_width = 0;
  matrix->root_value = NULL;
  
  parse_value_error_jump_point = NULL;
  parse_value_error = 0;
  
  matrix->root_value = build_value_tree (argv[2]);
  if (matrix->root_value == NULL) 
  {
    Tcl_AppendResult (interp, value_error_string, (char *) NULL);
    return TCL_ERROR;
  }
  
  if (argc > 3) 
  {
    if (configure_matrix (interp, canvas, item, argc-3, argv+3, 0) != TCL_OK) 
    {
      delete_matrix (canvas, item, Tk_Display (Tk_CanvasTkwin (canvas)));
      return TCL_ERROR;
    }
  }
  
  return TCL_OK;
}

/*---------------------------------------------------------------------------*/

static int matrix_coords (Tcl_Interp *interp, 
			  Tk_Canvas canvas, 
			  Tk_Item *item, 
			  int argc, 
			  string_t argv[])
/* Implement "coords" widget command on matrices. */
{
  matrix_t *matrix = (matrix_t *) item;
  char x[TCL_DOUBLE_SPACE], y[TCL_DOUBLE_SPACE];
  
  if (argc == 0) 
  {
    Tcl_PrintDouble (interp, matrix->x, x);
    Tcl_PrintDouble (interp, matrix->y, y);
    Tcl_AppendResult (interp, x, " ", y, (char *) NULL);
  } 
  else if (argc == 2) 
  {
    if (Tk_CanvasGetCoord (interp, canvas, argv[0], &matrix->x) != TCL_OK
	|| Tk_CanvasGetCoord (interp, canvas, argv[1], &matrix->y) != TCL_OK)
      return TCL_ERROR;
    
    compute_matrix_bounding_box (matrix);
  } 
  else 
  {
    sprintf (interp->result, 
	     "wrong # coordinates: expected 0 or 2, got %d", argc);
    return TCL_ERROR;
  }
  return TCL_OK;
 }

/*---------------------------------------------------------------------------*/

#define MAX_STATIC_POINTS 4

static void display_value (Display *display,
			   matrix_t *matrix,
			   Drawable drawable,
			   matrix_value_t *value,
			   long_t origin_x,
			   long_t origin_y,
			   long_t x, 
			   long_t y,
			   long_t width,
			   long_t height)
/* Draw a matrix value in a given drawable. */
{
  matrix_value_t *element;
  XPoint static_points[MAX_STATIC_POINTS];
  long_t element_origin_x, element_origin_y, element_bottom;
  
  /* If (sub)value is out of bounds, no need to draw it. */
  if (origin_x >= x + width || origin_y >= y + height
      || origin_x + value->width <= x || origin_y + value->height <= y)
    return;
  
  switch (value->value_type) 
  {
  case RECORD_TYPE:
    static_points[0].x = origin_x + RECORD_LEFT_BORDER - 1;
    static_points[0].y = origin_y + 1;
    
    static_points[1].x = origin_x + 1;
    static_points[1].y = origin_y + 1;
    
    static_points[2].x = origin_x + 1;
    static_points[2].y = origin_y + value->height - 2;
    
    static_points[3].x = origin_x + RECORD_LEFT_BORDER - 1;
    static_points[3].y = origin_y + value->height - 2;
    
    XDrawLines (display, drawable, matrix->gc, static_points, 4, 
		CoordModeOrigin);
      
    static_points[0].x = origin_x + value->width - RECORD_RIGHT_BORDER;
    static_points[0].y = origin_y + 1;
    
    static_points[1].x = origin_x + value->width - 2;
    static_points[1].y = origin_y + 1;
    
    static_points[2].x = origin_x + value->width - 2;
    static_points[2].y = origin_y + value->height - 2;
    
    static_points[3].x = origin_x + value->width - RECORD_RIGHT_BORDER;
    static_points[3].y = origin_y + value->height - 2;
    
    XDrawLines (display, drawable, matrix->gc, static_points, 4,
		CoordModeOrigin);
    
    element_origin_x = (origin_x + RECORD_LEFT_BORDER + 
			max_attribute_width (value) 
			+ matrix->space_width);
    element_origin_y = origin_y + RECORD_TOP_BORDER;
    
    for (element = value->first; element != NULL; element = element->next)
    {
      draw_string (display, drawable, matrix, 
		   origin_x + RECORD_LEFT_BORDER,
		   element_origin_y 
		   + (element->height - matrix->line_height) / 2, 
		   element->attribute);
	  
      display_value (display, matrix, drawable, element, 
		     element_origin_x, element_origin_y, x, y, width, height);
      element_origin_y += element->height;
    }
    break;

  case LIST_TYPE:
  {
    long_t  angle_bracket_width = value->height / ANGLE_BRACKET_RATIO;
    
    static_points[0].x = origin_x + angle_bracket_width;
    static_points[0].y = origin_y + 1;
    
    static_points[1].x = origin_x + 1;
    static_points[1].y = origin_y + value->height / 2;
    
    static_points[2].x = origin_x + angle_bracket_width;
    static_points[2].y = origin_y + value->height - 2;
    
    XDrawLines (display, drawable, matrix->gc, static_points, 3, 
		CoordModeOrigin);
    
    static_points[0].x = (origin_x + value->width - angle_bracket_width - 1);
    static_points[0].y = origin_y + 1;
    
    static_points[1].x = origin_x + value->width - 2;
    static_points[1].y = origin_y + value->height / 2;
    
    static_points[2].x = (origin_x + value->width 
			  - angle_bracket_width - 1);
    static_points[2].y = origin_y + value->height - 2;
    
    XDrawLines (display, drawable, matrix->gc, static_points, 3, 
		CoordModeOrigin);
    
    element_origin_x = origin_x + angle_bracket_width + 2;
    element_bottom = origin_y + value->height - LIST_BOTTOM_BORDER;
    
    for (element = value->first; element != NULL; element = element->next)
    {
      display_value (display, matrix, drawable, element, 
		     element_origin_x, element_bottom - element->height,
		     x, y, width, height);
      element_origin_x += element->width;
      
      if (element->next != NULL) 
      {
	draw_string (display, drawable, matrix, element_origin_x, 
		     element_bottom - matrix->line_height, ",");
	element_origin_x += (matrix->comma_width + matrix->space_width);
      }
    }
    break;
  }
      
  case STRING_TYPE:
    draw_string (display, drawable, matrix, origin_x, origin_y, value->string);
    break;
    
  default:
    error ("unexpected node type");
  }
}

/*---------------------------------------------------------------------------*/

static void display_matrix (Tk_Canvas canvas,
			    Tk_Item *item,
			    Display *display,
			    Drawable drawable,
			    int x, 
			    int y, 
			    int width, 
			    int height)
/* Draw a matrix in a given drawable. */
{
  matrix_t *matrix = (matrix_t *) item;
  short origin_x, origin_y;
  
  Tk_CanvasDrawableCoords (canvas, matrix->x, matrix->y, 
			   &origin_x, &origin_y);
  display_value (display, matrix, drawable, matrix->root_value, 
		 origin_x, origin_y, 
		 x + origin_x - matrix->x, y + origin_y - matrix->y,
		 width, height);
}

/*---------------------------------------------------------------------------*/

static double matrix_to_point (Tk_Canvas canvas, 
			       Tk_Item *item, 
			       double point[])
/* Return the distance between the point (<point>[0], <point>[1])
 * and the matrix <item>. If the point is inside, return 0. */
{
   matrix_t *matrix = (matrix_t *) item;
   double diff_x, diff_y;

   if (point[0] < matrix->header.x1)
     diff_x = matrix->header.x1 - point[0];
   else if (point[0] > matrix->header.x2)
     diff_x = point[0] - matrix->header.x1;
   else 
     diff_x = 0;

   if (point[1] < matrix->header.y1)
     diff_y = matrix->header.y1 - point[1];
   else if (point[1] > matrix->header.y2)
     diff_y = point[1] - matrix->header.y1;
   else 
     diff_y = 0;


   return sqrt (diff_x * diff_x + diff_y * diff_y);
}

/*---------------------------------------------------------------------------*/

static int matrix_to_area (Tk_Canvas canvas, Tk_Item *item, double rect[])
/* Return -1 if <item> lies entirely outside <rect>,
 * 0 if it overlaps, and 1 if it is entirely inside. */
{
   matrix_t *matrix = (matrix_t *) item;

   if (matrix->header.x2 < rect[0] ||
       matrix->header.x1 > rect[2] ||
       matrix->header.y2 < rect[1] || 
       matrix->header.y1 > rect[3])
     return -1;

   if (matrix->header.x1 >= rect[0] &&
       matrix->header.x2 <= rect[2] &&
       matrix->header.y1 >= rect[1] &&
       matrix->header.y2 <= rect[3])
     return 1;

   return 0;
}

/*---------------------------------------------------------------------------*/

static void scale_matrix (Tk_Canvas canvas,
			  Tk_Item *item,
			  double origin_x, 
			  double origin_y,
			  double scale_x,
			  double scale_y)
/* Rescale a matrix item. */
{
  /* matrices can't be scaled */
}

/*---------------------------------------------------------------------------*/

static void translate_matrix (Tk_Canvas canvas,
			      Tk_Item *item,
			      double delta_x, 
			      double delta_y)
/* Move matrix <item> by <delta_x> and <delta_y>. */
{
   matrix_t *matrix = (matrix_t *) item;

   matrix->x += delta_x;
   matrix->y += delta_y;

   matrix->header.x1 = (int) matrix->x;
   matrix->header.y1 = (int) matrix->y;
   matrix->header.x2 = (int) (matrix->x + matrix->root_value->width);
   matrix->header.y2 = (int) (matrix->y + matrix->root_value->height);
}

/*---------------------------------------------------------------------------*/

static void line_to_ps (Tcl_Interp *interp,
			Tk_Canvas canvas,
			matrix_t *matrix,
			double *points,
			long_t num_of_points)
/* Print line in postscript format. */
{
   char buffer[30];

   Tk_CanvasPsPath (interp, canvas, points, num_of_points);
   sprintf (buffer, "%d setlinewidth\n", matrix->line_width);
   Tcl_AppendResult (interp, buffer, NULL);
   Tcl_AppendResult (interp, "stroke\n", NULL);
}

/*---------------------------------------------------------------------------*/

static void string_to_ps (Tcl_Interp *interp,
			  Tk_Canvas canvas,
			  matrix_t *matrix,
			  string_t string,
			  long_t origin_x,
			  long_t origin_y)
/* Print <string> in postscript format at <origin_x>/<origin_y>. */
{
  char buffer[60];
  char *s;

  sprintf (buffer, "%.15g %.15g moveto ", 
	   (double) origin_x,
	   Tk_CanvasPsY (canvas, (double) origin_y + matrix->ascent));
  Tcl_AppendResult (interp, buffer, NULL);
  
  sprintf (buffer, "(");
  for (s = string; *s != EOS; s++)
  {
    if (*s == '(' || *s == ')' || *s == '\\')
      sprintf (buffer, "\\%c", *s);
    else
      sprintf (buffer, "%c", *s);
  }
  
  sprintf (buffer, "(%s) show\n", string);
  Tcl_AppendResult (interp, buffer, NULL);
}

/*---------------------------------------------------------------------------*/

static void value_to_postscript (Tcl_Interp *interp,
				 Tk_Canvas canvas,
				 matrix_t *matrix,
				 matrix_value_t *value,
				 long_t origin_x,
				 long_t origin_y)
/* Generate Postscript code for a value. */
{
  matrix_value_t *element;
  double static_points[2 * MAX_STATIC_POINTS];
  long_t element_origin_x, element_origin_y, element_bottom;
  
  switch (value->value_type) 
  {
  case RECORD_TYPE:
    static_points[0] = origin_x + RECORD_LEFT_BORDER - 1;
    static_points[1] = origin_y + 1;
    
    static_points[2] = origin_x + 1;
    static_points[3] = origin_y + 1;
    
    static_points[4] = origin_x + 1;
    static_points[5] = origin_y + value->height - 2;
    
    static_points[6] = origin_x + RECORD_LEFT_BORDER - 1;
    static_points[7] = origin_y + value->height - 2;
    
    line_to_ps (interp, canvas, matrix, static_points, 4);
    
    static_points[0] = origin_x + value->width - RECORD_RIGHT_BORDER;
    static_points[1] = origin_y + 1;
    
    static_points[2] = origin_x + value->width - 2;
    static_points[3] = origin_y + 1;
    
    static_points[4] = origin_x + value->width - 2;
    static_points[5] = origin_y + value->height - 2;
    
    static_points[6] = origin_x + value->width - RECORD_RIGHT_BORDER;
    static_points[7] = origin_y + value->height - 2;
    
    line_to_ps (interp, canvas, matrix, static_points, 4);
    
    element_origin_x = (origin_x + RECORD_LEFT_BORDER 
			+ max_attribute_width (value) 
			+ matrix->space_width);
    element_origin_y = origin_y + RECORD_TOP_BORDER;
    
    for (element = value->first; element != NULL; element = element->next)
    {
      string_to_ps (interp, canvas, matrix, 
		    element->attribute,
		    origin_x + RECORD_LEFT_BORDER,
		    element_origin_y 
		    + (element->height - matrix->line_height)/2);
      
      value_to_postscript (interp, canvas, matrix, element, 
			   element_origin_x, element_origin_y);
      
      element_origin_y += element->height;
    }
    break;
    
  case LIST_TYPE:
  {
    long_t angle_bracket_width = value->height / ANGLE_BRACKET_RATIO;
    
    static_points[0] = origin_x + angle_bracket_width;
    static_points[1] = origin_y + 1;
    
    static_points[2] = origin_x + 1;
    static_points[3] = origin_y + value->height / 2;
    
    static_points[4] = origin_x + angle_bracket_width;
    static_points[5] = origin_y + value->height - 2;
    
    line_to_ps (interp, canvas, matrix, static_points, 3);
    
    static_points[0] = origin_x + value->width - angle_bracket_width-1;
    static_points[1] = origin_y + 1;
    
    static_points[2] = origin_x + value->width - 2;
    static_points[3] = origin_y + value->height / 2;
    
    static_points[4] = origin_x + value->width - angle_bracket_width-1;
    static_points[5] = origin_y + value->height - 2;
    
    line_to_ps (interp, canvas, matrix, static_points, 3);
    
    element_origin_x = origin_x + angle_bracket_width + 2;
    element_bottom = origin_y + value->height - LIST_BOTTOM_BORDER;
    
    for (element = value->first; element != NULL; element = element->next)
    {
      value_to_postscript (interp, canvas, matrix, element, 
			   element_origin_x, 
			   element_bottom - element->height);
      
      element_origin_x += element->width;
      
      if (element->next != NULL) 
      {
	string_to_ps (interp, canvas, matrix, ",",
		      element_origin_x, 
		      element_bottom - matrix->line_height);
	element_origin_x += (matrix->comma_width 
			     + matrix->space_width);
      }
    }
    break;
  }
  
  case STRING_TYPE:
    string_to_ps (interp, canvas, matrix, value->string, origin_x, origin_y);
    break;
    
  default:
    error ("unexpected node type");
  }
}

/*---------------------------------------------------------------------------*/

static int matrix_to_postscript (Tcl_Interp *interp,
				 Tk_Canvas canvas,
				 Tk_Item *item,
				 int prepass)
/* Generate Postscript code for a matrix. */
{
  matrix_t *matrix = (matrix_t *) item;
  
  if (prepass) 
    return TCL_OK;

  if (matrix->color == NULL) 
    return TCL_OK;

  if (Tk_CanvasPsFont (interp, canvas, matrix->font) != TCL_OK)
    return TCL_ERROR;
  
  if (Tk_CanvasPsColor (interp, canvas, matrix->color) != TCL_OK) 
    return TCL_ERROR;
  
  value_to_postscript (interp, canvas, matrix, matrix->root_value, 
		       matrix->header.x1, matrix->header.y1);
  
  return TCL_OK;
}

/*---------------------------------------------------------------------------*/

/* procedures of matrix item type that can be invoked by generic item code */
Tk_ItemType matrix_type = 
{
  "matrix",                       /* name */
  sizeof (matrix_t),              /* itemSize */
  create_matrix,                  /* createProc */
  config_specs,                   /* configSpecs */
  configure_matrix,               /* configureProc */
  matrix_coords,                  /* coordProc */
  delete_matrix,                  /* deleteProc */
  display_matrix,                 /* displayProc */
  0,                              /* alwaysRedraw */
  matrix_to_point,                /* pointProc */
  matrix_to_area,                 /* areaProc */
  matrix_to_postscript,           /* postscriptProc */
  scale_matrix,                   /* scaleProc */
  translate_matrix,               /* translateProc */
  (Tk_ItemIndexProc *) NULL,      /* indexProc */
  (Tk_ItemCursorProc *) NULL,     /* icursorProc */
  (Tk_ItemSelectionProc *) NULL,  /* selectionProc */
  (Tk_ItemInsertProc *) NULL,     /* insertProc */
  (Tk_ItemDCharsProc *) NULL,     /* dTextProc */
  (Tk_ItemType *) NULL            /* nextPtr */
};

/*---------------------------------------------------------------------------*/

int Matrix_Init (Tcl_Interp *interp)
/* Initialise the package. */
{
  /* Insert <matrix_type> into the type list. */
  Tk_CreateItemType (&matrix_type);

  return TCL_OK;
}
