/********************************************************************************
*                                                                               *
*                            V i s u a l   C l a s s                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 1999 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXVisual.h,v 1.3 1999/10/01 21:23:42 jeroen Exp $                        *
********************************************************************************/
#ifndef FXVISUAL_H
#define FXVISUAL_H


enum FXVisualOptions {
  VISUAL_DEFAULT      = 0,            // Default visual
  VISUAL_MONOCHROME   = 1,            // Must be monochrome visual
  VISUAL_BEST         = 2,            // Best (deepest) visual
  VISUAL_INDEXCOLOR   = 4,            // Palette visual
  VISUAL_GRAYSCALE    = 8,            // Gray scale visual
  VISUAL_TRUECOLOR    = 16,           // Must be true color visual
  VISUAL_OWNCOLORMAP  = 32,           // Allocate private colormap
  VISUAL_DOUBLEBUFFER = 64,           // Double-buffered [FXGLVisual]
  VISUAL_STEREO       = 128,          // Stereo [FXGLVisual]
  VISUAL_NOACCEL      = 256           // No hardware acceleration [for broken h/w]
  };


enum FXVisualType {
  VISUALTYPE_UNKNOWN,                 // Undetermined visual type
  VISUALTYPE_MONO,                    // Visual for drawing into 1-bpp surfaces
  VISUALTYPE_TRUE,                    // True color
  VISUALTYPE_INDEX,                   // Index [palette] color
  VISUALTYPE_GRAY                     // Gray scale
  };
  

// Visual describes pixel format of a drawable
class FXAPI FXVisual : public FXId {
  FXDECLARE(FXVisual)
  friend class FXApp;
  friend class FXWindow;
  friend class FXGLCanvas;
  friend class FXImage;
  friend class FXIcon;
  friend class FXBitmap;
  friend class FXDCWindow;
protected:
  FXuint        flags;                  // Visual flags
  FXuint        hint;                   // Depth Hint
  FXuint        depth;                  // Visual depth, bits/pixel
  FXuint        numred;                 // Number of reds
  FXuint        numgreen;               // Number of greens
  FXuint        numblue;                // Number of blues
  FXuint        numcolors;              // Total number of colors
  FXuint        maxcolors;              // Maximum number of colors
  FXuint        rgborder;               // Order of red/green/blue
  FXVisualType  type;                   // Visual type
  
#ifndef FX_NATIVE_WIN32
  
protected:
  Visual       *visual;                 // Application visual
  XVisualInfo  *info;                   // More info [FXGLVisual]
  FXID          colormap;               // Color map
  GC            gc;                     // Graphic context matching visual
  FXPixel       redmax;                 // Largest red value
  FXPixel       greenmax;               // Largest green value
  FXPixel       bluemax;                // Largest blue value
  FXPixel       redmask;                // Red plane mask
  FXPixel       greenmask;              // Green plane mask
  FXPixel       bluemask;               // Blue plane mask
  FXuchar       redshift;               // Red shift
  FXuchar       greenshift;             // Green shift
  FXuchar       blueshift;              // Blue shift
  FXPixel      *red_to_pix;             // Red to pixel lookup table
  FXPixel      *green_to_pix;           // Green to pixel lookup table
  FXPixel      *blue_to_pix;            // Blue to pixel lookup table
  FXuchar      *pix_to_red;             // Find red value from pixel
  FXuchar      *pix_to_green;           // Find green value from pixel
  FXuchar      *pix_to_blue;            // Find blue value from pixel
  FXPixel      *alloced;                // Allocated pixels in the colormap
  FXuint        nalloced;               // Number of allocated colors
  FXbool        freemap;                // We allocated the map
protected:
  static const FXuint dither[16];       // Dither table
protected:
  GC makegc();
  void setuptruecolor();
  void setupdirectcolor();
  void setuppseudocolor();
  void setupstaticcolor();
  void setupgrayscale();
  void setupstaticgray();
  void setupstaticmono();
  void setuppixmapmono();
  void setupcolormap();
protected:
  
  // True color
  inline FXPixel pixel_true(FXuint r,FXuint g,FXuint b){
    return red_to_pix[(redmax*r+127)/255] | green_to_pix[(greenmax*g+127)/255] | blue_to_pix[(bluemax*b+127)/255];
    }
  
  inline FXPixel pixel_true_dither(FXuint r,FXuint g,FXuint b,FXuint x,FXuint y){
    register FXuint d=dither[((y&3)<<2)|(x&3)];
    return red_to_pix[(redmax*r+d)/255] | green_to_pix[(greenmax*g+d)/255] | blue_to_pix[(bluemax*b+d)/255];
    }
  
  inline FXColor color_true(FXPixel pix){
    return FXRGB(pix_to_red[(pix&redmask)>>redshift],pix_to_green[(pix&greenmask)>>greenshift],pix_to_blue[(pix&bluemask)>>blueshift]);
    }

    
  // Monochrome
  inline FXPixel pixel_mono(FXuint r,FXuint g,FXuint b){
    return red_to_pix[(r+g+b)/765];
    }
  
  inline FXPixel pixel_mono_dither(FXuint r,FXuint g,FXuint b,FXuint x,FXuint y){
    return red_to_pix[(r+g+b+3*dither[((y&3)<<2)|(x&3)])/765];
    }
  
  inline FXColor color_mono(FXPixel pix){
    return FXRGB(pix_to_red[pix],pix_to_green[pix],pix_to_blue[pix]);
    }
  
  
  // Gray color
  inline FXPixel pixel_gray(FXuint r,FXuint g,FXuint b){
    return red_to_pix[(redmax*(r+g+b)+382)/765];
    }
  
  inline FXPixel pixel_gray_dither(FXuint r,FXuint g,FXuint b,FXuint x,FXuint y){
    return red_to_pix[(redmax*(r+g+b)+3*dither[((y&3)<<2)|(x&3)])/765];
    }
  
  inline FXColor color_gray(FXPixel pix){
    return FXRGB(pix_to_red[pix],pix_to_green[pix],pix_to_blue[pix]);
    }
  
  
  // Pseudo color
  inline FXPixel mix(FXuint r,FXuint g,FXuint b){
    return (r*numgreen+g)*numblue+b;
    }
  
  inline FXPixel pixel_index(FXuint r,FXuint g,FXuint b){
    return red_to_pix[mix((redmax*r+127)/255, (greenmax*g+127)/255, (bluemax*b+127)/255)];
    }
  
  inline FXPixel pixel_index_dither(FXuint r,FXuint g,FXuint b,FXuint x,FXuint y){
    register FXuint d=dither[((y&3)<<2)|(x&3)];
    return red_to_pix[mix((redmax*r+d)/255, (greenmax*g+d)/255, (bluemax*b+d)/255)];
    }
  
  inline FXColor color_index(FXPixel pix){
    return FXRGB(pix_to_red[pix],pix_to_green[pix],pix_to_blue[pix]);
    }
  
#else
  
protected:
  HPALETTE hPalette;
  PIXELFORMATDESCRIPTOR *info;
  int pixelformat;

protected:
  HPALETTE createAllPurposePalette();
  
#endif
  
protected:
  FXVisual();
public:
  
  // Construct default visual
  FXVisual(FXApp* a,FXuint flgs,FXuint d=32);

  // Initialize
  virtual void init();
  
  // Get flags
  FXuint getFlags() const { return flags; }

  // Get depth
  FXuint getDepth() const { return depth; }

  // Get number of colors
  FXuint getNumColors() const { return numcolors; }

  // Get number of reds
  FXuint getNumRed() const { return numred; }

  // Get number of greens
  FXuint getNumGreen() const { return numred; }

  // Get number of blues
  FXuint getNumBlue() const { return numred; }

  // Get RGB order
  FXuint getRGBOrder() const { return rgborder; }

  // Get device pixel value for color
  FXPixel getPixel(FXColor clr);
  
  // Get color value for device pixel value
  FXColor getColor(FXPixel pix);
  
  // Set maximum number of colors to allocate
  void setMaxColors(FXuint maxcols);
  
  // Get maximum number of colors
  FXuint getMaxColors() const { return maxcolors; }

  // Get visual type
  FXVisualType getType() const { return type; }

  // Save
  virtual void save(FXStream& store) const;
  
  // Load
  virtual void load(FXStream& store);
  
  // Destroy
  virtual ~FXVisual();
  };
  

#endif
