// ---------------------------------------------------------------------------
// - Vector.cpp                                                              -
// - standard object library - dynamic vector class implementation           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Real.hpp"
#include "Vector.hpp"
#include "Method.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // -------------------------------------------------------------------------
  // - vector class section                                                  -
  // -------------------------------------------------------------------------

  // create an empty vector

  Vector::Vector (void) {
    d_size   = 0;
    d_length = 0;
    p_vector = nilp;
  }

  // create a vector with an original size
  
  Vector::Vector (const long size) {
    if (size < 0) throw Exception ("size-error","in vector constructor");
    d_size   = size;
    d_length = 0;
    p_vector = new (Object*)[d_size];
    for (long i = 0; i < d_size; i++)
      p_vector[i] = nilp;
  }
  
  // copy constructor for this vector

  Vector::Vector (const Vector& that) {
    d_size   = that.d_size;
    d_length = that.d_length;
    p_vector = new (Object*)[d_size];
    for (long i = 0; i < d_length; i++)
      p_vector[i] = Object::iref (that.p_vector[i]);
  }

  // destroy this vector

  Vector::~Vector (void) {
    for (long i = 0; i < d_length; i++)
      Object::dref (p_vector[i]);
    delete [] p_vector;
  }

  // return the class name

  String Vector::repr (void) const {
    return "Vector";
  }

  // assignment operator for this class

  Vector& Vector::operator = (const Vector& that) {
    // protect against this = that
    if (this == &that) return *this;
    
    // clean vector first
    if (d_length != 0) {
      for (long i = 0; i < d_length; i++)
      Object::dref (p_vector[i]);
      delete [] p_vector;
    }

    // copy old to new
    d_size   = that.d_size;
    d_length = that.d_length;
    p_vector = new (Object*)[d_size];
    for (long i = 0; i < d_length; i++)
    p_vector[i] = Object::iref (that.p_vector[i]);
    return *this;
  }

  // add a new element in this vector

  void Vector::append (Object* object) {
    // check if we have to resize the vector
    if (d_length >= d_size) {
      long size = (d_size <= 0) ? 1 : d_size * 2;
      Object** vector = new (Object*)[size];
      for (long i = 0; i < d_length; i++)
	vector[i] = p_vector[i];
      delete [] p_vector;
      d_size   = size;
      p_vector = vector;
    }

    // set the object in this vector
    p_vector[d_length++] = Object::iref (object);
  }
  
  // set an object at a certain position in this vector. The old object is
  // destroyed.
  
  void Vector::set (const long index, Object* object) {
    // check that we are bounded
    if (index >= d_length) 
      throw Exception ("index-error","in vector set");
    
    // set the object
    Object::dref (p_vector[index]);
    p_vector[index] = Object::iref (object);
  }

  // get an object at a certain position
  
  Object* Vector::get (const long index) const {
    // check that we are bounded
    if (index >= d_length) 
      throw Exception ("index-error","in vector get");
    // get the object
    return p_vector[index];
  }
  
  // get the number of element in this vector

  long Vector::length (void) const {
    return d_length;
  }

  // return a new vector iterator

  Iterator* Vector::makeit (void) {
    return new VectorIterator (this);
  }

  // get an integer value from an index

  t_long Vector::getint (const long index) const {
    Object*  obj  = get (index);
    Integer* iobj = dynamic_cast <Integer*> (obj);
    if (iobj == nilp) 
      throw Exception ("type-error", "looking for integer but got",
		       Object::repr (obj));
    return iobj->toInteger ();
  }

  // get a real value from an index

  t_real Vector::getreal (const long index) const {
    Object* obj  = get (index);
    Real*   robj = dynamic_cast <Real*> (obj);
    if (robj == nilp) 
      throw Exception ("type-error", "looking for real but got",
		       Object::repr (obj));
    return robj->toReal ();
  }

  // get a real value from an index (either from an integer)

  t_real Vector::getireal (const long index) const {
    Object*   obj = get (index);
    Integer* iobj = dynamic_cast <Integer*> (obj);
    if (iobj != nilp) return iobj->toInteger ();
    Real* robj = dynamic_cast <Real*> (obj);
    if (robj != nilp) return robj->toReal ();
    // not found
    throw Exception ("type-error", "looking for real but got", 
		     Object::repr (obj));
  }

  // get a boolean value from an index

  bool Vector::getbool (const long index) const {
    Object*   obj = get (index);
    Boolean* bobj = dynamic_cast <Boolean*> (obj);
    if (bobj == nilp) 
      throw Exception ("type-error", "looking for boolean but got",
		       Object::repr (obj));
    return bobj->toBoolean ();
  }

  // get a character value from an index

  char Vector::getchar (const long index) const {
    Object*     obj = get (index);
    Character* cobj = dynamic_cast <Character*> (obj);
    if (cobj == nilp) 
      throw Exception ("type-error", "looking for character but got",
		       Object::repr (obj));
    return cobj->toCharacter ();
  }

  // get a string value from an index

  String Vector::getstring (const long index) const {
    Object*  obj = get (index);
    String* sobj = dynamic_cast <String*> (obj);
    if (sobj == nilp) 
      throw Exception ("type-error", "looking for string but got",
		       Object::repr (obj));
    return *sobj;
  }

  // generate a vector of arguments

  Vector* Vector::eval (Interp* interp, Nameset* nset, Cons* args) {
    long len = 0;
    if ((args == nilp) || ((len = args->length ()) == 0)) return nilp;
    Vector* result = new Vector (len);
  
    // loop in the cons cell and accumulate arguments
    try {
      while (args != nilp) {
	Object* car = args->getcar ();
	if (car == nilp) 
	  result->append ((Object*) nilp);
	else
	  result->append (car->eval (interp,nset));
	args = args->getcdr ();
      }
    } catch (...) {
      delete result;
      throw;
    }
    return result;
  }

  // create a new vector in a generic way

  Object* Vector::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Vector;
    return new Vector (*argv);
  }

  // evaluate this vector with a member name

  Object* Vector::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply a vector method by name

  Object* Vector::apply (Interp* interp, Nameset* nset, const String& name,
			 Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "length") && (argc == 0)) {
      delete argv;
      return new Integer (length ());
    }
    if ((name == "get-iterator") && (argc == 0)) {
      delete argv;
      return makeit ();
    }

    // dispatch 1 argument
    if ((name == "get") && (argc == 1)) {
      Object* result = get (argv->getint (0));
      delete argv;
      return result;
    }
    if ((name == "append") && (argc == 1)) {
      Object* result = argv->get (0);
      append (result);
      delete argv;
      return result;
    }

    // dispatch 2 arguments
    if ((name == "set") && (argc == 2)) {
      t_long val = argv->getint (0);
      Object* result = argv->get (1);
      set (val, result);
      delete argv;
      return result;
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }

  // -------------------------------------------------------------------------
  // - vector iterator class section                                         -
  // -------------------------------------------------------------------------

  // create a new vector iterator

  VectorIterator::VectorIterator (Vector* vec) {
    p_vector = vec;
    Object::iref (vec);
    begin ();
  }

  // destroy this vector iterator

  VectorIterator::~VectorIterator (void) {
    Object::dref (p_vector);
  }

  // return the class name

  String VectorIterator::repr (void) const {
    return "VectorIterator";
  }

  // reset the iterator to the begining

  void VectorIterator::begin (void) {
    d_index = 0;
  }

  // reset the iterator to the end

  void VectorIterator::end (void) {
    long len = p_vector->d_length;
    d_index  = (len == 0) ? 0 : len - 1;
  }

  // go to the next object

  void VectorIterator::next (void) {
    if (++d_index >= p_vector->d_length) 
      d_index = p_vector->d_length;
  }

  // go to the previous object

  void VectorIterator::prev (void) {
    if (--d_index < 0) d_index = 0;
  }

  // get the object at the current position

  Object* VectorIterator::getobj (void) {
    if (d_index >= p_vector->d_length) return nilp;
    return p_vector->get (d_index);
  }

  // return true if the iterator is at the end

  bool VectorIterator::isend (void) {
    if (d_index >= p_vector->d_length) return true;
    return false;
  }
}
