// ---------------------------------------------------------------------------
// - Integer.hpp                                                             -
// - standard object library - integer class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_INTEGER_HPP
#define ALEPH_INTEGER_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

#ifndef  ALEPH_VECTOR_HPP
#include "Vector.hpp"
#endif

namespace aleph {

  /// The Integer class is the object version of the native integer
  /// operation for the aleph subsystem. All operations are made with at least
  /// a 64 bits representation (i.e long long). Standard constructors and
  /// operators are provided. The Integer class is also a literal class, so
  /// the toLiteral and repr method are implemented virtually.
  /// @author amaury darsch

  class Integer : public Literal {
  private:
    /// the integer representation
    t_long d_value;

  public:
    /// create a new default integer
    Integer (void);

    /// create a new integer from a long long integer
    /// @param value the value to create
    Integer (const t_long value);

    /// create a new integer from a string
    /// @param value the value to convert
    Integer (const String& value);

    /// copy constructor for this integer
    /// @param that the integer class to copy
    Integer (const Integer& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this integer
    String toLiteral (void) const;

    /// @return a string representation of this integer
    String toString (void) const;

    /// @return a native integer representation
    t_long toInteger (void) const;

    /// assign an integer with a native value
    /// @param value the value to assign
    Integer& operator = (const long value);

    /// Assign an integer with a native value
    /// @param value the value to assign
    Integer& operator = (const Integer& value);

    /// compare this integer with a native value
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const long value) const;

    /// compare this integer with a native value
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const long value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are less
    bool operator < (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are less or equal
    bool operator <= (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are greater
    bool operator > (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are greater or equal
    bool operator >= (const Integer& value) const;
    
    /// add two integers together
    /// @param x the first argument to add
    /// @param y the second argument to add
    /// @return a new integer sum of the previous one
    friend Integer operator + (const Integer& x, const Integer& y);

    /// add an integer to this one
    /// @param x the argument to add
    /// @return this added integer
    Integer& operator += (const Integer& x);

    /// substract an integer to this one
    /// @param x the argument to substract
    /// @return this added integer
    Integer& operator -= (const Integer& x);

    /// substract two integers together
    /// @param x the first argument to substract
    /// @param y the second argument to substract
    /// @return a new integer difference of the previous one
    friend Integer operator - (const Integer& x, const Integer& y);

    /// compute the opposite of the integer
    /// @param x the integer to oppose
    /// @return a new integer opposite of the argument
    friend Integer operator - (const Integer& x);

    /// multiply two integers together
    /// @param x the first argument to multiply
    /// @param y the second argument to multiply
    /// @return a new integer product of the previous one
    friend Integer operator * (const Integer& x, const Integer& y);

    /// multiply an integer with this one
    /// @param x the argument to multiply
    /// @return this multiplied integer
    Integer& operator *= (const Integer& x);

    /// divide two integers together
    /// @param x the numertaor
    /// @param y the denumerator
    /// @return the division of the arguments  
    friend Integer operator / (const Integer& x, const Integer& y);

    /// @return the absolute value of this number
    Integer abs (void) const;

    /// @return the remainder of this value with the argument
    Integer mod (const Integer& x) const;

    /// operate this integer with another object
    /// @param type the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);

    /// generate a new integer
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// set an object to this integer
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate a integer member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this integer with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the real class a friend
    friend class Real;
  };
}

#endif
