// ---------------------------------------------------------------------------
// - InputTerm.cpp                                                           -
// - standard object library - terminal input stream class implementation    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cio.hxx"
#include "cterm.hxx"
#include "cstring.hxx"
#include "Method.hpp"
#include "Boolean.hpp"
#include "InputTerm.hpp"
#include "Exception.hpp"

namespace aleph {

  // this function computes the maximum size of the temporary buffer we must
  // allocate when reading wide characters and special functions
  static long get_mtlen (char** tinfo) {
    if (!tinfo) return 0;
    long max = 0;
    for (long i = 0; i < ITERM_MAX; i++) {
      long len = c_strlen (tinfo[i]);
      max = (max < len) ? len : max;
    }
    return max;
  }
  
  // this function check if we have a match between the buffer and the term
  // info sequence. it returns the number of matches
  static long match_tinfo (char** tinfo, char* buffer, long index, 
			   long* match) {
    if (!tinfo) return 0;
    long count = 0;
    for (long i = 0; i < ITERM_MAX; i++) {
      if (c_strncmp (tinfo[i], buffer, index) == true) {
	count++;
	*match = i;
      }
    }
    return count;
  }
  
  // Create a new default input stream. By default the input stream is
  // mapped to the default standard input
  
  InputTerm::InputTerm (void)  {
    d_sid    = c_stdin ();
    d_eof    = false;
    p_attr   = nilp;
    p_tinfo  = c_rtinfo  (true);
    d_mtlen  = get_mtlen (p_tinfo);
  }
  
  // destroy this terminal class
  
  InputTerm::~InputTerm (void) {
    c_stattr (d_sid, p_attr);
    c_ftattr (p_attr);
    for (int i = 0; i < ITERM_MAX; i++)
      delete [] p_tinfo[i];
    delete [] p_tinfo;
  }

  // return the class name

  String InputTerm::repr (void) const {
    return "InputTerm";
  }

  // read the next available character

  char InputTerm::read (void) {  
    // check for the pushback buffer
    if (d_buffer.length () != 0)
      return d_buffer.read ();
    
    // check if we are at the end of file
    if (d_eof == true) return eofc;
    
    // read the next character on the stream
    char byte  = '\0';
    long count = 0;
    if ((count = c_read (d_sid, &byte,1)) < 0) 
      throw Exception ("read-error", "interrupt during read operation");
    
    // check for eof
    if ((count == 0) || (byte == eofc)) {
      d_eof = true;
      return eofc;
    }
    return byte;
  }

  // return a wide character 

  t_word InputTerm::wread (void) {
    long match = 0;
    long count = 0;
    long index = 0;
    char buffer[d_mtlen + 1];
    
    // read the first character
    buffer[index]   = read ();
    buffer[index+1] = '\0';
    
    // try to match it
    if ((count = match_tinfo (p_tinfo, buffer, 1, &match)) == 0) 
      return (0x00ff & buffer[0]);
    
    // may be we have a match
    if (count == 1) return (0x1000 + (t_word) match);
    
    // now we have a match but it is not complete - let's continue
    // we install a timeout for the read - if we timeout we pushback the buffer
    // sequence if not we continue to match
    
    while (++index <= d_mtlen) {
      count = 0;
      if (c_select (d_sid,1,500) == false) break;
      buffer[index]   = read ();
      buffer[index+1] = '\0';
      count = match_tinfo (p_tinfo, buffer, index + 1, &match);
      if ((count == 0) || ((count == 1) && c_strcmp (p_tinfo[match],buffer))) 
	break;
    }
    
    // at this stage, we either have a match, or we timed-out. If we timed-out
    // count is null and index indicates the amount of characters to pushback
    // if count is 1, this is the match, else we pushback the buffer and return
    // the first character
    if (count != 1) {
      pushback (buffer);
      return (0x00ff & read ());
    }
    return (0x1000 + (t_word) match);
  }
  
  // return true if we are at the eof

  bool InputTerm::iseof (void) const {
    return d_eof;
  }
  
  // return true if the terminal is a tty
  
  bool InputTerm::istty (void) const {
    return c_istty (d_sid);
  }
  
  // save the terminal state

  void InputTerm::save (void) {
    p_attr = c_gtattr (d_sid);
  }
  
  // restore the terminal state
  
  void InputTerm::restore (void) {
    c_stattr (d_sid, p_attr);
  }
  
  // put the terminal in non canonical mode
  
  bool InputTerm::nocanon (void) {
    return c_stcanon (d_sid);
  }

  // create a new input term in a generic way

  Object* InputTerm::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 0) 
      throw Exception ("argument-error", 
		       "invalid arguments with with input term"); 
    return new InputTerm;
  }

  // evaluate this input term with a member name

  Object* InputTerm::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this input term with a method name

  Object* InputTerm::apply (Interp* interp, Nameset* nset, const String& name,
			    Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "valid-p") && (argc == 0)) {
      delete argv;
      return new Boolean (!iseof ());
    }
    if ((name == "eof-p") && (argc == 0)) {
      delete argv;
      return new Boolean (iseof ());
    }

    // input method call
    Object* result = nilp;
    try {
      result =  Input::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }
}
