// ---------------------------------------------------------------------------
// - HashTable.cpp                                                           -
// - standard object library - hash table class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "HashTable.hpp"
#include "Exception.hpp"

namespace aleph {
  
  // the hash table bucket
  struct s_bucket {
    // the object name
    String d_name;
    // the object 
    Object* p_object;
    // next record in the list
    s_bucket* p_next;
    // simple constructor
    s_bucket (void) {
      p_object = nilp;
      p_next   = nilp;
    }
    // simple destructor
    ~s_bucket (void) {
      Object::dref (p_object);
      delete p_next;
    }
  };
  
  // find a bucket by name given its root bucket
  static inline s_bucket* getbucket (s_bucket* bucket, const String& name) {
    // simple check as fast as we can
    if (bucket == nilp) return nilp;
    // loop until we have a match
    while (bucket != nilp) {
      if (bucket->d_name == name) return bucket;
      bucket = bucket->p_next;
    }
    // no bucket found
    return nilp;
  }
  
  // extract a bucket by name given its root bucket . This procedure remove the
  // bucket if it is found and maintain the link list.
  static inline s_bucket* rmbucket (s_bucket** root, const String& name) {
    s_bucket* bucket = *root;
    // simple check as fast as we can
    if (bucket == nilp) return nilp;
    // first case for the root bucket
    if (bucket->d_name == name) {
      *root = bucket->p_next;
      bucket->p_next = nilp;
      return bucket;
    }
    // loop until we have a match
    while (bucket->p_next != nilp) {
      if (bucket->p_next->d_name == name) {
	s_bucket* result = bucket->p_next;
	bucket->p_next = result->p_next;
	result->p_next = nilp;
	return result;
      }
      bucket = bucket->p_next;
    } 
    // no node found
    return nilp;
  }
  
  // Create a new hash table with a default size of 128 elements
  
  HashTable::HashTable (void) {
    // build the array
    d_size   = 64;
    d_thrs   = 44;
    d_count  = 0;
    p_table  = new (s_bucket*)[d_size];
    
    // initialize the table with null pointers
    for (long i = 0; i < d_size; i++)
      p_table[i] = nilp;
  }
  
  // Create a new hash table with a predefined size
  
  HashTable::HashTable (const long size) {
    // build the array - threshold at 70%
    d_size   = size;
    d_thrs   = (size * 7) / 10;
    d_count  = 0;
    p_table  = new (s_bucket*)[d_size];
    
    // initialize the table with null pointers
    for (long i = 0; i < d_size; i++)
      p_table[i] = nilp;
  }
  
  // delete this hash table but not the objects, norr the parent
  
  HashTable::~HashTable (void) {
    if (p_table != nilp) {
      for (long i = 0; i < d_size; i++) delete p_table[i];
      delete [] p_table;
    }
  }

  // return the class name

  String HashTable::repr (void) const {
    return "HashTable";
  }

  // set or create an object in this table
  
  void HashTable::add (const String& name, Object* object) {
    // protect the object
    Object::iref (object);
    // compute the hash value
    long hid = name.hashid () % d_size;
    // look for existing symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) {
      Object::dref (bucket->p_object);
      bucket->p_object = object;
      return;
    }
    // the bucket does not exist, create it 
    bucket           = new s_bucket;
    bucket->d_name   = name;
    bucket->p_object = object;
    bucket->p_next   = p_table[hid];
    p_table[hid]     = bucket;
    if (++d_count > d_thrs) resize (d_size * 2);
  }
  
  // get an object by name. If the name is not found, nilp is returned
  
  Object* HashTable::get (const String& name) const {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) return bucket->p_object;;
    return nilp;
  }

  // get an object by name. If the name is not found an exception is raised

  Object* HashTable::lookup (const String& name) const {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) return bucket->p_object;;
    throw Exception ("name-error", "name not found", name);
  }
  
  // return true if a name exists in this table

  bool HashTable::exists (const String& name) const {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) return true;
    return false;
  }
  
  // remove an object by name. 
  
  void HashTable::remove (const String& name) {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // extract the bucket
    s_bucket* bucket = rmbucket (&p_table[hid],name);
    delete bucket;
    d_count--;
  }
  
  // return a vector of names defined in this hash table
  
  Strvec HashTable::names (void) const {
    Strvec result;
    
    // loop inside the primary table
    for (long i = 0; i < d_size; i++) {
      s_bucket* bucket = p_table[i];
      if (bucket == nilp) continue;
      do {
	result.add (bucket->d_name);
      } while ((bucket = bucket->p_next) != nilp);
    }
    return result;
  }

  // resize the hash table by creating a new one
  
  void HashTable::resize (const long size) {
    // check for the size
    if (size < d_size) return;
    
    // get a vector of names
    Strvec data = this->names ();
    long len  = data.length ();
    if (len == 0) return;
    
    // create a new hash table with a double size
    HashTable* table = new HashTable (size);
    for (long i = 0; i < len; i++) {
      String name = data.get (i);
      table->add (name,this->get(name));
    }
    
    // clean the current hashtable
    for (long i = 0; i < d_size; i++) delete p_table[i];
    delete [] p_table;
    
    // restore from new to this one
    d_size  = table->d_size;
    d_thrs  = (d_size * 7) / 10;
    d_count = table->d_count;
    p_table = table->p_table;
    
    // reset the temporary hash table and delete it
    table->d_size  = 0;
    table->d_count = 0;
    table->p_table = nilp;
    delete table;
  }
}
