// ---------------------------------------------------------------------------
// - Predasio.cpp                                                            -
// - aleph:sio library - predicates implementation                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "System.hpp"
#include "Boolean.hpp"
#include "Terminal.hpp"
#include "Predasio.hpp"
#include "Directory.hpp"
#include "InputFile.hpp"
#include "Exception.hpp"
#include "OutputFile.hpp"

namespace aleph {

  // this procedure checks that we have one argument only and returns
  // the evaluated object
  static inline Object* get_obj (Interp* interp, Nameset* nset, Cons* args,
                                 const String& pname) {
    Object* car = nilp;
    if ((args == nilp) || (args->length () != 1))
      throw Exception ("argument-error", "illegal arguments with predicate",
                       pname);
    car = args->getcar ();
    return (car == nilp) ? nilp : car->eval (interp,nset);
  }

  // odirp: directory object predicate

  Object* asio_odirp  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "directory-p");
    if (dynamic_cast <Directory*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // inputp: input stream predicate

  Object* asio_inputp  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "input-p");
    if (dynamic_cast <Input*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // ifilep: input file predicate

  Object* asio_ifilep  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "input-file-p");
    if (dynamic_cast <InputFile*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // ofilep: output file predicate

  Object* asio_ofilep  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "output-file-p");
    if (dynamic_cast <OutputFile*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // outputp: output stream predicate

  Object* asio_outputp  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "output-p");
    if (dynamic_cast <Output*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // itermp: input term predicate

  Object* asio_itermp  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "input-term-p");
    if (dynamic_cast <InputTerm*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // otermp: output term predicate

  Object* asio_otermp  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "output-term-p");
    if (dynamic_cast <OutputTerm*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  } 

  // termp: terminal predicate

  Object* asio_termp  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "terminal-p");
    if (dynamic_cast <Terminal*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // filep: check for a regular file

  Object* asio_filep  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj  = get_obj (interp, nset, args, "file-p");
    String* sval = dynamic_cast <String*> (obj);
    if (sval == nilp) 
      throw Exception ("argument-error", "invalid object with file-p",
		       Object::repr (obj));
    return new Boolean (System::isfile (*sval));
  }

  // filep: check for a directory

  Object* asio_dirp  (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj  = get_obj (interp, nset, args, "dir-p");
    String* sval = dynamic_cast <String*> (obj);
    if (sval == nilp) 
      throw Exception ("argument-error", "invalid object with dir-p",
		       Object::repr (obj));
    return new Boolean (System::isdir (*sval));
  }  
}
