// ---------------------------------------------------------------------------
// - Class.hpp                                                               -
// - aleph engine - class class definition                                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_CLASS_HPP
#define  ALEPH_CLASS_HPP

#ifndef  ALEPH_LOCALSET_HPP
#include "Localset.hpp"
#endif

namespace aleph {

  /// The Class class is the class used to model aleph class object. The aleph
  /// model is based on a single inheritance mechanism. A class binds only
  /// a local set - which somehow is equivalent to c++ static data member.
  /// When an instance of that class is created, it is the instance initial
  /// method which is responsible to define the local data members.
  /// @author amaury darsch

  class Class : public Object {
  private:
    /// the class local set
    Localset* p_cset;
    /// the list of data members
    Strvec d_mdata;

  public:
    /// create a new class 
    Class (void);

    /// destroy this class class
    ~Class (void);

    /// @return the class name
    String repr (void) const;

    /// add a new data member definition
    /// @param name the name to add
    void madd (const String& name);

    /// @return the vector of data member names
    inline const Strvec& getmdata (void) const {
      return d_mdata;
    }

    /// @return an object in the class localset but do not evaluate
    inline Object* find (const String& name) const {
      return p_cset->find (name);
    }

    /// operate this class with another object
    /// @param type the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);

    /// set an object as a const static object by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to define as const
    /// @param object the object to set
    Object* cdef (Interp* interp, Nameset* nset, const String& name, 
		  Object* object);

    /// set an object as a static object by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to define as const
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, const String& name, 
		  Object* object);

    /// evaluate a static member by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the member name to evaluate in this object
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// create a new instance with some arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset 
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, Cons* args);

    /// apply this class by name with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the copy constructor private
    Class (const Class&);
    // make the assignment operator private
    Class& operator = (const Class&);
  };
}

#endif
