/*  XNECVIEW - a program for visualizing NEC2 structure and gain files
 *
 *  Copyright (C) 1998-1999, P.T. de Boer -- pa3fwm@amsat.org
 *
 *  Distributed on the conditions of the GPL: see the files README and
 *  COPYING, which accompany this source file.
 *
 *  This module contains all X-windows related stuff.
 */


#include  <X11/StringDefs.h> 
#include  <X11/Intrinsic.h> 
#include  <X11/Core.h> 
#include  <X11/Xaw/Box.h> 
#include  <X11/Xlib.h> 
#include  <X11/keysym.h> 
#include  <X11/Shell.h> 
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "xnecview.h"

#include "icon.xbm"

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/AsciiText.h>


Display *display;
Drawable window;
Widget drawing;
int scrnr;
GC gc;

Pixmap drawbuf;      /* picture buffer used for double buffering code */
Widget msgwidget;    /* text widget in top right corner, used for several messages */
XFontStruct *fontinfo;  /* info about the font used */
int fontheight;

int redraw=1;        /* flag which signifies need for redrawing */
int dragging=0;      /* flag to indicate that user is dragging the structure */
int quick;           /* flag to indicate that drawing should be rough but quick */


/* color values for several items on screen: */
XColor c_bg;
XColor c_axis;
XColor c_wire;
XColor c_surf;
XColor c_back;
XColor c_gain;
XColor c_scale;
XColor c_exci;
XColor c_load;


/* some macros taken from xfig-3.1.4, slightly modified: */
#define FirstArg(name, val) \
	{ XtSetArg(Args[0], (name), (val)); ArgCount=1;}
#define NextArg(name, val) \
	{ XtSetArg(Args[ArgCount], (name), (val)); ArgCount++;}
#define GetValues(n)	XtGetValues(n, Args, ArgCount)
#define SetValues(n)	XtSetValues(n, Args, ArgCount)
int ArgCount;
Arg Args[10];

/* -------------------- X windows stuff: drawing ---------------------------------------- */

void X_SetLineAttributes(unsigned int a,int b,int c,int d)
{
   XSetLineAttributes(display, gc, a,b,c,d);
}

void X_DrawLine(double a,double b,double c,double d)
{
   XDrawLine(display,drawbuf,gc,(int)(a+0.5),(int)(b+0.5),(int)(c+0.5),(int)(d+0.5));
}

void X_SetForeground(XColor *xc)
{
   XSetForeground(display,gc,xc->pixel);
}

void X_ClearWindow()
{
   X_SetForeground(&c_bg);
   XFillRectangle(display,drawbuf,gc, 0,0, winsizex, winsizey);
}

void X_DrawStringLL(double a,double b,char *s)    /* draw string at coordinates specifying lower left corner of text */
{
   XDrawString(display,drawbuf,gc,(int)(a+0.5),(int)(b+0.5)-fontinfo->descent,s,strlen(s));
}

void X_DrawStringUL(double a,double b,char *s)    /* draw string at coordinates specifying upper left corner of text */
{
   XDrawString(display,drawbuf,gc,(int)(a+0.5),(int)(b+0.5)+fontinfo->ascent,s,strlen(s));
}

Outdev outX={ X_SetLineAttributes, X_DrawLine, X_SetForeground, X_ClearWindow, X_DrawStringLL, X_DrawStringUL };



/* -------------------- other X windows stuff (except for drawing) ---------------------- */

char GSnumbers[][32]={"0 -1 -3 -6 -10 dB","0 -3 -10 -20 -30 dB","0 -10 -20 -30 dB"};

void upd_msg(void)
{
   char s[32];
   if (scaleplot && gainplot!=GPnone) {
      FirstArg(XtNlabel,GSnumbers[gainscale]);
   } else {
      sprintf(s,"phi=%4i theta=%3i",(int)(phi+0.5),(int)(theta+0.5)); 
      FirstArg(XtNlabel,s);
   }
   SetValues(msgwidget);
}



void redisplay_event(Widget w,XtPointer client,XExposeEvent *ev)
{
   if(ev->count != 0) return;
   redraw=1;
}


void resize_event(Widget w,XtPointer client,XConfigureEvent *ev)
{
   if(ev->type != ConfigureNotify) return;
   winsizex=ev->width;
   winsizey=ev->height;
   if (!unbuffered) {
      XFreePixmap(display,drawbuf);
      drawbuf=XCreatePixmap(display,window,winsizex,winsizey, DefaultDepth(display,scrnr));
   }
   calcproj();
   redraw=1;
}


int lastx=0;
int lasty=0;
int origx=0;
int origy=0;

void buttonpress_event(Widget w,XtPointer client,XButtonEvent *ev)
{
   origx=lastx=ev->x;
   origy=lasty=ev->y;
}


void buttonrelease_event(Widget w,XtPointer client,XButtonEvent *ev)
{
   if (dragging) {
      dragging=0;
      if (quick) {
         quick=0;
         redraw=1;
      }
      return;
   }

   if (ev->button==Button1) {
      zoom*=1.4142;
      trx-=(double)(ev->x-winsizex/2)/zoom/winsize;
      try-=(double)(ev->y-winsizey/2)/zoom/winsize;
      calcproj();
      redraw=1;
   }

   if (ev->button==Button2) { 
      zoom=ini_zoom;
      phi=ini_phi;
      theta=ini_theta;
      scaleplot=0;
      trx=ini_trx;
      try=ini_try;
      upd_msg();
      calcproj();
      redraw=1;
   }

   if (ev->button==Button3) {
      zoom/=1.4142;
      calcproj();
      redraw=1;
   }
}


void motion_event(Widget w,XtPointer client,XMotionEvent *ev)
{
   int dx,dy; 
   int pos_x,pos_y,root_x,root_y; /* for return from XQueryPointer */
   unsigned int keys_buttons;
   Window root,child;

   /* Get the pointer position*/
   if (!XQueryPointer(display,ev->window,&root,&child,
			   &root_x,&root_y,&pos_x,&pos_y,
			   &keys_buttons)) return;  /* Pointer on another screen */

   /* Don't allow motion which is too small*/
   if (abs(pos_x-origx)+abs(pos_y-origy)<2) return;

   /* Calculate motion vector */
   dx=pos_x-lastx;
   dy=pos_y-lasty;

   if (ev->state & (Button1Mask|Button2Mask)) {
      theta-=180.*(double)dy/(double)winsizey;
      phi-=180.*(double)dx/(double)winsizex;
      scaleplot=0;
      quick=(ev->state&Button2Mask);   /* Button 1 full drag, Button 2 quick drag */
   } else if (ev->state&Button3Mask) {
      trx+=(double)(dx)/zoom/winsize;
      try+=(double)(dy)/zoom/winsize;
      quick=1;
   } else return;

   upd_msg();
   calcproj();
   redraw=1;

   dragging=1;

   lastx=pos_x;
   lasty=pos_y;
}


void keypress_event(Widget w,XtPointer client,XKeyPressedEvent *ev)
{
   KeySym key;

   key=XLookupKeysym(ev,0);
   if (key==XK_Q || key==XK_q) {
      XFreeGC(display, gc);
      if (!unbuffered) {
      	XFreePixmap(display,drawbuf);
      }
      exit(0);
   }
   if (key==XK_R || key==XK_r || key==XK_period) {
      reread();
      redraw=1;
   }
}


void show_drawbuf()
{
   if (unbuffered) return;
   XCopyArea(display, drawbuf, window, gc, 0,0,winsizex,winsizey,0,0);
}


void getcolor(char *name,XColor *xc)
{
   XColor xe;
   Colormap cm;

   if (DefaultDepth(display,scrnr)==1) {     /* on 1bpp displays, choose black for everything except the background */
      if (strcmp(name,C_BG)) name="black";
      else name="white";
   }

   cm=DefaultColormap(display,scrnr);
   if (!XAllocNamedColor(display,cm,name,xc,&xe)) {
      /* if allocation failed, use black */
      xc->red=0;
      xc->blue=0;
      xc->green=0;
      xc->pixel=BlackPixel(display,scrnr);
   }
}

void cmd_quit(Widget w,XtPointer client,XButtonEvent *ev)
{
   XFreeGC(display, gc);
   exit(0);
}

void cmd_reload(Widget w,XtPointer client,XButtonEvent *ev)
{
   reread();
   redraw=1;
}

void cmd_X(Widget w,XtPointer client,XButtonEvent *ev)
{
   phi=0; theta=90; redraw=1;
   scaleplot=1;
   calcproj();
   upd_msg();
}

void cmd_Y(Widget w,XtPointer client,XButtonEvent *ev)
{
   phi=90; theta=90; redraw=1;
   scaleplot=1;
   calcproj();
   upd_msg();
}

void cmd_Z(Widget w,XtPointer client,XButtonEvent *ev)
{
   phi=0; theta=0; redraw=1;
   scaleplot=1;
   calcproj();
   upd_msg();
}

char GSnames[][8]={"lin","arrl","log"};

void cmd_gainscale(Widget w,XtPointer client,XButtonEvent *ev)
{
   if (++gainscale >=numGS) gainscale=0;
   FirstArg(XtNlabel, GSnames[gainscale]);
   SetValues(w);
   if (numphi>0) { extr/=GAINSIZE; process_nec_output(); }
   upd_msg();
   redraw=1;
}

char GPnames[][8]={"none","slice","frame"};

void cmd_gainplot(Widget w,XtPointer client,XButtonEvent *ev)
{
   if (++gainplot >=numGP) gainplot=0;
   FirstArg(XtNlabel, GPnames[gainplot]);
   SetValues(w);
   upd_msg();
   redraw=1;
}

char SPnames[][8]={"none","struct","+tags"};

void cmd_structplot(Widget w,XtPointer client,XButtonEvent *ev)
{
   if (++structplot >=numSP) structplot=0;
   FirstArg(XtNlabel, SPnames[structplot]);
   SetValues(w);
   redraw=1;
}


Widget export_popup;
Widget export_filename;

void cmd_export_cancel()
{
   XtPopdown(export_popup);
}

void cmd_export_export()
{
   char *s;
   Dimension width;
   FirstArg(XtNwidth,&width);
   GetValues(export_filename);
   s=malloc(width+4);
   if (!s) return;
   FirstArg(XtNstring,&s);
   GetValues(export_filename);
   if (write_postscript(s)) fprintf(stderr,"Error writing postscript\n");
   free(s);
   XtPopdown(export_popup);
}

String asciitext_translations = "<Key>Return: exp()\nCtrl<Key>J: exp()\nCtrl<Key>M: exp()\n";

void cmd_export(Widget w,XtPointer client,XButtonEvent *ev)
{
   Widget wi,pa1,pa2;
   Position xp,yp;
   char *s,*p;

   XtTranslateCoords(w,(Position)0,(Position)0,&xp,&yp);
   FirstArg(XtNx,xp-50);
   NextArg(XtNy,yp+10);
   export_popup=XtCreatePopupShell("xnecview: export",transientShellWidgetClass,w,Args,ArgCount);

   pa1=XtCreateManagedWidget("p1",panedWidgetClass,export_popup,NULL,0);

   FirstArg(XtNshowGrip, False);
   NextArg(XtNeditType, XawtextEdit);
   NextArg(XtNscrollVertical, XawtextScrollNever);
   s=malloc(strlen(inputfilename)+8);
   strcpy(s,inputfilename);
   p=strrchr(s,'.');
   if (!p) p=s+strlen(s);
   strcpy(p,".eps");
   NextArg(XtNstring, s);
   export_filename=XtCreateManagedWidget("",asciiTextWidgetClass,pa1,Args,ArgCount);
   free(s);
   XtOverrideTranslations(export_filename,XtParseTranslationTable(asciitext_translations));

   FirstArg(XtNshowGrip, False);
   NextArg(XtNorientation, XtorientHorizontal);
   pa2=XtCreateManagedWidget("p2",panedWidgetClass,pa1,Args,ArgCount);
   FirstArg(XtNshowGrip, False);
   wi=XtCreateManagedWidget("Export",commandWidgetClass,pa2,Args,ArgCount);
   XtAddCallback(wi, XtNcallback, (XtCallbackProc)cmd_export_export,NULL);
   FirstArg(XtNshowGrip, False);
   wi=XtCreateManagedWidget("Cancel",commandWidgetClass,pa2,Args,ArgCount);
   XtAddCallback(wi, XtNcallback, (XtCallbackProc)cmd_export_cancel,NULL);

   XtPopup(export_popup,XtGrabNonexclusive);
}


/* -------------------- X initialisation ------------------------------------------------ */

XtActionsRec actionTable[] = { {"exp", (XtActionProc)cmd_export_export} };

void initX(int *argcp,char **argv)
{
   Widget toplevel,pane1,pane2,w,wgs;
   Pixmap icon;

   winsize=winsizex=winsizey=ini_winsize;

   toplevel = XtInitialize(argv[0],"xnecview", NULL, 0, argcp, argv);
   pane1 = XtCreateManagedWidget("pane1",panedWidgetClass,toplevel,NULL,0);
   XtAppAddActions(XtWidgetToApplicationContext(toplevel), actionTable, XtNumber(actionTable));

   FirstArg(XtNshowGrip, False);
   NextArg(XtNorientation, XtorientHorizontal);
   pane2 = XtCreateManagedWidget("pane2",panedWidgetClass,pane1,Args,ArgCount);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget("quit",commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_quit, NULL);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget("reload",commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_reload, NULL);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget("export",commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_export, NULL);

   FirstArg(XtNshowGrip, False);
   NextArg(XtNinternalWidth, 1);
   w = XtCreateManagedWidget("",labelWidgetClass,pane2,Args,ArgCount);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget(SPnames[structplot],commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_structplot, NULL);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget(GPnames[gainplot],commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_gainplot, NULL);

   FirstArg(XtNshowGrip, False);
   wgs = XtCreateManagedWidget("arrl",commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(wgs, ButtonPressMask, FALSE, (XtEventHandler)cmd_gainscale, NULL);

   FirstArg(XtNshowGrip, False);
   NextArg(XtNinternalWidth, 1);
   w = XtCreateManagedWidget("",labelWidgetClass,pane2,Args,ArgCount);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget("X",commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_X, NULL);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget("Y",commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_Y, NULL);

   FirstArg(XtNshowGrip, False);
   w = XtCreateManagedWidget("Z",commandWidgetClass,pane2,Args,ArgCount);
   XtAddEventHandler(w, ButtonPressMask, FALSE, (XtEventHandler)cmd_Z, NULL);

   FirstArg(XtNshowGrip, False);
   msgwidget = XtCreateManagedWidget("phi=360  theta=180",labelWidgetClass,pane2,Args,ArgCount);


   drawing = XtCreateManagedWidget("xnecview",coreWidgetClass, pane1, NULL, 0);

   FirstArg(XtNheight, winsizey);
   NextArg(XtNwidth, winsizex);
   XtSetValues(drawing, Args, ArgCount);

   XtAddEventHandler(drawing, ExposureMask, FALSE, (XtEventHandler)redisplay_event, NULL);
   XtAddEventHandler(drawing, StructureNotifyMask, FALSE, (XtEventHandler)resize_event, NULL);
   XtAddEventHandler(drawing, ButtonPressMask, FALSE, (XtEventHandler)buttonpress_event, NULL);
   XtAddEventHandler(drawing, ButtonReleaseMask, FALSE, (XtEventHandler)buttonrelease_event, NULL);
   XtAddEventHandler(drawing, PointerMotionHintMask |ButtonMotionMask, FALSE, (XtEventHandler)motion_event, NULL);
   XtAddEventHandler(drawing, KeyPressMask, FALSE, (XtEventHandler)keypress_event, NULL);

   XtRealizeWidget(toplevel);

   display = XtDisplay(drawing);
   window = XtWindow(drawing);
   scrnr = DefaultScreen(display);
   gc = XCreateGC(display, window, 0, NULL);

   icon=XCreateBitmapFromData(display,window,icon_bits,icon_width,icon_height);
   FirstArg(XtNiconPixmap,icon);
   SetValues(toplevel);

   fontinfo=XLoadQueryFont(display,XFONT);
   if (fontinfo==NULL) fontinfo=XLoadQueryFont(display,"fixed");
   if (fontinfo==NULL) { fprintf(stderr,"Can't load fonts '%s' or 'fixed'.\n",XFONT); exit(1); }
   fontheight=fontinfo->ascent+fontinfo->descent;
   XSetFont(display,gc,fontinfo->fid);

   if (!unbuffered) drawbuf=XCreatePixmap(display,window,winsizex,winsizey, DefaultDepth(display,scrnr));
   else drawbuf=window;
   
   XSetBackground(display, gc, WhitePixel(display,scrnr));

   getcolor("white",&c_bg);
   getcolor(C_AXIS,&c_axis);
   getcolor(C_WIRE,&c_wire);
   getcolor(C_SURF,&c_surf);
   getcolor(C_BACK,&c_back);
   getcolor(C_GAIN,&c_gain);
   getcolor(C_SCALE,&c_scale); 
   getcolor(C_EXCI,&c_exci); 
   getcolor(C_LOAD,&c_load); 
  
   FirstArg(XtNlabel, GSnames[gainscale]);
   SetValues(wgs);
   upd_msg();
   out = &outX;
   XClearWindow(display,window);
}


/* -------------------- main loop (wait for event, process event, redisplay) ------------ */

int Pending(void)
{
   XEvent xe;
   if (!XtPending()) return 0;
   XtNextEvent(&xe);
   XtDispatchEvent(&xe);
   return redraw;
}

void mainX(void)
{
   for (;;) {
      XEvent xe;
      XtNextEvent(&xe);
      XtDispatchEvent(&xe);
      while (redraw && !XtPending()) {
         redraw=0;
         draw_all(1);
         show_drawbuf();
      }
   }
}
