/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <math.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "terraform.h"
#include "TFWindowHandler.h"
#include "FileIO.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"
#include "GuiColormap.h"
#include "GuiDialogPreview.h"
#include "GuiDialogReallyQuit.h"
#include "HeightFieldExport.h"
#include "HeightFieldGenRandom.h"
#include "HeightFieldIO.h"
#include "MenuDefs.h"
#include "TFDialogContourLines.h"
#include "TFDialogCraters.h"
#include "TFDialogErode.h"
#include "TFDialogFill.h"
#include "TFDialogFlowmap.h"
#include "TFDialogGaussianHill.h"
#include "TFDialogFold.h"
#include "TFDialogGenSSynth.h"
#include "TFDialogGenSubdiv.h"
#include "TFDialogLinearScale.h"
#include "TFDialogMerge.h"
#include "TFDialogMirror.h"
#include "TFDialogOptions.h"
#include "TFDialogPPov.h"
#include "TFDialogPrintOptions.h"
#include "TFDialogRender.h"
#include "TFDialogRescale.h"
#include "TFDialogRotate.h"
#include "TFDialogRoughSmooth.h"
#include "TFDialogTerrace.h"
#include "TFDialogTransform.h"
#include "TFGui.h"					// circular dependency
#include "TFOptions.h"
#include "TFPreviewDialog.h"
#include "Timer.h"
#include "MathRandom.h"
#include "agename.h"
#include "strrep.h"

#ifdef HAVE_IMLIB
#include "GuiDialogAboutDissolve.h"
#endif


extern int readhf (HeightField **HF, char *fname, bool haveGUI);

static int		cs_fileselMode;	// flag for file selection pointer


#define LOAD		0
#define SAVE		1
#define BANDSIZE	15


/*
 *  constructor: initialize everything 
 */
TFWindowHandler::TFWindowHandler (FlexArray *wList) 
		: TFWindowBase (wList)
		// default constructor is OK for TFWindow
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ TFWindowHandler\n");

	p_HF = NULL;
	p_HFD = NULL;
	p_fileSelection = NULL;
	p_genSSynthDialog=p_genSubdivDialog=p_placePovDialog=p_renderDialog=NULL;
	p_contLinesDialog=p_erodeDialog=p_flowmapDialog=p_rescaleDialog = NULL;
	p_cratersDialog=p_fillDialog=p_foldDialog=p_gaussHillDialog=
		p_linearScaleDialog=p_mirrorDialog=p_rotateDialog=
		p_roughSmoothDialog=p_terraceDialog=p_transformDialog = NULL;
	d_fastWire = FALSE;
	d_rotate = FALSE;
	d_autoRotate = FALSE;

#ifdef HAVE_OPENGL
	p_openGLWindow = NULL;
#endif
	
	activateMenus (FALSE);
	gdk_flush ();

	//connect_to_method (this->delete_event, this, &TFWindowHandler::delete_event_impl);
}


/*
 *  destructor: clean up 
 */
TFWindowHandler::~TFWindowHandler ()
{
	if (p_HF) 		delete p_HF;
	if (p_HFD) 		delete p_HFD;
	if (p_fileSelection) 	delete p_fileSelection;
	if (p_contLinesDialog)	delete p_contLinesDialog;
	if (p_cratersDialog)	delete p_cratersDialog;
	if (p_erodeDialog) 	delete p_erodeDialog;
	if (p_flowmapDialog) 	delete p_flowmapDialog;
	if (p_genSSynthDialog) 	delete p_genSSynthDialog;
	if (p_genSubdivDialog) 	delete p_genSubdivDialog;
	if (p_placePovDialog) 	delete p_placePovDialog;
	if (p_renderDialog) 	delete p_renderDialog;
	if (p_rescaleDialog) 	delete p_rescaleDialog;
	if (p_fillDialog) 	delete p_fillDialog;
	if (p_foldDialog) 	delete p_foldDialog;
	if (p_linearScaleDialog) delete p_linearScaleDialog;
	if (p_mirrorDialog) 	delete p_mirrorDialog;
	if (p_roughSmoothDialog) delete p_roughSmoothDialog;
	if (p_terraceDialog)	delete p_terraceDialog;
	if (p_transformDialog)	delete p_transformDialog;
	if (p_rotateDialog) 	delete p_rotateDialog;
	
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- TFWindowHandler\n");
}


/*
 *  idleRedrawCallback: loop through TFWindowBase::s_winList and find the desired HF 
 */
gint TFWindowHandler::idleRedrawCallback ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "Idle Redraw ...\n");
	drawHeightField ();
	return 0;
}


/*
 *  waitForChildreCallback: wait on a child process to avoid zombie processes
 */
gint TFWindowHandler::waitForChildrenCallback ()
{
	//printf ("Waiting for children\n");

	int	status;
	pid_t	wpid;

	wpid = waitpid (-1, &status, WNOHANG);
	if (wpid > 0)
		return 0;

	return 1;
}


/*
 *  autoRotateCallback: iterate through 1 step of the rotation
 */
gint TFWindowHandler::autoRotateCallback ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "Auto Rotate ...\n");

	if (p_HFD->rotateFastView (0.0, (M_PI/180)*1, 0.0))
		{
		// !!!!!!!!! set incorrectly selected check box back to FALSE
		d_autoRotate = FALSE;
		}
	else
		drawHeightField ();

	return d_autoRotate;
}


/*
 *  calibrateAutoRotate: figure out an acceptable delay on the fast 
 * 	wireframe rotation. If we just rotate as fast as we can, we'll 
 * 	flood the CPU/XServer ...
 */
gint TFWindowHandler::calibrateAutoRotate (int nSamples)
{
	Timer		timer;
	long		eusec;		// elapsed microseconds
	struct timeval  *etime;		// elapsed time

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "Calibrating AutoRotate ...");

	timer.start ();
	for (int i=0; i<nSamples; i++)
		{
		autoRotateCallback ();
		gdk_flush ();
		}
	timer.stop ();
	etime = timer.getElapsedTime ();
	eusec = etime->tv_usec + etime->tv_sec*1000000;
	eusec /= nSamples;

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW,  " done\n");

	return eusec;
}


/*
 *  setHeightField: set the Height Field we draw 
 */
void TFWindowHandler::setHeightField (HeightField *HF)
{
	int	x, y;
	bool	upd = FALSE;

	SanityCheck::bailout ((!HF), "HF==NULL", "TFWindowHandler::setHeightField");

	if (!p_HF)
		upd = TRUE;
		
	if (p_HF && p_HF!=HF)
		delete p_HF;
	p_HF=HF;

	if (p_HFD)
		delete p_HFD;
	p_HFD=new HeightFieldDraw (p_HF, this);
	p_HFD->setScale (1.0, &x, &y);
	if (upd)
		p_HFD->updateParams ();

	if (this->d_useLinearColormap)
		p_HFD->setColormap (this->p_linearColorMap);
	else
		p_HFD->setColormap (this->p_bandColorMap);

	if (GlobalTrace::isSet(GlobalTrace::TRACE_DEBUG))
		p_HFD->printSettings();

	//p_HFD->draw ();
	d_win.set_title (p_HF->getName());
	connect_to_method (Gtk_Main::timeout(50), this, &TFWindowHandler::idleRedrawCallback);
	activateMenus (TRUE);
}


/*
 *  drawHeightField: draw and then reset fore- and background colors
 */
void TFWindowHandler::drawHeightField ()
{
	SanityCheck::bailout ((!p_HF), "p_HF==NULL", "TFWindowHandler::drawHeightField");
	SanityCheck::bailout ((!p_HFD), "p_HFD==NULL", "TFWindowHandler::drawHeightField");

	p_HFD->draw ();
	this->setColors (&(this->p_linearColorMap->d_black));
}



/*
 *  fileMenuCallback: process a menu choice in the file menu
 */
void TFWindowHandler::fileMenuCallback (string m)
{
	int		rc=0;
	char		*menuitem = const_cast<char*>(m.c_str()),
			*aboutText[] = { _("brought to you by"), 
					"r.gasch@chello.nl", };

	if (!strcmp (menuitem, _(MENU_FILE_NEW_SYNTHESIS)))
		{
		if (!p_genSSynthDialog)
			p_genSSynthDialog = new TFDialogGenSSynth ();
		else
			{
			TFDialogGenSSynth	*TFDGen;

			TFDGen = dynamic_cast<TFDialogGenSSynth*>(p_genSSynthDialog);
			TFDGen->reset ();
			//TFDGen->set_title (_("New Height Field"));
			}
		
		p_genSSynthDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_NEW_SUBDIVIDE)))
		{
		if (!p_genSubdivDialog)
			p_genSubdivDialog = new TFDialogGenSubdiv ();
		else
			{
			TFDialogGenSubdiv 	*TFDGen;

			TFDGen = dynamic_cast<TFDialogGenSubdiv*>(p_genSubdivDialog);
			TFDGen->reset ();
			//TFDGen->set_title ("New Height Field");
			}
		
		p_genSubdivDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_NEW_RANDOM)))
		{
		if (!p_HF)
			p_HF = new HeightField ();

		HeightFieldGenRandom::generate (p_HF, this->width());
		setHeightField (p_HF);
		//p_HFD->draw ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_OPEN)))
		{
		cs_fileselMode=LOAD;			// 0 = load, 1 = save
		fileSelection ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_MERGE)))
		{
		TFDialogMerge 	*mergeDialog = TFGui::getMergeDialog ();
		mergeDialog->rebuildCList ();
		mergeDialog->initHeightFields (this);
		mergeDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_RELOAD)))
		{
		if (p_HF && !p_HF->isSaved())
			{
			if (!TFWindowBase::s_quitDialog)
				TFWindowBase::s_quitDialog = new GuiDialogReallyQuit ("", "");
			else
				TFWindowBase::s_quitDialogConnection.disconnect ();

			TFWindowBase::s_quitDialog->set_title (p_HF->getName());
			TFWindowBase::s_quitDialog->setLabel (_("Unsaved Changes! Reload Anyways?"));
			TFWindowBase::s_quitDialogConnection = connect_to_method (
				TFWindowBase::s_quitDialog->getYesButton()->clicked, 
				this, &TFWindowHandler::reloadCallback);
			TFWindowBase::s_quitDialog->show ();
			}
		else
		if (p_HF)
			reloadCallback ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_SAVE)))
		{
		if (p_HF)
			{
			HeightFieldIO		*HFIO;

			HFIO = new HeightFieldIO (p_HF->getName());
			rc = HFIO->write (p_HF);
			}
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_SAVEAS)))
		{
		if (p_HF)
			{
			cs_fileselMode=SAVE;			// 0 = load, 1 = save
			fileSelection ();
			}
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_CLOSE)))
		{
		if (p_HF && !p_HF->isSaved())
			{
			if (!TFWindowBase::s_quitDialog)
				TFWindowBase::s_quitDialog = new GuiDialogReallyQuit ("", "");
			else
				TFWindowBase::s_quitDialogConnection.disconnect ();

			TFWindowBase::s_quitDialog->set_title (p_HF->getName());
			TFWindowBase::s_quitDialog->setLabel (_("Unsaved Changes! Close Anyways?"));
			TFWindowBase::s_quitDialogConnection = connect_to_method 
				(TFWindowBase::s_quitDialog->getYesButton()->clicked, 
				 this, &TFWindowHandler::closeCallback);
			TFWindowBase::s_quitDialog->show ();
			}
		else
		if (p_HF)
			closeCallback ();
		}
#ifdef HAVE_OPENGL
	else
	if (!strcmp (menuitem, _(MENU_FILE_EXPORT_OPENGL)))
		{
		if (!p_openGLWindow)
			p_openGLWindow = new TFWindowOpenGL (p_HF);
		p_openGLWindow->show ();
		}
#endif
	else
	if (!strcmp (menuitem, _(MENU_FILE_EXPORT_POVRAY)))
		{
		HeightFieldExport	*HFE = new HeightFieldExport (p_HF);
		float			slvl, yscale;

		if (TFOptions::s_fillSea)
			slvl = TFOptions::s_sealevel;
		else
			slvl = 0;

		if (TFOptions::s_POVhalfYscale)
			yscale = TFOptions::s_yscale*.5;
		else
			yscale = TFOptions::s_yscale;

		HFE->renderPOV ();
		delete HFE;
		connect_to_function (Gtk_Main::timeout(1000), 
				&TFWindowHandler::waitForChildrenCallback);
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_ABOUT)))
		{
		if (!TFWindowBase::s_aboutDialog)
			{
#ifdef HAVE_IMLIB
			char	buf[256];
			sprintf (buf, "%s/terraform-logo.jpg", TF_DATADIR);
			FileIO	*fio = new FileIO (buf);
			if (fio->exists())
				TFWindowBase::s_aboutDialog = new GuiDialogAboutDissolve ("About Terraform", buf, aboutText, 2);
			else
				fprintf (stderr, _("Error opening %s\n"), buf);
			delete fio;
#else
			TFWindowBase::s_aboutDialog = new GuiDialogAbout ("About Terraform", NULL, aboutText, 2);
#endif
			if (TFWindowBase::s_aboutDialog)
				TFWindowBase::s_aboutDialog->buildDialogWindow ();
			}

		if (TFWindowBase::s_aboutDialog)
			TFWindowBase::s_aboutDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_OPTIONS)))
		{
		//if (p_HF)
		//	{
			if (!TFWindowBase::s_optionsDialog)
				TFWindowBase::s_optionsDialog = new TFDialogOptions (TFWindowBase::s_winList);
			else
				TFWindowBase::s_optionsDialog->show ();
		//	}
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_PRINT_SETTINGS)))
		{
		if (!this->s_printOptionsDialog)
			this->s_printOptionsDialog = new TFDialogPrintOptions ();
		else
			this->s_printOptionsDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_FILE_QUIT)))
		{
		if (!TFGui::allWindowsSaved())
			{
			if (!TFWindowBase::s_quitDialog)
				TFWindowBase::s_quitDialog = new GuiDialogReallyQuit ("", "");
			else
				TFWindowBase::s_quitDialogConnection.disconnect ();

			TFWindowBase::s_quitDialog->set_title (_("Really Quit Terraform?"));
			TFWindowBase::s_quitDialog->setLabel (_("Unsaved Changes! Exit anyway?"));
			TFWindowBase::s_quitDialogConnection = connect_to_method (
				TFWindowBase::s_quitDialog->getYesButton()->clicked, 
				this, &TFWindowHandler::destroyCallback);
			TFWindowBase::s_quitDialog->show ();
			}
		else
			destroyCallback ();
		}
	else
		SanityCheck::bailout (TRUE, "uncaught <file> menuitem", "TFWindowHandler::fileMenuCallback");

	if (rc)
		fprintf (stderr, _("Warning: last command returned an Error (%d)\n"), rc);
	fflush (stdout);
}


/*
 *  hfMenuCallback: process a menu choice in the height field menu
 */
void TFWindowHandler::hfMenuCallback (string m)
{
	char		*menuitem = const_cast<char*>(m.c_str());

	if (!p_HF)
		return;

	if (!strcmp (menuitem, _(MENU_HF_CRATERS)))
		{
		if (!p_cratersDialog) 
			p_cratersDialog = new TFDialogCraters (p_HF, p_HFD);
		p_cratersDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_FILL)))
		{
		if (!p_fillDialog) 
			p_fillDialog = new TFDialogFill (p_HF, p_HFD);
		p_fillDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_FLOWMAP)))
		{
		if (!p_flowmapDialog) 
			p_flowmapDialog = new TFDialogFlowmap (p_HF, p_HFD);
		p_flowmapDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_FOLD)))
		{
		if (!p_foldDialog)
			p_foldDialog = new TFDialogFold (p_HF, p_HFD);
		p_foldDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_GAUSS_HILL)))
		{
		if (!p_gaussHillDialog)
			p_gaussHillDialog= new TFDialogGaussianHill (p_HF, p_HFD);
		p_gaussHillDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_INVERT)))
		{
		if (p_HF)
			{
			HeightFieldOps	*HFO;

			HFO = new HeightFieldOps (p_HF);
			HFO->invert ();
			p_HFD->draw ();
			delete HFO;
			}
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_LINSCALE)))
		{
		if (!p_linearScaleDialog) 
			p_linearScaleDialog = new TFDialogLinearScale (p_HF, p_HFD);
		p_linearScaleDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_MIRROR)))
		{
		if (!p_mirrorDialog)
			p_mirrorDialog = new TFDialogMirror (p_HF, p_HFD);
		p_mirrorDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_RESCALE)))
		{
		if (!p_rescaleDialog) 
			p_rescaleDialog = new TFDialogRescale (p_HF, p_HFD);
		p_rescaleDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_ROTATE)))
		{
		if (!p_rotateDialog) 
			p_rotateDialog = new TFDialogRotate (p_HF, p_HFD);
		p_rotateDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_ROUGHSMOOTH)))
		{
		if (!p_roughSmoothDialog) 
			p_roughSmoothDialog = new TFDialogRoughSmooth (p_HF, p_HFD);
		p_roughSmoothDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_TERRACE)))
		{
		if (!p_terraceDialog)
			p_terraceDialog = new TFDialogTerrace (p_HF, p_HFD);
		p_terraceDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_TRANSFORM)))
		{
		if (!p_transformDialog)
			p_transformDialog = new TFDialogTransform (p_HF, p_HFD);
		p_transformDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_CLONE)))
		{
		HeightField 	*HF;
		char		*newName = TFGui::getNewName(p_HF->getName());

		HF = new HeightField (p_HF->getCopyOfData(),
			p_HF->getWidth(), p_HF->getHeight(), newName);
		free (newName);

		TFGui::addHeightField (HF);
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_EXPORT_CONTOUR)))
		{
		if (!p_contLinesDialog)
			p_contLinesDialog = new TFDialogContourLines (p_HF, p_HFD);
		p_contLinesDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_CRATERS)))
		{
		if (!p_contLinesDialog)
			p_contLinesDialog = new TFDialogContourLines (p_HF, p_HFD);
		p_contLinesDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_ERODE)))
		{
		if (!p_erodeDialog)
			p_erodeDialog = new TFDialogErode (p_HF, p_HFD);
		p_erodeDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_PLACEPOV)))
		{
		if (!p_placePovDialog)
			p_placePovDialog = new TFDialogPPov (p_HF);
		p_placePovDialog->show ();
		}
	else
	if (!strcmp (menuitem, _(MENU_HF_RENDER)))
		{
		if (!p_renderDialog)
			p_renderDialog = new TFDialogRender (p_HF, p_HFD);
		p_renderDialog->show ();
		}
	else
		SanityCheck::bailout ("uncaught <hf> menuitem", "TFWindowHandler::fileMenuCallback");

	fflush (stdout);
}


/*
 *  viewMenuCallback: process a menu choice in the view menu
 */
void TFWindowHandler::viewMenuCallback (string m)
{
	char		*menuitem = const_cast<char*>(m.c_str());
	long		delay;

	if (!p_HFD)
		return ;

	if (!strcmp (menuitem, _(MENU_VIEW_FASTWIRE)))
		{
		Gtk_Widget		*w;

		if (d_fastWire)
			d_fastWire = FALSE;
		else
			d_fastWire = TRUE;
		w = static_cast<Gtk_Widget*>(this->p_menuHash->lookup(_(MENU_VIEW_AUTOROTATE)));
		w->set_sensitive (d_fastWire);
		p_HFD->setFastWire (d_fastWire);
		if (p_HFD->getMode()==WIRE)
			drawHeightField ();
		}
	else
	if (!strcmp (menuitem, _(MENU_VIEW_AUTOROTATE)))
		{
		if (d_autoRotate)
			d_autoRotate = FALSE;
		else
			{
			d_autoRotate = TRUE;
			delay = (long int)(calibrateAutoRotate(10)/1000*1.1);
			connect_to_method (Gtk_Main::timeout(delay), this, 
					&TFWindowHandler::autoRotateCallback);
			}
		}
	else
	if (!strcmp (menuitem, _(MENU_VIEW_2DPLANE)) ||
	    !strcmp (menuitem, _(MENU_VIEW_3DWIRE))  ||
	    !strcmp (menuitem, _(MENU_VIEW_3DHEIGHT))||
	    !strcmp (menuitem, _(MENU_VIEW_3DLIGHT)) )
		{
		if (!p_HFD->setMode (menuitem))
			drawHeightField ();
		}
	else
		SanityCheck::bailout (TRUE, "uncaught <view> menuitem", "TFWindowHandler::fileMenuCallback");

	fflush (stdout);
}

/*
 *  colormapMenuCallback: process a menu choice in the colormap menu
 */
void TFWindowHandler::colormapMenuCallback (string m)
{
	char		*menuitem = const_cast<char*>(m.c_str());

	if (!strcmp (menuitem, _(MENU_CMAP_GRAYSCALE)))
		{
		this->d_useLinearColormap = TRUE;
		if (p_HFD)
			{
			if (!p_HFD->setColormap (this->p_linearColorMap))
				p_HFD->draw ();
			}
		}
	else
	if (!strcmp (menuitem, _(MENU_CMAP_COLORBANDS)))
		{
		this->d_useLinearColormap = FALSE;
		if (p_HFD)
			{
			if (!p_HFD->setColormap (this->p_bandColorMap)) 
				p_HFD->draw ();
			}
		}
	else
	if (!strcmp (menuitem, _(MENU_CMAP_DESERT)))
		{
		this->d_useLinearColormap = FALSE;
		if (p_HFD)
			{
			if (!p_HFD->setColormap (this->p_bandColorMap2))
				p_HFD->draw ();
			}
		}
	else
	if (!strcmp (menuitem, _(MENU_CMAP_REDHOT)))
		{
		this->d_useLinearColormap = TRUE;
		if (p_HFD)
			{
			if (!p_HFD->setColormap (this->p_linearColorMap2))
				p_HFD->draw ();
			}
		}
	else
		SanityCheck::bailout (TRUE, "uncaught <colormap> menuitem", "TFWindowHandler::fileMenuCallback");

	fflush (stdout);
}


/*
 *  destroyCallback: handle a destroy event 
 */
void TFWindowHandler::destroyCallback ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "Entering destroyCallback\n"); 

	closeCallback ();

	if (TFOptions::s_settingsAreChanged)
		{
		TFOptions	*tfOpt = new TFOptions ();

		tfOpt->install ();
		delete tfOpt;
		}

	if (GlobalTrace::isSet(GlobalTrace::TRACE_VERBOSE))
		g_print ("Shutting Down ...\n");
	gtk_exit (0);
	exit (0);
}


/*
 *  closeCallback: delete the height field
 */
void TFWindowHandler::closeCallback ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "Entering closeCallback\n"); fflush (stdout);

	if (TFWindowBase::s_quitDialog)
		TFWindowBase::s_quitDialog->hide ();

	if (p_HFD)
		p_HFD->clear ();

	activateMenus (FALSE);

	if (this->d_useLinearColormap)
		this->setColors (&(this->p_linearColorMap->d_black));
	else
		this->setColors (&(this->p_bandColorMap->d_black));
	this->drawRectangle (0, 0, this->width(), this->height(), TRUE);
	this->sync ();
	TFGui::scheduleDelete (p_HF);
	connect_to_function (Gtk_Main::timeout(100), &TFGui::cleanup);
}


/*
 *  reloadCallback: process a reload request 
 */
void TFWindowHandler::reloadCallback ()
{
	if (TFWindowBase::s_quitDialog)
		TFWindowBase::s_quitDialog->hide ();

	if (p_HF)
		{
		HeightField		*HF = NULL;

		readhf (&HF, p_HF->getName(), TRUE);
		setHeightField (HF);
		}
}


/*
 *  fileSelectionOK: user pressed OK in file selection dialog
 */
void TFWindowHandler::fileSelectionOK ()
{
	gchar 	*fname;
	int	rc;

	p_fileSelection->hide ();

	fname = const_cast<char*>(p_fileSelection->get_filename().c_str());
	//g_print ("%s\n", fname);

	//create_progress_bar (fname);

	if (cs_fileselMode == LOAD)
		{
		HeightField	*HF=NULL;

		rc = readhf (&HF, fname, TRUE);
		setHeightField (HF);
		}
	else
		{
		HeightFieldIO		*HFIO;

		HFIO = new HeightFieldIO (fname);
                rc = HFIO->write (p_HF, p_HFD);
		delete HFIO;
		}

	if (rc)
		fprintf (stderr, _("Warning: last command returned an Error (%d)\n"), rc);

	activateMenus (TRUE);
}


/*
 *  fileSelectionCancel: user pressed Cancel in file selection dialog
 */
void TFWindowHandler::fileSelectionCancel ()
{
	p_fileSelection->set_filename ("");
	p_fileSelection->hide ();
}


/*
 *  fileSelection: open a file selection dialog
 */
void TFWindowHandler::fileSelection ()
{
	if (!p_fileSelection)
		{
		p_fileSelection = new Gtk_FileSelection ("");
		p_fileSelection->set_position (GTK_WIN_POS_MOUSE);

		connect_to_method (p_fileSelection->get_ok_button()->clicked, 
					this, &TFWindowHandler::fileSelectionOK);
		connect_to_method (p_fileSelection->get_cancel_button()->clicked,
					this, &TFWindowHandler::fileSelectionCancel);
		}

	if (cs_fileselMode==LOAD)
		p_fileSelection->set_title ("Load Height Field");
	else
		p_fileSelection->set_title ("Save Height Field");

	p_fileSelection->show ();
}


/*
 *  checkMenus: check the menu status for menus whch are dependant 
 * 	on specific conditions. 
 */
void TFWindowHandler::checkMenus ()
{
	Gtk_ObjectHandle<Gtk_MenuItem> 	m;
	string				*s;

	s = new string (strrep(_(MENU_FILE_MERGE), "_", ""));
	m = p_itemFactory->get_menuitem_widget (*s);
	if (s_winList->getSize() > 1)
		m->set_sensitive (TRUE);
	else
		m->set_sensitive (FALSE);
	delete s;

	s = new string (strrep(_(MENU_FILE_RELOAD), "_", ""));
	m = p_itemFactory->get_menuitem_widget (*s);
	if (p_HF->isLoadedFromFile())
		m->set_sensitive (TRUE);
	else
		m->set_sensitive (FALSE);
	delete s;
}


/*
 *  activateMenus: activate the menus which need to be activated. 
 * 	FIXME: this is old code and can probably (don't need it for multi-win). 
 */
void TFWindowHandler::activateMenus (bool activate)
{
	Gtk_Widget	**w;
	int		c=0, i; 

	if (SanityCheck::warning((!this->p_menuHash->get_size()), "size==0", "TFWindowHandler::activateMenus"))
		return;
	
	w = new Gtk_Widget*[this->p_menuHash->get_size()];

	if (!activate || (p_HF && p_HF->isLoadedFromFile()))
		w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_RELOAD)));
	if (!activate || s_winList->getSize() > 1)
		w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_MERGE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_SAVE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_SAVEAS)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_CLOSE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_EXPORT_POVRAY)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_OPTIONS)));
	if (activate)
		w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_FILE_ABOUT)));
	//w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_BRANCH)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_CRATERS)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_FILL)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_FLOWMAP)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_FOLD)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_GAUSS_HILL)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_INVERT)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_LINSCALE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_MIRROR)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_RESCALE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_ROTATE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_ROUGHSMOOTH)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_TERRACE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_TRANSFORM)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_EXPORT_CONTOUR)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_CRATERS)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_ERODE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_HF_PLACEPOV)));
	//w[c++] = static_cast<Gtk_Widgt*> (this->p_menuHash->lookup (_(MENU_VIEW_BRANCH)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_VIEW_2DPLANE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_VIEW_3DWIRE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_VIEW_3DHEIGHT)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_VIEW_3DLIGHT)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_VIEW_3DLIGHT)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_VIEW_FASTWIRE)));
	if (!activate)
		w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_VIEW_AUTOROTATE)));
	//w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_CMAP_BRANCH)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_CMAP_COLORBANDS)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_CMAP_DESERT)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_CMAP_GRAYSCALE)));
	w[c++] = static_cast<Gtk_Widget*> (this->p_menuHash->lookup (_(MENU_CMAP_REDHOT)));

	for (i=0; i<c; i++)
		{
		if (!w[i])
			printf ("activateMenu Error: w[%d]=NULL\n", i);
		else
			w[i]->set_sensitive (activate);
		}

	delete w;
}


//
// Overloaded event implementations...
//
gint TFWindowHandler::button_press_event_impl (GdkEventButton *e)
{
	// button 1 starts rotating 
	if (e->button == 1)
		{
		d_rotate = TRUE;
		d_rotStartPosx = (int)e->x;	// Note current x position
		d_rotStartPosy = (int)e->y;	// Note current y position
		}
	else
	// button 2 resets the view 
	if (e->button == 2)
		{
		d_rotate = FALSE;
		if (p_HFD)
			{
			p_HFD->resetView ();
			p_HFD->draw (TRUE);
			}
		}
	else
	if (e->button == 3 && !TFOptions::s_fixedMenus)
		{
		this->d_menuHandle->popup (NULL, NULL, e->button, e->time);
		}

	return Gtk_Widget::button_press_event_impl(e);
}


gint TFWindowHandler::button_release_event_impl (GdkEventButton *e)
{
	//printf ("Button Release\n"); fflush (stdout);
	// Rotation completed
	d_rotate = FALSE;

	return Gtk_Widget::button_release_event_impl(e);
}


gint TFWindowHandler::motion_notify_event_impl (GdkEventMotion *e)
{
	int 		x, y;
	GdkModifierType	state;

	//printf ("Motion Notify\n"); fflush (stdout);

	if (d_rotate && d_fastWire && p_HFD->getMode() == WIRE) 
		{
		if (e->is_hint) 
			gdk_window_get_pointer(e->window, &x, &y, &state);
		else 
			{
			x = int (e->x);
			y = int (e->y);
			//state = e->state;
			}

		//printf ("Motion Notify (%d, %d)\n", x, y); fflush (stdout);
		// Apply rotation
		p_HFD->rotateFastView (0.0, (float)((d_rotStartPosx-x)/500.0), 0.0);
		if (p_HFD->d_fastDoRotZ)
			p_HFD->rotateFastView (0.0, 0.0, (float)((y-d_rotStartPosy)/500.0));
		d_rotStartPosx = int(e->x);
		d_rotStartPosy = int(e->y);
		p_HFD->draw (FALSE);
		}
	return Gtk_Widget::motion_notify_event_impl(e);
}


gint TFWindowHandler::delete_event_impl (GdkEventAny *e)
{
	printf ("HERE\n"); fflush (stdout);
	this->hide ();

	TFGui::scheduleDelete (p_HF);
	connect_to_function (Gtk_Main::timeout(100), &TFGui::cleanup);
        return (1);
}

