/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include "FileIO.h"
#include "HeightFieldExport.h"
#include "HeightFieldIO.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"
#include "MathTrig.h"
#include "PrintOptions.h"
#include "RenderOptions.h"
#include "TFOptions.h"
#include "tf_flexarrCOORD2.h"
#include "strrep.h"
#include "terraform.h"


#define NO_FLOW		-987654321		// marker for 'undefined'
#ifndef MIN
# define MIN(a,b)	(a<b ? a : b)
#endif
#ifndef MAX
# define MAX(a,b)	(a>b ? a : b)
#endif 


/*
 *  constructor: initialize everything
 */
HeightFieldExport::HeightFieldExport (HeightField *HF) 
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldExport\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldExport::HeightFieldExport");
	//SanityCheck::bailout ((!HF->d_hf), "HF->d_hf==NULL", "HeightFieldExport::HeightFieldExport");

	p_HF = HF;
}


/*
 *  destructor: clean up 
 */
HeightFieldExport::~HeightFieldExport ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldExport\n");
}


/*
 *  printContourLines: iterate through all the lists and items and 
 * 	print them. 
 */
void HeightFieldExport::printContourLines (TFGListFlexArr *listMain, 
					   HeightFieldDraw *HFD)
{
	SanityCheck::bailout ((!listMain), "listMain==NULL", "HeightFieldExport::printContourLine");
	SanityCheck::bailout ((!HFD), "HFD==NULL", "HeightFieldExport::printContourLinesToPS");

	int		limMain=listMain->length(),
			limLine;
	COORD2		*coord2;
	TFFlexArrayCoord2 *arrLine=NULL;

	for (int i=0; i<limMain; i++)
		{
		arrLine = static_cast<TFFlexArrayCoord2*>(listMain->nth(i));
		SanityCheck::bailout ((!arrLine), "arrLine==NULL", "HeightFieldExport::printContourLine");
		limLine = arrLine->getSize ();
		printf ("******* Contour line %d of %d (length=%d) ********* \n", i, limMain, limLine); fflush (stdout);

		for (int j=0; j<limLine; j++)
			{
			coord2 = static_cast<COORD2*>(arrLine->El(j));
			printf ("(%d,%d) ; ", (int)coord2->x, (int)coord2->y);
			}

		printf ("\n\n\n");
		}
}


/*
 *  exportContourLinesToPS: iterate through all the lists and items and 
 * 	convert them to Postscript output. 
 * 	This code donated by Cloister Bell. 
 */
void HeightFieldExport::exportContourLinesToPS (TFGListFlexArr *listMain, 
					HeightFieldDraw *HFD, char *fname) 
{
	SanityCheck::bailout ((!listMain), "listMain==NULL", "HeightFieldExport::exportContourLinesToPS");
	SanityCheck::bailout ((!HFD), "HFD==NULL", "HeightFieldExport::exportContourLinesToPS");
	SanityCheck::bailout ((!fname), "fname==NULL", "HeightFieldExport::exportContourLinesToPS");
	SanityCheck::bailout ((!strlen(fname)), "strlen(fname)==0", "HeightFieldExport::exportContourLinesToPS");

	int		i=0, 
			limMain=listMain->length(),
			limLine;
	COORD2		*coord2;
	TFFlexArrayCoord2 *arrLine=NULL;
	FileIO		*fio = new FileIO (fname);
	FILE		*fptr;
	float 		xsf,				// x scale factor
			ysf;				// y scale factor
        GdkColor        *color;                         // color at (x,y)
	char            buf[160];                       // debug output

	if (!fio)
		return;
	fio->open ("w");
	fptr = fio->getFilePtr ();

	if (!fptr)
		return;

	// write postscript header
        static char *static_ps_header[] = 
		{
		"/inch {72 mul} def\n",
		"/l {lineto} def /m {moveto} def\n",
		"/n {newpath} def /s {stroke} def\n",
                "/c {setrgbcolor} def /g {setgray} def\n",
                "/p {closepath} def /r {rlineto} def\n",
		"0 setlinewidth\n",
		"/Times-Roman findfont 10 scalefont setfont \n",
		"\0"
        	};

	fprintf(fptr, "%%!PS-Adobe-3.0 -- Created by Terraform %d.%d.%d\n",
		TF_MAJOR_VERSION, TF_MINOR_VERSION, TF_MICRO_VERSION);

        while(static_ps_header[i][0] != '\0')
		fprintf(fptr, static_ps_header[i++]);

	// move the coordinate system to the right place / orientation:
        if(PrintOptions::s_orientation == 'P') 		// portrait
	  {
	  fprintf(fptr, "%f inch %f %f sub inch translate 1 -1 scale\n",
		  PrintOptions::s_marginL, PrintOptions::s_paperHeight,
		  PrintOptions::s_marginT);
	  // calculate scale factors for portrait mode
	  xsf = ((PrintOptions::s_paperWidth - PrintOptions::s_marginL - 
		  PrintOptions::s_marginR) * 72.0) / (float)p_HF->getWidth();
	  ysf = ((PrintOptions::s_paperHeight - PrintOptions::s_marginT - 
		  PrintOptions::s_marginB) * 72.0) / (float)p_HF->getWidth();
	  }
        else 
	  { // PRINT_ORIENTATION_LANDSCAPE
          fprintf(fptr, "%f inch %f inch translate 90 rotate 1 -1 scale \n",
                  PrintOptions::s_marginT, PrintOptions::s_marginL); 
	  // calculate scale factors for landscape mode
	  xsf = ((PrintOptions::s_paperHeight - PrintOptions::s_marginL - 
		  PrintOptions::s_marginR) * 72.0) / (float)p_HF->getWidth();
	  ysf = ((PrintOptions::s_paperWidth - PrintOptions::s_marginT - 
		  PrintOptions::s_marginB) * 72.0) / (float)p_HF->getWidth();
	  }

        // scale the output to fit the paper size - margins.
	fprintf(fptr, "%f dup scale\n", min(xsf, ysf));

	// At this point, the postscript coordinate system has been 
	// transformed so that we can work naturally in units of HF 
	// grid cells. Nothing after this point should be done in 
	// points, inches, or any other physical units.

	// draw a box around the output
	fprintf(fptr, "n 0 0 m %d 0 l %d %d l 0 %d l p s\n",
		p_HF->getWidth(), p_HF->getWidth(), p_HF->getHeight(), p_HF->getHeight());

	// This will probably depend on contour lines being presented in 
	// low to high altitude order:

	// process each contour line:
	for (int i=0; i<limMain; i++)
		{
		arrLine = static_cast<TFFlexArrayCoord2*>(listMain->nth(i));
		SanityCheck::bailout ((!arrLine), "arrLine==NULL", "HeightFieldExport::printContourLine");
		limLine = arrLine->getSize ();

		// we need at least two points in the flexarray in order to 
		// make a valid postscript path
		if(limLine >= 2 )
			{
			fprintf(fptr, "n "); 			// newpath
			coord2 = static_cast<COORD2*>(arrLine->El(0));
			fprintf(fptr, "%d %d m ",		// moveto
				(int)coord2->x, (int)coord2->y);
			// do a lineto for each point in the contour.
			for (int j=1; j < limLine; j++)
				{
				coord2 = static_cast<COORD2*>(arrLine->El(j));
				fprintf(fptr, "%d %d l ",	// lineto
					(int)coord2->x, (int)coord2->y);
				}
			if(PrintOptions::s_color == 'C') // color mode
				{ 
				color = HFD->getColor2d((int)coord2->x, (int)coord2->y);
				if(color)  // 'c' is "setrgbcolor"
					{
					fprintf(fptr, "%f %f %f c ",
						(float)color->red/MAXBRIGHT,
						(float)color->green/MAXBRIGHT,
						(float)color->blue/MAXBRIGHT);
					sprintf(buf, "elv: %5.3f color: %d r: %f g: %f b: %f\n",
						p_HF->getEl((int)coord2->x,(int)coord2->y), 
						(unsigned int)color, 
						(float)color->red/MAXBRIGHT,
						(float)color->blue/MAXBRIGHT,
						(float)color->green/MAXBRIGHT);
					GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, buf);
					}
				else
				        fprintf(fptr, "0 0 0 c "); // black.
				}
			else 
			if(PrintOptions::s_color == 'G') // grayscale mode
				fprintf(fptr, "%f g ", // setgray
					p_HF->getEl(p_HF->Pos2Off(coord2->x,
							       coord2->y)));

                        fprintf(fptr, "s "); 			// stroke
			}
		}

	// draw map legend.  only relevant in color or grayscale modes.
#define SQUARESIZE 18
	if(PrintOptions::s_color == 'C' || PrintOptions::s_color == 'G')
		{
		if(PrintOptions::s_orientation == 'P')		// portrait
			fprintf(fptr, "0 %d translate ", p_HF->getHeight() + 18);
		else                           // PRINT_ORIENTATION_LANDSCAPE
			fprintf(fptr, "%d 0 translate ", p_HF->getWidth() + 18);
	
		for(i=0; i < 10; i++) // 10 should really be # of contour lev's
			{
			fprintf(fptr, "n %d 0 m ", i*SQUARESIZE);
			fprintf(fptr, "0 %d r %d 0 r 0 %d r p ",
				SQUARESIZE, SQUARESIZE, -SQUARESIZE);
			if(PrintOptions::s_color == 'C')
				{
				color = HFD->getColorElv((float)i/10.0 + .05);
				if(color)
					fprintf(fptr, "%f %f %f c ",
						(float)color->red/MAXBRIGHT,
						(float)color->green/MAXBRIGHT,
						(float)color->blue/MAXBRIGHT);
				else
					fprintf(fptr, "0 0 0 c ");
				}
			else // grayscale
				fprintf(fptr, "%f g ", (float)i/10.0);

			fprintf(fptr, "fill gsave 0 g ");
			sprintf(buf, "%3.2f -- %3.2f",
				(float)i/10.0,
				(float)(i+1)/10.0);
			fprintf(fptr, "90 rotate 1 -1 scale %d %d moveto ", 
				(SQUARESIZE * 3) / 2,
				i * SQUARESIZE + SQUARESIZE / 2);
			fprintf(fptr, "(%s) show grestore ", buf);
			}
		}

	// print postamble
	fprintf(fptr, "showpage\n");

	fio->close ();
}



/* 
 * renderPov: render a HF in POVRay. This routines forks and the parent returns
 */
int HeightFieldExport::renderPOV ()
{
	RenderOptions 	*rOpt = p_HF->getRenderOptions ();
	FileIO		*fpov;

	//SanityCheck::bailout ((!rOpt), "rOpt==NULL", "HeightFieldExport::renderPOV");
	if (!rOpt)
		{
		rOpt = new RenderOptions (p_HF->getName());
		p_HF->setRenderOptions (rOpt);
		}

	// see if the master POV file exists
	fpov = new FileIO (rOpt->getPOVFile());	
	if (!fpov->exists())
		{
		SanityCheck::warning ("povFile doesn't exist", "HeightFieldExport::renderPOV");
		//printf ("POV file: %s\n", rOpt->getPOVFile());
		delete fpov;
		return -1;
		}
	delete fpov;

	// fork: child process renders, parent returns
	int pid = fork ();
	if (pid)
		return (0);

	// declare variables
	FileIO	*fHF = NULL;			// HF filename
	UI	width, height; 			// size
	int	rc;				// return code
	char	buf1[256],			// HF POV parameter filename buffer
		buf2[512],			// command buffer 
		outfile[256];			// command buffer 

	// original filename in comments is assumed world.tga
	fHF = new FileIO (p_HF->getName()); 

	// determine location of working files 
	if (rOpt->getKeepFiles())
		{
		sprintf (buf1, "%s_tf.pov", fHF->getBasename()); // world.pov
		sprintf (outfile, "%s_tf.tga", fHF->getBasename()); 
		}
	else
		{
		char 	*tmpdir;

		tmpdir = getenv ("TMPDIR");
		if (!tmpdir)
			tmpdir = "/tmp";			// /tmp/world.pov
		sprintf (buf1, "%s/%s_tf.pov", tmpdir, fHF->getBasename());
		sprintf (outfile, "%s/%s_tf.tga", tmpdir, fHF->getBasename()); 
		}

	// write the POV parameter file 
	rOpt->writePOVFile (buf1);

	// write HF
	HeightFieldIO		*HFIO = new HeightFieldIO (p_HF->getName());
	HFIO->write (p_HF);
	delete HFIO;

	// create the POV command string
	rOpt->getSize (&width, &height);
	if (TFOptions::s_POV30)
		{
		if (GlobalTrace::isSet (GlobalTrace::TRACE_DEBUG))
			sprintf (buf2, "%s +W%d +H%d +O%s +i %s ", 
				TFOptions::s_POVexec, width, height, 
				outfile, buf1);
		else
			sprintf (buf2, "%s +W%d +H%d +O%s +i %s > /dev/null 2>&1",
				TFOptions::s_POVexec, width, height, 
				outfile, buf1);
		}
	else
		{
		if (GlobalTrace::isSet (GlobalTrace::TRACE_DEBUG))
			sprintf (buf2, "%s +D +P +W%d +H%d +O%s +i %s ",
				TFOptions::s_POVexec, width, height, 
				outfile, buf1);
		else
			sprintf (buf2, "%s +D +P +W%d +H%d +O%s +i %s > /dev/null 2>&1",
				TFOptions::s_POVexec, width, height, 
				outfile, buf1);
		}
	//printf ("%s\n", buf2);
	rc = system (buf2);

	// clean up
	if (!rOpt->getKeepFiles())
		{
		unlink (buf1);
		unlink (p_HF->getName());
		}

	delete fHF;

	SanityCheck::warning (rc, "povray command returned an error", "HeightFieldExport::renderPOV");

	 // have to call _exit to avoid closing file descriptors/X connection
        _exit (0);
}


