/***************************************************************************
                          rfonts.h  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/****************************************************************************
** rfonts.h 1998/10/09 A. Mustun RibbonSoft 
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#ifndef RFONTS_H
#define RFONTS_H

#include <qstringlist.h>

#include "rprgdef.h"
#include "rlayoutdef.h"

class RElement;


/*! Structur for one font: */
//
struct RFont {
  RElement* letter[256];         // 256 Pointers to all letters
  int       elNum[256];          // 256 values for the element number for every letter
  float     letWidth[256];       // 256 values for the width of every letter
  char      fontName[32];        // name of this font
  float     letterSpacingFactor; // Factor for Letter Spacing (fontHeight*letterSpacingFactor=letterSpacing)
  float     wordSpacingFactor;   // Factor for Word Spacing (fontHeight*wordSpacingFactor=wordSpacing)
  float     lineDistanceFactor;  // Factor for Line Distance (fontHeight*lineDistanceFactor=lineDistance)
};


/*! Font handling (single instance) */
//
class RFonts
{

protected:
  RFonts();
  
public:
  ~RFonts();

  /** Initialize the fonts */
  void    init();

  /*! Pointer to the one and only RFonts instance */
  static  RFonts* getRFonts();
  /*! Reset all fonts (set values to zero) This function does NOT delete the letter elements! */
  void    reset();
  /*! Get the Font Number from its name */
  int     getFontNumber(const char* _name);
  /*! Get the Fonts name from its number */
  char*   getFontName(int _num);
  /*! Get the Fonts letterSpacingFactor from its number */
  float   getFontLetterSpacingFactor(int _num);
  /*! Get the Fonts wordSpacingFactor from its number */
  float   getFontWordSpacingFactor(int _num);
  /*! Get the Fonts lineDistanceFactor from its number */
  float   getFontLineDistanceFactor(int _num);
  /*! Get the total number of fonts found */
  int     getNumFonts() { return numFonts; }
  /*!
      Get width of a letter
      \param _font font number of the letter
      \param _letter ascii code of the letter
      \param _height font height
  */
  float   getLetterWidth(int _font,
                         unsigned char _letter, 
                         float _height);
  /*!
      Get width of a string which can contain line breaks
      \param _font Font
      \param _text String
      \param _height Font height
      \param _letSpace Letter spacing
      \param _wordSpace Word spacing
  */
  float   getTextWidth(int _font,
                       const unsigned char* _text,
                       float _height, 
                       float _letSpace, 
                       float _wordSpace);
  /*!
      Get width of a line of a string which contains line breaks
      \param _font Font
      \param _text String
      \param _index Line number
      \param _height Font height
      \param _letSpace Letter spacing
      \param _wordSpace Word spacing
      \param _fixedWidth Use same width for all letters (as 'W')
  */
  float   getTextLineWidth(int _font,
                           const unsigned char* _text,
                           int _index,
                           float _height, 
                           float _letSpace, 
                           float _wordSpace,
                           bool _fixedWidth=false);
  /*!
      Get max. height of a string which can contain line breaks
      \param _font Font
      \param _text String
      \param _height Font height
      \param _lineDist Line distance
  */
  float   getTextHeight(int _font,
                        const unsigned char* _text,
                        float _height, 
                        float _lineDist);
  /*!
      Get a single element from a letter
      \param _font   Font
      \param _letter Letter number
      \param _height Font height
      \param _x      x offset
      \param _y      y offset
      \param _angle  angle
      \param _align  Alignment flags (E_LEFT / E_RIGHT / E_CENTER)
      \param _index  Index of the element in the letter
      \param _el     Pointer to the memory where the element gets saved
  */
  void    getLetterElement(int _font,
                           unsigned char _letter, 
                           float _height,
                           float _x, float _y,
                           float _angle,
                           uint _align,
                           int _index,
                           RElement* _el);
  /*!
      Get number of elements in a letter
      \param _letter Letter
      \param _font   Font
  */
  int     getLetterElementNumber(int _font,
                                 unsigned char _letter);

  /*!
      Check if a Font number is valid
      \param _font The font number
   */
  bool    isFontValid(int _font) { if(_font>=0 && _font<numFonts) return true;
                                   else                           return false; }
  /*!
      Check if a Letter index is valid
      \param _let The letter index
  */
  bool    isLetterValid(unsigned char _letter) { return true; }

  /*!
      Check if a Letter is available (has at least one element)
      \param _let The letter index
  */
  bool    isLetterAvailable(int _font, unsigned char _letter);


private:
  /*! Delete all Fonts */
  void deleteFonts();
  /** Lookup all fonts and store the paths in list fontPathList */
  void lookupFonts( QString _fontPath );
  /*! Read Fonts from all directories  */
  void readFonts();

protected:
  /*! The one and only instance of RFonts */
  static RFonts* theOneAndOnlyRFonts;

  /** List of all font paths of all fonts found */
  QStringList fontPathList;

  /*! The single Fonts */
  RFont* font;          // Pointer to single fonts
  /*! Number of Fonts */
  int    numFonts;      // number of fonts
  

};


#endif

// EOF


