// copyright (c) 1997,1998 stephen f. white
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

import java.util.*;
import java.net.*;		// for URL stuff
import vrml.*;
import vrml.field.*;
import vrml.node.*;

public class JSAISceneInterface extends Script implements SceneInterface {
    TalkerFrame		talkerFrame;
    Browser		browser;		// running VRML browser
    Dispatcher		dispatcher;

    // script eventOut's
    SFBool		isConnected;

    // rootNode eventIn's
    MFNode		children;
    MFNode		addChildren;
    MFNode		removeChildren;

    // viewpoint EventIn's
    SFVec3f		viewpointPosition;
    SFRotation		viewpointOrientation;
    SFBool		viewpointBind;

    // intern'ed strings for eventIn names
    static final String	positionStr = "position".intern();
    static final String	orientationStr = "orientation".intern();

    public void initialize()
    {
	String		hostname = "";
	int		port;

	Node rootNode = (Node) ((SFNode) getField("rootNode")).getValue();
	addChildren = (MFNode) rootNode.getEventIn("addChildren");
	removeChildren = (MFNode) rootNode.getEventIn("removeChildren");
	children = (MFNode) rootNode.getExposedField("children");

	Node viewpoint = (Node) ((SFNode) getField("viewpoint")).getValue();
	viewpointPosition = (SFVec3f) viewpoint.getExposedField("position");
	viewpointOrientation = (SFRotation) viewpoint.getExposedField("orientation");
	viewpointBind = (SFBool) viewpoint.getEventIn("set_bind");
	isConnected = (SFBool) getEventOut("isConnected");

	browser = getBrowser();
	try {
	    hostname = new URL(browser.getWorldURL()).getHost();
	} catch (MalformedURLException e) {
	}
	port = ((SFInt32) getField("port")).getValue();
	MFString	avatarNamesField = (MFString) getField("avatarNames");
	MFString	avatarURLsField = (MFString) getField("avatarURLs");
	String[]	avatarNames = new String[avatarNamesField.getSize()];
	String[]	avatarURLs = new String[avatarURLsField.getSize()];
	avatarNamesField.getValue(avatarNames);
	avatarURLsField.getValue(avatarURLs);
	dispatcher = new Dispatcher(this);
	talkerFrame = new TalkerFrame(dispatcher, hostname, port, avatarNames, avatarURLs);
	talkerFrame.show();
	dispatcher.setTalker(talkerFrame.panel);
    }

    public void processEvent(vrml.Event e) {

	String		s = e.getName().intern();

	if (s == positionStr) {
	    float[] vec = new float[3];
	    ((ConstSFVec3f) e.getValue()).getValue(vec);
	    dispatcher.sendPosition(new VSFVec3f(vec));
	} else if (s == orientationStr) {
	    float[] vec = new float[4];
	    ((ConstSFRotation) e.getValue()).getValue(vec);
	    dispatcher.sendOrientation(new VSFRotation(vec));
	} else {
	    dispatcher.log(" unknown event:  " + e);
	}
    }

    public void onStatus(String str)
    {
	talkerFrame.setStatus(str);
    }

    public void onError(String str)
    {
	dispatcher.log("error:  " + str);
	new Alert(talkerFrame, "Error", str).show();
    }

    public void addObject(VRMLObject obj)
    {
	addChildren.setValue(((JSAIObject) obj).nodes);
    }

    public void removeObject(VRMLObject obj)
    {
	removeChildren.setValue(((JSAIObject) obj).nodes);
    }

    public void removeAllObjects()
    {
	children.setValue(new Node[0]);
    }

    public VRMLObject createObject(int vid, String url,
   				   VRMLObjectObserver observer)
    {
	return new JSAIObject(vid, url, browser, observer);
    }

    public void setViewpointPosition(VSFVec3f position)
    {
	viewpointBind.setValue(false);
	viewpointPosition.setValue(position.getValue());
	viewpointBind.setValue(true);
    }

    public void setViewpointOrientation(VSFRotation orientation)
    {
	viewpointBind.setValue(false);
	viewpointOrientation.setValue(orientation.getValue());
	viewpointBind.setValue(true);
    }

    public void onConnect() {
	isConnected.setValue(true);
    }

    public void onDisconnect() {
	isConnected.setValue(false);
    }
}
