// copyright (c) 1998 Jeff Sonstein
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

import java.awt.*;

public class BorderPanel extends Panel {

  /** Border types */
  public final static int SOLID = 0;
  public final static int RAISED = 1;
  public final static int LOWERED = 2;
  public final static int IN = 3;
  public final static int OUT = 4;

  /** Alignment types */
  public final static int LEFT = 0;
  public final static int CENTER = 1;
  public final static int RIGHT = 2;

  /** Defaults */
  public final static int DEFAULT_STYLE = IN;
  public final static int DEFAULT_THICKNESS = 10;
  public final static int DEFAULT_SOLID_THICKNESS = 4;
  public final static int DEFAULT_RAISED_THICKNESS = 2;
  public final static int DEFAULT_LOWERED_THICKNESS = 2;
  public final static int DEFAULT_IN_THICKNESS = 10;
  public final static int DEFAULT_OUT_THICKNESS = 10;
  public final static int DEFAULT_GAP = 5;
  public final static Color DEFAULT_COLOR = Color.black;
  public final static Font DEFAULT_FONT = new Font("TimesRoman", Font.PLAIN, 14);
  public final static int DEFAULT_ALIGNMENT = LEFT;

  /** private variables */
  private int style;
  private int thickness;
  private int gap;
  private Color color;
  private Font font;
  private String text;
  private int alignment;

  /** default constructor */
  public BorderPanel() {
    style       = DEFAULT_STYLE;
    thickness   = DEFAULT_THICKNESS;
    gap         = DEFAULT_GAP;
    color       = DEFAULT_COLOR;
    text        = null;
    font        = DEFAULT_FONT;
    alignment   = DEFAULT_ALIGNMENT;
  }

  /** @param style for new BorderPanel */
  public BorderPanel(int style) {
    this();
    // set thickness appropriate to this style
    int thickness;
    switch (style) {
      case SOLID: thickness = DEFAULT_SOLID_THICKNESS; break;
      case RAISED: thickness = DEFAULT_RAISED_THICKNESS; break;
      case LOWERED: thickness = DEFAULT_LOWERED_THICKNESS; break;
      case IN: thickness = DEFAULT_IN_THICKNESS; break;
      case OUT: thickness = DEFAULT_OUT_THICKNESS; break;
      default: thickness = DEFAULT_THICKNESS;
    }
    this.style = style;
    this.thickness = thickness;
  }

  /** @param text  caption for new BorderPanel */
  public BorderPanel(String text) {
    this();
    this.text = text;
  }

  /** @param color  for new BorderPanel */
  public BorderPanel(Color color) {
    this();
    this.color = color;
    this.thickness = thickness;
  }

  /** returns insets */
  public Insets insets() {
    int adjustment = 0;
    // adjust for text string
    if (style == IN || style == OUT) {
      if (text != null && text.length() > 0) {
        try {
          // set font and get info
          int height = getGraphics().getFontMetrics(font).getHeight();
          if (height > thickness) {
            adjustment = height - thickness;
          }
        }
        catch (Exception e) {
        }
      }
    }
    // return appropriate insets
    int dist = thickness + gap;
    return new Insets(dist + adjustment, dist, dist, dist);
  }

  public BorderPanel setStyle(int style) {
    // set the style and re-layout the panel
    // set thickness appropriate to this style
    int thickness;
    switch (style) {
      case SOLID: thickness = DEFAULT_SOLID_THICKNESS; break;
      case RAISED: thickness = DEFAULT_RAISED_THICKNESS; break;
      case LOWERED: thickness = DEFAULT_LOWERED_THICKNESS; break;
      case IN: thickness = DEFAULT_IN_THICKNESS; break;
      case OUT: thickness = DEFAULT_OUT_THICKNESS; break;
      default: thickness = DEFAULT_THICKNESS;
    }
    this.style = style;
    this.thickness = thickness;
    layout();
    repaint();
    return this;
  }

  public int getStyle() {
    return style;
  }

  public BorderPanel setThickness(int thickness) {
    if (thickness > 0) {
      this.thickness = thickness;
      layout();
      repaint();
    }
    return this;
  }

  public int getThickness() {
    return thickness;
  }

  public BorderPanel setGap(int gap) {
    if (gap > -1) {
      this.gap = gap;
      layout();
      repaint();
    }
    return this;
  }

  public int getGap() {
    return gap;
  }

  /**
  *   sets color for SOLID or text color for etched
  *
  *   @param color 
  */
  public BorderPanel setColor(Color color) {
    this.color = color;
    if (style == SOLID || style == IN || style == OUT) {
      repaint();
    }
    return this;
  }

  /** gets color for SOLID or text color for etched */
  public Color getColor() {
    return color;
  }

  /** sets font for etched */
  public BorderPanel setTextFont(Font font) {
    // set font
    if (font != null) {
      this.font = font;
      if (style == IN || style == OUT) {
        layout();
        repaint();
      }
    }
    return this;
  }

  /** gets font if etched */
  public Font getTextFont() {
    return font;
  }

  /** sets text if etched */
  public BorderPanel setText(String text) {
    this.text = text;
    if (style == IN || style == OUT) {
      layout();
      repaint();
    }
    return this;
  }

  /** gets text if etched */
  public String getText() {
    return text;
  }

  /**
  *   sets alignment if etched 
  *
  *   @param alignment The new alignment.
  */
  public BorderPanel setAlignment(int alignment) {
    this.alignment = alignment;
    if (style == IN || style == OUT) {
      layout();
      repaint();
    }
    return this;
  }

  public int getAlignment() {
    return alignment;
  }

  public void paint(Graphics g) {
    // get current dimensions
    Dimension size = size();
    int width = size.width;
    int height = size.height;
    // set colors
    Color light = getBackground().brighter().brighter();
    Color dark = getBackground().darker().darker();
    // Draw border
    switch (style) {
      case RAISED:    // 3D Border (in or out)
      case LOWERED:
        Color topleft = null;
        Color bottomright = null;
        // set colors
        if (style == RAISED) {
          topleft = light;
          bottomright = dark;
        }
        else {
          topleft = dark;
          bottomright = light;
        }
        // draw border
        g.setColor(topleft);
        for (int i = 0; i < thickness; i++) {
          g.drawLine(i, i, width - i - 2, i);
          g.drawLine(i, i + 1, i, height - i - 1);
        }
        g.setColor(bottomright);
        for (int i = 0; i < thickness; i++) {
          g.drawLine(i + 1, height - i - 1, width - i - 1, height - i - 1);
          g.drawLine(width - i - 1, i, width - i - 1, height - i - 2);
        }
        break;
      case IN:    // Etched Border (in or out)
      case OUT:
        int adjust1 = 0;
        int adjust2 = 0;
        // set font and get info
        Font oldfont = g.getFont();
        g.setFont(font);
        FontMetrics fm = g.getFontMetrics();
        int ascent = fm.getAscent();
        // set adjustment
        if (style == IN) {
          adjust1 = 1;
        }
        else {
          adjust2 = 1;
        }
        // Calculate adjustment for text
        int adjustment = 0;
        if (text != null && text.length() > 0) {
          if (ascent > thickness) {
            adjustment = (ascent - thickness) / 2;
          }
        }
        // always draw light rectangle first
        int x = thickness / 2;
        int y = thickness / 2 + adjustment;
        int w = width - thickness - 1;
        int h = height - thickness - 1 - adjustment;
        // draw rectangles
        g.setColor(light);
        g.drawRect(x + adjust1, y + adjust1, w, h);
        g.setColor(dark);
        g.drawRect(x + adjust2, y + adjust2, w, h);
        // draw text, if applicable
        if (text != null && text.length() > 0) {
          // calculate drawing area
          int fontheight = fm.getHeight();
          int strwidth = fm.stringWidth(text);
          int textwidth = width - 2 * (thickness + 5);
          if (strwidth > textwidth) {
            strwidth = textwidth;
          }
          // calculate offset for alignment
          int offset;
          switch (alignment) {
            case CENTER:
              offset = (width - strwidth) / 2;
              break;
            case RIGHT:
              offset = width - strwidth - thickness - 5;
              break;
            case LEFT:
            default: // assume left alignment if invalid
              offset = thickness + 5;
              break;
          }
          // clear drawing area and set clipping region
          g.clearRect(offset - 5, 0, strwidth  + 10, fontheight);
          g.clipRect(offset, 0, strwidth, fontheight);
          // draw text
          g.setColor(color);
          g.drawString(text, offset, ascent);
          // restore old clipping area
          g.clipRect(0, 0, width, height);
        }
        g.setFont(oldfont);
        break;
      case SOLID:
      default: // assume SOLID
        g.setColor(color);
        for (int i = 0; i < thickness; i++) {
          g.drawRect(i, i, width - 2 * i - 1, height - 2 * i - 1);
        }
    }
  }

  public String toString() {
    StringBuffer str = new StringBuffer("BorderPanel[");
    // style
    str.append("style=");
    switch (style) {
      case SOLID: str.append("SOLID"); break;
      case RAISED: str.append("RAISED"); break;
      case LOWERED: str.append("LOWERED"); break;
      case IN: str.append("IN"); break;
      case OUT: str.append("OUT"); break;
      default: str.append("unknown");
    }
    str.append(",");
    // thickness
    str.append("thickness=");
    str.append(thickness);
    str.append(",");
    // gap
    str.append("gap=");
    str.append(gap);
    str.append(",");
    // color
    str.append(color);
    str.append(",");
    // font
    str.append(font);
    str.append(",");
    // text
    str.append("text=");
    str.append(text);
    str.append(",");
    // alignment
    str.append("alignment=");
    switch (alignment) {
      case LEFT: str.append("LEFT"); break;
      case CENTER: str.append("CENTER"); break;
      case RIGHT: str.append("RIGHT"); break;
      default: str.append("unknown");
    }
    str.append("]");
    return str.toString();
  }

}

