/*
                            XSW Object Matching

	Functions:

	int DBIsObjectGarbage(long object_num)
	int DBIsObjectNameIndex(char *name)
	double DBGetObjectVisibility(long object_num)
	double DBGetObjectVisibilityPtr(xsw_object_struct *obj_ptr)

	long MatchWeaponsLock(
		long ref_obj,
		long start_obj,
		double range
	)
	long MatchIntercept(long object_num, char *arg)
	long MatchInterceptByNumber(long ref_obj, long tar_obj)
	long MatchObjectByName(char *name, int type)
	int MatchIFF(long ref_obj, long tar_obj)
	int MatchIFFPtr(
	        xsw_object_struct *ref_obj_ptr,
	        xsw_object_struct *tar_obj_ptr
	)

	void DBSetupGarbageObject()

	---

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "../include/string.h"
#include "../include/reality.h"
#include "../include/mathutil.h"
#include "../include/isrefs.h"
#include "../include/dbmatch.h"


xsw_object_struct garbage_object;


/*
 *      Checks if object is allocated and garbage.
 *
 *      Returns true if not allocated or garbage.
 */
int DBIsObjectGarbage(long object_num)
{
        if((object_num < 0) ||
           (object_num >= total_objects) ||
           (xsw_object == NULL)
        )
            return(-1);
        else if(xsw_object[object_num] == NULL)
            return(-1);
        else if(xsw_object[object_num]->type <= XSW_OBJ_TYPE_GARBAGE)
            return(1);
        else
            return(0);
}

/*
 *	Checks if the object name is in index notation (returns true)
 *	or a regular name (returns false).
 */
int DBIsObjectNameIndex(char *name)
{
	if(name == NULL)
	    return(0);
	else if(*name == '#')
	    return(1);
	else
	    return(0);
}

/*           
 *      Returns the object's visibility.
 */         
double DBGetObjectVisibility(long object_num)
{
        if(DBIsObjectGarbage(object_num))
            return(0.00);
        else 
            return(DBGetObjectVisibilityPtr(xsw_object[object_num]));
}

double DBGetObjectVisibilityPtr(xsw_object_struct *obj_ptr)
{
        if(obj_ptr == NULL)
            return(0.00);

        if(obj_ptr->type == XSW_OBJ_TYPE_AREA)
            return(obj_ptr->cur_visibility);
	else if(obj_ptr->loc_type == XSW_LOC_TYPE_NEBULA)
	    return(obj_ptr->cur_visibility * VISIBILITY_NEBULA);
        else
            return(obj_ptr->cur_visibility);
}


/*
 *      returns the next XSW Object number that is within
 *      range range of object ref_object.   ref_object should be the object
 *      that is doing the weapons lock, and may never be -1.
 *      start_object maybe -1 or any object number less than total_objects.
 *      range should be the the maximum scanner range of ref_object.
 *      Returns the next XSW Object number of a valid (nongarbage or nonerror)
 *      object withing range range of ref_object starting from start_object,
 *      or returns -1 if no match can be made.
 */
long MatchWeaponsLock(long ref_obj, long start_obj, double range) 
{
	long i, f;
	xsw_object_struct *obj_ptr, *ref_obj_ptr;


	if(DBIsObjectGarbage(ref_obj))
	    return(-1);
	else
	    ref_obj_ptr = xsw_object[ref_obj];

	/* Sanitize start object (can be -1). */
        if(start_obj < -1)
            start_obj = -1;

	/* range must be positive. */
	if(range < 0)
	    range = 1;


	/* Begin search, note that start_obj could be -1. */
	for(i = start_obj + 1, f = -1;
            i < total_objects;
            i++
	)
	{
	    obj_ptr = xsw_object[i];

	    if(obj_ptr == NULL)
		continue;
	    if(obj_ptr->type <= XSW_OBJ_TYPE_GARBAGE)
		continue;

/*
	    if((obj_ptr->type == XSW_OBJ_TYPE_WEAPON) ||
               (obj_ptr->type == XSW_OBJ_TYPE_STREAMWEAPON) ||
               (obj_ptr->type == XSW_OBJ_TYPE_SPHEREWEAPON) ||
               (obj_ptr->type == XSW_OBJ_TYPE_AREA)
	    )
		continue;
 */

	    /* Don't match referance object. */
	    if(obj_ptr == ref_obj_ptr)
		continue;

	    /* See if object is in range. */
	    if(Mu3DInRangePtr(obj_ptr, ref_obj_ptr,
		    range * DBGetObjectVisibilityPtr(obj_ptr)
	       )
	    )
	    {
	        /* Checks passed, this is the object we want. */
	        f = i;
	        break;
	    }
	}


	return(f);
}



/*
 *	Tries to match object specified in arg.  If arg
 *	is "" or "#off" then -1 is returned.   If the object specified in arg
 *	is found and is of type XSW_OBJ_TYPE_STATIC then that object's number
 *	is returned.  If the object is found but not of type static, then it 
 *	is checked to see if it is within scanner range of object_num and if 
 *	it is, then it's number is returned or of not in range then -1 is    
 *	returned.
 */
long MatchIntercept(long object_num, char *arg)
{
	char name[XSW_OBJ_NAME_MAX];
	long target_obj;
	xsw_object_struct *obj_ptr;
	xsw_object_struct *tar_obj_ptr;


	if(arg == NULL)
	    return(-1);

	if(DBIsObjectGarbage(object_num))
            return(-1);
	else
	    obj_ptr = xsw_object[object_num];


	/* Sanitize copy over name to be searched. */
	strncpy(name, arg, XSW_OBJ_NAME_MAX);
	name[XSW_OBJ_NAME_MAX - 1] = '\0';
	StringStripSpaces(name);


	/* Turn intercept off? */
	if(!strcasecmp(name, "#off") ||
	   (name[0] == '\0')
	)
	    return(-1);


	/* Match target_obj. */
	target_obj = MatchObjectByName(name, -1);
	if(DBIsObjectGarbage(target_obj))
	    return(-1);
	else
            tar_obj_ptr = xsw_object[target_obj];


	/* Cannot intercept itself. */
	if(target_obj == object_num)
	    return(-1);


	/* Following types of objects may not be intercepted. */
	if((tar_obj_ptr->type == XSW_OBJ_TYPE_STREAMWEAPON) ||
           (tar_obj_ptr->type == XSW_OBJ_TYPE_SPHEREWEAPON)
	)
	    return(-1);

        /* Following objects don't need range check. */
        if((tar_obj_ptr->type == XSW_OBJ_TYPE_STATIC) ||
	   (tar_obj_ptr->type == XSW_OBJ_TYPE_HOME) ||
           (tar_obj_ptr->type == XSW_OBJ_TYPE_AREA)
	)
            return(target_obj);


        /* Is target_obj within scanner range of object_num? */
        if(Mu3DInRangePtr(obj_ptr, tar_obj_ptr,
                obj_ptr->scanner_range *
                DBGetObjectVisibilityPtr(tar_obj_ptr)
            )
        )
	    return(target_obj);
	else
	    return(-1);
}


/*
 *	Checks if tar_obj is within intercept range of ref_obj,
 *	and that both are valid.  Returns tar_obj if everything
 *	checks out okay or -1 if not.
 */
long MatchInterceptByNumber(long ref_obj, long tar_obj)
{
        xsw_object_struct *obj_ptr;
        xsw_object_struct *tar_obj_ptr;


        /* Error checks. */
        if(DBIsObjectGarbage(ref_obj))
            return(-1);
        else
            obj_ptr = xsw_object[ref_obj];

        if(DBIsObjectGarbage(tar_obj))
            return(-1);
        else
            tar_obj_ptr = xsw_object[tar_obj];


        /* ref_obj and tar_obj cannot be the same. */
        if(ref_obj == tar_obj)
            return(-1);


        /* Skip these types of objects. */
        if((tar_obj_ptr->type == XSW_OBJ_TYPE_WEAPON) ||
           (tar_obj_ptr->type == XSW_OBJ_TYPE_STREAMWEAPON) ||
           (tar_obj_ptr->type == XSW_OBJ_TYPE_SPHEREWEAPON)
        )
            return(-1);


        /* Following objects always matchable regardless of distance. */
        if((tar_obj_ptr->type == XSW_OBJ_TYPE_STATIC) ||
	   (tar_obj_ptr->type == XSW_OBJ_TYPE_HOME) ||
	   (tar_obj_ptr->type == XSW_OBJ_TYPE_AREA)
	)
            return(tar_obj);


        /* If none of the above, then do scanner range check. */
        if(Mu3DInRangePtr(obj_ptr, tar_obj_ptr,
                obj_ptr->scanner_range *
                DBGetObjectVisibilityPtr(tar_obj_ptr)
            )
        )
	    return(tar_obj);
	else
            return(-1);
}


/*
 *	Matches object by name, return its number or -1 on error or
 *	no match.
 *
 *	The search can be narrowed down to a certain type of object
 *	or -1 for all types.
 */
long MatchObjectByName(char *name, int type)
{
	long object_num;
	xsw_object_struct **obj_ptr;


	/* Name cannot be NULL or contain no characters. */
	if(name == NULL)
	    return(-1);
	if(name[0] == '\0')
	    return(-1);


	/* Skip leading spaces in name. */
	while(isblank(*name))
	    name++;

	/* Number matching? */
	if((*name == '#') ||
           (isdigit(*name))
	)
	{
	    /* Skip leading '#' characters. */
	    while(*name == '#')
		name++;

	    /* Get object number. */
	    object_num = atol(name);

	    /* Check if object_num is valid. */
	    if((object_num < 0) || (object_num >= total_objects))
		return(-1);
	    if(xsw_object[object_num] == NULL)
		return(-1);
	    if(xsw_object[object_num]->type <= XSW_OBJ_TYPE_GARBAGE)
		return(-1);

            /* Match all types of objects? */
	    if(type == -1)
		return(object_num);

	    /* Match specific type. */
	    if(xsw_object[object_num]->type == type)
		return(object_num);
	    else
		return(-1);
	}


	/* Go through XSW objects list. */
	for(object_num = 0, obj_ptr = xsw_object;
            object_num < total_objects;
            object_num++, obj_ptr++
	)
	{
	    if(*obj_ptr == NULL) continue;
	    if((*obj_ptr)->type <= XSW_OBJ_TYPE_GARBAGE) continue;

	    /* Object must characters in its name. */
	    if((*obj_ptr)->name[0] == '\0') continue;

	    /* Match all types? */
	    if(type == -1)
	    {
		/* Partial name match. */
		if(strstr((*obj_ptr)->name, name) == NULL)
		    continue;

		/* Got match. */
		return(object_num);
	    }
	    /* Match specific type. */
	    else
	    {
		/* Match type first. */
		if((*obj_ptr)->type != type)
		    continue;

                /* Partial name match. */
                if(strstr((*obj_ptr)->name, name) == NULL)
                    continue;

                /* Got match. */
                return(object_num);
	    }
	}


	/* No match. */
	return(-1);
}

/*
 *	Returns IFF of ref_obj to tar_obj.
 */
int MatchIFF(long ref_obj, long tar_obj)
{
	if(DBIsObjectGarbage(ref_obj))
	    return(IFF_UNKNOWN);
	else if(DBIsObjectGarbage(tar_obj))
	    return(IFF_UNKNOWN);
	else
	    return(MatchIFFPtr(
		xsw_object[ref_obj],
		xsw_object[tar_obj]
	    ));
}

/*
 *	Returns IFF of ref_obj_ptr to tar_obj_ptr
 */
int MatchIFFPtr(
	xsw_object_struct *ref_obj_ptr,
	xsw_object_struct *tar_obj_ptr
)
{
	char *strptr_ref, *strptr_tar;


	if((ref_obj_ptr == NULL) ||
           (tar_obj_ptr == NULL)
	)
	    return(IFF_UNKNOWN);

	strptr_ref = ref_obj_ptr->empire;
	strptr_tar = tar_obj_ptr->empire;

	if((strptr_ref == NULL) ||
           (strptr_tar == NULL) ||
           (*strptr_ref == '\0') ||
	   (*strptr_tar == '\0')
	)
	    return(IFF_UNKNOWN);

	if(strcasecmp(strptr_ref, strptr_tar))
	    return(IFF_UNKNOWN);
	else
	    return(IFF_FRIENDLY);
}

/*
 *	Sets up the global garbage object.
 */
void DBSetupGarbageObject()
{
	xsw_object_struct *obj_ptr;


	obj_ptr = &garbage_object;

	obj_ptr->client_options = 0;
	obj_ptr->server_options = 0;

        memset(obj_ptr->name, '\0', XSW_OBJ_NAME_MAX);
        memset(obj_ptr->empire, '\0', XSW_OBJ_PASSWORD_MAX); 
	memset(obj_ptr->password, '\0', XSW_OBJ_EMPIRE_MAX);

        obj_ptr->last_updated = 0;

        obj_ptr->type = XSW_OBJ_TYPE_GARBAGE;
	obj_ptr->loc_type = XSW_LOC_TYPE_SPACE;
        obj_ptr->imageset = ISREF_DEFAULT;
        obj_ptr->owner = -1;
        obj_ptr->size = 1;
        obj_ptr->locked_on = -1;  
        obj_ptr->intercepting_object = -1;

        obj_ptr->scanner_range = 8.0;
        obj_ptr->sect_x = 0;
        obj_ptr->sect_y = 0;
        obj_ptr->sect_z = 0;
        obj_ptr->x = 0;
        obj_ptr->y = 0;
        obj_ptr->z = 0;
        obj_ptr->heading = 0;
        obj_ptr->pitch = 0;
        obj_ptr->bank = 0;
	memset(&obj_ptr->attitude_vector_compoent, 0x00, sizeof(xsw_vector_compoent_struct));
        obj_ptr->velocity = 0;
        obj_ptr->velocity_max = 0;
        obj_ptr->velocity_heading = 0;
        obj_ptr->velocity_pitch = 0;
        obj_ptr->velocity_bank = 0;
        memset(&obj_ptr->momentum_vector_compoent, 0x00, sizeof(xsw_vector_compoent_struct));
        obj_ptr->thrust_dir = 3.1415927;
        obj_ptr->thrust = 0;   
        obj_ptr->thrust_power = 0;   
        obj_ptr->thrust_power = 0;
        obj_ptr->throttle = 0;
	obj_ptr->engine_state = ENGINE_STATE_ON;
        obj_ptr->turnrate = 0.0003;
        obj_ptr->hp = 1;
        obj_ptr->hp_max = 1;
        obj_ptr->power = 0;
        obj_ptr->power_max = 0;
        obj_ptr->power_purity = 1.0;
        obj_ptr->core_efficency = 0;
        obj_ptr->antimatter = 0;
        obj_ptr->antimatter_max = 0;
        obj_ptr->shield_state = SHIELD_STATE_NONE;
        obj_ptr->shield_frequency = XSW_DEF_SHIELD_FREQ;
        obj_ptr->selected_weapon = -1;
        obj_ptr->total_weapons = 0;
        obj_ptr->birth_time = 0;
        obj_ptr->lifespan = -1;
        obj_ptr->cloak_state = CLOAK_STATE_NONE;   
        obj_ptr->cloak_strength = 0.0;
        obj_ptr->shield_visibility = 0.0;
        obj_ptr->visibility = 1.0;
	obj_ptr->cur_visibility = 1.0;
        obj_ptr->damage_control = DMGCTL_STATE_OFF;
	obj_ptr->com_channel = XSW_DEF_COM_CHANNEL;
	obj_ptr->ai_flags = 0;

	obj_ptr->tractored_object = NULL;
	obj_ptr->total_tractored_objects = 0;

        obj_ptr->permission.uid = DEFAULT_UID;
        obj_ptr->permission.gid = DEFAULT_GID;

        obj_ptr->animation.interval = 500;
        obj_ptr->animation.last_interval = 0;
        obj_ptr->animation.current_frame = 0;
        obj_ptr->animation.total_frames = 1;
        obj_ptr->animation.cycle_count = 0;
        obj_ptr->animation.cycle_times = -1;	/* Loop infinatly. */

        obj_ptr->weapons = NULL;
	obj_ptr->score = NULL;
	obj_ptr->eco = NULL;


	return;
}
