## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import os, re, sys, types, Tkinter

# PySol imports
from mfxutil import destruct, Struct, KwStruct                      #bundle#
from gamedb import GAME_DB, GI                                      #bundle#

# Toolkit imports
from tkwidget import _ToplevelDialog, MfxDialog                     #bundle#
from tktree import Leaf, Node, TreeInCanvas                         #bundle#


# /***********************************************************************
# //
# ************************************************************************/

class SelectGameLeaf(Leaf):
    def __init__(self, tree, parent, text, gameid, selected=0):
        self.gameid = gameid
        Leaf.__init__(self, tree, parent, text, selected)

    def drawSymbol(self, x, y):
        return self.tree.canvas.create_image(
            x, y, image=self.tree.data.img[2+(self.gameid<=0)], anchor="nw")

    def updateSymbol(self):
        pass


class SelectGameNode(Node):
    def __init__(self, tree, text, select, expanded=0, parent=None):
        Node.__init__(self, tree, parent, text, expanded)
        self.select = select

    def getContents(self):
        if self.subnodes is not None:
            return self.subnodes
        ##print self.whoami()
        if type(self.select) == types.TupleType or type(self.select) == types.ListType:
            return self.select
        contents = []
        for id in self.tree.data.games:
            gi = GAME_DB.get(id)
            if self.select(gi):
                node = SelectGameLeaf(self.tree, self, gi.name, gi.id)
                contents.append(node)
        if not contents:
            return self.tree.data.no_games
        return contents


    def drawSymbol(self, x, y):
        return self.tree.canvas.create_image(
            x, y, image=self.tree.data.img[self.expanded], anchor="nw")

    def updateSymbol(self):
        pass


# /***********************************************************************
# //
# ************************************************************************/

class SelectGameData:
    def __init__(self):
        self.games = GAME_DB.getGamesIdSortedByName()
        self.no_games = [ SelectGameLeaf(None, None, "(no games)", -1), ]
        g_by_type = []
        for name, select_func in GI.SELECT_GAME_BY_TYPE:
            name = re.sub(r'&', '', name)
            g_by_type.append(SelectGameNode(None, name, select_func))
        self.rootnodes = (
            SelectGameNode(None, "All Games", lambda g: 1, expanded=0),
            SelectGameNode(None, "Popular Games", lambda g, gi=GI: g.id in GI.POPULAR_GAMES, expanded=0),
            SelectGameNode(None, "by Game Type", tuple(g_by_type), expanded=1),
            SelectGameNode(None, "by Number of Decks", (
                SelectGameNode(None, "1 deck games",   lambda g: g.decks == 1),
                SelectGameNode(None, "2 deck games",   lambda g: g.decks == 2),
                SelectGameNode(None, "3 deck games",   lambda g: g.decks == 3),
                SelectGameNode(None, "4 deck games",   lambda g: g.decks == 4),
            )),
            SelectGameNode(None, "by Compatibility", (
                SelectGameNode(None, "Atari ST Patience", lambda g: g.id in GI.COMPAT_ATARI_GAMES),
                SelectGameNode(None, "Gnome AisleRiot", lambda g: g.id in GI.COMPAT_GNOME_GAMES),
                SelectGameNode(None, "KDE Patience", lambda g: g.id in GI.COMPAT_KPAT_GAMES),
                SelectGameNode(None, "xpat2", lambda g: g.id in GI.COMPAT_XPAT2_GAMES),
            )),
            SelectGameNode(None, "Other Categories", (
                SelectGameNode(None, "Games for Children (very easy)", lambda g, gi=GI: g.id in GI.CHILDREN_GAMES),
                SelectGameNode(None, "Open Games (all cards visible)", lambda g, gi=GI: g.id in GI.OPEN_GAMES),
            )),
        )
#%ifndef BUNDLE
        # FIXME: provide info in registerGame()
        dummy = (
            SelectGameNode(None, "by Game Time", (
                SelectGameNode(None, "Quick",          lambda g: g.game_time == GI.TIME_QUICK),
                SelectGameNode(None, "Medium",         lambda g: g.game_time == GI.TIME_MEDIUM),
                SelectGameNode(None, "Long",           lambda g: g.game_time == GI.TIME_LONG),
            )),
            SelectGameNode(None, "by Skill Level", (
                SelectGameNode(None, "Chance only",    lambda g: g.game_time == GI.SOC_CHANCE_ONLY),
                SelectGameNode(None, "Mostly chance",  lambda g: g.game_time == GI.SOC_MOSTLY_CHANCE),
                SelectGameNode(None, "Moderate",       lambda g: g.game_time == GI.SOC_MODERATE),
                SelectGameNode(None, "Mostly skill",   lambda g: g.game_time == GI.SOC_MOSTLY_SKILL),
                SelectGameNode(None, "Difficult",      lambda g: g.game_time == GI.SOC_DIFFICULT),
            )),
            SelectGameNode(None, "by Chance of Winning", (
                SelectGameNode(None, "Very low"   ,    lambda g: g.chance_of_winning == GI.COW_VERY_LOW),
                SelectGameNode(None, "Low",            lambda g: g.chance_of_winning == GI.COW_LOW),
                SelectGameNode(None, "Medium",         lambda g: g.chance_of_winning == GI.COW_MEDIUM),
                SelectGameNode(None, "High",           lambda g: g.chance_of_winning == GI.COW_HIGH),
                SelectGameNode(None, "Very high",      lambda g: g.chance_of_winning == GI.COW_VERY_HIGH),
            )),
            SelectGameNode(None, "by Difficulty Level", (
                SelectGameNode(None, "Easy",           lambda g: g.game_time == GI.GD_EASY),
                SelectGameNode(None, "Medium",         lambda g: g.game_time == GI.GD_MEDIUM),
                SelectGameNode(None, "Hard",           lambda g: g.game_time == GI.GD_HARD),
            )),
        )
#%endif
        self.img = (Tkinter.PhotoImage(data="""
R0lGODlhEAAOAPIFAAAAAICAgMDAwP//AP///4AAAAAAAAAAACH5BAEAAAUALAAAAAAQAA4AAAOL
WLrcGxA6FoYYYoRZwhCDMAhDFCkBoa6sGgBFQAzCIAzCIAzCEACFAEEwEAwEA8FAMBAEAIUAYSAY
CAaCgWAgGAQAhQBBMBAMBAPBQDAQBACFAGEgGAgGgoFgIBgEAAUBBAIDAgMCAwIDAgMCAQAFAQQD
AgMCAwIDAgMCAwEABSaiogAKAKeoqakFCQA7"""), Tkinter.PhotoImage(data="""
R0lGODlhEAAOAPIFAAAAAICAgMDAwP//AP///4AAAAAAAAAAACH5BAEAAAUALAAAAAAQAA4AAAN3
WLrcHBA6Foi1YZZAxBCDQESREhCDMAiDcFkBUASEMAiDMAiDMAgBAGlIGgQAgZeSEAAIAoAAQTAQ
DAQDwUAwAEAAhQBBMBAMBAPBQBAABACFAGEgGAgGgoFgIAAEAAoBBAMCAwIDAgMCAwEAAApERI4L
jpWWlgkAOw=="""), Tkinter.PhotoImage(data="""
R0lGODdhEAAOAPIAAAAAAAAAgICAgMDAwP///wAAAAAAAAAAACwAAAAAEAAOAAADTii63DowyiiA
GCHrnQUQAxcQAAEQgAAIg+MCwkDMdD0LgDDUQG8LAMGg1gPYBADBgFbs1QQAwYDWBNQEAMHABrAR
BADBwOsVAFzoqlqdAAA7"""), Tkinter.PhotoImage(data="""
R0lGODdhEAAOAPIAAAAAAAAAgICAgMDAwP8AAP///wAAAAAAACwAAAAAEAAOAAADVCi63DowyiiA
GCHrnQUQAxcUQAEUgAAIg+MCwlDMdD0LgDDQBE3UAoBgUCMUCDYBQDCwEWwFAUAwqBEKBJsAIBjQ
CDRCTQAQDKBQAcDFBrjf8Lg7AQA7"""))


# /***********************************************************************
# //
# ************************************************************************/

class SelectGameTree(TreeInCanvas):
    data = None

    def __init__(self, dialog, parent, gameid, font, width=-1, height=-1):
        self.dialog = dialog
        self.gameid = gameid
        if self.data is None:
            # update class variable
            self.__class__.data = SelectGameData()
        for node in self.data.rootnodes:
            node.tree = self
        #
        disty = 16
        if width < 0:
            width = 400
        if height < 0:
            height = 20 * disty
            if parent and parent.winfo_screenheight() >= 600:
                height = 25 * disty
            if parent and parent.winfo_screenheight() >= 800:
                height = 30 * disty
        TreeInCanvas.__init__(self, parent, self.data.rootnodes,
                              c_width=width, c_height=height)
        self.style.distx = 20
        self.style.disty = disty
        self.style.width = 16
        self.style.height = 14
        if font:
            self.style.font = font
        self.draw()

    def getContents(self, node):
        return node.getContents()

    def singleClick(self, event=None):
        node = self.findNode()
        if node is None:
            return
        ##print "Pressed", node
        if isinstance(node, Leaf):
            if node.gameid > 0:
                self.gameid = node.gameid
                self.dialog.mDone(0)
            #node.selected = not node.selected
            ##node.updateSymbol()
        else:
            node.expanded = not node.expanded
            ##node.updateSymbol()
            self.redraw()


# /***********************************************************************
# //
# ************************************************************************/

class SelectGameDialog(MfxDialog):
    def __init__(self, parent, title, gameid, **kw):
        kw = self.initKw(kw)
        _ToplevelDialog.__init__(self, parent, title, kw.resizable)
        top_frame, bottom_frame = self.createFrames(kw)
        self.createBitmaps(top_frame, kw)
        #
        self.top.wm_minsize(200, 200)
        self.gameid = gameid
        self.num = kw.default
        self.tree = SelectGameTree(self, top_frame, gameid, font=kw.font)
        self.tree.sc.frame.pack(fill=Tkinter.BOTH, expand=1, padx=kw.padx, pady=kw.pady)
        #
        focus = self.createButtons(bottom_frame, kw)
        focus = self.tree.sc.frame
        self.mainloop(focus, kw.timeout)

    def destroy(self):
        self.tree.sc.unbind()
        MfxDialog.destroy(self)

    def initKw(self, kw):
        kw = KwStruct(kw,
            strings=("Cancel",), default=-1,
            separatorwidth=2, resizable=1,
            font=None,
            padx=10, pady=10,
            buttonpadx=10, buttonpady=5,
            d_width=600, d_height=600,
        )
        return MfxDialog.initKw(self, kw)

    def mDone(self, num):
        self.gameid = self.tree.gameid
        MfxDialog.mDone(self, num)

