## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys

# PySol imports
if sys.modules.has_key("pysoltk"):                                  #bundle#
    from gamedb import registerGame, GameInfo, GI                   #bundle#
    from util import *                                              #bundle#
    from stack import *                                             #bundle#
    from game import Game                                           #bundle#
    from layout import Layout                                       #bundle#
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint #bundle#
    from hint import KlondikeType_Hint                              #bundle#


# /***********************************************************************
# // Gypsy
# ************************************************************************/

class Gypsy(Game):
    Layout_Method = Layout.gypsyLayout
    Talon_Class = DealRowTalonStack
    Foundation_Class = SS_FoundationStack
    RowStack_Class = AC_RowStack
    Hint_Class = KlondikeType_Hint

    def createGame(self, **layout):
        # create layout
        l, s = Layout(self), self.s
        kwdefault(layout, rows=8, waste=0, texts=1)
        apply(self.Layout_Method, (l,), layout)
        self.setSize(l.size[0], l.size[1])
        # create stacks
        s.talon = self.Talon_Class(l.s.talon.x, l.s.talon.y, self)
        for r in l.s.foundations:
            s.foundations.append(self.Foundation_Class(r.x, r.y, self, suit=r.suit))
        for r in l.s.rows:
            s.rows.append(self.RowStack_Class(r.x, r.y, self))
        # default
        l.defaultAll()

    def startGame(self):
        for i in range(2):
            self.s.talon.dealRow(flip=0, frames=0)
        self.s.talon.dealRow()

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.color != card2.color and
                (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank))


# /***********************************************************************
# // Giant
# ************************************************************************/

class Giant_Foundation(SS_FoundationStack):
    def canMovePile(self, cards):
        if not SS_FoundationStack.canMovePile(self, cards):
            return 0
        # can only move cards if the Talon is empty
        return len(self.game.s.talon.cards) == 0


class Giant(Gypsy):
    Foundation_Class = Giant_Foundation

    def startGame(self):
        self.s.talon.dealRow()


# /***********************************************************************
# // Irmgard
# ************************************************************************/

class Irmgard_Talon(TalonStack):
    # A single click deals 9 (or 7) new cards to the RowStacks.
    def dealCards(self):
        if self.cards:
            if len(self.cards) > 7:
                c = self.dealRow()
            else:
                c = self.dealRow(self.game.s.rows[1:8])
            return c
        return 0


class Irmgard(Gypsy):
    Layout_Method = Layout.harpLayout
    Talon_Class = Irmgard_Talon

    def createGame(self):
        Gypsy.createGame(self, rows=9, playcards=19)

    def startGame(self):
        r = self.s.rows
        for i in range(1, 5):
            self.s.talon.dealRow(rows=r[i:len(r)-i], flip=0, frames=0)
        self.s.talon.dealRow()


# /***********************************************************************
# // Miss Milligan
# ************************************************************************/

class MissMilligan_Reserve(AC_RowStack):
    def acceptsPile(self, from_stack, cards):
        if not AC_RowStack.acceptsPile(self, from_stack, cards):
            return 0
        # note that this reserve stack accepts sequences
        # the stack and the Talon must be empty
        return len(self.cards) == 0 and len(self.game.s.talon.cards) == 0

    def getBottomImage(self):
        return self.game.app.images.getReserveBottom()


class MissMilligan(Gypsy):
    RowStack_Class = KingAC_RowStack
    ReserveStack_Class = MissMilligan_Reserve

    def createGame(self, reserves=1):
        # create layout
        l, s = Layout(self), self.s

        # set window
        self.setSize(l.XM + 9*l.XS, l.YM + (1+max(4, reserves))*l.YS)

        # create stacks
        x, y = l.XM, l.YM
        s.talon = self.Talon_Class(x, y, self)
        for i in range(8):
            x = x + l.XS
            s.foundations.append(self.Foundation_Class(x, y, self, i/2, max_move=0))
        x, y = l.XM, y + l.YS
        rx, ry = x + l.XS - l.XM/2, y - l.YM/2
        for i in range(reserves):
            s.reserves.append(self.ReserveStack_Class(x, y, self))
            y = y + l.YS
        if s.reserves:
            self.setRegion(s.reserves, (-999, ry, rx - 1, 999999))
        else:
            rx = -999
        x, y = l.XM, l.YM + l.YS
        for i in range(8):
            x = x + l.XS
            s.rows.append(self.RowStack_Class(x, y, self))
        self.setRegion(s.rows, (rx, ry, 999999, 999999))

        # define stack-groups
        l.defaultStackGroups()

    def startGame(self):
        self.s.talon.dealRow()


# /***********************************************************************
# // Milligan Harp
# ************************************************************************/

class MilliganHarp(MissMilligan):
    RowStack_Class = Yukon_AC_RowStack

    def createGame(self):
        MissMilligan.createGame(self, reserves=0)

    def startGame(self):
        for i in range(len(self.s.rows)):
            self.s.talon.dealRow(rows=self.s.rows[i+1:], flip=0, frames=0)
        self.s.talon.dealRow()


# /***********************************************************************
# // Milligan Cell
# ************************************************************************/

class MilliganCell(MissMilligan):
    ReserveStack_Class = ReserveStack

    def createGame(self):
        MissMilligan.createGame(self, reserves=4)

    def startGame(self):
        self.s.talon.dealRow()


# register the game
registerGame(GameInfo(1, Gypsy, "Gypsy",
                      GI.GT_GYPSY, 2, 0))
registerGame(GameInfo(65, Giant, "Giant",
                      GI.GT_GYPSY, 2, 0))
registerGame(GameInfo(3, Irmgard, "Irmgard",
                      GI.GT_GYPSY, 2, 0))
registerGame(GameInfo(62, MissMilligan, "Miss Milligan",
                      GI.GT_GYPSY, 2, 0))
registerGame(GameInfo(68, MilliganHarp, "Milligan Harp",
                      GI.GT_GYPSY, 2, 0))
registerGame(GameInfo(78, MilliganCell, "Milligan Cell",
                      GI.GT_GYPSY, 2, 0))

