## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys

# PySol imports
if sys.modules.has_key("pysoltk"):                                  #bundle#
    from gamedb import registerGame, GameInfo, GI                   #bundle#
    from util import *                                              #bundle#
    from stack import *                                             #bundle#
    from game import Game                                           #bundle#
    from layout import Layout                                       #bundle#
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint #bundle#


# /***********************************************************************
# //
# ************************************************************************/

class Fan_Hint(CautiousDefaultHint):
    # FIXME: demo is not too clever in this game

    def _preferHighRankMoves(self):
        return 1


# /***********************************************************************
# // Fan
# ************************************************************************/

class Fan(Game):
    Talon_Class = InvisibleTalonStack
    Foundation_Class = SS_FoundationStack
    RowStack_Class = KingSS_RowStack
    Hint_Class = Fan_Hint

    #
    # game layout
    #

    def createGame(self, rows=(5,5,5,3), playcards=9):
        # create layout
        l, s = Layout(self, XOFFSET=10), self.s

        # set window
        # (set size so that at least 9 cards are fully playable)
        w = max(2*l.XS, l.XS+(playcards-1)*l.XOFFSET)
        ##print 2*l.XS, w
        self.setSize(l.XM + max(rows)*w, l.YM + (1+len(rows))*l.YS)

        # create stacks
        x, y = l.XM + w, l.YM
        for i in range(4):
            s.foundations.append(self.Foundation_Class(x, y, self, i))
            x = x + w
        for i in range(len(rows)):
            x, y = l.XM, y + l.YS
            for j in range(rows[i]):
                stack = self.RowStack_Class(x, y, self, max_move=1, max_accept=1)
                stack.CARD_XOFFSET, stack.CARD_YOFFSET = l.XOFFSET, 0
                s.rows.append(stack)
                x = x + w
        x, y = self.width - l.XS, self.height - l.YS
        s.talon = self.Talon_Class(x, y, self)

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def startGame(self):
        for i in range(2):
            self.s.talon.dealRow(rows=self.s.rows[:17], frames=0)
        self.s.talon.dealRow()
        assert len(self.s.talon.cards) == 0

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.suit == card2.suit and
                (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank))

    def getHighlightPilesStacks(self):
        return ()


# /***********************************************************************
# // Scotch Patience
# ************************************************************************/

class ScotchPatience(Fan):
    Foundation_Class = AC_FoundationStack
    RowStack_Class = StackWrapper(RK_RowStack, base_rank=NO_RANK)


# /***********************************************************************
# // Shamrocks
# ************************************************************************/

class Shamrocks_RowStack(BasicRowStack):
    def acceptsPile(self, from_stack, cards):
        if not BasicRowStack.acceptsPile(self, from_stack, cards):
            return 0
        c1, c2 = self.cards[-1], cards[0]
        return c1.rank == (c2.rank + 1) % self.cap.mod or c2.rank == (c1.rank + 1) % self.cap.mod


class Shamrocks(Fan):
    RowStack_Class = StackWrapper(Shamrocks_RowStack, base_rank=NO_RANK, max_cards=3)

##    def shallHighlightMatch(self, stack1, card1, stack2, card2):
##        return (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank)


# /***********************************************************************
# // La Belle Lucie (Midnight Oil)
# ************************************************************************/

class LaBelleLucie_Talon(TalonStack):
    def canDealCards(self):
        return self.round != self.max_rounds

    def dealCards(self):
        # move all cards to the Talon, shuffle and redeal
        num_cards = 0
        assert len(self.cards) == 0
        for r in self.game.s.rows:
            if r.cards:
                num_cards = num_cards + len(r.cards)
                self.game.moveMove(len(r.cards), r, self, frames=0)
        assert len(self.cards) == num_cards
        if num_cards == 0:          # game already finished
            return 0
        # shuffle
        self.game.shuffleStackMove(self)
        # redeal
        self.game.nextRoundMove(self)
        n = num_cards
        k = n % len(self.game.s.rows)
        if k == 0: k = len(self.game.s.rows)
        while n > 0:
            for r in self.game.s.rows:
                n = n - 1
                frames = (0, 4)[k > n]
                self.game.moveMove(1, self, r, frames=frames)
                if n == 0:
                    break
        # done
        assert len(self.cards) == 0
        return num_cards


class LaBelleLucie(Fan):
    Talon_Class = StackWrapper(LaBelleLucie_Talon, max_rounds=3)
    RowStack_Class = StackWrapper(SS_RowStack, base_rank=NO_RANK)


# /***********************************************************************
# // Trefoil
# ************************************************************************/

class Trefoil(LaBelleLucie):
    def createGame(self):
        Fan.createGame(self, rows=(5,5,5,1))

    def _shuffleHook(self, cards):
        # move Aces to bottom of the Talon (i.e. last cards to be dealt)
        return self._shuffleHookMoveToBottom(cards, lambda c: (c.rank == 0, c.suit))

    def startGame(self):
        for i in range(2):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        self.s.talon.dealRow(rows=self.s.foundations)
        assert len(self.s.talon.cards) == 0


# /***********************************************************************
# // Beleaguered Castle
# ************************************************************************/

class BeleagueredCastle(Fan):

    #
    # game layout
    #

    def createGame(self, playcards=13):
        # create layout
        l, s = Layout(self, XOFFSET=12), self.s

        # set window
        # (set size so that at least 13 cards are fully playable)
        w = max(3*l.XS, l.XS+(playcards-1)*l.XOFFSET)
        x0 = l.XM
        x1 = x0 + w + 2*l.XM
        x2 = x1 + l.XS + 2*l.XM
        x3 = x2 + w + l.XM
        self.setSize(x3, l.YM + 4*l.YS)

        # create stacks
        x, y, = x1, l.YM
        for i in range(4):
            s.foundations.append(SS_FoundationStack(x, y, self, i, max_move=0))
            y = y + l.YS
        for x, y in ((x0, l.YM), (x2, l.YM)):
            for i in range(4):
                stack = RK_RowStack(x, y, self, max_move=1, max_accept=1)
                stack.CARD_XOFFSET, stack.CARD_YOFFSET = l.XOFFSET, 0
                s.rows.append(stack)
                y = y + l.YS
        x, y = self.width - l.XS, self.height - l.YS
        s.talon = InvisibleTalonStack(x, y, self)

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def _shuffleHook(self, cards):
        # move Aces to bottom of the Talon (i.e. last cards to be dealt)
        return self._shuffleHookMoveToBottom(cards, lambda c: (c.rank == 0, c.suit))

    def startGame(self):
        for i in range(5):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        self.s.talon.dealRow(rows=self.s.foundations)
        assert len(self.s.talon.cards) == 0


# /***********************************************************************
# // Black Hole
# ************************************************************************/

class BlackHole_Foundation(AbstractFoundationStack):
    def acceptsPile(self, from_stack, cards):
        if not AbstractFoundationStack.acceptsPile(self, from_stack, cards):
            return 0
        # check the rank
        if self.cards:
            r1, r2 = self.cards[-1].rank, cards[0].rank
            return (r1 + 1) % self.cap.mod == r2 or (r2 + 1) % self.cap.mod == r1
        return 1


class BlackHole(Game):
    RowStack_Class = StackWrapper(ReserveStack, max_accept=0, max_cards=3)
    Hint_Class = Fan_Hint

    #
    # game layout
    #

    def createGame(self, playcards=5):
        # create layout
        l, s = Layout(self, XOFFSET=12), self.s

        # set window
        w = max(2*l.XS, l.XS+(playcards-1)*l.XOFFSET)
        self.setSize(l.XM + 5*w, l.YM + 4*l.YS)

        # create stacks
        y = l.YM
        for i in range(5):
            x = l.XM + i*w
            s.rows.append(self.RowStack_Class(x, y, self))
        for i in range(2):
            y = y + l.YS
            for j in (0, 1, 3, 4):
                x = l.XM + j*w
                s.rows.append(self.RowStack_Class(x, y, self))
        y = y + l.YS
        for i in range(4):
            x = l.XM + i*w
            s.rows.append(self.RowStack_Class(x, y, self))
        for r in s.rows:
            r.CARD_XOFFSET = l.XOFFSET
            r.CARD_YOFFSET = 0
        x, y = l.XM + 2*w, l.YM + 3*l.YS/2
        s.foundations.append(BlackHole_Foundation(x, y, self, ANY_SUIT, dir=0, mod=13, max_move=0, max_cards=52))
        x, y = l.XM + 4*w, self.height - l.YS
        s.talon = InvisibleTalonStack(x, y, self)

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def _shuffleHook(self, cards):
        # move Ace to bottom of the Talon (i.e. last cards to be dealt)
        return self._shuffleHookMoveToBottom(cards, lambda c: (c.id == 13, c.suit), 1)

    def startGame(self):
        for i in range(2):
            self.s.talon.dealRow(frames=0)
        self.s.talon.dealRow()
        self.s.talon.dealRow(rows=self.s.foundations)
        assert len(self.s.talon.cards) == 0


# register the game
registerGame(GameInfo(56, Fan, "Fan",
                      GI.GT_FAN_TYPE, 1, 0))
registerGame(GameInfo(87, ScotchPatience, "Scotch Patience",
                      GI.GT_FAN_TYPE, 1, 0))
registerGame(GameInfo(57, Shamrocks, "Shamrocks",
                      GI.GT_FAN_TYPE, 1, 0))
registerGame(GameInfo(82, LaBelleLucie, "La Belle Lucie",
                      GI.GT_FAN_TYPE, 1, 2))
registerGame(GameInfo(88, Trefoil, "Trefoil",
                      GI.GT_FAN_TYPE, 1, 2))
registerGame(GameInfo(34, BeleagueredCastle, "Beleaguered Castle",
                      GI.GT_FAN_TYPE, 1, 0))
registerGame(GameInfo(98, BlackHole, "Black Hole",
                      GI.GT_FAN_TYPE, 1, 0))

