/* Groundhog -- a simple logic game
 * Copyright (C) 1998-1999 Maurits Rijk
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <strstream.h>
#include <time.h>
#include <unistd.h>

#include "about_dialog.h"
#include "ball.h"
#include "game.h"
#include "highscore_dialog.h"
#include "new_game_dialog.h"
#include "new_highscore_dialog.h"
#include "options_dialog.h"
#include "pocket.h"
#include "solved_dialog.h"
#include "tube_pair.h"

#include "highscore.xpm"
#include "new.xpm"

Game* Game::_instance = 0;

#if GTK_MAJOR_VERSION > 1 || GTK_MINOR_VERSION > 0

GtkItemFactoryEntry Game::_menu_items[] = {
   {"/_File", 0, 0, 0, "<Branch>"},
   {"/File/_New...", "<control>N", (GtkItemFactoryCallback) Game::OnNew, 0, 0},
   {"/File/Show _Highscore...", 0, 
    (GtkItemFactoryCallback) Game::OnShowHighscore, 0, 0},
   {"/File/_Options...", 0, (GtkItemFactoryCallback) Game::OnOptions, 0, 0},
   {"/File/sep1", 0, 0, 0, "<Separator>"},
   {"/File/_Quit", "<control>Q", (GtkItemFactoryCallback) Game::OnQuit, 0, 0},
   {"/_Help", 0, 0, 0, "<LastBranch>"},
   {"/Help/_About...", 0, (GtkItemFactoryCallback) Game::OnAbout, 0, 0}
};

#else

GtkMenuEntry Game::_menu_items[] =
{
   {"<Main>/File/New...", "<control>N", Game::OnNew, 0},
   {"<Main>/File/Show Highscore...", 0, Game::OnShowHighscore, 0},
   {"<Main>/File/Options...", 0, Game::OnOptions, 0},
   {"<Main>/File/<separator>", 0, 0, 0},
   {"<Main>/File/Quit", "<control>Q", Game::OnQuit, 0},
   {"<Main>/Help/About...", 0, Game::OnAbout, 0}
};

#endif

Game::Game(int argc, char** argv)
{
   _mode = Game::Intermediate;
   _nr_of_moves = 0;
   _seconds = 0;
   _timeout = 0;
   _instance = this;
   gtk_init(&argc, &argv);
   
   GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(window), "Groundhog");
   gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, TRUE);

   gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		      GTK_SIGNAL_FUNC(Game::DeleteEvent), 0);

   gtk_signal_connect(GTK_OBJECT(window), "destroy",
		      GTK_SIGNAL_FUNC(Game::OnQuit), 0);
   gtk_widget_show(window);

   // Set tooltips support
   _tooltips = gtk_tooltips_new();
   GdkColormap* colormap = gdk_window_get_colormap(window->window);
   static GdkColor yellow;
   yellow.red = 61669;
   yellow.green = 59113;
   yellow.blue = 35979;
   gdk_color_alloc(colormap, &yellow);
   gtk_tooltips_set_colors(_tooltips, &yellow, 
			   &window->style->fg[GTK_STATE_NORMAL]);

   GtkWidget* main_vbox = gtk_vbox_new(FALSE, 1);
   gtk_container_border_width(GTK_CONTAINER(main_vbox), 1);
   gtk_container_add(GTK_CONTAINER(window), main_vbox);
   gtk_widget_show(main_vbox);

   int nmenu_items = sizeof(_menu_items) / sizeof(_menu_items[0]);
#if GTK_MAJOR_VERSION > 1 || GTK_MINOR_VERSION > 0
   GtkItemFactory* factory;
   GtkAccelGroup* accel_group = gtk_accel_group_new();

   factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accel_group);
   gtk_item_factory_create_items(factory, nmenu_items, _menu_items, 0);
   gtk_accel_group_attach(accel_group, (GTK_OBJECT(window)));

   GtkWidget* menubar = gtk_item_factory_get_widget(factory, "<main>");
   gtk_box_pack_start(GTK_BOX(main_vbox), menubar, FALSE, TRUE, 0);
   gtk_widget_show(menubar);
#else
   GtkMenuFactory* factory;
   GtkMenuFactory* subfactory;
   
   factory = gtk_menu_factory_new(GTK_MENU_FACTORY_MENU_BAR);
   subfactory = gtk_menu_factory_new(GTK_MENU_FACTORY_MENU_BAR);
   
   gtk_menu_factory_add_subfactory(factory, subfactory, "<Main>");
   gtk_menu_factory_add_entries(factory, _menu_items, nmenu_items);

   gtk_window_add_accelerator_table(GTK_WINDOW(window), subfactory->table);

   // Right justify Help entry
   GtkMenuPath* help = gtk_menu_factory_find(factory, "<Main>/Help");
   gtk_menu_item_right_justify(GTK_MENU_ITEM(help->widget));

   gtk_box_pack_start(GTK_BOX(main_vbox), subfactory->widget, FALSE, TRUE, 0);
   gtk_widget_show(subfactory->widget);
#endif
   
   // Create toolbar
   GtkWidget* handlebox = gtk_handle_box_new();
   gtk_box_pack_start(GTK_BOX(main_vbox), handlebox, FALSE, FALSE, 5);
   
   _toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
   gtk_tooltips_set_colors(GTK_TOOLBAR(_toolbar)->tooltips, &yellow, 
			   &window->style->fg[GTK_STATE_NORMAL]);
   gtk_container_border_width(GTK_CONTAINER(_toolbar), 5);
   gtk_toolbar_set_space_size(GTK_TOOLBAR(_toolbar), 5);
   gtk_container_add(GTK_CONTAINER(handlebox), _toolbar);

   // Create icons
   GdkPixmap* icon;
   GdkBitmap* mask;
   GtkWidget* new_button;
   GtkWidget* iconw;

   GtkStyle* style = gtk_widget_get_style(window);
   icon = gdk_pixmap_create_from_xpm_d(window->window, &mask,
				       &style->bg[GTK_STATE_NORMAL], 
				       new_xpm);
   iconw = gtk_pixmap_new(icon, mask);
   new_button = 
      gtk_toolbar_append_item(GTK_TOOLBAR(_toolbar),
			      "New", "Start new game",
			      "Private", iconw,
			      GTK_SIGNAL_FUNC(Game::OnNew), 0);

   icon = gdk_pixmap_create_from_xpm_d(window->window, &mask,
				       &style->bg[GTK_STATE_NORMAL], 
				       highscore_xpm);
   iconw = gtk_pixmap_new(icon, mask);
   new_button = 
      gtk_toolbar_append_item(GTK_TOOLBAR(_toolbar),
			      "Highscore", "Show highscore",
			      "Private", iconw,
			      GTK_SIGNAL_FUNC(Game::OnShowHighscore), 0);

   gtk_widget_show(_toolbar);
   gtk_widget_show(handlebox);

   // Create play field

   GtkWidget* hbox = gtk_hbox_new(FALSE, 1);
   gtk_container_add(GTK_CONTAINER(main_vbox), hbox);
   gtk_widget_show(hbox);

   _frame = gtk_frame_new(0);
   gtk_frame_set_shadow_type(GTK_FRAME(_frame), GTK_SHADOW_IN);

   gtk_container_add(GTK_CONTAINER(hbox), _frame);
   gtk_widget_show(_frame);

   // create a 5x5 table
   _table = gtk_table_new(7, 7, TRUE);
   gtk_container_border_width(GTK_CONTAINER(_table), 10);
   
   // put the table in the frame
   gtk_container_add(GTK_CONTAINER(_frame), _table);
   gtk_widget_show(_table);

   GtkWidget* vbox = gtk_vbox_new(FALSE, 1);
   gtk_container_add(GTK_CONTAINER(hbox), vbox);
   gtk_widget_show(vbox);
   
   // Create '#moves' label
   _moves = gtk_label_new("");
   gtk_container_add(GTK_CONTAINER(vbox), _moves);
   gtk_widget_show(_moves);
   DisplayMoves();

   // Create 'time' label
   _time = gtk_label_new("");
   gtk_container_add(GTK_CONTAINER(vbox), _time);
   gtk_widget_show(_time);
   DisplayTime();
    
   // create 'go' button
   _go = gtk_button_new_with_label("Go");
   gtk_widget_set_sensitive(_go, FALSE);
   gtk_widget_set_usize(_go, 100, 50);
   gtk_signal_connect(GTK_OBJECT(_go), "clicked",
		      GTK_SIGNAL_FUNC(Game::GoCB), (gpointer) this);
   gtk_container_add(GTK_CONTAINER(vbox), _go);
   SetTooltip(_go, "Play move");
   gtk_widget_show(_go);
   
   _high_score.Read();
}

gint 
Game::DeleteEvent(GtkWidget* widget, GdkEvent* event, gpointer data)
{
   return FALSE;
}

void
Game::SetTooltip(GtkWidget* widget, const char* text)
{
   gtk_tooltips_set_tip(_tooltips, widget, text, NULL);
}

void
Game::DisplayMoves()
{
   char info[128];
   ostrstream ost(info, sizeof(info));

   ost << "Moves: " << _nr_of_moves << ends;
   gtk_label_set(GTK_LABEL(_moves), info);
}

void
Game::DisplayTime()
{
   char info[128];
   ostrstream ost(info, sizeof(info));

   ost << "Time: " << _seconds << ends;
   gtk_label_set(GTK_LABEL(_time), info);
}

void
Game::StopTime()
{
   if (_timeout)
      gtk_timeout_remove(_timeout);
}

gint
Game::Timeout(gpointer data)
{
   Game* game = (Game*) data;
   game->_seconds++;
   game->DisplayTime();
   return TRUE;
}

bool
Game::IsNewHighScore()
{
   if (_mode == Game::Beginner)
      return _high_score.IsNewBeginnerHighScore(_seconds);
   else if (_mode == Game::Intermediate)
      return _high_score.IsNewIntermediateHighScore(_seconds);
   else if (_mode == Game::Expert)
      return _high_score.IsNewExpertHighScore(_seconds);
   return false;
}

void
Game::GoCB(GtkWidget *widget, gpointer data)
{
   ((Game*) data)->Go();
}

void
Game::Go()
{
   MoveBalls();
   _nr_of_moves++;
   DisplayMoves();
   if (IsSolved()) {
      StopTime();
      gtk_widget_set_sensitive(_go, FALSE);
      if (IsNewHighScore()) {
	 static NewHighScoreDialog* dialog = new NewHighScoreDialog(this);
	 dialog->DoDialog();
      } else {
	 static SolvedDialog* dialog = new SolvedDialog();
	 dialog->SetLabel(_nr_of_moves);
	 dialog->DoDialog();
      }
   }
}

void
Game::OnNew(GtkWidget *widget, gpointer data)
{
   static NewGameDialog* dialog = new NewGameDialog(Game::_instance);
   dialog->DoDialog();
}

void
Game::NewGame(int rows, int columns)
{
   if (rows == 2 && columns == 2) {
      _mode = Game::Beginner;
   } else if (rows == 5 && columns == 5) {
      _mode = Game::Intermediate;
   } else if (rows == 9 && columns == 9) {
      _mode = Game::Expert;
   } else {
      _mode = Game::UserDefined;
   }
   _nr_of_moves = 0;
   _seconds = 0;
   _rows = rows;
   _columns = columns;
   _tube_pairs.MakeEmpty();
   _balls.MakeEmpty();
   _pockets.MakeEmpty();

   DisplayMoves();
   DisplayTime();

   gtk_widget_destroy(GTK_WIDGET(_table));
   _table = gtk_table_new(_rows + 2, _columns + 2, TRUE);
   gtk_container_border_width(GTK_CONTAINER(_table), 10);
   gtk_container_add(GTK_CONTAINER(_frame), _table);
   gtk_widget_show(_table);

   (void) Construct(_columns, _rows, 1, 1);

   Shuffle();
   StopTime();
   _timeout = gtk_timeout_add(1000, Game::Timeout, this);
   gtk_widget_set_sensitive(_go, TRUE);
}

void
Game::AddHighScore(const char* name)
{
   Score score(name, _seconds);
   if (_mode == Game::Beginner)
      _high_score.AddBeginnerHighScore(score);
   else if (_mode == Game::Intermediate)
      _high_score.AddIntermediateHighScore(score);
   else
      _high_score.AddExpertHighScore(score);
   _high_score.Write();

}

void
Game::ShowHighScore()
{
   static HighScoreDialog* dialog = new HighScoreDialog();
   dialog->SetHighScore(_high_score);
   if (_mode == Game::Beginner || _mode == Game::UserDefined)
      dialog->ShowBeginnerTab();
   else if (_mode == Game::Intermediate)
      dialog->ShowIntermediateTab();
   else
      dialog->ShowExpertTab();
   dialog->DoDialog();
}

void
Game::OnShowHighscore(GtkWidget *widget, gpointer data)
{
   _instance->ShowHighScore();
}

void
Game::OnOptions(GtkWidget *widget, gpointer data)
{
   static OptionsDialog* dialog = new OptionsDialog(Game::_instance);
   dialog->DoDialog();
}

void
Game::EnableTooltips(bool enable)
{
   gtk_toolbar_set_tooltips(GTK_TOOLBAR(_toolbar), enable);
   if (enable)
      gtk_tooltips_enable(_tooltips);
   else
      gtk_tooltips_disable(_tooltips);
}

void
Game::OnQuit(GtkWidget *widget, gpointer data)
{
   gtk_exit(0);
}

void
Game::OnAbout(GtkWidget *widget, gpointer data)
{
   static AboutDialog* dialog = new AboutDialog();
   dialog->DoDialog();
}

void
Game::ConnectBottom(TubePair* p1, TubePair* p2)
{
   p1->ConnectBottom(p2);
   TubePair* next1 = p1->GetRightPair();
   TubePair* next2 = p2->GetRightPair();
   if (next1 && next2)
      ConnectBottom(next1, next2);
}

void
Game::ConnectRight(TubePair* p1, TubePair* p2)
{ 
   p1->ConnectRight(p2);
   TubePair* next1 = p1->GetBottomPair();
   TubePair* next2 = p2->GetBottomPair();
   if (next1 && next2)
      ConnectRight(next1, next2);
}

TubePair*
Game::Construct(int m, int n, int row, int column)
{
   TubePair* pair = new TubePair(_table, row, column);
   _tube_pairs.Add(pair);

   if (row == 1) {
      Pocket* pocket = new UpperPocket(_table, 0, column);
      Ball* ball = new Ball(Ball::Green);
      pocket->PutBall(ball);

      _pockets.Add(pocket);
      _balls.Add(ball);
      pair->ConnectTop(pocket);
   }
   if (row == _rows) {
      Pocket* pocket = new LowerPocket(_table, _rows + 1, column);
      Ball* ball = new Ball(Ball::Yellow);
      pocket->PutBall(ball);

      _pockets.Add(pocket);
      _balls.Add(ball);
      pair->ConnectBottom(pocket);
   }
   if (column == 1) {
      Pocket* pocket = new LeftPocket(_table, row, 0);
      Ball* ball = new Ball(Ball::Red);
      pocket->PutBall(ball);

      _pockets.Add(pocket);
      _balls.Add(ball);
      pair->ConnectLeft(pocket);
   }
   if (column == _columns) {
      Pocket* pocket = new RightPocket(_table, row, _columns + 1);
      Ball* ball = new Ball(Ball::Blue);
      pocket->PutBall(ball);

      _pockets.Add(pocket);
      _balls.Add(ball);
      pair->ConnectRight(pocket);
   }


   if (m == 1 && n == 1) {
      ;				// Do nothing
   } else if (m == 1) {
      TubePair* rest = Construct(1, n - 1, row + 1, column);
      pair->ConnectBottom(rest);
   } else if (n == 1) {
      TubePair* rest = Construct(m - 1, 1, row, column + 1);
      pair->ConnectRight(rest);
   } else {			// m > 1 && n > 1
      TubePair* new_row = Construct(m - 1, 1, row, column + 1);
      TubePair* new_column = Construct(1, n - 1, row + 1, column);
      TubePair* rest = Construct(m - 1, n - 1, row + 1, column + 1);
      pair->ConnectRight(new_row);
      pair->ConnectBottom(new_column);
      ConnectBottom(new_row, rest);
      ConnectRight(new_column, rest);
   }
   return pair;
}

void
Game::Build()
{
   _rows = 5;
   _columns = 5;

   (void) Construct(_columns, _rows, 1, 1);
   gtk_main();
}

void
Game::Shuffle()
{
   srand(time(0) ^ getpid());
   for (int i = 0; i < 10; i++) {
      _tube_pairs.Shuffle();
      MoveBalls();
   }
   _tube_pairs.Shuffle();
}

void
Game::MoveBalls()
{
   _pockets.Eject();
   while (_balls.Move())
      ;
}

void
Game::Start()
{
   MoveBalls();
}

bool
Game::IsSolved()
{
   return _pockets.HasRightBalls();
}
