/* 
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "gnome_chess.h"
#include <pwd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "gtkutil.h"
#include "serverlist.h"

extern struct engine *first_chess_programm;
extern void check_prefs_dir(void);
extern char *get_xdir(void);
extern char *cfg_get_str(char *cfg, char *var, char *dest);

GtkWidget *slwin = 0;
GtkWidget *sllist;
GtkWidget *entry_server;
GtkWidget *entry_port;
GtkWidget *entry_user;
GtkWidget *entry_password;
GtkWidget *entry_comment;

struct serverentry *new_serverentry(void);
struct defaultserv dserv[] = 
{
     {"chessclub.com", 5000, "nobody","ICC Chess Server"},
     {"freechess.org", 5000, "nobody", "FICS Chess Server"},

     {0,0,0,0}
};

/* Prototypes for functions used internally */
void add_defaults(void);
char *read_next_server_entry(char *my_cfg, struct serverentry *serv);
void load_serverentrys(void);
void save_serverentrys(void);
void add_server_entry(struct serverentry *serv);
void connect_clicked(GtkWidget *wid, void *data);
void new_server_clicked(GtkWidget *wid, void *data);
void delete_server_clicked(GtkWidget *wid, void *data);
void sl_row_unselected(GtkWidget *clist, gint row, gint column, GdkEventButton *even);
void sl_row_selected(GtkWidget *clist, gint row, gint column, GdkEventButton *even, void *data);
void move_up_clicked(GtkWidget *igad, void *data);
void move_dn_clicked(GtkWidget *igad, void *data);
void handle_server(GtkWidget *igad);
void handle_port(GtkWidget *igad);
void handle_user(GtkWidget *igad);
void handle_password(GtkWidget *igad);
void handle_comment(GtkWidget *igad);
void open_server_list(GtkWidget *widd, void *data);
int close_server_list(GtkWidget *win, int destroy);


void server_cb(GtkWidget *widget, void *data) {
	open_server_list(widget, data);
}

void add_defaults(void)
{
   struct serverentry serv;
   int i = 0;

   serv.password[0] = 0;
   while(1)
   {
      if(dserv[i].port == 0) return;
      strcpy(serv.server, dserv[i].server);
      serv.port = dserv[i].port;
      strcpy(serv.user, dserv[i].user);
      strcpy(serv.comment, dserv[i].comment);
      add_server_entry(&serv);
      i++;
   }
}

char *read_next_server_entry(char *my_cfg, struct serverentry *serv)
{
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "servername ", serv->server);
   if(my_cfg)
   {
      my_cfg = cfg_get_str(my_cfg, "port ", serv->user);
      serv->port = atoi(serv->user);
   }
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "user ", serv->user);
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "password ", serv->password);
   if(my_cfg) my_cfg = cfg_get_str(my_cfg, "comment ", serv->comment);

   return my_cfg;
}

void load_serverentrys(void)
{
   struct serverentry serv;
   struct stat st;
   int fh;
   char *cfg, *my_cfg;
   char file[256];

   snprintf(file, sizeof file, "%s/servers.conf", get_xdir());
   fh = open(file, O_RDONLY);
   if(fh != -1)
   {
      fstat(fh, &st);
      cfg = malloc(st.st_size);
      if(cfg)
      {
	 read(fh, cfg, st.st_size);
	 my_cfg = cfg;
	 while((my_cfg = read_next_server_entry(my_cfg, &serv)))
	 {
	    add_server_entry(&serv);
	 }
	 free(cfg);
      }
      close(fh);
   } else
     add_defaults();
}

void save_serverentrys(void)
{
   int fh, i = 0;
   char file[256], buf[1024];
   char *servname, *port, *user, *pass, *comment;

   check_prefs_dir();

   snprintf(file, sizeof file, "%s/servers.conf", get_xdir());
   fh = open(file, O_TRUNC | O_WRONLY | O_CREAT, 0600);
   if(fh != -1)
   {
      while(1)
      {
	 if(!gtk_clist_get_text(GTK_CLIST(sllist), i, 0, &servname)) break;
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 1, &port);
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 2, &user);
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 3, &pass);
	 gtk_clist_get_text(GTK_CLIST(sllist), i, 4, &comment);
	 snprintf(buf, sizeof buf,
		  "servername = %s\n"
		  "port = %d\n"
		  "user = %s\n"
		  "password = %s\n"
		  "comment = %s\n\n",
		  servname, atoi(port), user,
		  pass, comment);
	 write(fh, buf, strlen(buf));
	 i++;
      }
      close(fh);
   }
}

void add_server_entry(struct serverentry *serv)
{
   char tbuf[16];
   gchar *new[1][6];

   sprintf(tbuf, "%d", serv->port);

   new[0][0] = serv->server;
   new[0][1] = tbuf;
   new[0][2] = serv->user;
   new[0][3] = serv->password;
   new[0][4] = serv->comment;

   gtk_clist_append(GTK_CLIST(sllist), new[0]);
}

void connect_clicked(GtkWidget *wid, void *data)
{
   char *server, *portstr, *u, *p;
   int port, row;
   char arg[132];	

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      if(!gtk_clist_get_text(GTK_CLIST(sllist), row, 0, &server)) return;
      gtk_clist_get_text(GTK_CLIST(sllist), row, 1, &portstr);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 2, &u);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 3, &p); 
      port = atoi(portstr);
      strcpy(arg, server);

	first_chess_programm = engine_ics_new(arg, u, p, portstr, NULL);
	close_server_list(0, TRUE);
   }
}

void delete_server_clicked(GtkWidget *wid, void *data)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      gtk_clist_unselect_all(GTK_CLIST(sllist));
      gtk_clist_remove((GtkCList*)sllist, row);
   }
}

void new_server_clicked(GtkWidget *wid, void *data)
{
   int i;
   gchar *new[1][5];
   new[0][0] = "EDIT ME";
   new[0][1] = "5000";
   new[0][2] = "nobody";
   new[0][3] = "";
   new[0][4] = "";
   i = gtk_clist_append(GTK_CLIST(sllist), new[0]);
   gtk_clist_select_row(GTK_CLIST(sllist), i, 0);
   gtk_clist_moveto(GTK_CLIST(sllist), i, 0, 0.5, 0);
}

void sl_row_unselected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even)
{
   gtk_entry_set_text(GTK_ENTRY(entry_server), "");
   gtk_entry_set_text(GTK_ENTRY(entry_port), "");
   gtk_entry_set_text(GTK_ENTRY(entry_user), "");
   gtk_entry_set_text(GTK_ENTRY(entry_password), "");
   gtk_entry_set_text(GTK_ENTRY(entry_comment), "");
}

void sl_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, void *data)
{
   gchar *server, *user, *port, *comment, *password;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      gtk_clist_get_text(GTK_CLIST(sllist), row, 2, &user);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 0, &server);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 1, &port);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 3, &password);
      gtk_clist_get_text(GTK_CLIST(sllist), row, 4, &comment);

      gtk_entry_set_text(GTK_ENTRY(entry_server), server);
      gtk_entry_set_text(GTK_ENTRY(entry_port), port);
      gtk_entry_set_text(GTK_ENTRY(entry_user), user);  
      gtk_entry_set_text(GTK_ENTRY(entry_password), password);
      gtk_entry_set_text(GTK_ENTRY(entry_comment), comment);

      if(even && even->type == GDK_2BUTTON_PRESS)
      {
	 connect_clicked(clist, NULL);
      }
   } else
     sl_row_unselected(0, 0, 0, 0);
}

void move_up_clicked(GtkWidget *igad, void *data)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1 && row > 0)
   {
      gtk_clist_freeze(GTK_CLIST(sllist));
      gtk_clist_swap_rows(GTK_CLIST(sllist), row-1, row);
      gtk_clist_thaw(GTK_CLIST(sllist));
      row--;
      if(gtk_clist_row_is_visible(GTK_CLIST(sllist), row) != GTK_VISIBILITY_FULL)
	gtk_clist_moveto(GTK_CLIST(sllist), row, 0, 0.1, 0);
   }
}

void move_dn_clicked(GtkWidget *igad, void *data)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *temp;
      if(!gtk_clist_get_text(GTK_CLIST(sllist), row+1, 0, &temp)) return;
      gtk_clist_freeze(GTK_CLIST(sllist));
      gtk_clist_swap_rows(GTK_CLIST(sllist), row, row+1);
      gtk_clist_thaw(GTK_CLIST(sllist));
      row++;
      if(!gtk_clist_row_is_visible(GTK_CLIST(sllist), row) != GTK_VISIBILITY_FULL)
	gtk_clist_moveto(GTK_CLIST(sllist), row, 0, 0.9, 0);
   } 
}

void handle_server(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *server = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 0, server);
   }
}

void handle_port(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *port = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 1, port);
   }
}

void handle_user(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *chan = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 2, chan);
   }
}

void handle_password(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *pw = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 3, pw);
   }
}

void handle_comment(GtkWidget *igad)
{
   int row;

   row = gtkutil_clist_selection(sllist);
   if(row != -1)
   {
      char *comment = gtk_entry_get_text((GtkEntry *)igad);
      gtk_clist_set_text((GtkCList*)sllist, row, 4, comment);
   }
}

void open_server_list(GtkWidget *widd, void *data)
{
   static gchar *titles[] = {_("Server"), _("Port"), _("Username"), _("Password"), _("Comment")};
   GtkWidget *vbox, *box, *wid;

   if(slwin)
   {
      close_server_list(0, TRUE);
      return;
   }

   slwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(slwin), _("Gnome-Chess: Server List"));
   gtk_signal_connect(GTK_OBJECT(slwin), "delete_event",
                       GTK_SIGNAL_FUNC(close_server_list), 0);
   gtk_window_set_policy(GTK_WINDOW(slwin), TRUE, TRUE, FALSE);

   vbox = gtk_vbox_new(0,2);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(slwin), vbox);
   gtk_widget_show(vbox);

   sllist = gtkutil_clist_new(5, titles, vbox, GTK_POLICY_ALWAYS,
			      sl_row_selected, NULL,
			      sl_row_unselected, NULL);
   gtk_widget_set_usize(sllist, -1, 200);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 0, 165);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 1, 40);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 2, 90);
   gtk_clist_set_column_width(GTK_CLIST(sllist), 3, 60);

   box = gtk_hbox_new(0,0);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   gtkutil_label_new(_("Server Name:"), box);

   entry_server = gtkutil_entry_new(99, box, handle_server, 0);

   gtkutil_label_new(_("Port:"), box);

   entry_port = gtkutil_entry_new(14, box, handle_port, 0);
   gtk_widget_set_usize(entry_port, 80, -1);

   box = gtk_hbox_new(0,5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   gtkutil_label_new(_("User Name:"), box);

   entry_user = gtkutil_entry_new(85, box, handle_user, 0);

   gtkutil_label_new(_("Password:"), box);

   entry_password = gtkutil_entry_new(85, box, handle_password, 0);
   gtk_widget_set_usize(entry_password, 80, -1);

   gtkutil_label_new(_("Comment:"), box);

   entry_comment = gtkutil_entry_new(99, box, handle_comment, 0);
   gtk_widget_set_usize(entry_comment, 160, -1);

   wid = gtk_hseparator_new();
   gtk_box_pack_start(GTK_BOX(vbox), wid, FALSE, FALSE, 8);
   gtk_widget_show(wid);

   box = gtk_hbox_new(0, 2);
   gtk_container_set_border_width(GTK_CONTAINER(box), 2);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 2);
   gtk_widget_show(box);

   gtkutil_button_new(_("Connect"), 0, 0, connect_clicked, NULL, box);
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_NEW, _("New Server"),
			  new_server_clicked, NULL, box);
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_CLOSE, _("Delete Server"),
			  delete_server_clicked, NULL, box);
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_UP, _("Move Up"),
			  move_up_clicked, NULL, box);
   gtkutil_stock_button(slwin, GNOME_STOCK_PIXMAP_DOWN, _("Move Dn"),
			  move_dn_clicked, NULL, box);

   load_serverentrys();

   gtk_widget_show(slwin);
}

int close_server_list(GtkWidget *win, int destroy)
{
   save_serverentrys();
   if(destroy && slwin) gtk_widget_destroy(slwin);
   slwin = 0;
   return 0;
}

