/*
 *  Xtend is Copyright (C) 1998 David M. Shaw <dshaw@jabberwocky.com>
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include <time.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>
#include <strings.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include "parse.h"
#include "util.h"
#include "x10.h"

extern int errno,verbose,special,statusfile;
extern unsigned short okaytowipe;
extern unsigned char addresstable[16][16];
extern char *file,*port,*lockdir;

char parsefunction(char *byte);
void parseaddress(char byte);
void setcm11atime(int fd);

static char RCSID[]= "@(#) $Id: util.c,v 2.13 1998-04-05 21:58:48-04 dshaw Exp $";

void printbytes(char *buf,int length)
{
	int x=(int)RCSID;

	for(x=0;x<length;x++)
		{
		printf(" 0x%.2X",buf[x]);
		}

	printf("\n");
}

void getbytes(int fd,char *buf,int count)
{
	ssize_t	bytes=0,index=0;

	if(verbose>8)
		printf("Getting a message of %d (0x%X) bytes\n",count,count);

	while(index!=count)
		{
		bytes=read(fd,&buf[index],count-index);
		index+=bytes;
		}
}

int waitforpoll(int fd)
{
	fd_set	descs,zero;
struct	timeval timeout;
	int	err;
	ssize_t	bytes=0;
unsigned char	buf[MAXBUF];

	FD_ZERO(&zero);

	if(verbose>8)
		{
		printf("\nWaiting for poll.. ");
		fflush(stdout);
		}

	while(bytes!=1 || *buf!=POLL)
		{
	/* this is so we don't have to busy wait while waiting for our poor
	   user to actually DO something */

		FD_ZERO(&descs);
		FD_SET(fd,&descs);

		err=select(fd+1,&descs,NULL,NULL,NULL);
		if(err==-1)
			{
			if(errno==EINTR)
				return 0;

			perror("select");
			exit(-1);
			}
		bytes=read(fd,buf,sizeof(buf));

		/* this is arguably foolish, but how else can one
		   handle receiving an EOF continually?
		   It only happens when using a heyu file (of course),
		   and actually is how 'tail(1)' handles the same thing. */

		timeout.tv_sec=HEYU_TIMEOUT;
		timeout.tv_usec=0;

		if(bytes==0 && select(0,&zero,&zero,&zero,&timeout)==-1)
			{
			if(errno==EINTR)
				return 0;

			perror("select-timeout");
			exit(-1);
			}

		if(verbose>9)
			printf("%d bytes, starting with [%X]\n",bytes,buf[0]);

		/* was there a power failure? */
		/* we have to do this or the CM11a won't give us data */
		/* Of course, only do it if we are talking to the port
		   ourselves.  Otherwise, heyu will handle it for us */
		if(*buf==POWERFAIL && file==NULL)
			setcm11atime(fd);
		}

	if(verbose>8)
		printf("got it!\n");

	/* only send an ack if we're handling the serial stuff
	   ourselves */

	if(file==NULL)
		{
		if(verbose>8)
			{
			printf("Sending ACK.. ");
			fflush(stdout);
			}

		*buf=ACK;
		bytes=write(fd,buf,1);

		if(verbose>8)
			printf("done!\n");
		}
	return 1;
}

int getmessage(int fd,char *buf)
{
	int	bytes=0;
	char	count;

	/* get the number of bytes we need to pull in */
	while(bytes==0)
		{
		bytes=read(fd,&count,1);
		}

	if(verbose>8)
		printf("A message of size %d (0x%X)\n",count,count);

	getbytes(fd,buf,count);
	return count;
}

void handlemessage(char *buf,char length)
{
/* the first byte is the bitmask for commands or addresses */
/* 1 bit ==function & 0 bit == address */
	char	mask=buf[0];
	char	index;
	char	bit=0x01;

	if(verbose>8)
		{
		printf("Mask: 0x%X. Raw bytes: ",mask);
		printbytes(&buf[1],length-1);
		}

/* now we work our way through whatever we have in the buffer, checking
 the mask bit to see if it's a command or an address */
	for(index=0;index<length-1;index++)
		{
		if(bit & mask)
			{
			if(parsefunction(&buf[index+1]))
				{
				/* this is in case of dim/bright commands
				   which are glued into the buffer sort of
				   willy-nilly */
				index++;
				bit = bit << 1;
				}
			}
		else
			{
			parseaddress(buf[index+1]);
			}

		bit = bit << 1;
		}
}

void wipeaddresstable(int house)
{
	int x;

	for(x=0;x<16;x++)
		{
		/* this just marks the "addressed" bit as clear */
		addresstable[house][x]&=~ADDRESSBIT;
		}

	okaytowipe&=~(0x01 << house);

	if(verbose>6)
		printf("Wiping address table for house %d\n",house);
}

void lockfile(char *port,char *lockdir,int action)
{
	char	full[PATH_MAX];
	int	temp;
	pid_t	pid;
	char	*device=rindex(port,'/')+1;
	char	lock[26]="LCK..";

	strncat(lock,device,20);
	lock[25]='\0';

/* this is more cosmetic than necessary, but hey, I never like having two
   slashes in a row if I can avoid it */

	if(lockdir[strlen(lockdir)-1]=='/')
		lockdir[strlen(lockdir)-1]='\0';

/* this is actually not as foolish as it looks!  We know that lockdir
   can't be more than PATH_MAX-27 characters from the parsing function */

	strcpy(full,lockdir);
	strcat(full,"/");
	strcat(full,lock);

	if(action==DO_LOCK)
		{
		/* lock */
		if(verbose>2)
			printf("Creating lockfile %s\n",full);
		temp=open(full,O_EXCL|O_CREAT|O_WRONLY,0666);
		if(temp==-1)
			{
			if(errno==EEXIST)
				{
				/* the lockfile exists */
				/* is it for a dead process? */
				temp=open(full,O_RDONLY);
				if(temp==-1 || read(temp,lock,11)==-1
				|| sscanf(lock,"%10d",(int *)&pid)!=1)
					{
					fprintf(stderr,"Couldn't access lockfile %s\n",full);
					exit(-1);
					}

				if(verbose>3)
					printf("Lockfile exists as %d\n",(int)pid);

				/* is it dead? */
				if(kill(pid,0)==-1 && errno==ESRCH)
					{
					if(verbose>3)
						printf("Pid %d is dead.  Breaking lock.\n",(int)pid);
					close(temp);
					if(unlink(full)==-1)
						{
						fprintf(stderr,"Couldn't break lockfile %s: %s\n",full,strerror(errno));
						exit(-1);
						}

					lockfile(port,lockdir,action);
					return;
					}
				fprintf(stderr,"Lockfile %s already exists!\n",full);
				exit(-1);
				}

			fprintf(stderr,"Couldn't create lockfile %s: %s\n",full,strerror(errno));
			exit(-1);
			}

		sprintf(lock,"%10d\n",(int)getpid());

		if(write(temp,lock,11)==-1)
			{
			fprintf(stderr,"Couldn't write to lockfile %s: %s\n",full,strerror(errno));
			exit(-1);
			}
		close(temp);
		}
	else
		{
		/* unlock */
		if(verbose>2)
			printf("Removing lockfile %s\n",full);
		if(unlink(full)==-1)
			{
			fprintf(stderr,"Couldn't remove lockfile %s: %s\n",full,strerror(errno));
			exit(-1);
			}
		}
}

void interrupt(int signal)
{
	if(verbose>2)
		printf("Caught signal %d\n",signal);

	if(signal==SIGHUP)
		special=1;
	else
		special=2;

	setsignals();
}

void setsignals(void)
{
	signal(SIGTERM,interrupt);
	signal(SIGINT,interrupt);
	signal(SIGHUP,interrupt);
}

void shutdown(void)
{
	if(verbose>0)
		printf("Caught signal - shutting down\n");

	/* remove the lockfile, if necessary */
	if(file==NULL)
		lockfile(port,lockdir,DO_UNLOCK);

	/* close the status file */
	close(statusfile);
	statusfile=-1;
}

void setcm11atime(int fd)
{
	char message[7];
struct	tm *now;
	int x;
time_t	clock=time(NULL);

	if(verbose>2)
		printf("Responding to a power failure request\n");

	now=localtime(&clock);

	message[0]=SETCLOCK;
	message[1]=now->tm_sec;
	/* just in case of leap seconds */
	if(message[1]>59)
		message[1]=0;
	message[2]=now->tm_min;
	/* this is weird - the minute field takes the alternate
	   hours */
	if(((float)now->tm_hour/2)!=(now->tm_hour/2))
		message[2]+=60;
	message[3]=now->tm_hour/2;
	message[5]=0x01; /* Saturday.  We'll fix it later */
	message[6]=0x60; /* this is house A - it doesn't matter really */

	/* bump the mask along to get the right day of the week */
	for(x=now->tm_wday;x<6;x++)
		message[5]=message[5] << 1;

	if(now->tm_yday>256)
		{
		/* this is odd... the MSB of the year is part of the
		   day of the week byte */
		message[4]=now->tm_yday-256;
		message[5]=message[5] | 0x80;
		}
	else
		message[4]=now->tm_yday;

	if(write(fd,message,7)==-1)
		{
		fprintf(stderr,"Couldn't write date to CM11a: %s\n",strerror(errno));
		exit(-1);
		}
}

int changeenv(void)
{
	char	varname[8]="X10_xxx",value[4],set[12];
	int	err=0,house,unit;

	for(house=0;house<16;house++)
		for(unit=0;unit<16;unit++)
			{
			varname[4]=house+65;

			if(unit>8)
				{
				varname[5]=49;
				varname[6]=unit+39;
				}
			else
				{
				varname[5]=unit+49;
				varname[6]='\0';
				}

#ifdef NO_APPL_ENV
			/* don't bother to inject into the environment
			   if it's just to tell us that it's an
			   appliance */
			if(addresstable[house][unit]&APPLBIT)
				continue;
#endif
			if(addresstable[house][unit]!=0)
				{
				sprintf(value,"%d",addresstable[house][unit]);

				/* I'd rather use setenv here, as it is more
				   efficient, but POSIX doesn't have the darn
				   thing */

				strcpy(set,varname);
				strcat(set,"=");
				strcat(set,value);

				if(putenv(set)==-1)
					err=-1;

				if(verbose>4)
					printf("Setting $%s=%s (%d)\n",varname,value,addresstable[house][unit]);
				}
			}
	return err;
}

void updatestatusfile(int house,int unit)
{
	off_t offset=((house*16)+unit)*2;

	if(verbose>4)
		printf("Updating status file for %c%d (at %d) to be %d\n",house+65,unit+1,(int)offset,addresstable[house][unit]);

	if(lseek(statusfile,offset,SEEK_SET)==-1)
		{
		perror("couldn't seek in the status file");
		return;
		}

	if(write(statusfile,&addresstable[house][unit],1)==-1)
		{
		perror("couldn't write to the status file");
		return;
		}
}
