#ifndef SET_HH
#define SET_HH

// Copyright (c) 1996-1999 The University of Cincinnati.
// All rights reserved. 

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Author: Dale E. Martin          dmartin@ece.uc.edu

//---------------------------------------------------------------------------
//
// $Id: set.hh,v 1.4 1999/03/23 20:27:42 dmartin Exp $
//
//---------------------------------------------------------------------------
#include <iostream.h>
#include "savant.hh"
#include "dl_list.hh"
#include "IIR_Declaration.hh"
#include "IIR_TypeDefinition.hh"

template <class type>
class set{
  // @BeginExternalProseDescription
  // This class maintains a distinct set of pointers to the objects it's 
  // templatized on.  Functions are provided to add and reome things from
  // the set, to get the number of objects in the set, and walk the members
  // of the set.  NOTE:  Since only pointers are kept within the set, care
  // must be taken when "delete"ing members of the of set and so forth.
  // @EndExternalProseDescription

public:
  set( int starting_size = 4 );
  set( type *, int starting_size = 4 );
  set( set<type> & );

  void init( int starting_size );

  virtual ~set();
  
  // Methods to add members to the set
  virtual void add( type * );
  // Add the elements of this list to the set
  void add( dl_list<type> * );
  // Add the elements of this set to the set
  void add( set<type> * );

  void remove( type * );
  int num_elements();
  type *get_element();
  type *get_next_element();

  // "New" a list, and put the elements of the set into it.
  dl_list<type> *make_list();

  // See if this thingy is in the set.
  bool in_set( type * );

  // Reset the list so it's prisitine once again...
  void reset( int );

  set<type> &operator=( set<type> &to_copy );

  void dump();
  
  void reduce_set( IIR_Boolean (IIR::*constraint_function)() );

protected:
  int num;
  type **data_array;

private:
  void init( int starting_size, bool need_to_clear_array );

  virtual int find( type * );
  
  void duplicate_set( set<type> & );

  int array_size;
  int get_pos;
};


template <class type>
inline
void 
set<type>::init( int starting_size ){
  init( starting_size, true );
}

template <class type>
inline
void 
set<type>::init(int starting_size, bool need_to_clear_array ){

  if( starting_size == 0 ){
    starting_size = 1;
  }

  get_pos = 0;

  array_size = starting_size;
  num = 0;

  data_array = new type*[array_size];

  if( need_to_clear_array == true ){
    memset( data_array, 0, array_size );
  }
}

template <class type>
inline
set<type>::set(int starting_size){
  init( starting_size );
}

template <class type>
inline
set<type>::set( type *first_element, int starting_size ){
  init( starting_size );

  add( first_element );
}

template <class type>
inline
void 
set<type>::duplicate_set( set<type> &to_copy ){
  delete [] data_array;

  int to_copy_num = to_copy.num_elements();
  init( to_copy_num, false );
  memcpy( data_array, to_copy.data_array, to_copy_num*sizeof( type * ) );
  num = to_copy_num;
}

template <class type>
inline
set<type>::set( set<type> &to_copy ){
  // Assure we're not deleting garbage in the call to duplicate set
  data_array = 0;
  duplicate_set( to_copy );
}

template <class type>
inline
set<type> &
set<type>::operator=( set<type> &to_copy ){
  duplicate_set( to_copy );

  return *this;
}

template <class type>
inline
void 
set<type>::reset(int starting_size){
  delete [] data_array;

  init( starting_size );
}

template <class type>
inline
set<type>::~set(){
  delete [] data_array;
}

template <class type>
inline
void 
set<type>::add( type *to_add ){
  if( in_set( to_add ) == false ){
    if( num == array_size - 1 ){
      // Then we already have a full array - allocate more space.
      type **temp_array = new type*[ 2*array_size ];
      
      memcpy( temp_array, data_array, num*sizeof( type * ) );

      delete [] data_array;
      data_array = temp_array;
      array_size = 2*array_size;
    }

    data_array[ num ] = to_add;
    num++;
  }
  //  else {
  // cout << *to_add << " already in set" << endl;
  //  }
}

template <class type>
inline
void 
set<type>::add( dl_list<type> *list_to_add ){
  type *current = list_to_add->first();
  while( current != NULL ){
    add( current );
    current = list_to_add->successor( current );
  }
}


template <class type>
inline
void 
set<type>::add( set<type> *set_to_add ){
  type *current = set_to_add->get_element();
  while( current != NULL ){
    add( current );
    current = set_to_add->get_next_element();
  }
}

template <class type>
inline
void 
set<type>::remove( type *to_remove ){
  int pos = find( to_remove );
  if( pos != -1 ){
    // Then we actually had one of these in our set.
    
    // This is so our next "get_element" does the right thing...
    if( get_pos >= pos ){
      get_pos--;
    }

    // Shift all of the pointers down a slot.    
    memmove( data_array + pos, data_array + pos + 1, (num - pos - 1) * sizeof( type * ) );
//     int i;
//     for( i = pos; i < num - 1; i++ ){
//       data_array[ i ] = data_array[ i + 1 ];
//     }

    num--;
  }
}

template <class type>
inline
int
set<type>::find( type *to_find ){
  int i;
  bool found_it = false;

  for( i = 0; i < num; i++ ){
    if( data_array[i] == to_find ){
      found_it = true;
      break;
    }
  }

  if( found_it == true ){
    return i;
  }
  else{
    return -1;
  }
}

template <class type>
inline
bool 
set<type>::in_set( type *test_for ){
  int pos = find( test_for );

  if( pos == -1 ){
    return false;
  }
  else{
    return true;
  }
}

template <class type>
inline
type *
set<type>::get_element(){
  if( num > 0 ){
    get_pos = 0;
    return data_array[ get_pos ];
  }
  else{
    return NULL;
  }
}

template <class type>
inline
type *
set<type>::get_next_element(){
  get_pos++;
  if( get_pos >= num ){
    return NULL;
  }
  else{
    return data_array[ get_pos ];
  }
}

template <class type>
inline
int 
set<type>::num_elements(){
  return num;
}

template <class type>
inline
dl_list<type> *
set<type>::make_list(){
  dl_list<type> *retval = new dl_list<type>;

  int i;
  for( i = 0; i < num; i++ ){
    retval->append( data_array[i] );
  }

  return retval;
}

template <class type>
inline
void 
set<type>::dump(){
  int i;
  for( i = 0; i < num_elements(); i++ ){
    if( data_array[i] == NULL ){
      cout << "NULL" << endl;
    }
    else{
      //      cout << *data_array[i] << endl;
    }
  }
}

inline 
void 
set<IIR_Declaration>::reduce_set( IIR_Boolean (IIR::*constraint_function)() ){
  IIR_Declaration *current = get_element();
  while( current != NULL ){
    if( (current->*constraint_function)() == FALSE ){
      remove( current );
    }
    current = get_next_element();
  }
}

inline 
void 
set<IIR_TypeDefinition>::reduce_set( IIR_Boolean (IIR::*constraint_function)() ){
  IIR_TypeDefinition *current = get_element();
  while( current != NULL ){
    if( (current->*constraint_function)() == FALSE ){
      remove( current );
    }
    current = get_next_element();
  }
}


// This function takes two pointers to sets, performs an intersection on the
// sets, and creates a new set (allocates memory) holding the intersection.
// The two sets that are passed in are unaffected.  (Aside from their 
// internal position pointer being modified.)
template <class type>
inline
set<type> *
set_intersect( set<type> *a, set<type> *b ){
  set<type> *retval = new set<type>;

  if (a == NULL || b == NULL) {
    return retval;
  }

  set<type> *larger_set, *smaller_set;
  
  if( a->num_elements() > b->num_elements() ){
    larger_set = a;
    smaller_set = b;
  }
  else{
    larger_set = b;
    smaller_set = a;  
  }

  // Walk set a, and see if each member is in b.  If it is, then add it
  // to the return set.
  int bound = larger_set->num_elements();
  int i;

  type *current = larger_set->get_element();  
  for( i = 0; i < bound; i++ ){
    if( smaller_set->in_set( current ) == true ){
      retval->add( current );
    }
    current = larger_set->get_next_element();
  }

  return retval;
}

#endif
