// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ececs.uc.edu
//          Malolan Chetlur     mal@ececs.uc.edu
//          Timothy J. McBrayer tmcbraye@ececs.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ececs.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_SensitizedProcessStatement.cc,v 1.4 1999/07/23 21:08:12 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIRScram_SensitizedProcessStatement.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_Identifier.hh"
#include "IIR_Label.hh"
#include "IIR_Name.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_WaitStatement.hh"
#include "error_func.hh"
#include "set.hh"

IIRScram_SensitizedProcessStatement::~IIRScram_SensitizedProcessStatement() {}


void 
IIRScram_SensitizedProcessStatement::_type_check(){
  IIR *signal_decl;

  IIR_Designator *current_designator = sensitivity_list.first();
  while( current_designator != NULL ){
    ASSERT( current_designator->get_kind() == IIR_DESIGNATOR_EXPLICIT );

    IIR *current_signal_name =
      ((IIR_DesignatorExplicit *)current_designator)->get_name();

    // If a signal in the sensitivity list can't be resolved, it will
    // complain about it in this function call...
    signal_decl = _resolve_signal_name( current_signal_name );
    if( signal_decl != NULL ){
      // This might leak a name...
      ((IIR_DesignatorExplicit *)current_designator)->set_name( signal_decl );
    }
    current_designator = sensitivity_list.successor( current_designator );
  }
}


IIR *
IIRScram_SensitizedProcessStatement::_resolve_signal_name( IIR *signal_name ){
  IIR *retval = NULL;
  
  if( signal_name->_is_resolved() == TRUE ){
    return signal_name;
  }

  IIR *orig_name = signal_name;

  while( signal_name->get_kind() == IIR_INDEXED_NAME ){
    signal_name = ((IIR_Name *)signal_name)->get_prefix();
  }

  set<IIR_Declaration> *signal_decls = signal_name->_symbol_lookup();

  if( signal_decls == NULL ){
    report_undefined_symbol( signal_name );
    return NULL;
  }

  IIR_Declaration *current_decl = signal_decls->get_element();
  while (current_decl != NULL) {
    if(current_decl->_is_signal() == false) {
      signal_decls->remove(current_decl);
    }
    current_decl = signal_decls->get_next_element();
  }

  switch (signal_decls->num_elements()) {
  case 0: {
    ostrstream err;
    err << "Signal " << *signal_name << " is undeclared in this scope" << ends;
    report_error(this, err);
    break;
  }
  case 1: {
    IIR_Declaration *sig_decl = signal_decls->get_element();
    // Everything but signals have been eliminated, meaning, this must be an array
    // access.
    IIR *signal_name_transformed = NULL;
    if( orig_name->get_kind() == IIR_INDEXED_NAME ){
      set<IIR_TypeDefinition> *orig_types = orig_name->_get_rval_set();

      if( orig_types == NULL ){
	report_undefined_symbol( orig_name );
      }

      switch( orig_types->num_elements() ){
      case 1:{
	IIR_TypeDefinition *right_type = orig_types->get_element();
	signal_name_transformed = orig_name->_semantic_transform( right_type );
	signal_name_transformed->_type_check( right_type );
	
	retval = signal_name_transformed->_rval_to_decl( right_type );

	break;
      }
      default:{
	report_ambiguous_error( orig_name, orig_types );
      }
      }
    }
    else{
      signal_name_transformed = signal_name->_semantic_transform( sig_decl->_get_subtype() );
      signal_name_transformed->_type_check( sig_decl->_get_subtype() );      
      retval = signal_name_transformed->_decl_to_decl( sig_decl );
    }

    if( retval != NULL ){
      ASSERT( retval->_is_resolved() == TRUE );
    }

    break;
  }
  default:
    ostrstream err;
    report_ambiguous_error( signal_name, signal_decls);
    break;    
  }

  delete signal_decls;
  if( retval->_is_iir_declaration() == TRUE ){
    IIR_Attribute *my_attribute =  ((IIR_Declaration *)retval)->_get_attribute_name();
    if( my_attribute != NULL ){
      retval = (IIR *)my_attribute;
    }
  }
  return retval;
}


void 
IIRScram_SensitizedProcessStatement::_publish_vhdl(ostream &_vhdl_out) {
  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ": ";
  }
    
  if (get_postponed() == true) {
    _vhdl_out << "postponed ";
  }
    
  _vhdl_out << "process";

  if(sensitivity_list.num_elements() != 0) {
    _vhdl_out << " (";
    sensitivity_list._publish_vhdl(_vhdl_out);
    _vhdl_out << ")\n";
  }
    
  process_declarative_part._publish_vhdl_decl(_vhdl_out);
  _vhdl_out << "\nbegin\n";
  process_statement_part._publish_vhdl(_vhdl_out);
  _vhdl_out << "end";
  if (get_postponed() == true) {
    _vhdl_out << " postponed";
  }
  _vhdl_out << " process";
  if (get_label() != NULL) {
    _vhdl_out << " ";
    get_label()->_publish_vhdl(_vhdl_out);
  }
  _vhdl_out << ";\n\n";
}


void
IIRScram_SensitizedProcessStatement::_publish_cc() {
  IIR* process;

  process = _transmute();
  process->_publish_cc();
}


IIR * 
IIRScram_SensitizedProcessStatement::_transmute() {
  IIR_ProcessStatement* pstmt = new IIR_ProcessStatement;
  IIR_WaitStatement* wstmt = new IIR_WaitStatement;

  copy_location( this, wstmt );
  wstmt->sensitivity_list = sensitivity_list;

  copy_location( this, pstmt );
  pstmt->set_label(get_label());
  pstmt->set_postponed(get_postponed());
  pstmt->process_declarative_part = process_declarative_part;
  pstmt->process_statement_part = process_statement_part;
  pstmt->process_statement_part.append(wstmt);

  return pstmt;  
}


void
IIRScram_SensitizedProcessStatement::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  sensitivity_list._get_list_of_input_signals(list);
  process_statement_part._get_list_of_input_signals(list);
}


#ifdef PROCESS_COMBINATION
void 
IIRScram_SensitizedProcessStatement::
_static_elaborate(IIR_ArchitectureDeclaration *arch,
		  IIR_DeclarationList *cfglist,
		  char *hier_location) {
  ostrstream newname;
  IIR_Label *label;
  IIR_Char *text;

  newname << hier_location;
  label = get_label();
  if (label != NULL) {
    text = label->get_declarator()->get_text();
    for( IIR_Int32 i = 0; i < label->get_declarator()->get_text_length(); i++ ){
      newname << text[i];
    }
  }
  newname << ends;
  text = newname.str();

#ifdef DEBUG_ELAB
  cout << "elaborated a sensitized process named " << text << "\n";
#endif
  
  ASSERT(label->get_declarator()->get_kind() == IIR_IDENTIFIER);
  ((IIR_Identifier *)label->get_declarator())->release();
  label->set_declarator(IIR_Identifier::get(text, strlen(text)));
}
#endif


IIR *
IIRScram_SensitizedProcessStatement::_clone() {
  IIR *cstmt;
  cstmt = _transmute();
  cstmt = cstmt->_clone();
  return cstmt;
}

