// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_FunctionCall.cc,v 1.4 1999/10/21 15:13:34 dmadhava Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_FunctionCall.hh"
#include "IIR_AssociationElement.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "StandardPackage.hh"

IIRScram_FunctionCall::~IIRScram_FunctionCall() {}

set<IIR_Declaration> *
IIRScram_FunctionCall::_symbol_lookup(){
  ASSERT( get_implementation() != NULL );
  return new set<IIR_Declaration>( get_implementation() );
}

ostream &
IIRScram_FunctionCall::_print( ostream &os ){
  ASSERT(get_implementation() != NULL);

  os << *get_implementation()->get_declarator();
  if(parameter_association_list.num_elements() != 0) {
    os << "(";
    os << parameter_association_list;
    os << ") ";
  }

  return os;
}

void 
IIRScram_FunctionCall::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT(get_implementation() != NULL);

  get_implementation()->get_declarator()->_publish_vhdl(_vhdl_out);
  if(parameter_association_list.num_elements() != 0) {
    _vhdl_out << "(";
    parameter_association_list._publish_vhdl_without_formals(_vhdl_out);
    _vhdl_out << ") ";
  }
}

void
IIRScram_FunctionCall::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  parameter_association_list._build_sensitivity_list(sensitivity_list);
}

void
IIRScram_FunctionCall::_add_decl_into_cgen_symbol_table() {
  IIR_AssociationElement *elem = parameter_association_list.first();

  while(elem != NULL) {
    if(elem->_get_actual() != NULL) {
      elem->_get_actual()->_add_decl_into_cgen_symbol_table();
    }
    elem = parameter_association_list.successor(elem);
  }
}

IIR_FunctionCall *
IIRScram_FunctionCall::_build_function_call( IIR_SubprogramDeclaration *func_decl,
					     IIR_IndexedName * ){
  IIR_FunctionCall *retval = new IIR_FunctionCall;
  copy_location( func_decl, retval );
  retval->set_implementation( func_decl );
  return retval;
}

void
IIRScram_FunctionCall::_publish_cc_as_operator() {
  ASSERT ( get_implementation() != NULL );
  get_implementation()->_publish_cc();
  _cc_out << "(";
  parameter_association_list._publish_cc();

  if ((IIR_TextLiteral::_cmp(get_implementation()->get_declarator(), "\"&\"") == 0) && (parameter_association_list._are_all_scalars() == TRUE)) {
    ASSERT ( _get_subtype() != NULL );
    _cc_out << ", (";
    _get_subtype()->_publish_cc_type_name();
    _cc_out << " *) NULL";
  }
  
  _cc_out << ")";
}
    
void 
IIRScram_FunctionCall::_publish_cc() {
  ASSERT(get_implementation() != NULL);
  
  if ((get_implementation()->_is_operator() == TRUE) &&
      (get_implementation()->_is_implicit_declaration() == TRUE)) {
    _publish_cc_as_operator();
    return;
  }
  
  IIR_TextLiteral *declarator = get_implementation()->get_declarator();
  if((IIRScram_Identifier::_cmp(declarator, "now") == 0)) {
    //Eventually has to add the check if the function now() is
    //from standard package
    // &&
    //     (get_implementation()->_get_declarative_region())){
    // _cc_out << "savantnow";
    get_implementation()->_publish_cc();
    return;
  }
  else {
    // Special handling is required for "now"call within a subprogram.
    if(_is_currently_publishing_subprogram() == TRUE) {
      if(IIRScram_Identifier::_cmp(declarator, "now") == 0) {
	_cc_out << "processPtr->";
	get_implementation()->_publish_cc();
	_cc_out << "(";
      } else {
	// Pass on the process ptr to any function that is being called from
	// this function.
	get_implementation()->_publish_cc();
	_cc_out << "(processPtr";
      }
    } else {
      get_implementation()->_publish_cc();
      switch(_get_currently_publishing_unit()) {
      case IIRScram::ENTITY_DECL:
      case IIRScram::ARCHITECTURE_DECL:
      case IIRScram::PACKAGE:
      case IIRScram::PACKAGE_BODY:
      case IIRScram::PROCESS_STATE:
      case IIRScram::CONFIGURATION_DECLARATION:	
      case IIRScram::GENERIC_MAP:
	_cc_out << "(NULL";
	break;
      case IIRScram::PROCESS:
      case IIRScram::BLOCK:
	_cc_out << "(this";
	break;
      default:
	cerr << "Yet to Consider Function call invocation in the Publishing Unit: "  << (int)_get_currently_publishing_unit() << endl;
	cerr << "Assuming publishing a process statement:" << endl;
	_cc_out << "(this";
	break;
      }
    }
    
    if(parameter_association_list.num_elements() != 0) {
      //     parameter_association_list._publish_cc_subprogram_arguments();
      _cc_out << ", ";
      parameter_association_list._publish_cc();
    }
    _cc_out << ")";
  }
}
void 
IIRScram_FunctionCall::_publish_cc_elaborate() {
  _publish_cc();
}

// This method is called from
// IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_Add.
// This happens ONLY when this is used as a type conversion function.
// Then, this function call MUST have ONLY one argument.
void 
IIRScram_FunctionCall::_publish_cc_elaborate_arg() {
  ASSERT(parameter_association_list.num_elements() == 1);
  parameter_association_list.first()->_publish_cc_elaborate();
}

void 
IIRScram_FunctionCall::_publish_cc_bounds() {
  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->get_kind() == IIR_FUNCTION_DECLARATION );

  ((IIR_FunctionDeclaration *)get_implementation())->get_return_type()->_publish_cc_bounds();
}
  
void 
IIRScram_FunctionCall::_type_check( set<IIR_TypeDefinition> *possible_types ){
  ASSERT(get_implementation() != NULL);
  IIR_SubprogramDeclaration *my_decl = get_implementation();
  
  if(parameter_association_list._is_resolved() == FALSE) {
    parameter_association_list._resolve_and_order( &my_decl->interface_declarations, NULL );
  }
  
  parameter_association_list._fill_in_defaults( this, &my_decl->interface_declarations );

  ASSERT( parameter_association_list._is_resolved() == TRUE );
}

void
IIRScram_FunctionCall::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  parameter_association_list._get_list_of_input_signals(list);

  // There isn't any real type checking - it was all done during the
  // semantic_transform of the indexed_name.
}

IIR *
IIRScram_FunctionCall::_rval_to_decl( IIR_TypeDefinition *my_type ){
  // These are some simple sanity checks...
  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->_get_subtype() != NULL );
  ASSERT( my_type != NULL );
  ASSERT(  get_implementation()->_get_subtype()->_is_compatible( my_type ) != NULL );

  return (IIR *)this;
}

IIR_TypeDefinition *
IIRScram_FunctionCall::_get_subtype(){
  ASSERT( get_implementation() != NULL );
  ASSERT(get_implementation()->get_kind() == IIR_FUNCTION_DECLARATION);
  IIR_FunctionDeclaration* func_decl = (IIR_FunctionDeclaration*)get_implementation();
  return func_decl->get_return_type();  
}

set<IIR_TypeDefinition> *
IIRScram_FunctionCall::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  ASSERT( get_implementation() != NULL );

  return get_implementation()->_get_rval_set(constraint_function);
}

void
IIRScram_FunctionCall::_publish_cc_initialization_value()
{
  _publish_cc();
}

void
IIRScram_FunctionCall::_publish_cc_first_objectParameter()
{
  // This function cannot be called on function call nodes with no
  // parameters. This function is for TCs that usually have a parameter
  ASSERT (parameter_association_list.first() != NULL);

  parameter_association_list.first()->_publish_cc_first_objectParameter();
}

void
IIRScram_FunctionCall::_publish_cc_kernel_type()
{
  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->get_kind() == IIR_FUNCTION_DECLARATION );
  
  ((IIR_FunctionDeclaration *)get_implementation())->get_return_type()->_publish_cc_kernel_type();
}

IIR*
IIRScram_FunctionCall::_clone() {
  IIR_FunctionCall *funccall = new IIR_FunctionCall();

  IIR_Expression::_clone(funccall);
  funccall->set_implementation(get_implementation());
  
  IIR_AssociationElement *newassoc, *assoc = parameter_association_list.first();
  while (assoc != NULL) {
    newassoc = (IIR_AssociationElement*)assoc->_clone();
    funccall->parameter_association_list.append(newassoc);
    assoc = parameter_association_list.successor(assoc);
  }

  return funccall;
}

IIR_Mode
IIRScram_FunctionCall::_get_mode() {
  ASSERT(parameter_association_list.num_elements() == 1);
  ASSERT(parameter_association_list.first()->_is_signal() == TRUE);

  return parameter_association_list.first()->_get_actual()->_get_mode();
}

IIR_Declaration *
IIRScram_FunctionCall::_find_formal_declaration(){
  ASSERT( _is_resolved() == TRUE );
  ASSERT( parameter_association_list.num_elements() == 1 );
  ASSERT( parameter_association_list.first() != NULL );
  return parameter_association_list.first()->_find_formal_declaration();
}

IIR_Boolean 
IIRScram_FunctionCall::_is_globally_static_primary(){
  IIR_Boolean retval = TRUE;

  ASSERT( get_implementation() != NULL );
  ASSERT( get_implementation()->get_kind() == IIR_FUNCTION_DECLARATION );
  if( ((IIR_FunctionDeclaration *)get_implementation())->get_pure() == IIR_IMPURE_FUNCTION ){
    retval = FALSE;
  }
  
  if( parameter_association_list._is_globally_static_for_function() == FALSE ){
    retval = FALSE;
  }

  return retval;
}

IIR_Boolean
IIRScram_FunctionCall::_is_static_expression() {
  if (_is_globally_static_primary() == FALSE)  {
     return FALSE;
  }

  IIR_AssociationElement *current = parameter_association_list.first();
  while (current != NULL)  {
    if (current->_is_static_expression() == FALSE)  {
      return FALSE;
    }
    current = parameter_association_list.successor(current);
  }

  return TRUE;
}
