
// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_DisconnectSpecification.cc,v 1.2 1999/03/09 20:54:03 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include <strstream.h>
#include "IIRScram_DisconnectSpecification.hh"
#include "IIR_PhysicalSubtypeDefinition.hh"
#include "IIR_DesignatorList.hh"
#include "IIR_Designator.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "StandardPackage.hh"

IIRScram_DisconnectSpecification::~IIRScram_DisconnectSpecification(){}

void 
IIRScram_DisconnectSpecification::_publish_vhdl_decl(ostream &_vhdl_out) {
  _publish_vhdl( _vhdl_out );
}

void 
IIRScram_DisconnectSpecification::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT( guarded_signal_list._is_resolved() == TRUE );
  ASSERT( get_type_mark()->_is_resolved() == TRUE );
  ASSERT( get_time_expression()->_is_resolved() == TRUE );

  _vhdl_out << " disconnect ";
  guarded_signal_list._publish_vhdl(_vhdl_out);
  _vhdl_out << " : ";
  get_type_mark()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " after ";
  get_time_expression()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ";\n";
}

void 
IIRScram_DisconnectSpecification::_type_check( IIR_DeclarationList &declarative_region ){
  ASSERT( guarded_signal_list.num_elements() > 0 );
  ASSERT( get_time_expression() != NULL );
  ASSERT( get_type_mark() != NULL );
      
  set<IIR_TypeDefinition> type_mark_set( get_type_mark() );
  set<IIR_TypeDefinition> *signal_types = NULL;
  IIR_Designator *current_designator = guarded_signal_list.first();
  while( current_designator != NULL ){
    if( current_designator->get_kind() == IIR_DESIGNATOR_EXPLICIT ){
      signal_types = current_designator->_get_rval_set( &IIR::_is_signal );
      if( signal_types == NULL ){
	report_undefined_symbol( current_designator );
	goto next;
      }

      reconcile_sets( signal_types, &type_mark_set );
      switch( signal_types->num_elements() ){
      case 0:{
	ostrstream err;
	err << "No signal |" << *current_designator << "| of type << " << *get_type_mark()
	    << " defined in this scope." << ends;
	report_error( current_designator, err );
	goto next;
      }
      case 1:{
	IIR_TypeDefinition *correct_type = signal_types->get_element();
	IIR *new_designator = current_designator->_semantic_transform( correct_type );
	ASSERT( new_designator == current_designator );
	current_designator->_type_check( correct_type );
	new_designator = current_designator->_rval_to_decl( correct_type );
	ASSERT( new_designator == current_designator );

	if( new_designator->_is_locally_static() == FALSE ){
	  ostrstream err;
	  err << "Signal |" << *current_designator << "| is not locally static.  Only "
	      << "locally static signals are allowed in disconnection specifications." << ends;
	  report_error( current_designator, err );
	}
	if( new_designator->_is_guard_signal() == FALSE ){
	  ostrstream err;
	  err << "Signal |" << *current_designator << "| is not a guarded signal.  Only "
	      << "guard signals are allowed in disconnection specifications." << ends;
	  report_error( current_designator, err );
	}

	break;
      }
      default:{
	report_ambiguous_error( current_designator, signal_types );
	break;
      }
      }
    }
    else{
      ostrstream err;
      err << "No support for disconnection specifications with OTHERS or ALL yet!" << ends;
      report_error( current_designator, err );
      abort();
    } 
    next:
      delete signal_types;
      current_designator = guarded_signal_list.successor( current_designator );
  }

  set<IIR_TypeDefinition> *time_expression_types = get_time_expression()->_get_rval_set();
  set<IIR_TypeDefinition> *time_types = new set<IIR_TypeDefinition>(StandardPackage::time_type);

  if( time_expression_types == NULL ){
    report_undefined_symbol( get_time_expression() );
    goto finish;
  }
  
  ASSERT( time_types != NULL && time_types->num_elements() == 1 );
  reconcile_sets( time_expression_types, time_types );
  switch( time_expression_types->num_elements() ){
  case 0:{
    ostrstream err;
    err << "Expression |" << *get_time_expression() << "| must have builtin type TIME" << ends;
    report_error( get_time_expression(), err );
    goto finish;
  }
  case 1:{
    IIR_TypeDefinition *time_rval = time_expression_types->get_element();

    set_time_expression( get_time_expression()->_semantic_transform( time_rval ) );
    get_time_expression()->_type_check( time_rval );
    set_time_expression( get_time_expression()->_rval_to_decl( time_rval ) );

    break;
  }
  default:{
    report_ambiguous_error( get_time_expression(), time_expression_types );
    goto finish;
  }
  }

 finish:
  delete time_types;
  delete time_expression_types;
}
