// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Swaminathan Subramanian  ssubrama@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_Declaration.cc,v 1.7 1999/10/07 14:05:02 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_Declaration.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_Attribute.hh"
#include "IIR_AttributeDeclaration.hh"
#include "IIR_AttributeSpecification.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_DesignatorList.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_Label.hh"
#include "IIR_PackageDeclaration.hh"
#include "IIR_PackageBodyDeclaration.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_StringLiteral.hh"
#include "error_func.hh"
#include "symbol_table.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_LibraryDeclaration.hh"

extern symbol_table *cgen_sym_tab_ptr;

IIRScram_Declaration::IIRScram_Declaration(){
  // Assumes declarations aren't implict, and visible, by default.
  _visible_flag = TRUE;
  _implicit_flag = FALSE;
  attribute_name = NULL;
  declarative_region = NULL;
  mangledDeclarator = NULL;
  implicit_declarations = NULL;
  implemented_attributes_in_constructor = NULL;
  implemented_attributes_in_state = NULL;
  implemented_attributes_in_initstate = NULL;  
}


IIRScram_Declaration::~IIRScram_Declaration(){
  if (mangledDeclarator != NULL) {
    delete mangledDeclarator;
  }
}


IIRScram_Declaration::declaration_type 
IIRScram_Declaration::_get_type(){
  return ERROR;
}

IIR_AttributeSpecification*
IIRScram_Declaration::_get_attribute_specification(IIR* attr){
  ASSERT(attr->_get_prefix_declaration()->get_kind() 
	 == IIR_ATTRIBUTE_DECLARATION);
  IIR_AttributeDeclaration* attr_decl = 
    (IIR_AttributeDeclaration *) attr->_get_prefix_declaration();
  IIR_AttributeSpecificationList* attr_spec_list = _get_attribute_specification_list();
  ASSERT(attr_spec_list != NULL);
  IIR_AttributeSpecification* attr_spec = attr_spec_list->first();

  while(attr_spec != NULL) {
    if(attr_spec->_get_declaration() == attr_decl) {
      return attr_spec;
    }
    attr_spec = attr_spec_list->successor(attr_spec);
  }
  return NULL;
}


IIR_Boolean
IIRScram_Declaration::_is_homograph_of( IIR_Declaration *to_check ){
  IIR_Boolean retval;

  if( IIR_TextLiteral::_cmp( get_declarator(), to_check->get_declarator() ) == 0 ){
    int olCount = 0;
    if( _is_overloadable() == TRUE ){
      olCount++;
    }
    if( to_check->_is_overloadable() == TRUE ){
      olCount++;
    }

    switch( olCount ){
    case 0:
    case 1:{
      retval = TRUE;
      break;
    }
    case 2:{
      retval = FALSE;
      break;
    }
    default:{
      ostrstream err;
      err << "Internal error in IIRScram_Declaration::_is_homograph_of( IIR_Declaration"
	  << " *to_check ) - olCount == " << olCount << ends;
      report_error( this, err );
      abort();
    }
    }
    
  }
  return retval;
}


void 
IIRScram_Declaration::_publish_vhdl(ostream &_vhdl_out) {
  // This takes care of the problem where an implicit declaration name
  // gets published instead of an actual attribute name.
  if( _get_attribute_name() == NULL ){
    get_declarator()->_publish_vhdl(_vhdl_out);
  }
  else{
    _get_attribute_name()->_publish_vhdl(_vhdl_out);
  }
}


void 
IIRScram_Declaration::_publish_vhdl_decl(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl_decl(ostream &)");
}


void
IIRScram_Declaration::_publish_cc_addChild() {
  ASSERT(_is_signal() == TRUE);
  ASSERT(_current_publish_name != NULL);
  char *ptr = _current_publish_name; // Set in IIRScram_ProcessStatement

  if(_is_resolved_signal() == TRUE || _have_added_driver() == FALSE) {
    _cc_out << "{" << endl
	    << "VHDLType* ptr = new ";
    _publish_cc_type_name();
    _publish_cc_constructor_args();
    _cc_out << ";" << endl;
    if(_is_composite_resolved_signal() == TRUE) {
      _cc_out << "ptr->setParentCompositeType(ptr);" << endl;
    }
    _cc_out << "addChild(";

    if ((_current_publish_node != NULL) && (_current_publish_node->_is_concurrent_statement() == TRUE)) {
      ((IIR_ConcurrentStatement *) _current_publish_node)->_publish_cc_scoping_prefix( _get_declarative_region(), _current_publish_node );
    }
    
    _publish_cc_elaborate();
    _cc_out << ", *(ptr)" << ptr;
    //_set_driver_addition_flag(TRUE);
    _cc_out << "};" << endl;
    if(_is_resolved_signal() == TRUE) {
      _cc_out << "setResolutionFunctionId(";
      if ((_current_publish_node != NULL) && (_current_publish_node->_is_concurrent_statement() == TRUE)) {
	((IIR_ConcurrentStatement *) _current_publish_node)->_publish_cc_scoping_prefix( _get_declarative_region(), _current_publish_node );
      }
      _publish_cc_elaborate();
      _cc_out << ", ";
      _get_subtype()->_get_resolution_function()->_publish_cc_resolution_function_name();
      _cc_out << ");" << endl;
    }
  }
}


void
IIRScram_Declaration::_publish_cc_locatesignal() {
  ASSERT(_is_signal() == TRUE);
  _cc_out << "ptr = ";
  _get_declarator()->_publish_cc();
  _cc_out << ".locateSig(sigId);" << endl;
  _cc_out << "if( ptr != NULL) {" << endl
	  << "  return ptr;" << endl
	  << "}" << endl;
}


void 
IIRScram_Declaration::_publish_cc_declarative_region() {
  if(_get_declarative_region() != NULL) {
    IIR* my_decl_region = _get_declarative_region();
    switch(my_decl_region->get_kind()) {
    case IIR_FOR_LOOP_STATEMENT:
    case IIR_FUNCTION_DECLARATION:
    case IIR_PROCEDURE_DECLARATION:
    case IIR_PROCESS_STATEMENT:
    case IIR_SENSITIZED_PROCESS_STATEMENT:
    case IIR_ARCHITECTURE_DECLARATION:
    case IIR_ENTITY_DECLARATION:
      // Nothing needs to be done.
      break;
    case IIR_PACKAGE_DECLARATION:
      ((IIR_PackageDeclaration*) my_decl_region)->_publish_cc_package_name();
      _cc_out << "::";
      break;
    case IIR_PACKAGE_BODY_DECLARATION:
      ((IIR_PackageBodyDeclaration*) my_decl_region)->_publish_cc_package_name();
      _cc_out << "::";
      break;
    default:
      cerr << "IIRScram_Declaration::_publish_cc_declarative_region(): "
	   << my_decl_region->get_kind_text() << " not handled yet." << endl;
      break;
    }
  }
}

  
void
IIRScram_Declaration::_publish_cc_implicit_signal_attributes(){
  if( implicit_declarations != NULL ){
    IIR_Declaration* implied_declarations = implicit_declarations->get_element();
    while(implied_declarations != NULL) {
      if(cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	implied_declarations->_publish_cc_decl();
	if(implied_declarations->_is_signal() == TRUE) {
	  implied_declarations->_publish_cc_implicit_signal_attributes();
	}
      }
      implied_declarations = implicit_declarations->get_next_element();
    }
  }
}


void
IIRScram_Declaration::_publish_cc_implicit_state_objects_init(){
  ASSERT(_is_signal() == TRUE);
  
  if( implicit_declarations != NULL ){
    IIR_Declaration* implied_declarations = implicit_declarations->get_element();
    while(implied_declarations != NULL) {
      if(cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	implied_declarations->_publish_cc_state_object_init();
	if(implied_declarations->_is_signal() == TRUE) {
	  implied_declarations->_publish_cc_implicit_state_objects_init();
	}
      }
      implied_declarations = implicit_declarations->get_next_element();
    }
  }
}


void
IIRScram_Declaration::_publish_cc_implicit_signal_type() {
  IIR_Attribute* attribute = _get_attribute_name();
  ASSERT(_is_signal() == TRUE);
  attribute->_publish_cc_type_name();
}


void
IIRScram_Declaration::_publish_cc_name() {
  _get_declarator()->_publish_cc();
}


void 
IIRScram_Declaration::_publish_cc_declarator() {
  _get_declarator()->_publish_cc();
}


void
IIRScram_Declaration::_publish_cc_decl() {
  _report_undefined_scram_fn("publish_cc_decl()");
}


void
IIRScram_Declaration::_publish_cc_sigdest() {
  _report_undefined_scram_fn("publish_cc_sigdest()");
}


void
IIRScram_Declaration::_publish_cc_decl_with_constructor_args(){
  _report_undefined_scram_fn("_publish_cc_decl_with_constructor_args()");
}


void 
IIRScram_Declaration::_publish_cc_binding_name(ostream& outstream) {
  _report_undefined_scram_fn("_publish_cc_binding_name()");
}


void
IIRScram_Declaration::_publish_cc_init_signal() {
  _report_undefined_scram_fn("_publish_cc_init_signal()");
}


void
IIRScram_Declaration::_publish_cc_init_function() {
  _report_undefined_scram_fn("_publish_cc_init_sfunction()");
}


void
IIRScram_Declaration::_publish_cc_value() {
  _report_undefined_scram_fn("_publish_cc_value()");
}


void
IIRScram_Declaration::_publish_cc_constructor_args(){
  _report_undefined_scram_fn("_publish_cc_constructor_args");
}


void 
IIRScram_Declaration::_publish_cc_state_object_init(){
  _report_undefined_scram_fn("_publish_cc_state_object_init()");
}

void 
IIRScram_Declaration::_publish_cc_type_info(){
  _report_undefined_scram_fn("_publish_cc_type_info()");
}

void 
IIRScram_Declaration::_publish_cc_extern_type_info(){
  _report_undefined_scram_fn("_publish_cc_extern_type_info()");
}

void
IIRScram_Declaration::_publish_cc_implicit_signal_attributes_copying(){
  if( implicit_declarations != NULL ){
    IIR_Declaration* implied_declarations = implicit_declarations->get_element();
    while(implied_declarations != NULL) {
      if(cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	implied_declarations->_publish_cc_necessary_copying();
	if(implied_declarations->_is_signal() == TRUE) {
	  implied_declarations->_publish_cc_implicit_signal_attributes_copying();
	}
      }
      implied_declarations = implicit_declarations->get_next_element();
    }
  }
}

void
IIRScram_Declaration::_publish_cc_implicit_signal_attributes_read_or_write(const char *functionName, const char *streamName){
  if( implicit_declarations != NULL ){
    IIR_Declaration* implied_declarations = implicit_declarations->get_element();
    while(implied_declarations != NULL) {
      if(cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	implied_declarations->_publish_cc_read_or_write(functionName, streamName);
	if(implied_declarations->_is_signal() == TRUE) {
	  implied_declarations->_publish_cc_implicit_signal_attributes_read_or_write(functionName, streamName);
	}
      }
      implied_declarations = implicit_declarations->get_next_element();
    }
  }
}


void
IIRScram_Declaration::_publish_cc_necessary_copying() {
  _report_undefined_scram_fn("_publish_cc_necessary_copying()");
}

void
IIRScram_Declaration::_publish_cc_read_or_write(const char *, const char *) {
  _report_undefined_scram_fn("_publish_cc_read_or_write(const char *, const char *)");
}


void
IIRScram_Declaration::_publish_cc_constant_object_init() {
  _report_undefined_scram_fn("_publish_cc_constant_object_init()");
}


IIR_DesignatorList *
IIRScram_Declaration::_get_instantiation_list() {
  _report_undefined_scram_fn("_get_instantiation_list()");
  return NULL;
}


void
IIRScram_Declaration::_get_headers(set<IIR>&) {
  _report_undefined_scram_fn("_get_headers(set<IIR>&)");
}


IIR_Identifier *
IIRScram_Declaration::_get_mangled_declarator(){
  return mangledDeclarator;
}


void
IIRScram_Declaration::_set_mangled_declarator(char * mangledName){
  if ( mangledName != NULL ) {
    mangledDeclarator = IIR_Identifier::get(mangledName, strlen(mangledName));
  }
}


void
IIRScram_Declaration::_set_driver_addition_flag(IIR_Boolean) {
  _report_undefined_scram_fn("_set_driver_addition_flag(IIR_Boolean)");
}


bool 
IIRScram_Declaration::_check_param( IIR_TypeDefinition *decl, int arg_num ){
  return false;
}


// For the next two methods, since we're assuming that we've already
// correctly resolved everything, we'll just new a set with "this" in it.
set<IIR_Declaration> *
IIRScram_Declaration::_symbol_lookup(){
  return new set<IIR_Declaration>( (IIR_Declaration *)this );
}

set<IIR_Declaration> *
IIRScram_Declaration::_symbol_lookup( set<IIR_Declaration> *look_in ){
  return new set<IIR_Declaration>( (IIR_Declaration *)this );
}

IIR_Declaration*
IIRScram_Declaration::_get_prefix_declaration() {
  return ((IIR_Declaration*)this);
}


IIR_Boolean
IIRScram_Declaration::_is_composite_resolved_signal() {
  if(_is_signal() == TRUE) {
    if((_get_subtype()->_get_resolution_function() != NULL) &&
       ((_get_subtype()->_is_array_type() == TRUE)|| 
	(_get_subtype()->_is_array_type() == TRUE))) {
      return TRUE;
    }
    else {
      return FALSE;
    }
  }
  else {
    return FALSE;
  }
}


void
IIRScram_Declaration::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  switch(get_kind()) {
  case IIR_SIGNAL_DECLARATION:
  case IIR_SIGNAL_INTERFACE_DECLARATION:
    {
      // We do not need implicit signal declarations like s'quiet in this
      // list.
      if(_is_implicit_declaration() != TRUE) {
	list->add((IIR_Declaration*)this);
	if (get_value() != NULL) {
	  get_value()->_get_list_of_input_signals(list);
	}
	break;
      }
    }
  case IIR_ALIAS_DECLARATION:
    {
      IIR_AliasDeclaration* alias_decl = (IIR_AliasDeclaration*) this;
      if(alias_decl->_is_signal()) {
	list->add(alias_decl);
      }
      break;
    }
  }
}


void
IIRScram_Declaration::_get_signal_source_info(set<IIR_Declaration>* siginfo) {
  siginfo->add((IIR_Declaration*)this);
}


IIR_Boolean 
IIRScram_Declaration::_is_physical_type(){
  if( _get_subtype() != NULL ){
    return _get_subtype()->_is_physical_type();
  }
  else{
    return FALSE;
  }
}


// This funnction needs work.
//If the type is a composite type, the sub-elements may be resolved
//Subtype. In that case, the signal is considered as a resolved signnal
//This function is yet to take care of such cases
IIR_Boolean 
IIRScram_Declaration::_is_resolved_signal() {
  if(_get_subtype()->_get_resolution_function() != NULL) {
    return TRUE;
  }
  else {
    if(_get_subtype()->_is_iir_array_type_definition() == TRUE) {
      IIR_ArrayTypeDefinition* arr_type_def = (IIR_ArrayTypeDefinition *)_get_subtype();
      IIR_TypeDefinition *sub_element_type_def = arr_type_def->get_element_subtype();
      
      return sub_element_type_def->_is_resolved_signal();
    }
    else {
      return FALSE;
    }
  }
}


IIR_Boolean 
IIRScram_Declaration::_is_textio(){
  if(IIRScram_TextLiteral::_cmp(get_declarator(), "textio") == 0) {
    return TRUE;
  }
  else {
    return FALSE;
  }
}


IIR_Boolean 
IIRScram_Declaration::_is_standard(){
  if(IIRScram_TextLiteral::_cmp(get_declarator(), "standard") == 0) {
    return TRUE;
  }
  else {
    return FALSE;
  }
}


IIR_Boolean 
IIRScram_Declaration::_is_enumeration_type(){
  if( _get_subtype() != NULL ){
    return _get_subtype()->_is_enumeration_type();
  }
  else{
    return FALSE;
  }
}


IIR_Attribute *
IIRScram_Declaration::_get_attribute_name(){
  return attribute_name;
}


void 
IIRScram_Declaration::_set_attribute_name( IIR_Attribute *new_name ){
  attribute_name = new_name;
}

IIR_Declaration*
IIRScram_Declaration::_get_implicit_declaration_for_attribute ( IIR_Attribute *findAttrib ) {
  // Walk the list of implicit declarations and find if there is a function
  // declaration whose attribute is == findAttrib. If so return that
  // function declaration. Else return NULL. This function is called from
  // IIRScram_Attribute::_check_and_add_necessary_decls() to handle attributes
  // on indexed names.
    
  if( implicit_declarations != NULL ){
    IIR_Declaration* implied_declarations = implicit_declarations->get_element();
    while(implied_declarations != NULL) {
      if (implied_declarations->_get_attribute_name()->get_kind() == findAttrib->get_kind()) {
	return implied_declarations;
      }
      
      implied_declarations = implicit_declarations->get_next_element();
    }
  }

  return NULL;
}  

set<IIR_Declaration> *
IIRScram_Declaration::_get_implicit_declarations(){
  return implicit_declarations;
}


void 
IIRScram_Declaration::_set_implicit_declarations( set<IIR_Declaration> *new_set ){
  ASSERT( new_set != NULL );
  implicit_declarations = new_set;
}


IIR_Declaration*
IIRScram_Declaration::_get_signal_decl() {
  _report_undefined_scram_fn("_get_signal_decl()");
  return NULL;
}


IIR_Boolean
IIRScram_Declaration::_is_published_attribute_in_constructor(SignalAttribute attrib) {
  ASSERT(get_kind() == IIR_SIGNAL_DECLARATION ||
	 get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
  ASSERT(implemented_attributes_in_constructor != NULL);
  return implemented_attributes_in_constructor[int(attrib)];
}


void 
IIRScram_Declaration::_add_published_attribute_in_constructor(SignalAttribute attrib) {
  ASSERT(get_kind() == IIR_SIGNAL_DECLARATION ||
	 get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
  ASSERT(implemented_attributes_in_constructor != NULL &&
	 implemented_attributes_in_constructor[int(attrib)] == FALSE);

  implemented_attributes_in_constructor[int(attrib)] = TRUE;
}


IIR_Boolean
IIRScram_Declaration::_is_published_attribute_in_state(SignalAttribute attrib) {
  ASSERT(get_kind() == IIR_SIGNAL_DECLARATION ||
	 get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
  ASSERT(implemented_attributes_in_state != NULL);

  return implemented_attributes_in_state[int(attrib)];
}


void 
IIRScram_Declaration::_add_published_attribute_in_state(SignalAttribute attrib) {
  ASSERT(get_kind() == IIR_SIGNAL_DECLARATION ||
	 get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
  ASSERT(implemented_attributes_in_state != NULL &&
	 implemented_attributes_in_state[int(attrib)] == FALSE);

  implemented_attributes_in_state[int(attrib)] = TRUE;
}


IIR_Boolean
IIRScram_Declaration::_is_published_attribute_in_initstate(SignalAttribute attrib) {
  ASSERT(get_kind() == IIR_SIGNAL_DECLARATION ||
	 get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
  ASSERT(implemented_attributes_in_initstate != NULL);

  return implemented_attributes_in_initstate[int(attrib)];
}


const IIR_Char*
IIRScram_Declaration::_get_mangling_prefix() {
  return ((const IIR_Char *) "D");
}


void
IIRScram_Declaration::_mangle_declarator(){
  ostrstream mangledDeclarator;

  if ((_get_declarative_region() != NULL) && (this->get_kind() != IIR_LIBRARY_DECLARATION)) {
    if ((_get_declarative_region()->_is_textio() == FALSE) && (_get_declarative_region()->_is_standard() == FALSE))  {
      if (get_kind() == IIR_ARCHITECTURE_DECLARATION) {
	IIR_ArchitectureDeclaration* tempArch = (IIR_ArchitectureDeclaration *) this;
	ASSERT ( tempArch->get_kind() == IIR_ARCHITECTURE_DECLARATION );
	if (tempArch->_get_entity() != NULL) {
	  mangledDeclarator << *(tempArch->_get_entity()->_get_declarator());
	}
      }
      else {
	if (_get_declarative_region()->_get_declarator() != NULL) {
	  mangledDeclarator << *_get_declarative_region()->_get_declarator();
	}
      }
      
      mangledDeclarator << "_" << _get_mangling_prefix();
    }
  }

  if (_is_iir_library_declaration() == TRUE) {
    IIR_LibraryDeclaration *tempLib =
      (IIR_LibraryDeclaration *) this;
    ASSERT (tempLib->_is_iir_library_declaration() == TRUE);
    
    if (tempLib->_get_logical_name() != NULL) {
      mangledDeclarator << *(tempLib->_get_logical_name()) << ends;
    }
    else {
      mangledDeclarator << *get_declarator() << ends;
    }
  }
  else {
    mangledDeclarator << *get_declarator() << ends;
  }
  
  _set_mangled_declarator(mangledDeclarator.str());
}


void 
IIRScram_Declaration::_add_published_attribute_in_initstate(SignalAttribute attrib) {
  ASSERT(get_kind() == IIR_SIGNAL_DECLARATION ||
	 get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
  ASSERT(implemented_attributes_in_initstate != NULL &&
	 implemented_attributes_in_initstate[int(attrib)] == FALSE);

  implemented_attributes_in_initstate[int(attrib)] = TRUE;
}


ostream & 
IIRScram_Declaration::_print( ostream &os ){
  if( get_declarator() != NULL ){
    os << *get_declarator();
  }
  else{
    os << "<ANONYMOUS>";
  }

  return os;
}


IIR_TypeDefinition *
IIRScram_Declaration::_get_type_of_element( int ){
  _report_undefined_scram_fn("_get_type_of_element( int )");
  return NULL;
}


IIR_TypeDefinition *
IIRScram_Declaration::_get_type_of_param( int ){
  _report_undefined_scram_fn("_get_type_of_param( int )");
  return NULL;
}

IIR_TypeDefinition*
IIRScram_Declaration::_get_name_type() {
  return _get_subtype();
}


IIR_Int32 
IIRScram_Declaration::_get_num_indexes(){
  return 0;
}


IIR_Int32 
IIRScram_Declaration::_num_required_args(){
  return 0;
}


IIR_TextLiteral *
IIRScram_Declaration::_get_declarator() {
  if (_get_mangling_flag() == false)  {
    if (_get_currently_publishing_unit() != IIRScram::NONE_PARSING) {
      if (_get_mangled_declarator() == NULL) {
	_mangle_declarator();
      }
      return mangledDeclarator;
    }
    else {
      return get_declarator();
    }
  }
  else {
    if (_is_subprogram() == true) {
      if (_get_mangled_declarator() == NULL) {
	ostrstream subprogDeclarator;
	
	subprogDeclarator << "savant";
	
	if (get_declarator()->_is_string_literal() == TRUE) {
	  ((IIR_StringLiteral *) get_declarator())->_convert_function_name(subprogDeclarator);
	}
	else {
	  subprogDeclarator << *get_declarator();
	}
	
	subprogDeclarator << ends;
	
	_set_mangled_declarator(subprogDeclarator.str());
      }
      
      return _get_mangled_declarator();
    }
    else {
      return get_declarator();
    }
  }
}

IIR_TextLiteral *
IIRScram_Declaration::_get_prefix_string(){
  return get_declarator();
}


IIR*
IIRScram_Declaration::_clone() {
  _report_undefined_scram_fn("IIR *_clone()");
  return NULL;
}


void
IIRScram_Declaration::_clone(IIR *cl) {
  int i;
  IIRScram_Declaration *clone = (IIR_Declaration*)cl;
  IIRScram::_clone(clone);
  IIR_TextLiteral *declclone = (IIR_TextLiteral*)get_declarator()->_clone();
  clone->set_declarator(declclone);

  clone->_set_is_visible( _is_visible() );
  clone->_set_is_implicit( _is_implicit_declaration() );
  clone->_set_declarative_region( _get_declarative_region() );

  if ( _get_mangled_declarator() != NULL ) {
    clone->_set_mangled_declarator( _get_mangled_declarator()->get_text() );
  }

  IIR_Attribute *attrclone = _get_attribute_name();
  if (attrclone != NULL) {
    attrclone = (IIR_Attribute*)attrclone->_clone();
  }
  clone->_set_attribute_name(attrclone);

  if( implicit_declarations != NULL ){
    IIR_Declaration *impdecl = implicit_declarations->get_element();
    while (impdecl != NULL) {
      IIR_Declaration *impclone;
      impclone = (IIR_Declaration*)impdecl->_clone();
      if( clone->implicit_declarations == NULL ){
	clone->implicit_declarations = new set<IIR_Declaration>;
      }
      clone->implicit_declarations->add(impclone);
      impdecl = implicit_declarations->get_next_element();
    }
  }

  if ( implemented_attributes_in_constructor != NULL ) {
    clone->implemented_attributes_in_constructor = new IIR_Boolean[numSignalAttributes];
    for ( i = 0; i < numSignalAttributes; ++i ) {
      clone->implemented_attributes_in_constructor[i] = implemented_attributes_in_constructor[i];
    }
  }
  if ( implemented_attributes_in_state != NULL ) {
    clone->implemented_attributes_in_state = new IIR_Boolean[numSignalAttributes];
    for ( i = 0; i < numSignalAttributes; ++i ) {
      clone->implemented_attributes_in_state[i] = implemented_attributes_in_state[i];
    }
  }
  if ( implemented_attributes_in_initstate != NULL ) {
    clone->implemented_attributes_in_initstate = new IIR_Boolean[numSignalAttributes];
    for ( i = 0; i < numSignalAttributes; ++i ) {
      clone->implemented_attributes_in_initstate[i] = implemented_attributes_in_initstate[i];
    }
  }

}


#ifdef PROCESS_COMBINATION
void
IIRScram_Declaration::_static_elaborate_decl(char *prefix) {
  IIR_Identifier *new_id, *old_id;
  IIR_Char *value;
  IIR_Int32 length;
  ostrstream newname;

  ASSERT(get_declarator()->get_kind() == IIR_IDENTIFIER);

  old_id = (IIR_Identifier*)get_declarator();
  newname << prefix << *old_id << ends;
  value = newname.str();
  length = strlen(prefix) + old_id->get_text_length();
  new_id = IIR_Identifier::get(value, length);
  set_declarator(new_id);
#ifdef DEBUG_ELAB
  cout << "Changed declaration name |" << *old_id << "| to |" 
       << *new_id << "|\n";
#endif
  old_id->release();
}
#endif


IIR *
IIRScram_Declaration::_get_declarative_region(){
  return declarative_region;
}


void 
IIRScram_Declaration::_set_declarative_region( IIR *new_declarative_region ){
  declarative_region = new_declarative_region;
}


void 
IIRScram_Declaration::_add_to_declarative_region( set<IIR_Declaration> * ){
  _report_undefined_scram_fn("_add_to_declarative_region");
}


void 
IIRScram_Declaration::_add_to_declarative_region( IIR_DeclarationList &region,
						  set<IIR_Declaration> *set_to_add ){
  ASSERT( set_to_add != NULL );
  
  IIR_Declaration *current_decl = NULL;
#ifdef DEVELOPER_ASSERTIONS
  IIR_Declaration *decl_in_region = region.first();
  while( decl_in_region != NULL ){
    current_decl = set_to_add->get_element();
    while( current_decl != NULL ){
      ASSERT( current_decl != decl_in_region );
      current_decl = set_to_add->get_next_element();
    }
    decl_in_region = region.successor( decl_in_region );
  }
#endif

  current_decl = set_to_add->get_element();
  while( current_decl != NULL ){
    region.append( current_decl );
    current_decl = set_to_add->get_next_element();
  }
}


IIR_Boolean
IIRScram_Declaration::_in_process_statement() {
  if(_get_declarative_region()->get_kind() == IIR_PROCESS_STATEMENT) {
    return TRUE;
  }
  else {
    return FALSE;
  }
}

// This method is common for SignalDeclaration and
// SignalInterfaceDeclaration and hence defined here.
void
IIRScram_Declaration::_build_sensitivity_list(IIR_DesignatorList *sensitivity_list) {
  if(_is_signal() == FALSE) {
    return;
  }

  IIR_Designator *node = sensitivity_list->first();
  while (node != NULL) {
    ASSERT(node->get_kind() == IIR_DESIGNATOR_EXPLICIT);
    if (((IIR_DesignatorExplicit*)node)->get_name() == this) { break; }
    node = sensitivity_list->successor(node);
  }
  if (node == NULL) {
    IIR_DesignatorExplicit *new_designator = new IIR_DesignatorExplicit;
    copy_location( this, new_designator );
    new_designator->set_name(this);
    sensitivity_list->append(new_designator);
  }
}


void
IIRScram_Declaration::_add_declarations_in_initializations() {
  _report_undefined_scram_fn("_add_declarations_in_initializations()");
}


void
IIRScram_Declaration::_add_decl_into_cgen_symbol_table() {
  if((get_kind() != IIR_PHYSICAL_UNIT) && (_is_iir_element_declaration() == FALSE)) {
    _report_undefined_scram_fn("_add_decl_into_cgen_symbol_table()");
  }
}


void
IIRScram_Declaration::_add_declaration_and_open_scope( ){
  _get_symbol_table()->add_declaration( (IIR_Declaration *)this );
  _get_symbol_table()->open_scope( this );
}


void
IIRScram_Declaration::_add_declaration( ){
  _get_symbol_table()->add_declaration( (IIR_Declaration *)this );
}


void
IIRScram_Declaration::_close_scope( ){
  _get_symbol_table()->close_scope( this );
}


void 
IIRScram_Declaration::_make_interface_visible( symbol_table *sym_tab ){
  if( sym_tab == NULL ){
    _get_symbol_table()->make_visible( (IIR_Declaration *)this );
  }
  else{
    sym_tab->make_visible( (IIR_Declaration *)this );
  }
}


IIR_Declaration*
IIRScram_Declaration::_get_signal_prefix() {
  ASSERT(_is_signal() == TRUE);
  ASSERT(_is_implicit_declaration() == TRUE);

  IIR_Attribute* attached_attribute = _get_attribute_name();
  IIR* retval = NULL;

  ASSERT(attached_attribute != NULL);
  retval = attached_attribute->get_prefix();

  if (retval->_is_iir_declaration() == FALSE) {
    retval = retval->_get_prefix_declaration();
  }
  
  ASSERT( retval->_is_iir_declaration() == TRUE );

  return ((IIR_Declaration*)retval);
}


set<IIR_TypeDefinition> *
IIRScram_Declaration::_get_rval_set( IIR_Boolean (IIR::*constraint_function)() ){
  if (constraint_function != 0 ) {
    ASSERT((((IIR_Declaration *)this)->*constraint_function)() == TRUE );
  }
  ASSERT( _get_subtype() != 0 );
  return new set<IIR_TypeDefinition>( _get_subtype() );
}


IIR *
IIRScram_Declaration::_decl_to_decl( IIR_Declaration *new_decl ){
  ASSERT( new_decl == (IIR_Declaration *)this );
  return this;
}


void
IIRScram_Declaration::_clear() {
  for (int i = 0; i < numSignalAttributes;  i++) {
    implemented_attributes_in_constructor[i] = FALSE;
    implemented_attributes_in_state[i] = FALSE;
    implemented_attributes_in_initstate[i] = FALSE;
  }
}


IIR_Boolean
IIRScram_Declaration::_is_signal_in_outerscope() {
  if(_is_signal() == TRUE) {
    if((_get_currently_publishing_unit() == IIRScram::PROCEDURE) ||
       (_get_currently_publishing_unit() == IIRScram::FUNCTION)) {
      if((_get_declarative_region()->get_kind() != IIR_PROCEDURE_DECLARATION) &&
	 (_get_declarative_region()->get_kind() != IIR_FUNCTION_DECLARATION)) {
	return TRUE;
      }
    }
  }
  return FALSE;
}


void
IIRScram_Declaration::_set_scoping_prefix() {
  strstream prefix;
  IIR_Char* prefix_string = NULL;

  if((_get_currently_publishing_unit() == IIRScram::PROCEDURE) ||
     (_get_currently_publishing_unit() == IIRScram::FUNCTION) ||
     (_get_currently_publishing_unit() == PROCESS)) {
    switch (_get_declarative_region()->get_kind()) {
    case IIR_BLOCK_STATEMENT:
    case IIR_ENTITY_DECLARATION:
    case IIR_PROCESS_STATEMENT:
    case IIR_SENSITIZED_PROCESS_STATEMENT:
    case IIR_PACKAGE_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION: {
      IIR_ProcessStatement* process_ptr = _get_current_process_statement();
      if ( process_ptr == NULL ){
	break;
      }
      prefix << "((";
      process_ptr->_get_label()->_print(prefix);
      prefix << "_state *) ";

      if (_get_currently_publishing_unit() != PROCESS) {
	prefix << "((";
	process_ptr->_get_label()->_print(prefix);
	prefix << " *) processPtr)->";
      }
      
      prefix << "state->current)->";
      prefix << ends;
      prefix_string = prefix.str();
      _set_publish_prefix_string(prefix_string);
      break;
    }
    default:
      break;
    }
  }
}


void
IIRScram_Declaration::_reset_scoping_prefix() {
  if((_get_currently_publishing_unit() == IIRScram::PROCEDURE) ||
     (_get_currently_publishing_unit() == IIRScram::FUNCTION) ||
     (_get_currently_publishing_unit() == PROCESS)) {
    IIR_Char* ptr = _get_publish_prefix_string();
    if(ptr != NULL) {
      delete [] ptr;
    }
    _set_publish_prefix_string(NULL);
  }
}

void
IIRScram_Declaration::_publish_cc_anonymous_addChild() {
  ASSERT(_is_signal() == TRUE);
  ASSERT(_current_publish_name != NULL);
  char *ptr = _current_publish_name; // Set in IIRScram_ProcessStatement

  if(_is_resolved_signal() == TRUE || _have_added_driver() == FALSE) {
    _cc_out << "{" << endl
	    << "VHDLType* ptr = new ";
    _publish_cc_type_name();
    _publish_cc_constructor_args();
    _cc_out << ";" << endl;
    if(_is_composite_resolved_signal() == TRUE) {
      _cc_out << "ptr->setParentCompositeType(ptr);" << endl;
    }
    _cc_out << "addChild(";

    if ((_current_publish_node != NULL) && (_current_publish_node->_is_concurrent_statement() == TRUE)) {
      ((IIR_ConcurrentStatement *) _current_publish_node)->_publish_cc_scoping_prefix( _get_declarative_region(), _current_publish_node );
    }
    
    _publish_cc_elaborate();
    _cc_out << ", *(ptr)" << ptr;
    // This is just an anonymous driver. So flag is false. Only when a ral
    // driver ( with process id) is added, the flag is set to true
    //_set_driver_addition_flag(TRUE); 
    _cc_out << "};" << endl;
    if(_is_resolved_signal() == TRUE) {
      _cc_out << "setResolutionFunctionId(";
      if ((_current_publish_node != NULL) && (_current_publish_node->_is_concurrent_statement() == TRUE)) {
	((IIR_ConcurrentStatement *) _current_publish_node)->_publish_cc_scoping_prefix( _get_declarative_region(), _current_publish_node );
      }
      _publish_cc_elaborate();
      _cc_out << ", ";
      _get_subtype()->_get_resolution_function()->_publish_cc_resolution_function_name();
      _cc_out << ");" << endl;
    }
  }
}

