// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Swaminathan Subramanian ssubrama@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_TypeDefinition.cc,v 1.5 1999/10/26 16:44:46 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_FloatingSubtypeDefinition.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_IntegerSubtypeDefinition.hh"
#include "IIR_Name.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_ScalarTypeDefinition.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "IIR_StringLiteral.hh"
#include "IIR_Identifier.hh"
#include "IIR_ConstantInterfaceDeclaration.hh"
#include "IIRScram_TextLiteral.hh"
#include "symbol_table.hh"
#include "StandardPackage.hh"

class IIR_AccessTypeDefinition;

IIRScram_TypeDefinition::~IIRScram_TypeDefinition() {}

IIRScram_TypeDefinition::IIRScram_TypeDefinition(){
  my_attribute = NULL;
}


void
IIRScram_TypeDefinition::_publish_vhdl(ostream &_vhdl_out) {
  if (_get_declaration() != NULL) {
    _get_declaration()->_publish_vhdl(_vhdl_out);
  } else {
    ASSERT(get_base_type() != NULL);
    get_base_type()->_publish_vhdl(_vhdl_out);
  }
}


void
IIRScram_TypeDefinition::_publish_vhdl_decl(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl_decl(ostream &)");  
}


set<IIR_Declaration> *
IIRScram_TypeDefinition::_find_declarations( IIR_Name * ) {
  _report_undefined_scram_fn("_find_declarations( IIR_Name * )");  
  return NULL;
}

set<IIR_Declaration> *
IIRScram_TypeDefinition::_find_declarations( IIR_TextLiteral * ) {
  _report_undefined_scram_fn("_find_declarations( IIR_TextLiteral * )");  
  return NULL;
}

IIR_Boolean
IIRScram_TypeDefinition::_is_bit_type() {
  return FALSE;
}


IIR_Boolean
IIRScram_TypeDefinition::_is_subtype_decl() {
  IIR_Declaration* type_decl = _get_declaration();
  if(type_decl != NULL) {
    return type_decl->_is_subtype_decl();
  }
  else {
    return FALSE;
  }
}


IIR_Boolean
IIRScram_TypeDefinition::_is_line_type() {
  if(_get_declaration() != NULL) {
    return ( 0 ==IIR_TextLiteral::_cmp(_get_declaration()->get_declarator(), "line"));
  }
  else {
    return FALSE;
  }
}


IIR_Boolean
IIRScram_TypeDefinition::_is_boolean_type() {
  return FALSE;
}

IIR_Boolean
IIRScram_TypeDefinition::_is_kernel_type() {
  if ((IIRScram_TextLiteral::_cmp(_get_declarator(), "integer") == 0)        ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "boolean") == 0)        ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "severity_level") == 0) ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "positive") == 0)       ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "character") == 0)      ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "real") == 0)           ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "time") == 0)           ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "natural") == 0)        ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "bit") == 0)            ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "side") == 0)           ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "width") == 0)          ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "delay_length") == 0)   ||      (IIRScram_TextLiteral::_cmp(_get_declarator(), "file_open_kind") == 0) ||
      (IIRScram_TextLiteral::_cmp(_get_declarator(), "file_open_status") == 0)
      ) {
    return TRUE;
  }
  else {
    return FALSE;
  }
}

void
IIRScram_TypeDefinition::_publish_vhdl_range(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl_range(ostream &)");  
}


void
IIRScram_TypeDefinition::_publish_vhdl_constraint(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl_constraint(ostream &)");  
}


void
IIRScram_TypeDefinition::_publish_cc_left() {
  _report_undefined_scram_fn("_publish_cc_left()");  
}


void
IIRScram_TypeDefinition::_publish_cc_universal_left() {
  _report_undefined_scram_fn("_publish_cc_universal_left()");  
}


void
IIRScram_TypeDefinition::_publish_cc_right() {
  _report_undefined_scram_fn("_publish_cc_right()");  
}


IIR_Boolean
IIRScram_TypeDefinition::_is_ascending_range() {
  _report_undefined_scram_fn("_is_ascending_range()");  
  return true;
}


IIR_Boolean
IIRScram_TypeDefinition::_is_subtype() {
  return FALSE;
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_get_bottom_base_type(){
  if( get_base_type() == NULL ){
    return (IIR_TypeDefinition *)this;
  }
  else{
    return get_base_type()->_get_bottom_base_type();
  }
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_get_element_subtype(){
  _report_undefined_scram_fn("_get_element_subtype()");  
  return NULL;
}

IIR_ScalarTypeDefinition *
IIRScram_TypeDefinition::_get_index_subtype(){
  _report_undefined_scram_fn("_get_index_subtype()");  
  return NULL;
}


void  
IIRScram_TypeDefinition::_set_index_subtype( IIR_ScalarTypeDefinition *){
  _report_undefined_scram_fn("_set_index_subtype()");  
  abort();
}


void  
IIRScram_TypeDefinition::_set_element_subtype( IIR_TypeDefinition *){
  _report_undefined_scram_fn("_set_element_subtype()");  
  abort();
}


void
IIRScram_TypeDefinition::_publish_cc_decl_hh() {
  char *className;
  strstream classStream;

  classStream << "Savant";
  _get_declarator()->_print(classStream);
  classStream << "Type" << ends;

  className = classStream.str();
  
  _cc_out << "#ifndef " << className << "_HH\n";
  _cc_out << "#define " << className << "_HH\n\n";
  
  _cc_out << "class IntegerType;\n"
	  << "class EnumerationType;\n"
	  << "class SavantstringType;\n";

  _cc_out << "#include \"standard.hh\"\n";
  if (_current_entity_name != NULL){
    _cc_out << "#include \"";
    _cc_out << _current_entity_name << "_decls.hh\"\n";
  }

  if((get_base_type() != NULL) &&
     (get_base_type()->_is_anonymous() == FALSE)) {
    _cc_out << "#include \"";
    get_base_type()->_publish_cc_type_name();
    _cc_out << ".hh\"\n";
  }
  if(_get_resolution_function() != NULL) {
    _cc_out <<  "#include \"_savant_entity_elab.hh\"\n";
    _cc_out << "extern VHDLType* ";
    _get_resolution_function()->_publish_cc_resolution_function_name();
    _cc_out << "(VHDLKernelBase*, int, VHDLType**);\n";
  }
  _publish_cc_typeInfo();
  _publish_cc_headers();
  _publish_cc_class();

  if ((_is_iir_array_type_definition() == TRUE) && (_is_subtype_decl() == FALSE)) {
    _publish_cc_concatenation_operator_prototype();
  }
  
  if (_is_file_type() == TRUE) {
    _publish_cc_decl_predefined_procedures_prototypes();
  }
  
  //_publish_cc_decl_operators_prototypes();

  _cc_out << "#endif\n";
}


void
IIRScram_TypeDefinition::_publish_cc_headers() {
  _report_undefined_scram_fn("_publish_cc_headers()");  
}


void
IIRScram_TypeDefinition::_publish_cc_constructor_args() {
  _report_undefined_scram_fn("_publish_cc_constructor_args()");  
}


void
IIRScram_TypeDefinition::_publish_cc_class_event() {
  _report_undefined_scram_fn("_publish_cc_class_event()");
}


void
IIRScram_TypeDefinition::_publish_cc_class_last_event() {
  _report_undefined_scram_fn("_publish_cc_class_last_event()");
}


void
IIRScram_TypeDefinition::_publish_cc_bounds() {
  _cc_out << "defaultInfo";
}

void
IIRScram_TypeDefinition::_publish_cc() {
  _cc_out << "Savant"
	  << *_get_declarator()
	  << "Type";
}

void
IIRScram_TypeDefinition::_publish_cc_class() {
  _cc_out << "class ";
  _publish_cc();
  _cc_out << " : public ";
  _publish_cc_parent_type_name();
  _cc_out << " {\n";
  _cc_out << "public:\n ";
  if(_is_subtype_decl() == FALSE) {
    _publish_cc_data_members();
  }
  _publish_cc_decl_constructors();
  _publish_cc_decl_destructors();
  if((_is_scalar_type() == TRUE)&& (_is_access_type() != TRUE)) {
    _publish_cc_function_leftvalue();
  }
  if (_is_record_type() == TRUE) {
    _publish_cc_decl_operator_equalto();
  } else if(_is_array_type() == TRUE) {
    _publish_cc_decl_operator_equalto();
    _publish_cc_decl_operator_subscript();
  }
  
  _publish_cc_decl_type_attributes();
  _cc_out << "};\n";
}


void
IIRScram_TypeDefinition::_publish_cc_parent_type_name() {
  if((_is_subtype_decl() == TRUE) && 
     (_is_enumeration_type() == FALSE)) {
    ASSERT(_get_type_mark() != NULL);
    _get_type_mark()->_publish_cc_type_name();
  }
  else if(get_base_type() != NULL && (!get_base_type()->_is_anonymous())) {
    get_base_type()->_publish_cc_type_name();
  } else {
    _publish_cc_kernel_type();
  }
}


void
IIRScram_TypeDefinition::_publish_cc_function_leftvalue() {
  char *typeName;
  strstream typeStream;
  switch(get_kind()){
  case IIR_INTEGER_SUBTYPE_DEFINITION: 
  case IIR_INTEGER_TYPE_DEFINITION:
    typeStream << "rangeInfo" << ends;
    break;
  case IIR_FLOATING_SUBTYPE_DEFINITION:
  case IIR_FLOATING_TYPE_DEFINITION:
    typeStream << "realInfo" << ends;
    break;
  case IIR_ENUMERATION_SUBTYPE_DEFINITION:
  case IIR_ENUMERATION_TYPE_DEFINITION:
    typeStream << "enumInfo" << ends;
    break;
  case IIR_PHYSICAL_SUBTYPE_DEFINITION:
  case IIR_PHYSICAL_TYPE_DEFINITION:
    typeStream << "phyInfo" << ends;
    break;
  }
  typeName = typeStream.str();
  _cc_out << "const VHDLData& leftValue(";
  _cc_out << typeName << " rInfo";
  _cc_out << ") {\n";
  _cc_out << "return ((";
  _publish_cc_kernel_type();
  _cc_out << "::";
  _cc_out << "LEFT(rInfo)).getVHDLData());\n";
  _cc_out << "}\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_decl_relational_operators() {
  if((_is_file_type() != TRUE)&&(_is_subtype_decl() != TRUE)) {
    _publish_cc_equality_operator();
    _publish_cc_inequality_operator();    
  }
  if((_is_scalar_type() == TRUE) || (_is_discrete_type() == TRUE)) {
    _publish_cc_lessthan_operator();
    _publish_cc_lessThanOrEqualto_operator();
    _publish_cc_greaterThan_operator();
    _publish_cc_greaterThanOrEqualto_operator();
  }
}


void 
IIRScram_TypeDefinition::_publish_cc_decl_relational_operators_prototypes() {
  if((_is_file_type() != TRUE) && (_is_subtype_decl() != TRUE)) {
    _publish_cc_equality_operator_prototype();
    _publish_cc_inequality_operator_prototype();    
  }
  if((_is_scalar_type() == TRUE) || (_is_discrete_type() == TRUE)) {
    _publish_cc_lessthan_operator_prototype();
    _publish_cc_lessThanOrEqualto_operator_prototype();
    _publish_cc_greaterThan_operator_prototype();
    _publish_cc_greaterThanOrEqualto_operator_prototype();
  }
}


void 
IIRScram_TypeDefinition::_publish_cc_decl_arithmetic_operators_prototypes() {
  if(_is_numeric_type() == TRUE) {
    _publish_cc_plus_operator_prototype();
    _publish_cc_minus_operator_prototype();
    _publish_cc_unaryplus_operator_prototype();
    _publish_cc_unaryminus_operator_prototype();
    _publish_cc_abs_operator_prototype();
  }
  if((_is_single_dimensional_array_type() == TRUE)&&(_is_subtype_decl() != TRUE)) {
    _publish_cc_concatenation_operator_prototype();
  }
  if(_is_integer_type() == TRUE || _is_floating_type() == TRUE) {
    _publish_cc_multiply_operator_prototype();
    _publish_cc_divide_operator_prototype();
    _publish_cc_pow_operator_prototype();
  }
  // Assumption: Physical Type is either integer or floating type. -- SK
  if(_is_physical_type() == TRUE) {
    _publish_cc_multiply_operator_prototype();
    _publish_cc_divide_operator_prototype();
  }    
  if(_is_integer_type() == TRUE) {
    _publish_cc_mod_operator_prototype();
    _publish_cc_rem_operator_prototype();    
  }
}


void 
IIRScram_TypeDefinition::_publish_cc_decl_operators() {
  _report_undefined_scram_fn("_publish_cc_decl_operators()");
}


void 
IIRScram_TypeDefinition::_publish_cc_decl_arithmetic_operators() {
  _report_undefined_scram_fn("_publish_cc_decl_arithmetic_operator()");
}


void 
IIRScram_TypeDefinition::_publish_cc_decl_operators_prototypes() {
  if(_is_subtype_decl() == FALSE) {
    _publish_cc_decl_relational_operators_prototypes();
    _publish_cc_decl_arithmetic_operators_prototypes();
    if (_is_file_type() == TRUE) {
      _publish_cc_decl_predefined_procedures_prototypes();
    }
  }
}


void
IIRScram_TypeDefinition::_publish_cc_equality_operator_prototype() {
  _cc_out << "extern EnumerationType\n";
  _cc_out << "savantEqual(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";
}


void
IIRScram_TypeDefinition::_publish_cc_inequality_operator_prototype() {
  _cc_out << "extern EnumerationType\n";
  _cc_out << "savantNotEqual(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";
}


void
IIRScram_TypeDefinition::_publish_cc_plus_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\n";
  _cc_out << "savantPlus(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_plus_operator() {
  _report_undefined_scram_fn("_publish_cc_plus_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_minus_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantMinus(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& minus);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_minus_operator() {
  _report_undefined_scram_fn("_publish_cc_minus_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_unaryplus_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantUnaryPlus(const ";
  _publish_cc_type_name();
  _cc_out << "& ";
  _cc_out << "lhs);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_unaryplus_operator() {
  _report_undefined_scram_fn("_publish_cc_unaryplus_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_unaryminus_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantUnaryMinus(const ";
  _publish_cc_type_name();
  _cc_out << "& ";
  _cc_out << "lhs);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_unaryminus_operator() {
  _report_undefined_scram_fn("_publish_cc_unaryminus_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_abs_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantAbs(const ";
  _publish_cc_type_name();
  _cc_out << "& ";
  _cc_out << "lhs);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_abs_operator() {
  _report_undefined_scram_fn("_publish_cc_abs_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_concatenation_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantConcatenate(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";

  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantConcatenate(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  if (((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_scalar_type() == TRUE && ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_kernel_type()==FALSE){
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_type_name();
  }
  _cc_out << "& rhs);\n";

  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantConcatenate(const ";
  if (((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_scalar_type() == TRUE && ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_kernel_type() == FALSE){
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_type_name();
  }

  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";

  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantConcatenate(const ";
  if (((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_scalar_type() == TRUE && ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_kernel_type() == FALSE){
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_type_name();
  }

  _cc_out << "& lhs, const ";
  if (((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_scalar_type() == TRUE && ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_is_kernel_type() == FALSE){
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    ((IIR_ArrayTypeDefinition*)this)->get_element_subtype()->_publish_cc_type_name();
  }

  _cc_out << "& rhs, ";
  _publish_cc_type_name();
  _cc_out << "*);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_concatenation_operator() {
  _report_undefined_scram_fn("_publish_cc_concatenation_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_multiply_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantMultiply(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_multiply_operator() {
  _report_undefined_scram_fn("_publish_cc_multiply_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_divide_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantDivide(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_divide_operator() {
  _report_undefined_scram_fn("_publish_cc_divide_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_pow_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantPow(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const IntegerType& rhs);\n";;
}


void 
IIRScram_TypeDefinition::_publish_cc_pow_operator() {
  _report_undefined_scram_fn("_publish_cc_pow_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_mod_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantMod(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";
}


void 
IIRScram_TypeDefinition::_publish_cc_mod_operator() {
  _report_undefined_scram_fn("_publish_cc_mod_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_rem_operator_prototype() {
  _cc_out << "extern ";
  _publish_cc_type_name();
  _cc_out << "\nsavantRem(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs);\n";
}

void 
IIRScram_TypeDefinition::_publish_cc_rem_operator() {
  _report_undefined_scram_fn("_publish_cc_rem_operator()");
}


void
IIRScram_TypeDefinition::_publish_cc_lessthan_operator_prototype() {
  _cc_out << "extern EnumerationType\n";
  _cc_out << "savantLessThan(const ";
  _publish_cc_type_name();
  _cc_out << "&, const ";
  _publish_cc_type_name();
  _cc_out << "&);\n";
}


void
IIRScram_TypeDefinition::_publish_cc_lessThanOrEqualto_operator_prototype() {
  _cc_out << "extern EnumerationType\n";
  _cc_out << "savantLessThanOrEqual(const ";
  _publish_cc_type_name();
  _cc_out << "&, const ";
  _publish_cc_type_name();
  _cc_out << "&);\n";
}


void
IIRScram_TypeDefinition::_publish_cc_greaterThan_operator_prototype()
{
  _cc_out << "extern EnumerationType\n";
  _cc_out << "savantGreaterThan(const ";
  _publish_cc_type_name();
  _cc_out << "&, const ";
  _publish_cc_type_name();
  _cc_out << "&);\n";
}


void
IIRScram_TypeDefinition::_publish_cc_greaterThanOrEqualto_operator_prototype()
{
  _cc_out << "extern EnumerationType\n";
  _cc_out << "savantGreaterThanOrEqual(const ";
  _publish_cc_type_name();
  _cc_out << "&, const ";
  _publish_cc_type_name();
  _cc_out << "&);\n";
}


void
IIRScram_TypeDefinition::_publish_cc_decl_predefined_procedures_prototypes()
{
  _report_undefined_scram_fn("_publish_cc_decl_predefined_procedures_prototypes()");
}


void 
IIRScram_TypeDefinition::_publish_cc_equality_operator() {
  _report_undefined_scram_fn("_publish_cc_equality_operator()");
}


void 
IIRScram_TypeDefinition::_publish_cc_inequality_operator() {
  _report_undefined_scram_fn("_publish_cc_inequality_operator()");
}


void 
IIRScram_TypeDefinition::_publish_cc_lessthan_operator() {
  _report_undefined_scram_fn("_publish_cc_lessthan_operator()");
}

 
void 
IIRScram_TypeDefinition::_publish_cc_lessThanOrEqualto_operator() {
  _report_undefined_scram_fn("_publish_cc_lessThanOrEqualto_operator()");
}


void 
IIRScram_TypeDefinition::_publish_cc_greaterThan_operator() {
  _report_undefined_scram_fn("_publish_cc_greaterThan_operator()");
}


void 
IIRScram_TypeDefinition::_publish_cc_greaterThanOrEqualto_operator() {
  _report_undefined_scram_fn("_publish_cc_greaterThanOrEqualTo_operator()");
}


void 
IIRScram_TypeDefinition::_publish_cc_elaborate() {
  if (this->_is_scalar_type() == TRUE &&
      this->_is_kernel_type() == FALSE){
    _publish_cc_kernel_type();
  }
  else {
    _publish_cc_type_name();
  }
}


void
IIRScram_TypeDefinition::_publish_cc_data_members() {
  _report_undefined_scram_fn("_publish_cc_data_members()");  
}


void
IIRScram_TypeDefinition::_publish_cc_decl_constructors() {
  _report_undefined_scram_fn("_publish_cc_decl_constructors()");  
}


void
IIRScram_TypeDefinition::_publish_cc_type_constructors() {
  _report_undefined_scram_fn("_publish_cc_type_constructors()");  
}


void
IIRScram_TypeDefinition::_publish_cc_subtype_constructors() {
  _report_undefined_scram_fn("_publish_cc_subtype_constructors()");  
}


void
IIRScram_TypeDefinition::_publish_cc_decl_destructors() {
  _report_undefined_scram_fn("_publish_cc_decl_destructors()");  
}


void
IIRScram_TypeDefinition::_publish_cc_type_destructors() {
  _report_undefined_scram_fn("_publish_cc_type_destructors()");  
}


void
IIRScram_TypeDefinition::_publish_cc_subtype_destructors() {
  _report_undefined_scram_fn("_publish_cc_subtype_destructors()");  
}


void
IIRScram_TypeDefinition::_publish_cc_decl_operator_equalto() {
  _report_undefined_scram_fn("_publish_cc_decl_operator_equalto()");  
}


void
IIRScram_TypeDefinition::_publish_cc_decl_operator_subscript() {
  _report_undefined_scram_fn("_publish_cc_decl_operator_subscript()");  
}


void
IIRScram_TypeDefinition::_publish_cc_type_operator_equalto() {
  _report_undefined_scram_fn("_publish_cc_type_operator_equalto()");  
}


void
IIRScram_TypeDefinition::_publish_cc_subtype_operator_equalto() {
  _report_undefined_scram_fn("_publish_cc_subtype_operator_equalto()");  
}


void
IIRScram_TypeDefinition::_publish_cc_kernel_type() {
  _report_undefined_scram_fn("_publish_cc_kernel_type()");  
}


void
IIRScram_TypeDefinition::_publish_cc_decl_cc() {
  _report_undefined_scram_fn("_publish_cc_decl_cc()");  
}


void
IIRScram_TypeDefinition::_publish_cc_subelement_type() {
  _report_undefined_scram_fn("_publish_cc_subelement_type()");  
}


void
IIRScram_TypeDefinition::_publish_cc_anonymous_type_name() {
  _report_undefined_scram_fn("_publish_cc_anonymous_type_name()");  
}


void
IIRScram_TypeDefinition::_publish_cc_type_name() {
  IIR_Declaration *decl;
  decl = _get_declaration();
  if (_is_access_type() == TRUE) {
    _cc_out << "Savant";
    decl->_get_declarator()->_publish_cc();
    _cc_out << "Type";
    return;
  }
  if (_is_scalar_type() == TRUE) {
    _publish_cc_kernel_type();
  }
  else {
    if(decl != NULL) {
      _cc_out << "Savant";
      decl->_get_declarator()->_publish_cc();
      _cc_out << "Type";
    }
    else {
      // Well there has to be a base type somewhere that has a decl !!
      if (_get_base_type() != NULL) {
	_get_base_type()->_publish_cc_type_name();
      }
      else {
	// Some of the array types will have the type_mark filled in
	// in case of constrained array-type definitions. So check for that
	if (_get_type_mark() != NULL) {
	  _get_type_mark()->_publish_cc_type_name();
	}
      }
    }
  }
}

void
IIRScram_TypeDefinition::_publish_cc_base_type_name() {
  if(_is_subtype_decl() == TRUE) {
    if(_is_enumeration_type() == TRUE) {
      get_base_type()->_publish_cc_type_name();
    }
    else {
      ASSERT(_get_type_mark() != NULL);
      _get_type_mark()->_publish_cc_type_name();
    }
  }
  else if(get_base_type() != NULL && get_base_type()->_get_declaration() != NULL) {
    get_base_type()->_publish_cc_type_name();
  }
  else {
    _publish_cc_type_name();
  }
}


void
IIRScram_TypeDefinition:: _publish_cc_init_signal() {
  _report_undefined_scram_fn("_publish_cc_init_signal()");  
}


void
IIRScram_TypeDefinition:: _publish_cc_composite_init() {
  _report_undefined_scram_fn("_publish_cc_composite_init()");  
}


void
IIRScram_TypeDefinition::_publish_cc_range() {
  _report_undefined_scram_fn("_publish_cc_range()");  
}


ostream &
IIRScram_TypeDefinition::_print( ostream &os ){
  os << *_get_declarator();
  return os;
}


IIR_Int32
IIRScram_TypeDefinition::_get_num_indexes(){
  // This is over-ridden in IIR_ArrayTypeDefinition...
  return 0;
}


IIR_Boolean 
IIRScram_TypeDefinition::_is_anonymous(){
  if( _get_declaration() == NULL ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}


IIR_Boolean 
IIRScram_TypeDefinition::_is_element(){
  return is_element();
}


IIR_Boolean 
IIRScram_TypeDefinition::_is_resolved_type(){
  if (_get_resolution_function() != NULL) {
    return TRUE;
  }
  return FALSE;
}


IIR_Boolean 
IIRScram_TypeDefinition::is_element(){
  ASSERT( _is_access_type() == TRUE || _is_array_type() == FALSE );
  return FALSE;
}


void 
IIRScram_TypeDefinition::set_is_element( IIR_Boolean ){
  _report_undefined_scram_fn("IIRScram_TypeDefinition::set_is_element");
  abort();
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_get_type_of_element( int index ){
  // Repeated almost verbatim in IIRScram_FunctionDeclaration
  if( index > _get_num_indexes() ){
    return NULL;
  }
  
  IIR_TypeDefinition *current_subtype = (IIR_TypeDefinition *)this;
  if( index == 0 ){
    return current_subtype;
  }

  int i;
  for( i = 0; i < index - 1; i++ ){
    ASSERT( current_subtype->_is_array_type() == TRUE );

    current_subtype = current_subtype->_get_element_subtype();
  }
  
  IIR_TypeDefinition *retval = current_subtype->_get_element_subtype();
  
  return retval;
}

IIR_TypeDefinition*
IIRScram_TypeDefinition::_is_explicit_type_conversion_needed(IIR_TypeDefinition *targetType) {
  // This method checks to see if an explicit type conversion needs to be
  // code-generated. If so it returns the common type else it return NULL.

  if (targetType == this) {
    return NULL;
  }
  
  if (_is_base_type(targetType)) {
    // Target type is the base type of this type. No explicit type conversion
    // is needed.
    return NULL;
  }

  // return the most compaitable type.
  return _is_compatible(targetType);
}

IIR_Boolean
IIRScram_TypeDefinition::_is_base_type(IIR_TypeDefinition *_base) {
  IIR_TypeDefinition *base_type = _get_base_type();

  while (base_type != NULL) {
    if (base_type == _base) {
      return TRUE;
    }

    base_type = base_type->_get_base_type();
  }

  return FALSE;
}

void
IIRScram_TypeDefinition::_publish_cc_decl_type_attributes() {
  _report_undefined_scram_fn("_publish_cc_decl_type_attributes()");  
}


void
IIRScram_TypeDefinition::_publish_cc_necessary_decl_in_state(){
  // Dummy function. Do not put undef here.
}


void
IIRScram_TypeDefinition::_publish_cc_init_fields_for_signals() {
  // Dummy function. Do not put undefined func. here.
}


void
IIRScram_TypeDefinition::_publish_cc_set_resolution_function() {
  if(_get_resolution_function() != NULL) {
    _cc_out << "switch(objType) {\n"
	    << "case ObjectBase::SIGNAL_NETINFO: {\n"
	    << "  int res_fn_id = _savant_entity_elab::addResolutionFn(";
    _get_resolution_function()->_publish_cc_resolution_function_name();
    _cc_out << ");\n"
	    << "  setResolutionFunctionId(res_fn_id);\n"
	    << "  break;\n"
	    << "}\n"
	    << "case ObjectBase::SIGNAL:\n";
    if(_is_array_type() == TRUE || _is_record_type() == TRUE) {
      _cc_out  << "  setCompositeResolvedSignal(true);\n";
    }
    _cc_out << "case ObjectBase::VARIABLE:\n"
	    << "  setParentCompositeType(this);\n"
	    << "  break;\n"
	    << "default:\n"
	    << "  break;\n"
	    << "}\n";
  }
}


void
IIRScram_TypeDefinition::_add_decl_into_cgen_symbol_table() {
  // Dummy function to keep _report_undefined_scram_fn quiet.
}


IIR_FunctionDeclaration * 
IIRScram_TypeDefinition::_resolve_resolution_function( IIR_Name *resolution_function ){
  IIR_FunctionDeclaration *resolution_function_decl = NULL;

  set<IIR_Declaration> *func_decls;
  func_decls = resolution_function->_symbol_lookup( &IIR::_is_iir_function_declaration );
  if( func_decls == NULL ){
    report_undefined_symbol( resolution_function );
  }

  resolution_function_decl = (IIR_FunctionDeclaration *)resolve_if_one( func_decls );

  delete func_decls;

  return resolution_function_decl;
}


IIR_ScalarTypeDefinition *
IIRScram_TypeDefinition::_determine_type_of_bound( IIR *left_or_right ){
  IIR_ScalarTypeDefinition *retval = NULL;

  ASSERT( left_or_right != NULL );
  
  set<IIR_TypeDefinition> *left_or_right_types = left_or_right->_get_rval_set();
  if( left_or_right_types == NULL ){
    report_undefined_symbol( left_or_right );
    return NULL;
  }

  left_or_right_types->reduce_set( &IIR::_is_iir_scalar_type_definition );
  
  IIR_ScalarTypeDefinition *left_or_right_type = NULL;  
  switch( left_or_right_types->num_elements() ){
  case 0:{
    ostrstream err;
    err << "|" << *left_or_right << "| is not a scalar type and isn't valid in this context."
	<< ends;
    report_error( left_or_right, err );
    break;
  }
  
  case 1:{
    retval = (IIR_ScalarTypeDefinition *)left_or_right_types->get_element();
    ASSERT( retval->_is_scalar_type() == TRUE );
    break;
  }
    
  default:{
    if( left_or_right_types->in_set( StandardPackage::savant_universal_integer ) == TRUE ){
      left_or_right_type = StandardPackage::savant_universal_integer;
      retval = left_or_right_type;
    }
    else if( left_or_right_types->in_set( StandardPackage::savant_universal_real ) == TRUE ){
      left_or_right_type = StandardPackage::savant_universal_real;
      retval = left_or_right_type;
    }
    else{
      report_ambiguous_error( left_or_right, left_or_right_types );
      retval = (IIR_ScalarTypeDefinition *)left_or_right_types->get_element();
    }

    break;
  }
  }

  delete left_or_right_types;

  return retval;
}


IIR_ScalarTypeDefinition *
IIRScram_TypeDefinition::_construct_new_type( IIR_RangeTypeDefinition *temp_range ){
  // We have a range like 1 to 10, or -2.0 to 7.1. The only other legal
  // possibility is for this to be type'range.  (This method should only be
  // used for creating a new type in a type declaration.

  IIR_ScalarTypeDefinition *retval = NULL;
  
  IIR *left = temp_range->get_left();
  ASSERT( left != NULL );

  IIR_ScalarTypeDefinition *left_type = _determine_type_of_bound( left );

  IIR *right = temp_range->get_right();
  if( right != NULL ){
    IIR_ScalarTypeDefinition *base_type = NULL;
    IIR_ScalarTypeDefinition *constrained_subtype = NULL;

    IIR_ScalarTypeDefinition *right_type = _determine_type_of_bound( right );
    if( right_type == NULL ){
      return NULL;
    }

    if( left_type->_is_iir_integer_type_definition() == TRUE ){
      if( right_type->_is_iir_integer_type_definition() == TRUE ){
	base_type = new IIR_IntegerTypeDefinition();
	copy_location( temp_range, base_type );
      }
      else{
	ostrstream err;
	err << "|" << *left << " " << *temp_range->get_direction() << " " << *right
	    << "| is not a valid range in this context. |" << *right << "| is not an "
	    << "integer type, and |" << *left << "| is." << ends;
	report_error( temp_range, err );
	retval = NULL;
      }
    }
    else if ( left_type->_is_iir_floating_type_definition() == TRUE ){
      if( right_type->_is_iir_floating_type_definition() == TRUE ){
	base_type = new IIR_FloatingTypeDefinition();
	copy_location( temp_range, base_type );
      }
      else{
	ostrstream err;
	err << "|" << *left << " " << *temp_range->get_direction() << " " << *right
	    << "| is not a valid range in this context. |" << *right << "| is not an "
	    << "floating type, and |" << *left << "| is." << ends;
	report_error( temp_range, err );
	retval = NULL;
      }
    }
    else{
      ostrstream err;
      err << "Internal error in IIRScram_TypeDefinition::_construct_new_type - "
	  << "got a left that is a " << left_type->get_kind_text() << " and I don't know "
	  << "what to do!" << ends;
      report_error( temp_range, err );
      abort();
    }
    
    constrained_subtype = (IIR_ScalarTypeDefinition *)base_type->_get_new_subtype();
    base_type->_clone( constrained_subtype );

    left = left->_semantic_transform( left_type );
    left->_type_check( left_type );
    left = left->_rval_to_decl( left_type );

    right = right->_semantic_transform( right_type );
    right->_type_check( right_type );
    right = right->_rval_to_decl( right_type );

    constrained_subtype->set_left( left );
    constrained_subtype->set_right( right );
    constrained_subtype->set_direction( temp_range->get_direction() );
    constrained_subtype->set_base_type( base_type );
    retval = constrained_subtype;
  } 
  else{
    // Then the left better have been a 'range.
    ostrstream err;
    err << "'range temporarily not supported in this context!" << ends;
    report_error( temp_range, err );
    abort();
  }
  
  ASSERT( retval != NULL );
  ASSERT( retval->_is_resolved() == TRUE );
  
  return retval;
}

IIR*
IIRScram_TypeDefinition::_get_direction(){
  _report_undefined_scram_fn("_get_direction()");  
  return  NULL;
}


IIR_TypeDefinition*
IIRScram_TypeDefinition::_get_base_type(){
  _report_undefined_scram_fn("_get_base_type()");  
  return  NULL;
}


IIR*
IIRScram_TypeDefinition::_get_base_type_left(){
  _report_undefined_scram_fn("_get_base_type_left()");  
  return  NULL;
}


IIR*
IIRScram_TypeDefinition::_get_base_type_direction(){
  _report_undefined_scram_fn("_get_base_type_direction()");  
  return  NULL;
}


IIR*
IIRScram_TypeDefinition::_get_base_type_right(){
  _report_undefined_scram_fn("_get_base_type_right()");  
  return  NULL;
}


IIR_TextLiteral *
IIRScram_TypeDefinition::_get_declarator(){
  if( _get_declaration() != NULL ){
    return _get_declaration()->_get_declarator();
  }
  else{
    char *string = "<ANONYMOUS>";
    return IIR_Identifier::get( string, strlen( string ) );
  }
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_is_compatible( IIR_TypeDefinition *to_check ){
  
  ASSERT( this != NULL );

  IIR_TypeDefinition *left_subtype = to_check;

  // Check for NULL types.
  if( to_check == (IIR_TypeDefinition*)StandardPackage::savant_null_type_definition ){
    return (IIR_TypeDefinition *)this;
  }
  else if( this == (IIR_TypeDefinition*)StandardPackage::savant_null_type_definition ){
    return (IIR_TypeDefinition *)to_check;
  }

  // Check for "normal" compatibility.
  while( left_subtype != NULL ){
    IIR_TypeDefinition *right_subtype = (IIR_TypeDefinition *)this;    
    while( right_subtype != NULL ){
      if( left_subtype == right_subtype ){
	return left_subtype;
      }
      right_subtype = right_subtype->get_base_type();
    }
    
    left_subtype = left_subtype->get_base_type();
  }

  // Check for special cases, like universal ints being compared with
  // other integer types.
  return _check_special_compatible( to_check );
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_check_special_compatible( IIR_TypeDefinition * ){
  //  _report_undefined_scram_fn("_check_special_compatible( IIR_TypeDefinition * )");
  return NULL;
}


set<IIR_TypeDefinition> *
IIRScram_TypeDefinition::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  return new set<IIR_TypeDefinition>( (IIR_TypeDefinition *)this );
}


void 
IIRScram_TypeDefinition::_type_check( set<IIR_TypeDefinition> * ){
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_construct_new_subtype( IIR_Name *, IIR_ScalarTypeDefinition * ){
  _report_undefined_scram_fn("IIRScram_TypeDefinition::_construct_new_subtype( IIR *resolution_function, IIR_ScalarTypeDefinition *new_constraint )");  
  return NULL;
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_get_new_subtype( ){
  _report_undefined_scram_fn("IIRScram_TypeDefinition::_get_new_subtype( )");  
  return NULL;
}


IIR_FunctionDeclaration *
IIRScram_TypeDefinition::_get_resolution_function( ){
  return NULL; 
}


void
IIRScram_TypeDefinition::_set_resolution_function( IIR_FunctionDeclaration * ){
  _report_undefined_scram_fn("IIRScram_TypeDefinition::set_resolution_function( IIR_FunctionDeclaration * )");  
}


IIR_TypeDefinition *
IIRScram_TypeDefinition::_construct_new_subtype_resolution_function_only( IIR_Name *resolution_function,
									  IIR_ScalarTypeDefinition *new_constraint ){
  IIR_TypeDefinition *retval;
  IIR *cloned = _clone();

  ASSERT( cloned->_is_iir_type_definition() == TRUE );
  retval = (IIR_TypeDefinition *)cloned;

  retval->set_base_type( (IIR_TypeDefinition *)this );

  if( new_constraint != NULL ){
    ostrstream err;
    err << "Constraints may not be applied to record types." << ends;
    report_error( this, err );
  }

  IIR_FunctionDeclaration *resolution_function_decl;
  resolution_function_decl = _resolve_resolution_function( resolution_function );

  retval->_set_resolution_function( resolution_function_decl );

  return retval;
}


void 
IIRScram_TypeDefinition::_clone( IIR_TypeDefinition *my_clone ){
  my_clone->set_base_type( get_base_type() );
  IIR::_clone( my_clone );
}


IIR * 
IIRScram_TypeDefinition::_clone(  ){
  _report_undefined_scram_fn("IIRScram_TypeDefinition::_clone()");
  return NULL;
}

void
IIRScram_TypeDefinition::_publish_cc_type_info(){
  _report_undefined_scram_fn("_publish_cc_type_info()");
}

void
IIRScram_TypeDefinition::_publish_cc_object_type_info() {
  if (_is_anonymous() == FALSE) {
    _cc_out << ", Savant";
    _cc_out << *_get_declarator();
    _cc_out << "Type_info";
  }
  else {
    if (_is_kernel_type() == TRUE) {
      // Okay. If its a kernel type we can figure out what the subtype is
      // going to be from the base type!!
      ASSERT (_get_base_type() != NULL);
      _cc_out << ", ";
      _get_base_type()->_publish_cc();
      _cc_out << "_info";
    }
    else {
      if ((_is_scalar_type() == TRUE) && (_get_base_type() != NULL)) {
	_get_base_type()->_publish_cc_object_type_info();
      }
      else {
	// Okay. One last try. If it is a integer type, then, well use the
	// default SavantintegerType_info. Rest of the guys we cannot do much.
	if (_is_integer_type() == TRUE)  {
	  _cc_out << ", SavantintegerType_info";
	}
	else {
	  cerr << "Warning [IIRScram_TypeDefinition::"
	       << "_publish_cc_object_type_info()] :: Maybe missing type info"
	       << " for" << get_kind() << endl;
	}
      }
    }
  }
}

void
IIRScram_TypeDefinition::_publish_cc_extern_type_info(){
  _report_undefined_scram_fn("_publish_cc_extern_type_info()");
}

void 
IIRScram_TypeDefinition::_publish_cc_typeInfo(){
  IIR_TypeDefinition *type_def = (IIR_TypeDefinition *) this;
  if (this->_is_array_type() == TRUE && _is_access_type() == FALSE){
    _publish_cc_extern_type_info();
    if (this->_get_element_subtype()->_is_scalar_type() == TRUE &&
	this->_get_element_subtype()->_is_kernel_type() == FALSE){
      this->_get_element_subtype()->_publish_cc_extern_type_info();
    }
    return;
  }
  
  if(_is_scalar_type() == TRUE &&
     _is_access_type() == FALSE){
    _cc_out << "extern ";
    switch(get_kind()){
    case IIR_INTEGER_SUBTYPE_DEFINITION:
    case IIR_INTEGER_TYPE_DEFINITION:
      _cc_out << "rangeInfo ";
      break;
    case IIR_FLOATING_SUBTYPE_DEFINITION:
    case IIR_FLOATING_TYPE_DEFINITION:
      _cc_out << "realInfo ";
      break;
    case IIR_ENUMERATION_SUBTYPE_DEFINITION:
    case IIR_ENUMERATION_TYPE_DEFINITION:
      _cc_out << "enumInfo ";
      break;
    case IIR_PHYSICAL_SUBTYPE_DEFINITION:
    case IIR_PHYSICAL_TYPE_DEFINITION:
      _cc_out << "phyInfo ";
      break;
    }
    _cc_out << "Savant";
    type_def->_get_declaration()->_get_declarator()->_publish_cc();
    _cc_out << "Type_info;\n";
  }
}

void 
IIRScram_TypeDefinition::_come_into_scope( symbol_table *sym_tab, 
					   IIR_TypeDeclaration *type_decl ){
  
  // Type decl should _only_ be NULL for the universal operators.
  ASSERT( type_decl != NULL || this == StandardPackage::savant_universal_integer ||
	  this == StandardPackage::savant_universal_real );
  
  set<IIR_Declaration> *implicit_decls = NULL;

  if( type_decl != NULL ){
    implicit_decls = type_decl->_get_implicit_declarations();
  }

  if(  implicit_decls == NULL ){
    implicit_decls = new set<IIR_Declaration>();
    _build_implicit_operators( implicit_decls );

    if( type_decl != NULL ){
      type_decl->_set_implicit_declarations( implicit_decls );
    }
  }
  
  ASSERT( implicit_decls != NULL );
  ASSERT( implicit_decls->num_elements() > 0 );
  sym_tab->add_declaration( implicit_decls );
}


void 
IIRScram_TypeDefinition::_build_implicit_operators( set<IIR_Declaration> *add_to ){
  // So, ALL types have these relational operators defined.
  char *operators[] = { "\"=\"", "\"/=\"", NULL };
  _build_implicit_operators( operators, add_to, StandardPackage::boolean_type, this, this );
  if( _is_numeric_type() == TRUE ){
    char *addition_operators[] = { "\"+\"", "\"-\"", NULL };
    _build_implicit_operators( addition_operators, add_to, this, this, this );
    // This is unary +/-
    _build_implicit_operators( addition_operators, add_to, this, this );
    _build_implicit_operator( "\"abs\"", add_to, this, this );
  }
  
  if( this == StandardPackage::bit_type || this == StandardPackage::boolean_type ){
    _build_logical_operators( add_to );
  }
}

void 
IIRScram_TypeDefinition::_build_implicit_operator( char *op, 
						   set<IIR_Declaration> *add_to,
						   IIRScram_TypeDefinition *return_type,
						   IIRScram_TypeDefinition *left_type,
						   IIRScram_TypeDefinition *right_type ){

  IIR_FunctionDeclaration *new_operator = new IIR_FunctionDeclaration();
  new_operator->set_declarator( IIR_StringLiteral::get(op, strlen(op) ) );
  
  IIR_ConstantInterfaceDeclaration *left = new IIR_ConstantInterfaceDeclaration();
  left->set_declarator( IIR_Identifier::get("left", strlen("left")) );
  left->set_subtype( (IIR_TypeDefinition *)left_type );
  left->_set_is_visible( FALSE );
  new_operator->_set_is_implicit( TRUE );
  IIRScram::copy_location( this, left );
  new_operator->interface_declarations.append( left );
  
  if( right_type != NULL ){
    IIR_ConstantInterfaceDeclaration *right = new IIR_ConstantInterfaceDeclaration();
    right->set_declarator( IIR_Identifier::get("right", strlen("right") ) );
    right->set_subtype( (IIR_TypeDefinition *)right_type );
    right->_set_is_visible( FALSE );
    new_operator->_set_is_implicit( TRUE );
    IIRScram::copy_location( this, right );
    new_operator->interface_declarations.append( right );
  }

  new_operator->set_return_type( (IIR_TypeDefinition *)return_type );
  new_operator->_set_is_implicit( TRUE );
  IIRScram::copy_location( this, new_operator );
  
  add_to->add( new_operator );
}

void 
IIRScram_TypeDefinition::_build_implicit_operators( char *ops[], 
						    set<IIR_Declaration> *add_to,
						    IIRScram_TypeDefinition *return_type,
						    IIRScram_TypeDefinition *left_type,
						    IIRScram_TypeDefinition *right_type ){
  int i = 0;
  char *current = ops[i];
  while( current != NULL ){
    _build_implicit_operator( current, add_to, return_type, left_type, right_type );
    current = ops[++i];
  }
}

void 
IIRScram_TypeDefinition::_build_logical_operators( set<IIR_Declaration> *add_to ){
  char *logical_operators[] = {
    "\"and\"", "\"or\"", "\"nand\"", "\"nor\"", "\"xor\"", "\"xnor\"", 0
  };

  _build_implicit_operators( logical_operators, add_to, this, this, this );
  _build_implicit_operator( "\"not\"", add_to, this, this );
}

void 
IIRScram_TypeDefinition::_build_ordering_operators( set<IIR_Declaration> *add_to ){
  // Scalar types have these relational operators defined.
  char *operators[] = { "\"<\"", "\"<=\"", "\">\"", "\">=\"", NULL };

  _build_implicit_operators( operators, add_to, 
			     StandardPackage::boolean_type, this, this );
}
