/*$Id: d_mos.cc,v 15.19 1999/11/03 18:07:50 al Exp $ -*- C++ -*-
 * mos device basics
 * netlist syntax:
 * device:  mxxxx d g s b mname <device args> <model card args>
 * model:   .model mname NMOS <args>
 *	or  .model mname PMOS <args>
 */
#include "l_denoise.h"
#include "d_cap.h"
#include "d_admit.h"
#include "d_vccs.h"
#include "d_cs.h"
#include "d_res.h"
#include "ap.h"
#include "d_mos1.h"
#include "d_mos2.h"
#include "d_mos3.h"
#include "d_mos6.h"
/*--------------------------------------------------------------------------*/
//		MODEL_MOS_BASE::MODEL_MOS_BASE();
//	CARD*	MODEL_MOS_BASE::new_model(CS& cmd);

//		MOS_COMMON::MOS_COMMON();
//		MOS_COMMON::MOS_COMMON(const MOS_COMMON& p)
//	void	MOS_COMMON::parse(CS& cmd);
// 	void	MOS_COMMON::print(int,int)const;
//	void	MOS_COMMON::expand();

//		DEV_MOS::DEV_MOS();
//		DEV_MOS::DEV_MOS(const DEV_MOS& p);
//	void	DEV_MOS::parse(CS& cmd);
// 	void	DEV_MOS::print(int,int)const;
//	double	DEV_MOS::probe_tr_num(const std::string& what)const;
//	void	DEV_MOS::expand();
// 	bool	DEV_MOS::do_tr();
//	void	DEV_MOS::limit_mos(double,double,double);
/*--------------------------------------------------------------------------*/
int MODEL_MOS_BASE::Count = 0;
int MOS_COMMON::Count = -1;
int DEV_MOS::Count = 0;
static EVAL_MOS_Ids Eval_Ids(CC_STATIC);
static EVAL_MOS_Gmf Eval_Gmf(CC_STATIC);
static EVAL_MOS_Gmr Eval_Gmr(CC_STATIC);
static EVAL_MOS_Gds Eval_Gds(CC_STATIC);
static EVAL_MOS_Gmbf Eval_Gmbf(CC_STATIC);
static EVAL_MOS_Gmbr Eval_Gmbr(CC_STATIC);
static EVAL_MOS_Cgb Eval_Cgb(CC_STATIC);
static EVAL_MOS_Cgd Eval_Cgd(CC_STATIC);
static EVAL_MOS_Cgs Eval_Cgs(CC_STATIC);
static MOS_COMMON Default_MOS(CC_STATIC);
/*--------------------------------------------------------------------------*/
MODEL_MOS_BASE::MODEL_MOS_BASE()
{
  			/* inherit, then override diode defaults */
  cjo    = NOT_INPUT;	/* override 0.0 */
  pb     = 0.8;		/* override 1.0 */
  fcpb   = fc * pb;
			/* mos (not diode) defaults */
  rsh    = NOT_INPUT;
  rd     = NOT_INPUT;
  rs     = NOT_INPUT;
  cbd    = NOT_INPUT;
  cbs    = NOT_INPUT;
  is     = NOT_INPUT;
  js     = NOT_INPUT;
  cgso   = 0.0;
  cgdo   = 0.0;
  cgbo   = 0.0;
  calc_cj = false;
  polarity = pN;
  ++Count;
}
/*--------------------------------------------------------------------------*/
void MODEL_MOS_BASE::parse_params_base(CS& cmd)
{
  int level = 0;
  cmd.get("LEvel",  &level);	/* dummys */

  cmd.get("TNOM",   &_tnom,	mOFFSET, -ABS_ZERO);
  cmd.get("FC",     &fc);	/* diode params */
  cmd.get("PB",     &pb,	mPOSITIVE);
  cmd.get("CJ",     &cjo);
  cmd.get("MJ",     &mj);
  cmd.get("CJSw",   &cjsw);
  cmd.get("MJSw",   &mjsw);
  cmd.get("KF",     &kf);
  cmd.get("AF",     &af);

  cmd.get("IS",     &is);
  cmd.get("JS",     &js);

  cmd.get("RSH",    &rsh);
  cmd.get("RD",     &rd);
  cmd.get("RS",     &rs);
  cmd.get("CBD",    &cbd);
  cmd.get("CBS",    &cbs);
  cmd.get("CGSo",   &cgso);
  cmd.get("CGDo",   &cgdo);
  cmd.get("CGBo",   &cgbo);
}
/*--------------------------------------------------------------------------*/
void MODEL_MOS_BASE::post_parse_base()
{
  {if (_tnom == NOT_INPUT){
    _tnom = OPT::tnom;
  }else{
    untested();
  }}
  {if ((rs == NOT_INPUT)  &&  (rd != NOT_INPUT)){
    untested();
    error(bWARNING, long_label()+": rd input, but not rs. setting rs = 0.\n");
    rs = 0.;
  }else if ((rd == NOT_INPUT)  &&  (rs != NOT_INPUT)){
    untested();
    error(bWARNING, long_label()+": rs input, but not rd. setting rd = 0.\n");
    rd = 0.;
  }else{
    /* rd, rs are ok, either have both or neither */
  }}
  
  {if ((rsh != NOT_INPUT)  &&  (rd != NOT_INPUT)){
    error(bWARNING, long_label() + ": rsh - rs - rd conflict: using "
	  + (((rd <= 0.)  &&  (rs <= 0.)) ? "rsh" : "rs,rd") + '\n');
  }else if ((rsh == NOT_INPUT)  &&  (rd == NOT_INPUT)){
    rsh = 0.;
  }else{
    /* rsh, rd are ok, have one or other */
  }}
  
  {if (is == NOT_INPUT  &&  js == NOT_INPUT){
    is = mDEFAULT_is;
  }else if (is != NOT_INPUT  &&  js != NOT_INPUT){
    error(bWARNING, long_label() + ": is - js conflict\n");
  }else{
    /* is, js are ok, have one or other */
  }}

  fcpb = fc * pb;
}
/*--------------------------------------------------------------------------*/
void MODEL_MOS_BASE::print_base_begin(OMSTREAM where, int level)const
{
  where.setfloatwidth(7);
  where   << ".model  " << short_label();
  where   << ((polarity < 0) ? "  pmos  (" : "  nmos  (");
  where   <<   "level=" << level;

  where	  << "  tnom="	<< _tnom+ABS_ZERO;
  if (is != NOT_INPUT)
    where << "  is="	<< is;
  if (js != NOT_INPUT)
    where << "  js="	<< js;
  where   << "  fc="	<< fc;
  where   << "  pb="	<< pb;
  if (!calc_cj)
    where << "  cj="	<< cjo;
  where   << "  mj="	<< mj;
  where   << "  cjsw="	<< cjsw;
  where   << "  mjsw="	<< mjsw;

  if (rsh != NOT_INPUT)
    where << "  rsh="	<< rsh;
  if (rd != NOT_INPUT)
    where << "  rd="	<< rd;
  if (rs != NOT_INPUT)
    where << "  rs="	<< rs;
  if (cbd != NOT_INPUT)
    where << "  cbd="	<< cbd;
  if (cbs != NOT_INPUT)
    where << "  cbs="	<< cbs;
  where   << "  cgso="	<< cgso;
  where   << "  cgdo="	<< cgdo;
  where   << "  cgbo="	<< cgbo;

  if (kf != NOT_INPUT)
    where << "  kf="	<< kf;
  if (af != NOT_INPUT)
    where << "  af="	<< af;
}
/*--------------------------------------------------------------------------*/
void MODEL_MOS_BASE::print_base_mid(OMSTREAM where)const
{
  where   << ")\n*+(";
  if (calc_cj)
    where << "* cj="	<< cjo;
}
/*--------------------------------------------------------------------------*/
void MODEL_MOS_BASE::print_base_end(OMSTREAM where)const
{
  where << ")\n";
}
/*--------------------------------------------------------------------------*/
CARD* MODEL_MOS_BASE::new_model(CS& cmd)
{
  /* already parsed to know it is either NMos or PMos */
  cmd.skiplparen();
  cmd.stuck();
  int level = 1;
  cmd.pscan("LEvel").get("LEvel", &level);
  if (cmd.stuck()){
    cmd.warn(bWARNING, "no level specified, using 1");
  }
  switch (level){
  default:
  case 1: return new MODEL_MOS1;
  case 2: return new MODEL_MOS2;
  case 3: return new MODEL_MOS3;
  case 6: return new MODEL_MOS6;
  }
}
/*--------------------------------------------------------------------------*/
MOS_COMMON::MOS_COMMON(int c)
  :COMPONENT_COMMON(c),
   lo(NOT_INPUT),
   wo(NOT_INPUT),
   ad_in(NOT_INPUT),
   as_in(NOT_INPUT),
   pd(0.0),
   ps(0.0),
   nrd(1.0),
   nrs(1.0),
   off(false),
   icset(false),
   sb(c),
   db(c)
{
  for (int i = 0;  i < mNUM_INIT_COND;  i++){
    ic[i] = 0.;
  }
  ++Count;
  /* not all is initialized.  remainder filled in by expand */
}
/*--------------------------------------------------------------------------*/
MOS_COMMON::MOS_COMMON(const MOS_COMMON& p)
  :COMPONENT_COMMON(p),
   lo(p.lo),
   wo(p.wo),
   ad_in(p.ad_in),
   as_in(p.as_in),
   pd(p.pd),
   ps(p.ps),
   nrd(p.nrd),
   nrs(p.nrs),
   off(p.off),
   icset(p.icset),
   sb(p.sb, CC_STATIC),
   db(p.db, CC_STATIC)
{
  for (int i = 0;  i < mNUM_INIT_COND;  i++){
    ic[i] = 0.;
  }
  ++Count;
  /* not all is initialized.  remainder filled in by expand */
}
/*--------------------------------------------------------------------------*/
void MOS_COMMON::parse(CS& cmd)
{
  assert(!has_model());

  parse_modelname(cmd);
  if (cmd.is_pfloat()){				/* accept W/L notation for */
    wo = cmd.ctopf() * MICRON2METER;		/* dimensions (microns)    */
    {if (cmd.match('/')){
      untested();
      cmd.skip();
      lo = cmd.ctopf() * MICRON2METER;
    }else{
      untested();
    }}
  }
  cmd.stuck();
  do{
    cmd.get("L",   &lo,    mPOSITIVE);
    cmd.get("W",   &wo,    mPOSITIVE);
    cmd.get("AD",  &ad_in, mPOSITIVE);
    cmd.get("AS",  &as_in, mPOSITIVE);
    cmd.get("PD",  &pd,    mPOSITIVE);
    cmd.get("PS",  &ps,    mPOSITIVE);
    cmd.get("NRD", &nrd,   mPOSITIVE);
    cmd.get("NRS", &nrs,   mPOSITIVE);
  }while (cmd.more() && !cmd.stuck());
  cmd.check(bWARNING, "what's this?");
}
/*--------------------------------------------------------------------------*/
void MOS_COMMON::print(OMSTREAM where)const
{
  where << "  " << modelname();

  where.setfloatwidth(7);
  if (lo != NOT_INPUT)
    where << "  l="   << lo;
  if (wo != NOT_INPUT)
    where << "  w="   << wo;
  if (ad_in != NOT_INPUT)
    where << "  ad="  << ad_in;
  if (as_in != NOT_INPUT)
    where << "  as="  << as_in;
  if (pd != 0.)
    where << "  pd="  << pd;
  if (ps != 0.)
    where << "  ps="  << ps;
  where   << "  nrd=" << nrd;
  where   << "  nrs=" << nrs;
  
  if (icset){
    where << "  IC=";
    for (int i = 0;  i < mNUM_INIT_COND;  i++)
      where << ic[i] << ' ';
  }
  where << '\n';
}
/*--------------------------------------------------------------------------*/
const MODEL_MOS_BASE* MOS_COMMON::expand()
{
  const MODEL_MOS123* m = dynamic_cast<const MODEL_MOS123*>(attach_model());
  if (!m){
    untested();
    error(bERROR, "model "+modelname()+" is not a mosfet (NMOS or PMOS)\n");
  }
 
  {if (lo != NOT_INPUT){
    le = lo - 2. * m->ld;
  }else{
    untested();
    le = OPT::defl - 2. * m->ld;
  }}
  we = (wo != NOT_INPUT) ? wo : OPT::defw;
  
  cgate = m->cox * we * le;

  relxj = (m->xj != NOT_INPUT  &&  m->xj > 0.)
    ? .5 * m->xj / le
    : NOT_INPUT;
  // mos2 only -- need to move
  {if (cgate != 0){
    eta_1 = (kPI/4.) * E_SI * m->delta / cgate * le;
  }else{
    // reachable only where it doesn't matter
    eta_1 = 0;
  }}
  eta = eta_1 + 1.;
  eta_2 = eta / 2.;

  {if (m->rsh != NOT_INPUT  &&  m->rd <= 0.  &&  m->rs <= 0.){
    rd = m->rsh * nrd;
    rs = m->rsh * nrs;
  }else{
    untested();
    rd = (m->rd != NOT_INPUT) ? m->rd : 0.;
    rs = (m->rs != NOT_INPUT) ? m->rs : 0.;
  }}

  ad = (ad_in != NOT_INPUT) ? ad_in : OPT::defad;
  as = (as_in != NOT_INPUT) ? as_in : OPT::defas;
  {if (m->js == NOT_INPUT  ||  ad == 0.  ||  as == 0.){
    {if (m->is == NOT_INPUT){
      untested();
      idsat = issat = mDEFAULT_is;
      //error(bWARNING, "ignoring js, using default is\n");
    }else{
      idsat = issat = m->is;	/* this convoluted logic */
    }}				/* is for Spice compatibility */
  }else{
    idsat = m->js * ad;
    issat = m->js * as;
  }}

  db.set_modelname(modelname());
  db.area = ad;
  db.perim = pd;
  db.is = idsat;
  db.cj = m->cbd;	/* if NOT_INPUT diode will calculate it */
  db.cjsw = NOT_INPUT;
  db.attach(model());
  db.off = true;

  sb.set_modelname(modelname());
  sb.area = as;
  sb.perim = ps;
  sb.is = issat;
  sb.cj = m->cbs;	/* if NOT_INPUT diode will calculate it */
  sb.cjsw = NOT_INPUT;
  sb.attach(model());
  sb.off = true;

  return m;
}
/*--------------------------------------------------------------------------*/
DEV_MOS::DEV_MOS()
{
  attach_common(&Default_MOS);
  ids = gm = gds = gmb = vgs = vds = vbs = vdsat = vgst = von = 0.;
  cutoff = subthreshold = saturated = reversed = dbfwd = sbfwd = 
  	punchthru = false;
  Rs = Rd = NULL;
  Ddb = Dsb = NULL;
  Cgs = Cgd = Cgb = NULL;
  Gmbf = Gmbr = Gmf = Gmr = NULL;
  Yds = NULL;
  Ids = NULL;
  polarity = pN;
  ++Count;
}
/*--------------------------------------------------------------------------*/
DEV_MOS::DEV_MOS(const DEV_MOS& p):BASE_SUBCKT(p)
{
  ids = gm = gds = gmb = vgs = vds = vbs = vdsat = vgst = von = 0.;
  cutoff = subthreshold = saturated = reversed = dbfwd = sbfwd = 
  	punchthru = false;
  Rs = Rd = NULL;
  Ddb = Dsb = NULL;
  Cgs = Cgd = Cgb = NULL;
  Gmbf = Gmbr = Gmf = Gmr = NULL;
  Yds = NULL;
  Ids = NULL;
  polarity = p.polarity;
  ++Count;
}
/*--------------------------------------------------------------------------*/
void DEV_MOS::parse(CS& cmd)
{
  const MOS_COMMON* cc = prechecked_cast<const MOS_COMMON*>(common());
  assert(cc);
  MOS_COMMON* c = new MOS_COMMON(*cc);
  assert(c);
  
  parse_Label(cmd);
  parse_nodes(cmd,numnodes(),numnodes());
  c->parse(cmd);
  attach_common(c);
}
/*--------------------------------------------------------------------------*/
void DEV_MOS::print(OMSTREAM where, int)const
{
  const MOS_COMMON* c = prechecked_cast<const MOS_COMMON*>(common());
  assert(c);

  where << short_label();
  printnodes(where,numnodes());
  c->print(where);
}
/*--------------------------------------------------------------------------*/
double DEV_MOS::probe_tr_num(const std::string& what)const
{
  CS cmd(what);
  assert(subckt().exists());

  {if (cmd.pmatch("VDS")){
    return nDRAIN.v0() - nSOURCE.v0();
  }else if (cmd.pmatch("VGS")){
    return nGATE.v0() - nSOURCE.v0();
  }else if (cmd.pmatch("VBS")){
    return nBULK.v0() - nSOURCE.v0();
  }else if (cmd.pmatch("VGD")){
    untested();
    return nGATE.v0() - nDRAIN.v0();
  }else if (cmd.pmatch("VBD")){
    untested();
    return nBULK.v0() - nDRAIN.v0();
  }else if (cmd.pmatch("VSD")){
    untested();
    return nSOURCE.v0() - nDRAIN.v0();
  }else if (cmd.pmatch("VDM")){
    return (nDRAIN.v0() - nSOURCE.v0()
	    + nDRAIN.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VGM")){
    return (nGATE.v0() - nSOURCE.v0()
	    + nGATE.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VBM")){
    return (nBULK.v0() - nSOURCE.v0()
	    + nBULK.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VSM")){
    return (nSOURCE.v0() - nSOURCE.v0()
	    + nSOURCE.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VDG")){
    untested();
    return nDRAIN.v0() - nGATE.v0();
  }else if (cmd.pmatch("VBG")){
    untested();
    return nBULK.v0() - nGATE.v0();
  }else if (cmd.pmatch("VSG")){
    untested();
    return nSOURCE.v0() - nGATE.v0();
  }else if (cmd.pmatch("VDB")){
    untested();
    return nDRAIN.v0() - nBULK.v0();
  }else if (cmd.pmatch("VGB")){
    untested();
    return nGATE.v0() - nBULK.v0();
  }else if (cmd.pmatch("VSB")){
    untested();
    return nSOURCE.v0() - nBULK.v0();
  }else if (cmd.pmatch("VD")){
    return nDRAIN.v0();
  }else if (cmd.pmatch("VG")){
    return nGATE.v0();
  }else if (cmd.pmatch("VB")){
    return nBULK.v0();
  }else if (cmd.pmatch("VS")){
    return nSOURCE.v0();
  }else if (cmd.pmatch("Id")){
    return (Rd)
      ?   CARD::probe(Rd,"I")
      :   CARD::probe(Ids,"I")
	+ CARD::probe(Gmf,"I")
	+ CARD::probe(Gmr,"I")
	+ CARD::probe(Yds,"I")
	+ CARD::probe(Gmbf,"I")
	+ CARD::probe(Gmbr,"I")
	- CARD::probe(Cgd,"I")
	+ CARD::probe(Ddb,"I") * polarity;
  }else if (cmd.pmatch("IS")){
    return (Rs)
      ?   CARD::probe(Rs,"I")
      : - CARD::probe(Ids,"I")
	- CARD::probe(Gmf,"I")
	- CARD::probe(Gmr,"I")
	- CARD::probe(Yds,"I")
	- CARD::probe(Gmbf,"I")
	- CARD::probe(Gmbr,"I")
	- CARD::probe(Cgs,"I")
	+ CARD::probe(Dsb,"I") * polarity;
  }else if (cmd.pmatch("IG")){
    return CARD::probe(Cgs,"I")
         + CARD::probe(Cgd,"I")
	 + CARD::probe(Cgb,"I");
  }else if (cmd.pmatch("IB")){
    return - CARD::probe(Ddb,"I") * polarity
	   - CARD::probe(Dsb,"I") * polarity
	   - CARD::probe(Cgb,"I");
  }else if (cmd.pmatch("CGSOvl")){
    return CARD::probe(Cgs,"NV");
  }else if (cmd.pmatch("CGDOvl")){
    return CARD::probe(Cgd,"NV");
  }else if (cmd.pmatch("CGBOvl")){
    return CARD::probe(Cgb,"NV");
  }else if (cmd.pmatch("CGST")){
    return CARD::probe(Cgs,"EV");
  }else if (cmd.pmatch("CGDT")){
    return CARD::probe(Cgd,"EV");
  }else if (cmd.pmatch("CGBT")){
    return CARD::probe(Cgb,"EV");
  }else if (cmd.pmatch("CGSm")){
    return CARD::probe(Cgs,"EV") - CARD::probe(Cgs,"NV");
  }else if (cmd.pmatch("CGDm")){
    return CARD::probe(Cgd,"EV") - CARD::probe(Cgd,"NV");
  }else if (cmd.pmatch("CGBm")){
    return CARD::probe(Cgb,"EV") - CARD::probe(Cgb,"NV");
  }else if (cmd.pmatch("CBD")){
    return CARD::probe(Ddb,"Cap");
  }else if (cmd.pmatch("CBS")){
    return CARD::probe(Dsb,"Cap");
  }else if (cmd.pmatch("CGATE")){
    const MOS_COMMON* c = prechecked_cast<const MOS_COMMON*>(common());
    assert(c);
    return c->cgate;
  }else if (cmd.pmatch("GM")){
    return gm;
  }else if (cmd.pmatch("GDS")){
    return gds;
  }else if (cmd.pmatch("GMB")){
    return gmb;
  }else if (cmd.pmatch("VGST")){
    return vgst;
  }else if (cmd.pmatch("VON")){
    return von;
  }else if (cmd.pmatch("VDSAT")){
    return vdsat * polarity;
  }else if (cmd.pmatch("VTH")){
    return von * polarity;
  }else if (cmd.pmatch("VDSInt")){
    untested();
    return vds;
  }else if (cmd.pmatch("VGSInt")){
    untested();
    return vgs;
  }else if (cmd.pmatch("VBSInt")){
    untested();
    return vbs;
  }else if (cmd.pmatch("IDS")){
    return ids;
  }else if (cmd.pmatch("IDSTray")){
    return - CARD::probe(Cgd,"I")
           + CARD::probe(Ddb,"I") * polarity;
  }else if (cmd.pmatch("IDERror")){
    return  CARD::probe(Gmf,"I")
	  + CARD::probe(Gmr,"I")
	  + CARD::probe(Yds,"I")
	  + CARD::probe(Gmbf,"I")
	  + CARD::probe(Gmbr,"I");
  }else if (cmd.pmatch("P")){
    double power = 0.;
    for (CARD_LIST::const_iterator
	   ci = subckt().begin(); ci != subckt().end(); ++ci){
      power += CARD::probe(*ci,"P");
    }      
    return power;
  }else if (cmd.pmatch("PD")){
    double power = 0.;
    for (CARD_LIST::const_iterator
	   ci = subckt().begin(); ci != subckt().end(); ++ci){
      power += CARD::probe(*ci,"PD");
    }      
    return fixzero(power, probe_tr_num("P"));
  }else if (cmd.pmatch("PS")){
    double power = 0.;
    for (CARD_LIST::const_iterator
	   ci = subckt().begin(); ci != subckt().end(); ++ci){
      power += CARD::probe(*ci,"PS");
    }      
    return fixzero(power, probe_tr_num("P"));
  }else if (cmd.pmatch("REgion")){
    return static_cast<double>(
	  (!cutoff)
	+ (!subthreshold * 2)
	+ (saturated * 4)
	+ (sbfwd * 10)
	+ (dbfwd * 20)
	+ (punchthru * 40)
    ) * ((reversed)? -1 : 1);
  }else if (cmd.pmatch("Status")){
    untested();
    return static_cast<double>(converged() * 2);
//  }else if (cmd.pmatch("DTNew")){
//    return timef - time0;
//  }else if (cmd.pmatch("DTOld")){
//    return time0 - time1;
//  }else if (cmd.pmatch("TIMEF")){
//    return timef;
//  }else if (cmd.pmatch("TIME")){
//    return time0;
//  }else if (cmd.pmatch("TIMEO")){
//    return time1;
  }else{ /* bad parameter */
    return NOT_VALID;
  }}
  /*NOTREACHED*/
}
/*--------------------------------------------------------------------------*/
bool DEV_MOS::tr_needs_eval()
{
  {if (is_q_for_eval()){
    return false;
  }else if (!converged()){
    return true;
  }else{
    node_t eff_source((reversed) ? drainnode : sourcenode);
    node_t eff_drain((reversed) ? sourcenode : drainnode);
    return !(conchk(vds,polarity*(eff_drain.v0()-eff_source.v0()),OPT::vntol)
	     && conchk(vgs, polarity*(nGATE.v0()-eff_source.v0()),OPT::vntol)
	     && conchk(vbs, polarity*(nBULK.v0()-eff_source.v0()),OPT::vntol));
  }}
}
/* on STATUS::iter[iSTEP] ..............
 * This forces nobypass on the first iteration of a new time step.
 * It hides a BUG, probably in review.
 * If an entire time step is bypassed, review doesn't realize it is ok
 * so it re-queues the same time (zero time step), even though the
 * numbers are ok.
 * The reason this is necessary is that a capacitor inside a mosfet may
 * need evaluation (integration) even if the whole device is bypassed.
 */
/*--------------------------------------------------------------------------*/
bool DEV_MOS::do_tr()
{
  const MOS_COMMON* c = prechecked_cast<const MOS_COMMON*>(common());
  assert(c);
  const MODEL_MOS_BASE* m = prechecked_cast<const MODEL_MOS_BASE*>(c->model());
  assert(m);
  assert(subckt().exists());

  double Vds, Vgs, Vbs;
  
  sourcenode.map();	/* BUG:: why is this here??.... */
  drainnode.map();	/* because these nodes get missed in the usual map */

  bool was_cutoff = cutoff;
  bool was_subthreshold = subthreshold;
  bool was_saturated = saturated;
  bool was_reversed = reversed;
  bool was_dbfwd = dbfwd;
  bool was_sbfwd = sbfwd;

  {if (STATUS::iter[SIM::mode] <= 1){
    reversed = false;
    Vds = Vgs = Vbs = 0.;
    if (OPT::mosflags & 0100){
      untested();
      //Vds = m->vto;
    }
    if (OPT::mosflags & 0200){
      untested();
      //Vgs = m->vto;
    }
    if (OPT::mosflags & 0400  &&  nSOURCE.t != nBULK.t){
      untested();
      Vbs = -1;
    }
    if (nDRAIN.t == nGATE.t){
      Vds = Vgs;
    }
  }else if (reversed){
    Vds = polarity * volts_limited(sourcenode,drainnode);
    Vgs = polarity * volts_limited(nGATE,drainnode);
    Vbs = polarity * volts_limited(nBULK,drainnode);
  }else{
    Vds = polarity * volts_limited(drainnode,sourcenode);
    Vgs = polarity * volts_limited(nGATE,sourcenode);
    Vbs = polarity * volts_limited(nBULK,sourcenode);
  }}

  limit_mos(Vds, Vgs, Vbs);// also sets the new vds,vgs,vbs
  if (vds < 0){
    {if (nSOURCE.t == nBULK.t  ||  nDRAIN.t == nGATE.t){
      error(bTRACE, long_label() + ": cannot reverse\n");
      error(bTRACE, "vds=%g vgs=%g vbs=%g\n", vds, vgs, vbs);
    }else if (!(OPT::mosflags & 0001)){
      error(bTRACE, long_label() + ": reversing\n");
      error(bTRACE, "before: vds=%g vgs=%g vbs=%g\n", vds, vgs, vbs);
      reversed = !reversed;
      vgs -= vds;
      vbs -= vds;
      vds = -vds;
      error(bTRACE, "after: vds=%g vgs=%g vbs=%g\n", vds, vgs, vbs);
      if (OPT::dampstrategy & dsREVERSE){
	SIM::fulldamp = true;
	untested();
	error(bTRACE, long_label() + ":reverse damp\n");
      }
      {if (!(OPT::mosflags & 0040)){
	vbs = std::min(vbs,0.);
      }else{
	untested();
      }}
    }else{
      untested();
    }}
  }
  m->tr_eval(this);
  set_converged(subckt().do_tr());
  
  if (vds > 100){
    untested();
    error(bDEBUG,"%s:%d: ", long_label().c_str(), evaliter());
    error(bDEBUG,"vds=%e vgs=%e vbs=%e\n",
		  vds,   vgs,   vbs);
    error(bDEBUG,"+      ids=%e gm=%e gds=%e gmb=%e\n",
    			 ids,   gm,   gds,   gmb);
  }
  trace3(long_label(), vds, vgs, vbs);
  trace4("", ids, gm, gds, gmb);
  if (was_cutoff != cutoff  ||  was_subthreshold != subthreshold  
  	||  was_saturated != saturated  ||  was_reversed != reversed  
	||  was_dbfwd != dbfwd  ||  was_sbfwd != sbfwd){
    if (OPT::dampstrategy & dsDEVREGION){
      SIM::fulldamp = true;
    }
    #if defined(DO_TRACE)
      error(bTRACE,"%s:%d: region change\n", long_label().c_str(), evaliter());
    #endif
  }
  return converged();
}
/*--------------------------------------------------------------------------*/
/* limit_mos: do Spice style voltage limiting
 */
void DEV_MOS::limit_mos(double Vds, double Vgs, double Vbs)
{
					/* Spice style vgs limiting */
  {if (!(OPT::mosflags & 0010) && STATUS::iter[SIM::mode] > 1){
    //assert(vgst == vgs - von);
    {if (Vgs > vgs){			/* increasing */
      {if (vgst < 0){			/* was off */
	vgs = std::min(Vgs, von + .5);
      }else if (vgst < 3.5){		/* ??? */
	vgs = std::min(Vgs, von + 4.);
      }else{
	vgs = std::min(Vgs, 3.*vgs - 2.*von + 2.);
      }}
    }else{				/* decreasing */
      {if (vgst < 0){			/* off */
	vgs = std::max(Vgs, 3. * vgs - 2. * von - 2.);
      }else if (vgst < 3.5){
	vgs = std::max(Vgs, von - .5);
      }else{
	vgs = std::max(Vgs, von + 2.);
      }}
    }}
    //Vds += vgs - Vgs;			/* vds patch (per Spice) not done */
  }else{				/* because it usually makes it worse */
    vgs = Vgs;
  }}
  {if (nDRAIN.t == nGATE.t){		/* gd tied, limiting done */
    vds = Vds + (vgs - Vgs);		/* it seems that vds = vgs should */
    					/* work, but it will be a little off */
					/* if there is resistance */

					/* Spice style vds limiting */
  }else if (!(OPT::mosflags & 0020) && STATUS::iter[SIM::mode] > 1){
    {if (Vds <= vds){			/* decreasing */
      {if (vds < 3.5){
	vds = std::max(Vds,-.5);
      }else{
	vds = std::max(Vds,2.);
      }}
    }else{				/* increasing */
      {if (vds < 3.5){
	vds = std::min(Vds,4.);
      }else{
	vds = std::min(Vds, 3.*vds + 2.);
      }}
    }}
    //vgs += vds - Vds;
  }else{
    vds = Vds;
  }}

  {if (!(OPT::mosflags & 0040) && STATUS::iter[SIM::mode] > 1){
//    if (Vbs > 0.){
//      if (vbs >= 0.){
//        vbs = std::min(Vbs,vbs+.1);
//      }else{
//        vbs = 0.;
//      }
//    }else{
//      vbs = Vbs;
//    }
    vbs = std::min(Vbs,0.);
  }else{
    vbs = Vbs;
  }}
  if (OPT::dampstrategy & dsDEVLIMIT
      && (vgs != Vgs || vds != Vds || vbs != Vbs)){
    untested();
    SIM::fulldamp = true;
    error(bTRACE, long_label() + ":device limit damp\n");
  }
  #if defined(DO_TRACE)
    if (vgs != Vgs || vds != Vds || vbs != Vbs){
      trace1(long_label(), evaliter());
      trace3("prop", Vds, Vgs, Vbs);
      trace3("using", vds, vgs, vbs);
    }
  #endif
}
/*--------------------------------------------------------------------------*/
void DEV_MOS::expand()
{
  const MOS_COMMON* cc = prechecked_cast<const MOS_COMMON*>(common());
  assert(cc);
  MOS_COMMON* c = const_cast<MOS_COMMON*>(cc);
  assert(c);
  const MODEL_MOS_BASE* m = c->expand();
  assert(m);

  polarity = m->polarity;
  /* It seems to matter what order these parts are placed in.
   * Changing "push_front" to "push_back" changes the results.
   * Differences are small, probably insignificant,
   * probably due to differences in accumulated round-off errors.
   * The differences show with values close to zero,
   * that probably should be zero, but have values ~~1e-40 either way.
   * Not sure what is the best ordering, or if this is significant.
   */
							       /* resistors */
  {if (OPT::rstray && c->rs != 0.){
    untested();
    sourcenode.t = STATUS::newnode_model();
    {if (!Rs){
      Rs = new DEV_RESISTANCE;
      subckt().push_front(Rs);
      untested();
    }else{
      untested();
    }}
    Rs->set("Rs", this, NULL, c->rs, nSOURCE, sourcenode);
  }else{
    sourcenode = nSOURCE;
    if (Rs){
      subckt().erase(Rs);
      Rs = NULL;
      untested();
    }
  }}

  {if (OPT::rstray && c->rd != 0.){
    untested();
    drainnode.t = STATUS::newnode_model();
    {if (!Rd){
      Rd = new DEV_RESISTANCE;
      subckt().push_front(Rd);
      untested();
    }else{
      untested();
    }}
    Rd->set("Rd", this, NULL, c->rd, nDRAIN, drainnode);
  }else{
    drainnode = nDRAIN;
    if (Rd){
      subckt().erase(Rd);
      Rd = NULL;
      untested();
    }
  }}

  {if (nBULK.t != drainnode.t   &&   c->idsat != 0.){
    if (!Ddb){
      Ddb = new DEV_DIODE;
      subckt().push_front(Ddb);
    }
    switch (polarity){
    case pN:
      Ddb->set("Ddb", this, &(c->db), 0., nBULK, drainnode);
      break;
    case pP:
      Ddb->set("Ddb", this, &(c->db), 0., drainnode, nBULK);
      break;
    }
  }else{
    if (Ddb){
      subckt().erase(Ddb);
      Ddb = NULL;
      untested();
    }
  }}

  {if (nBULK.t != sourcenode.t   &&   c->issat != 0.){
    if (!Dsb){
      Dsb = new DEV_DIODE;
      subckt().push_front(Dsb);
    }
    switch (polarity){
    case pN:
      Dsb->set("Dsb", this, &(c->sb), 0., nBULK, sourcenode);
      break;
    case pP:
      Dsb->set("Dsb", this, &(c->sb), 0., sourcenode, nBULK);
      break;
    }
  }else{
    if (Dsb){
      subckt().erase(Dsb);
      Dsb = NULL;
      untested();
    }
  }}

  {if (OPT::cstray  &&  nGATE.t != sourcenode.t){
    if (!Cgs){
      Cgs = new DEV_CAPACITANCE;
      subckt().push_front(Cgs);
    }
    Cgs->set("Cgs", this, &Eval_Cgs, (m->cgso*c->we), nGATE, sourcenode);
  }else{
    {if (Cgs){
      subckt().erase(Cgs);
      Cgs = NULL;
      untested();
    }else{
      untested();
    }}
  }}

  {if (OPT::cstray  &&  nGATE.t != drainnode.t){
    if (!Cgd){
      Cgd = new DEV_CAPACITANCE;
      subckt().push_front(Cgd);
    }
    Cgd->set("Cgd", this, &Eval_Cgd, (m->cgdo*c->we), nGATE, drainnode);
  }else{
    if (Cgd){
      subckt().erase(Cgd);
      Cgd = NULL;
      untested();
    }
  }}

  {if (OPT::cstray  &&  nBULK.t != nGATE.t){
    if (!Cgb){
      Cgb = new DEV_CAPACITANCE;
      subckt().push_front(Cgb);
    }
    Cgb->set("Cgb", this, &Eval_Cgb, (m->cgbo*c->le), nGATE, nBULK);
  }else{
    {if (Cgb){
      subckt().erase(Cgb);
      Cgb = NULL;
      untested();
    }else{
      untested();
    }}
  }}

  {if (nBULK.t != sourcenode.t){
    if (!Gmbf){
      Gmbf = new DEV_VCCS;
      subckt().push_front(Gmbf);
    }
    Gmbf->set("Gmbf",this,&Eval_Gmbf,0.,drainnode,sourcenode,nBULK,sourcenode);
  }else{
    if (Gmbf){
      subckt().erase(Gmbf);
      Gmbf = NULL;
      untested();
    }
  }}

  {if (nBULK.t != drainnode.t){
    if (!Gmbr){
      Gmbr = new DEV_VCCS;
      subckt().push_front(Gmbr);
    }
    Gmbr->set("Gmbr",this,&Eval_Gmbr,0.,sourcenode,drainnode,nBULK,drainnode);
  }else{
    if (Gmbr){
      subckt().erase(Gmbr);
      Gmbr = NULL;
      untested();
    }
  }}
  
  if (!Yds){
    Yds = new DEV_ADMITTANCE;
    subckt().push_front(Yds);
  }
  
  Yds->set("Yds", this, &Eval_Gds, 0., drainnode, sourcenode);

  {if (nGATE.t != sourcenode.t){
    if (!Gmf){
      Gmf = new DEV_VCCS;
      subckt().push_front(Gmf);
    }
    Gmf->set("Gmf",this,&Eval_Gmf,0.,drainnode,sourcenode,nGATE,sourcenode);
  }else{
    {if (Gmf){
      subckt().erase(Gmf);
      Gmf = NULL;
      untested();
    }else{
      untested();
    }}
  }}

  {if (nGATE.t != drainnode.t){
    if (!Gmr){
      Gmr = new DEV_VCCS;
      subckt().push_front(Gmr);
    }
    Gmr->set("Gmr",this,&Eval_Gmr,0.,sourcenode,drainnode,nGATE,drainnode);
  }else{
    if (Gmr){
      subckt().erase(Gmr);
      Gmr = NULL;
      untested();
    }
  }}

  if (!Ids){
    Ids = new DEV_CS;
    subckt().push_front(Ids);
  }
  Ids->set("Ids", this, &Eval_Ids, 0., drainnode, sourcenode);

  assert(subckt().exists());
  subckt().expand();
  subckt().set_slave();

  assert(!constant()); /* because it is nonlinear */
}
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
