/*	$Id: variables.c,v 1.8 1997/10/30 18:29:42 sandro Exp $	*/

/*
 * Copyright (c) 1997
 *	Sandro Sigala, Brescia, Italy.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "zile.h"
#include "extern.h"
#include "hash.h"

/*
 * Default variables values table.
 */
static struct var_entry { char *var; char *val; } def_vars[] = {
#define X(zile_var, val) { zile_var, val },
#include "tbl_vars.h"
#undef X
};

static htablep var_table;

void
init_variables(void)
{
	struct var_entry *p;

	var_table = hash_table_build_default();

	for (p = &def_vars[0]; p < &def_vars[sizeof(def_vars) / sizeof(def_vars[0])]; p++)
		set_variable(p->var, p->val);
}

void
set_variable(char *var, char *val)
{
	hash_table_store(var_table, var, xstrdup(val));

	/* Force refresh of cached variables. */
	cur_tp->refresh_cached_variables();
}

void
unset_variable(char *var)
{
	hash_table_delete(var_table, var);
}

char *
get_variable(char *var)
{
	return hash_table_fetch(var_table, var);
}

int
is_variable_equal(char *var, char *val)
{
	char *v;
	return ((v = hash_table_fetch(var_table, var)) != NULL)
		&& !strcmp(v, val);
}

int
lookup_bool_variable(char *var)
{
	char *p;

	if ((p = hash_table_fetch(var_table, var)) != NULL)
		return strcmp(p, "true") == 0;

	minibuf_error("warning: used uninitialized variable `%s'", var);
	waitkey(2 * 1000);

	return FALSE;
}

static historyp
make_variable_history(void)
{
	int i, n;
	bucketp bucket;
	historyp hp;

	for (i = n = 0; i < var_table->size; ++i)
		for (bucket = var_table->table[i]; bucket != NULL; bucket = bucket->next)
			++n;
	hp = new_history(n, FALSE);
	for (i = n = 0; i < var_table->size; ++i)
		for (bucket = var_table->table[i]; bucket != NULL; bucket = bucket->next)
			hp->completions[n++] = xstrdup(bucket->key);

	return hp;
}

DEFUN("set-variable", set_variable)
/*+
Set a variable value to the user specified value.
+*/
{
	char *var, *val;
	historyp hp;

	hp = make_variable_history();
	var = minibuf_read_history("Set variable: ", "", hp);
	free_history(hp);
	if (var == NULL)
		return cancel();

	if (var[0] == '\0') {
		minibuf_error("%FCNo variable name given%E");
		return FALSE;
	}

	if ((val = minibuf_read("Set %s to value: ", "", var)) == NULL)
		return cancel();

	set_variable(var, val);

	/*
	 * The `tab-width' variable is also local to the buffer.
	 */
	if (!strcmp(var, "tab-width")) {
		if ((cur_bp->tab_width = atoi(val)) < 1) {
			cur_bp->tab_width = 8;
			minibuf_error("warning: wrong tab-width value `%s'", val);
			waitkey(2 * 1000);
		}
	}

	return TRUE;
}

static int
sort_func(const void *ptr1, const void *ptr2)
{
	return strcmp((*(bucketp *)ptr1)->key, (*(bucketp *)ptr2)->key);
}

DEFUN("list-variables", list_variables)
/*+
List defined variables.
+*/
{
	windowp wp, old_wp = cur_wp;
	bufferp bp;
	bucketp bucket, *sorted_table;
	int i, j, n;

	if ((wp = find_window("*Variable List*")) == NULL) {
		cur_wp = popup_window();
		cur_bp = cur_wp->bp;
		bp = find_buffer("*Variable List*", TRUE);
		switch_to_buffer(bp);
	} else {
		cur_wp = wp;
		cur_bp = wp->bp;
	}

	zap_buffer_content();
	cur_bp->flags = BFLAG_NEEDNAME | BFLAG_NOSAVE | BFLAG_NOUNDO | BFLAG_NOEOB;
	set_temporary_buffer(cur_bp);

	/*
	 * Count the hash table buckets.
	 */
	for (i = n = 0; i < var_table->size; ++i)
		for (bucket = var_table->table[i]; bucket != NULL; bucket = bucket->next)
			++n;

	/*
	 * Allocate space for the sorted table.
	 */
	sorted_table = (bucketp *)xmalloc(sizeof(bucketp) * n);

	/*
	 * Add the buckets to the sorted table.
	 */
	for (i = j = 0; i < var_table->size; ++i)
		for (bucket = var_table->table[i]; bucket != NULL; bucket = bucket->next)
			sorted_table[j++] = bucket;

	/*
	 * Sort the table.
	 */
	qsort(sorted_table, n, sizeof(bucketp), sort_func);

	bprintf("Global variables:\n\n");
	bprintf("%-30s %s\n", "Variable", "Value");
	bprintf("%-30s %s\n", "--------", "-----");

	/*
	 * Output the table.
	 */
	for (i = 0; i < n; i++)
		if (sorted_table[i]->data != NULL)
			bprintf("%-30s \"%s\"\n", sorted_table[i]->key, sorted_table[i]->data);

	free(sorted_table);

	bprintf("\nLocal buffer variables:\n\n");
	bprintf("%-30s %s\n", "Variable", "Value");
	bprintf("%-30s %s\n", "--------", "-----");
	bprintf("%-30s \"%d\"\n", "tab-width", cur_bp->tab_width);

	gotobob();

	cur_bp->flags |= BFLAG_READONLY;

	cur_wp = old_wp;
	cur_bp = old_wp->bp;

	return TRUE;
}
