/************************************************************************/
/*  Make XImages and/or Pixmaps from a bitmap.				*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	"appColor.h"
#   include	"appImage.h"

#   include	<debugon.h>

extern void _XInitImageFuncPtrs( XImage * xim );

/************************************************************************/
/*									*/
/*  Data structures for dithering.					*/
/*									*/
/************************************************************************/
typedef struct ColorValue
    {
    long	cvR;
    long	cvG;
    long	cvB;
    } ColorValue;

static ColorValue *	scanThisRow;
static ColorValue *	scanNextCor;

typedef void (*GetSourceRow) ( ColorValue * cv, const unsigned char * from,
						const BitmapDescription * bd );

typedef int (*PutScreenRow)(	unsigned char *		to,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		cor,
				AppColors *		ac );

static int dc1[512];
static int dc3[512];
static int dc5[512];
static int dc7[512];

static void scanFillDitherTables( void )
    {
    int		i;

    for ( i= -256; i < 0; i++ )
	{
	dc1[i+256]= -( ( 8- 1*i+ 8 )/ 16 );
	dc3[i+256]= -( ( 8- 3*i+ 8 )/ 16 );
	dc5[i+256]= -( ( 8- 5*i+ 8 )/ 16 );
	dc7[i+256]= -( ( 8- 7*i+ 8 )/ 16 );
	}

    for ( i= 0; i < 256; i++ )
	{
	dc1[i+256]= ( 1*i+ 8 )/ 16;
	dc3[i+256]= ( 3*i+ 8 )/ 16;
	dc5[i+256]= ( 5*i+ 8 )/ 16;
	dc7[i+256]= ( 7*i+ 8 )/ 16;
	}
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 8 image from rgb8 palette data.			*/
/*									*/
/************************************************************************/
static void scan_ipal8(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;
    unsigned	mask;

    switch( bd->bdBitsPerPixel )
	{
	case 8:
	    for ( col= 0; col < bd->bdPixelsWide; from++, cv++, col++ )
		{
		cv->cvR += bd->bdRGB8Palette[*from].rgb8Red;
		cv->cvG += bd->bdRGB8Palette[*from].rgb8Green;
		cv->cvB += bd->bdRGB8Palette[*from].rgb8Blue;
		}
	    return;
	case 4: case 2: case 1:
	    mask= ( 1 << bd->bdBitsPerPixel )- 1;

	    for ( col= 0; col < bd->bdBytesPerRow; from++, col++ )
		{
		int	shift;

		for ( shift= 8- bd->bdBitsPerPixel; shift >= 0;
					cv++, shift -= bd->bdBitsPerPixel )
		    {
		    int		val= ( *from >> shift ) & mask;

		    cv->cvR += bd->bdRGB8Palette[val].rgb8Red;
		    cv->cvG += bd->bdRGB8Palette[val].rgb8Green;
		    cv->cvB += bd->bdRGB8Palette[val].rgb8Blue;
		    }
		}
	    return;
	case 16:
	    {
	    unsigned short *	psh= (unsigned short *)from;

	    for ( col= 0; col < bd->bdPixelsWide; psh++, cv++, col++ )
		{
		cv->cvR += bd->bdRGB8Palette[*psh].rgb8Red;
		cv->cvG += bd->bdRGB8Palette[*psh].rgb8Green;
		cv->cvB += bd->bdRGB8Palette[*psh].rgb8Blue;
		}

	    return;
	    }
	default:
	    LDEB(bd->bdBitsPerPixel); return;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 8 image from rgb8 palette data with an alpha channel.	*/
/*									*/
/************************************************************************/

static void scan_ipal8a(	ColorValue *			cv,
				const unsigned char *		from,
				const BitmapDescription *	bd )
    {
    int		col;
    unsigned	mask;

    switch( bd->bdBitsPerPixel )
	{
	case 16:
	    for ( col= 0; col < bd->bdPixelsWide; from += 2, cv++, col++ )
		{
		cv->cvR += bd->bdRGB8Palette[*from].rgb8Red;
		cv->cvG += bd->bdRGB8Palette[*from].rgb8Green;
		cv->cvB += bd->bdRGB8Palette[*from].rgb8Blue;
		}
	    return;
	case 8:
	    mask= 0x0f;

	    for ( col= 0; col < bd->bdPixelsWide; from++, cv++, col++ )
		{
		int		val= ( *from >> 4 ) & mask;

		cv->cvR += bd->bdRGB8Palette[val].rgb8Red;
		cv->cvG += bd->bdRGB8Palette[val].rgb8Green;
		cv->cvB += bd->bdRGB8Palette[val].rgb8Blue;
		}
	    return;
	default:
	    LDEB(bd->bdBitsPerPixel); return;
	}

    return;
    }

static void scan_ibw1(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; from++, col += 8 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit++ )
	    {
	    unsigned int	val;

	    val= 255* ( ( (*from) >> ( 7- bit ) ) & 0x1 );

	    val= 255- val;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_iwb1(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; from++, col += 8 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit++ )
	    {
	    unsigned int	val;

	    val= 255* ( ( (*from) >> ( 7- bit ) ) & 0x1 );

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_ibw2(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; from++, col += 4 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 2 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xc0 ) + 0xc0/ 2 )/ 0xc0;

	    val= 255- val;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_iwb2(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; from++, col += 4 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 2 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xc0 ) + 0xc0/ 2 )/ 0xc0;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_ibw4(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; from++, col += 2 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 4 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xf0 ) + 0xf0/ 2 )/ 0xf0;

	    val= 255- val;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_iwb4(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; from++, col += 2 )
	{
	unsigned int	bit;

	for ( bit= 0; bit < 8; cv++, bit += 4 )
	    {
	    unsigned int	val;

	    val= ( (*from) << bit );
	    val= ( 255* ( val & 0xf0 ) + 0xf0/ 2 )/ 0xf0;

	    cv->cvR += val;
	    cv->cvG += val;
	    cv->cvB += val;
	    }
	}
    }

static void scan_ibw8(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; cv++, from++, col++ )
	{
	int		val= 255- *from;

	cv->cvR += val;
	cv->cvG += val;
	cv->cvB += val;
	}
    }

static void scan_iwb8(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; cv++, from++, col++ )
	{
	cv->cvR += *from;
	cv->cvG += *from;
	cv->cvB += *from;
	}
    }

/************************************************************************/
/*									*/
/*  Add data from one row of a 24 bits image to an acumulator array.	*/
/*									*/
/************************************************************************/
static void scan_rgb24(	ColorValue *			cv,
			const unsigned char *		from,
			const BitmapDescription *	bd )
    {
    int		col;

    for ( col= 0; col < bd->bdPixelsWide; cv++, col++ )
	{ cv->cvR += *(from++); cv->cvG += *(from++); cv->cvB += *(from++); }

    return;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 8 image from rgb data.					*/
/*									*/
/************************************************************************/
static int scanFillD8RgbRow(	unsigned char *		to,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		cor,
				AppColors *		ac )
    {
    int			col;
    unsigned int	col332;
    XColor		xc;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; cor++, col++ )
	    {
	    int		count= 0;
	    int		cw;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    r += cor->cvR; g += cor->cvG; b += cor->cvB;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    col332= C332( r, g, b );

	    if  ( ac->acColors[col332].pad == 1 )
		{
		if  ( appColorRgb332( &xc, ac, r, g, b ) )
		    { LDEB(col); return -1;	}
		}
	    else{ xc= ac->acColors[col332];	}

	    *(to++)= xc.pixel;

	    r -= xc.red/ 256;	r += 256;
	    g -= xc.green/ 256;	g += 256;
	    b -= xc.blue/ 256;	b += 256;

	    val += count;

	    cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
	    cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
	    cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

	    r= dc7[r]; g= dc7[g]; b= dc7[b];
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r += val->cvR; g += val->cvG; b += val->cvB;
		r += cor->cvR; g += cor->cvG; b += cor->cvB;

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		col332= C332( r, g, b );

		if  ( ac->acColors[col332].pad == 1 )
		    {
		    if  ( appColorRgb332( &xc, ac, r, g, b ) )
			{ LDEB(col); return -1;	}
		    }
		else{ xc= ac->acColors[col332];	}

		*(to++)= xc.pixel;

		r -= xc.red/ 256;	r += 256;
		g -= xc.green/ 256;	g += 256;
		b -= xc.blue/ 256;	b += 256;

		cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
		cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
		cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

		r= dc7[r]; g= dc7[g]; b= dc7[b];

		e -= d2; cor++;
		}

	    e += e2;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 1 image from rgb data.					*/
/*									*/
/*  Routines rely on the fact that they are called in scan line order	*/
/*  and that surplus memory has been allocated for the last row.	*/
/*									*/
/************************************************************************/

static int scanFillD1RgbRow_X(	unsigned char *		to,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		cor,
				AppColors *		ac )
    {
    int			col;
    unsigned int	col111;
    XColor		xc;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; cor++, col++ )
	    {
	    int		count= 0;
	    int		cw;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    r += cor->cvR; g += cor->cvG; b += cor->cvB;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    col111= C111( r, g, b );

	    if  ( ac->ac222Colors[col111].pad == 1 )
		{
		if  ( appColorRgb111( &xc, ac, r, g, b ) )
		    { LDEB(col); return -1;	}
		}
	    else{ xc= ac->ac222Colors[col111];	}

	    *(to++)= xc.pixel;

	    r -= xc.red/ 256;	r += 256;
	    g -= xc.green/ 256;	g += 256;
	    b -= xc.blue/ 256;	b += 256;

	    val += count;

	    cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
	    cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
	    cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

	    r= dc7[r]; g= dc7[g]; b= dc7[b];
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r += val->cvR; g += val->cvG; b += val->cvB;
		r += cor->cvR; g += cor->cvG; b += cor->cvB;

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		col111= C111( r, g, b );

		if  ( ac->ac222Colors[col111].pad == 1 )
		    {
		    if  ( appColorRgb111( &xc, ac, r, g, b ) )
			{ LDEB(col); return -1;	}
		    }
		else{ xc= ac->ac222Colors[col111];	}

		*(to++)= xc.pixel;

		r -= xc.red/ 256;	r += 256;
		g -= xc.green/ 256;	g += 256;
		b -= xc.blue/ 256;	b += 256;

		cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
		cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
		cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

		r= dc7[r]; g= dc7[g]; b= dc7[b];

		e -= d2; cor++;
		}

	    e += e2;
	    }
	}

    return 0;
    }

static int scanFillD1RgbRow_32MM(	unsigned char *		to,
					int			frWide,
					int			toWide,
					int			weight,
					ColorValue *		val,
					ColorValue *		cor,
					AppColors *		ac )
    {
    unsigned char *	row= to;
    int			col;

    if  ( scanFillD1RgbRow_X( to, frWide, toWide, weight, val, cor, ac ) )
	{ LDEB(1); return -1;	}

    for ( col= 0; col < toWide; to += 4, row += 32, col += 32 )
	{
	to[3]=	( row[ 0]  << 7 )	|
		( row[ 1]  << 6 )	|
		( row[ 2]  << 5 )	|
		( row[ 3]  << 4 )	|
		( row[ 4]  << 3 )	|
		( row[ 5]  << 2 )	|
		( row[ 6]  << 1 )	|
		( row[ 7]  << 0 )	;

	to[2]=	( row[ 8]  << 7 )	|
		( row[ 9]  << 6 )	|
		( row[10]  << 5 )	|
		( row[11]  << 4 )	|
		( row[12]  << 3 )	|
		( row[13]  << 2 )	|
		( row[14]  << 1 )	|
		( row[15]  << 0 )	;

	to[1]=	( row[16]  << 7 )	|
		( row[17]  << 6 )	|
		( row[18]  << 5 )	|
		( row[19]  << 4 )	|
		( row[20]  << 3 )	|
		( row[21]  << 2 )	|
		( row[22]  << 1 )	|
		( row[23]  << 0 )	;

	to[0]=	( row[24]  << 7 )	|
		( row[25]  << 6 )	|
		( row[26]  << 5 )	|
		( row[27]  << 4 )	|
		( row[28]  << 3 )	|
		( row[29]  << 2 )	|
		( row[30]  << 1 )	|
		( row[31]  << 0 )	;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 16 image from rgb data.				*/
/*									*/
/************************************************************************/
static int scanFillD16RgbRow(	unsigned char *		cto,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		cor,
				AppColors *		ac )
    {
    int			col;
    unsigned int	col555;
    XColor		xc;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    unsigned short *	to= (unsigned short *)cto;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; cor++, col++ )
	    {
	    int		count= 0;
	    int		cw;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    r += cor->cvR; g += cor->cvG; b += cor->cvB;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    if  ( ac->acVisualClass ==	TrueColor	||
		  ac->acVisualClass ==	DirectColor	)
		{ appColorRgbDirect( &xc, ac, r, g, b );	}
	    else{
		col555= C555( r, g, b );

		if  ( ac->acColors[col555].pad == 1 )
		    {
		    if  ( appColorRgb555( &xc, ac, r, g, b ) )
			{ LDEB(col); return -1;	}
		    }
		else{ xc= ac->acColors[col555];	}
		}

	    *(to++)= xc.pixel;

	    r -= xc.red/ 256;	r += 256;
	    g -= xc.green/ 256;	g += 256;
	    b -= xc.blue/ 256;	b += 256;

	    val += count;

	    cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
	    cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
	    cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

	    r= dc7[r]; g= dc7[g]; b= dc7[b];
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r += val->cvR; g += val->cvG; b += val->cvB;
		r += cor->cvR; g += cor->cvG; b += cor->cvB;

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		if  ( ac->acVisualClass ==	TrueColor	||
		      ac->acVisualClass ==	DirectColor	)
		    { appColorRgbDirect( &xc, ac, r, g, b );	}
		else{
		    col555= C555( r, g, b );

		    if  ( ac->acColors[col555].pad == 1 )
			{
			if  ( appColorRgb555( &xc, ac, r, g, b ) )
			    { LDEB(col); return -1;	}
			}
		    else{ xc= ac->acColors[col555];	}
		    }

		*(to++)= xc.pixel;

		r -= xc.red/ 256;	r += 256;
		g -= xc.green/ 256;	g += 256;
		b -= xc.blue/ 256;	b += 256;

		cor[ 0].cvR= dc5[r]; cor[ 0].cvG= dc5[g]; cor[ 0].cvB= dc5[b];
		cor[-1].cvR= dc3[r]; cor[-1].cvG= dc3[g]; cor[-1].cvB= dc3[b];
		cor[+1].cvR= dc1[r]; cor[+1].cvG= dc1[g]; cor[+1].cvB= dc1[b];

		r= dc7[r]; g= dc7[g]; b= dc7[b];

		e -= d2; cor++;
		}

	    e += e2;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 24 image from rgb data.				*/
/*									*/
/************************************************************************/
static int scanFillD24RgbRow(	unsigned char *		to,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		corx,
				AppColors *		ac )
    {
    int			col;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; col++ )
	    {
	    int		count= 0;
	    int		cw;

	    r= g= b= 0;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    *(to++)= r; *(to++)= g; *(to++)= b;

	    val += count;
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r= val->cvR; g= val->cvG; b= val->cvB;

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		*(to++)= r; *(to++)= g; *(to++)= b;

		e -= d2;
		}

	    e += e2;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fill a Depth 32 image from rgb data.				*/
/*									*/
/************************************************************************/
static int scanFillD32RgbRow(	unsigned char *		cto,
				int			frWide,
				int			toWide,
				int			weight,
				ColorValue *		val,
				ColorValue *		corx,
				AppColors *		ac )
    {
    int			col;

    long		r;
    long		g;
    long		b;

    int			e;
    int			d2;
    int			e2;

    Pixel *		to= (Pixel *)cto;
    XColor		xc;

    if  ( toWide <= frWide )
	{
	e2= 2* frWide;
	d2= 2* toWide;
	e= e2- toWide;

	r= g= b= 0;

	for ( col= 0; col < toWide; col++ )
	    {
	    int		count= 0;
	    int		cw;

	    r= g= b= 0;

	    while( e >= 0 )
		{
		r += val[count].cvR;
		g += val[count].cvG;
		b += val[count].cvB;

		count++; e -= d2;
		}

	    e += e2;

	    cw= count* weight;
	    r= r/cw; g= g/cw; b= b/cw;

	    if  ( r < 0 ) { r= 0;	}
	    if  ( g < 0 ) { g= 0;	}
	    if  ( b < 0 ) { b= 0;	}

	    if  ( r > 255 ) { r= 255;	}
	    if  ( g > 255 ) { g= 255;	}
	    if  ( b > 255 ) { b= 255;	}

	    appColorRgbDirect( &xc, ac, r, g, b );
	    *(to++)= xc.pixel;

	    val += count;
	    }
	}
    else{
	e2= 2* toWide;
	d2= 2* frWide;
	e= e2- frWide;

	r= g= b= 0;

	for ( col= 0; col < frWide; val++, col++ )
	    {
	    while( e >= 0 )
		{
		r= val->cvR; g= val->cvG; b= val->cvB;

		if  ( r < 0 ) { r= 0;		}
		if  ( g < 0 ) { g= 0;		}
		if  ( b < 0 ) { b= 0;		}

		if  ( r > 255 ) { r= 255;	}
		if  ( g > 255 ) { g= 255;	}
		if  ( b > 255 ) { b= 255;	}

		appColorRgbDirect( &xc, ac, r, g, b );
		*(to++)= xc.pixel;

		e -= d2;
		}

	    e += e2;
	    }
	}

    return 0;
    }

static void scanInitRow(	ColorValue *		cv,
				int			wide )
    {
    int			col;

    for ( col= 0; col < wide; cv++, col++ )
	{ cv->cvR= cv->cvG= cv->cvB= 0L;	}
    }

static int scanBuildImage(	AppBitmapImage *	abi,
				AppColors *		ac,
				unsigned char *		approximatedBuffer,
				int			bytesPerRow,
				BitmapDescription *	bd,
				Display *		dis,
				Colormap		cmap,
				int			toWide,
				int			toHigh,
				GetSourceRow		getSource,
				PutScreenRow		putScreen )
    {
    int				toRow;
    int				frRow;

    const unsigned char *	from;
    unsigned char *		to;

    int				frWide= bd->bdPixelsWide;
    int				frHigh= bd->bdPixelsHigh;

    int				e;
    int				d2;
    int				e2;

    /*
    long			avg;
    */

    if  ( toHigh <= frHigh )
	{
	e2= 2* frHigh;
	d2= 2* toHigh;
	e= e2- toHigh;

	scanInitRow( scanNextCor+ 1, toWide );

	frRow= 0;
	for ( toRow= 0; toRow < toHigh; toRow++ )
	    {
	    int		count= 0;

	    scanInitRow( scanThisRow+ 1, frWide );
	    to= approximatedBuffer+ toRow* bytesPerRow;

	    while( e >= 0 )
		{
		from= abi->abiBuffer+ frRow* bd->bdBytesPerRow;

		(*getSource)( scanThisRow+ 1, from, bd );

		frRow++; count++; e -= d2;
		}

	    e += e2;

	    if  ( (*putScreen)( to, frWide, toWide, count,
					scanThisRow+ 1, scanNextCor+ 1, ac ) )
		{ LDEB(toRow); free( approximatedBuffer ); return -1;	}

	    /*
	    avg= ( scanNextCor[0].cvR && scanNextCor[toWide+ 1].cvR )/ 2;
	    scanNextCor[0].cvR= scanNextCor[toWide+ 1].cvR= avg;
	    avg= ( scanNextCor[0].cvG && scanNextCor[toWide+ 1].cvG )/ 2;
	    scanNextCor[0].cvG= scanNextCor[toWide+ 1].cvG= avg;
	    avg= ( scanNextCor[0].cvB && scanNextCor[toWide+ 1].cvB )/ 2;
	    scanNextCor[0].cvB= scanNextCor[toWide+ 1].cvB= avg;
	    */
	    }
	}
    else{
	e2= 2* toHigh;
	d2= 2* frHigh;
	e= e2- frHigh;

	toRow= 0;
	for ( frRow= 0; frRow < frHigh; frRow++ )
	    {
	    from= abi->abiBuffer+ frRow* bd->bdBytesPerRow;

	    scanInitRow( scanThisRow+ 1, frWide );
	    (*getSource)( scanThisRow+ 1, from, bd );

	    while( e >= 0 )
		{
		to= approximatedBuffer+ toRow* bytesPerRow;

		if  ( (*putScreen)( to, frWide, toWide, 1,
				    scanThisRow+ 1, scanNextCor+ 1, ac ) )
		    { LDEB(toRow); free( approximatedBuffer ); return -1; }

		e -= d2; toRow++;

		/*
		avg= ( scanNextCor[0].cvR && scanNextCor[toWide+ 1].cvR )/ 2;
		scanNextCor[0].cvR= scanNextCor[toWide+ 1].cvR= avg;
		avg= ( scanNextCor[0].cvG && scanNextCor[toWide+ 1].cvG )/ 2;
		scanNextCor[0].cvG= scanNextCor[toWide+ 1].cvG= avg;
		avg= ( scanNextCor[0].cvB && scanNextCor[toWide+ 1].cvB )/ 2;
		scanNextCor[0].cvB= scanNextCor[toWide+ 1].cvB= avg;
		*/
		}

	    e += e2;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make a pixmap from a picture.					*/
/*									*/
/*  1)  Make sure there is enough space in the accumulators for		*/
/*	scaling and dithering. Two extra bytes are allocated to make	*/
/*	dithering code simpler. (No need to check for the border).	*/
/*	Seven extra bytes are allocated to remove the need for checking	*/
/*	for the borders in bit expansion code.				*/
/*  2)  Make sure that at least the 222 colors are available.		*/
/*  2a) Allocate an array of XColors and initialise it.			*/
/*  3)  Fill dither tables.						*/
/*  4)  Fill an XImage, the way depends on the bitmap.			*/
/*  5)  Store it in a Pixmap.						*/
/*									*/
/************************************************************************/

int appImgMakeImage(	Display *			dis,
			int				screen,
			XImage **			pPimage,
			int				toWide,
			int				toHigh,
			AppColors *			ac,
			AppBitmapImage *		abi )
    {
    Colormap		cmap= DefaultColormap( dis, screen );
    int			depth= DefaultDepth( dis, screen );

    unsigned char *	ab;
    BitmapDescription *	bd= &(abi->abiBitmap);

    XImage *		xim;

    Visual *		vis;

    int			pad= 8;
    int			bypr;

    int			col;

    ColorValue *	fresh;

    PutScreenRow	putRow= (PutScreenRow)0;

    unsigned int	one= 1;

    /*  1  */
    fresh= (ColorValue *)realloc( scanThisRow, ( bd->bdPixelsWide+ 9 )* sizeof(ColorValue) );
    if  ( ! fresh )
	{ LXDEB(bd->bdPixelsWide,fresh); return -1;	}
    scanThisRow= fresh;

    fresh= (ColorValue *)realloc( scanNextCor, ( toWide+ 9 )* sizeof(ColorValue) );
    if  ( ! fresh )
	{ LXDEB(bd->bdPixelsWide,fresh); return -1;	}
    scanNextCor= fresh;

    vis= DefaultVisual( dis, screen );

    if  ( ac->acVisualClass ==	TrueColor	||
	  ac->acVisualClass ==	DirectColor	)
	{
	bypr= toWide;
	if  ( depth > 8 )
	    { bypr *= 2;	}
	if  ( depth > 16 )
	    { bypr *= 2;	}

	ab= (unsigned char *)malloc( toHigh* bypr );
	if  ( ! ab )
	    { LLDEB(toWide,toHigh); return -1;	}

	xim= XCreateImage( dis, vis, depth, ZPixmap, 0, (char *)ab,
					    toWide, toHigh, pad, 0 );

	if  ( ! xim )
	    { LDEB(xim); free( ab ); return -1;	}

	switch( xim->bits_per_pixel )
	    {
	    case 32:
		putRow= scanFillD32RgbRow;
		break;
	    case 24:
		if  ( xim->bitmap_unit != 8 )
		    { LDEB(xim->bitmap_unit); }

		putRow= scanFillD24RgbRow;
		break;
	    case 16:
		putRow= scanFillD16RgbRow;
		break;
	    default:
		LDEB(xim->bits_per_pixel);
		return -1;
	    }
	}
    else{
	switch( depth )
	    {
	    case 1:
		pad= BitmapPad( dis );
		bypr= ( toWide+ pad- 1 )/ pad; bypr *= (pad/8);
		ab= (unsigned char *)malloc( toHigh* bypr+ toWide+ 32 );
		if  ( ! ab )
		    { LLDEB(toWide,toHigh); return -1;	}

		xim= XCreateImage( dis, vis, depth, XYPixmap, 0, (char *)ab,
						    toWide, toHigh, pad, 0 );

		if  ( ! xim )
		    { LDEB(xim); free( ab ); return -1;	}

		switch( xim->bitmap_unit )
		    {
		    case 32:
			if  ( xim->byte_order == MSBFirst )
			    {
			    if  ( xim->bitmap_bit_order == MSBFirst )
				{ putRow= scanFillD1RgbRow_32MM; break; }
			    else{
				LLDEB(xim->byte_order,xim->bitmap_bit_order);
				return -1;
				}
			    }
			else{
			    LLDEB(xim->byte_order,xim->bitmap_bit_order);
			    return -1;
			    }
		    default:
			LDEB(xim->bitmap_unit); return -1;
		    }

		break;
	    case 8:
		bypr= 4* ( ( toWide+ 3 )/ 4 );
		ab= (unsigned char *)malloc( toHigh* bypr );
		if  ( ! ab )
		    { LLDEB(toWide,toHigh); return -1;	}

		xim= XCreateImage( dis, vis, depth, ZPixmap, 0, (char *)ab,
						    toWide, toHigh, pad, 0 );

		if  ( ! xim )
		    { LDEB(xim); free( ab ); return -1;	}

		putRow= scanFillD8RgbRow;

		break;
	    case 16:
		bypr= 4* ( ( 2* toWide+ 3 )/ 4 );
		ab= (unsigned char *)malloc( toHigh* bypr );
		if  ( ! ab )
		    { LLDEB(toWide,toHigh); return -1;	}

		xim= XCreateImage( dis, vis, depth, ZPixmap, 0, (char *)ab,
						    toWide, toHigh, pad, 0 );

		if  ( ! xim )
		    { LDEB(xim); free( ab ); return -1;	}

		putRow= scanFillD16RgbRow;

		break;
	    case 32:
	    case 24:
	    default:
		LDEB(depth); return -1;
	    }
	}

    /*  2  */
    for ( col= 0; col < 64; col++ )
	{
	int		r, g, b;
	XColor		xc;

	r= ( col & 0x30 ) << 2;
	g= ( col & 0x0c ) << 4;
	b= ( col & 0x03 ) << 6;

	/*
	printf( "%4d-> %4d,%4d,%4d-> %4d %s\n",
			col, r, g, b, C222(r,g,b),
			ac->ac222Colors[C222(r,g,b)].pad?"free":"ALLOCATED" );
	*/

	if  ( appColorRgb222( &xc, ac, r, g, b ) )
	    { LDEB(col); return -1; }
	}

    if  ( *((unsigned char *)&one) )
	{ xim->byte_order= LSBFirst; }
    else{ xim->byte_order= MSBFirst; }

    _XInitImageFuncPtrs( xim );

    bypr= xim->bytes_per_line;

    /*  3  */
    if  ( ! dc1[0] )
	{ scanFillDitherTables();	}

    /*  4  */
    switch( bd->bdColorEncoding )
	{
	case BMcoRGB8PALETTE:
	    if  ( bd->bdHasAlpha )
		{
		if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
						toWide, toHigh,
						scan_ipal8a, putRow ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
						toWide, toHigh,
						scan_ipal8, putRow ) )
		    { LDEB(1); return -1;	}
		}
	    break;
	case BMcoBLACKWHITE:
	    switch( bd->bdBitsPerPixel )
		{
		case 1:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_ibw1, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 2:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_ibw2, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 4:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_ibw4, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 8:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_ibw8, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		default:
		    LLDEB(bd->bdColorEncoding,bd->bdBitsPerPixel); return -1;
		}
	    break;
	case BMcoWHITEBLACK:
	    switch( bd->bdBitsPerPixel )
		{
		case 1:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_iwb1, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 2:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_iwb2, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 4:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_iwb4, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		case 8:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_iwb8, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		default:
		    LLDEB(bd->bdColorEncoding,bd->bdBitsPerPixel);
		}
	    break;
	case BMcoRGB:
	    switch( bd->bdBitsPerSample )
		{
		case 8:
		    if  ( scanBuildImage( abi, ac, ab, bypr, bd, dis, cmap,
					    toWide, toHigh,
					    scan_rgb24, putRow ) )
			{ LDEB(1); return -1;	}
		    break;
		default:
		    LLDEB(bd->bdColorEncoding,bd->bdBitsPerSample); return -1;
		}
	    break;
	default:
	    LDEB(bd->bdColorEncoding); return -1;
	}

    *pPimage= xim; return 0;
    }

/************************************************************************/
/*									*/
/*  Convert a BitmapDescription to an XImage.				*/
/*									*/
/************************************************************************/

int appImgMakePixmap(	Display *			display,
			int				screen,
			Window				win,
			GC				gc,
			Pixmap *			pPixmap,
			int				toWide,
			int				toHigh,
			AppColors *			ac,
			AppBitmapImage *		abi )
    {
    XImage *		xim;

    int			depth= DefaultDepth( display, screen );

    if  ( appImgMakeImage( display, screen, &xim, toWide, toHigh, ac, abi ) )
	{ LLDEB(toWide,toHigh); return -1;	}

    /*  5  */
    *pPixmap= XCreatePixmap( display, win, toWide, toHigh, depth );

    if  ( ! *pPixmap )
	{ XDEB(*pPixmap); return -1;	}

    XPutImage( display, *pPixmap, gc, xim, 0, 0, 0, 0, toWide, toHigh );

    XDestroyImage( xim );

    return 0;
    }
