/************************************************************************/
/*									*/
/*  Ted, Screen drawing and forcing drawing through XClearArea().	*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>

#   include	"tedApp.h"
#   include	<X11/cursorfont.h>

#   include	<debugon.h>

# define DRDEB(dr) DEB(DEBFUN( "%s(%3d) %s=[%4d..%4d,%4d..%4d]\n", \
			     __FILE__, __LINE__, #dr, \
			     (dr)->drX0, (dr)->drX1, (dr)->drY0, (dr)->drY1 ))

# define LOG_REDRAWS	0

/************************************************************************/
/*									*/
/*  Get buffer positions for a text line.				*/
/*									*/
/************************************************************************/

static void tedLinePositions(	BufferPosition *	bpLineBegin,
				BufferPosition *	bpLineEnd,
				const BufferItem *	bi,
				int			line,
				const TextLine *	tl )
    {
    docInitPosition( bpLineBegin );
    docInitPosition( bpLineEnd );

    bpLineBegin->bpBi= bpLineEnd->bpBi= (BufferItem *)bi;
    bpLineBegin->bpLine= bpLineEnd->bpLine= line;
    bpLineBegin->bpParticule= tl->tlFirstParticule;
    bpLineEnd->bpParticule= tl->tlFirstParticule+ tl->tlParticuleCount;
    bpLineBegin->bpStroff= tl->tlStroff;
    bpLineEnd->bpStroff= tl->tlStroff+ tl->tlStrlen;

    return;
    }

/************************************************************************/
/*									*/
/*  Various Border drawing routines.					*/
/*									*/
/************************************************************************/

static void tedDrawHorizontalBorder(	const BorderProperties *	bp,
					const DrawingContext *		dc,
					const DocumentRectangle *	drClip,
					int				above,
					int				x0,
					int				x1,
					int				y )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    int			wide;
    int			thick= tedBorderThick( &wide, bp, add );

    if  ( thick > 0 )
	{
	if  ( above )
	    { y -= thick;	}

	if  ( y > drClip->drY1 )
	    { return;	}
	if  ( y+ thick < drClip->drY0 )
	    { return;	}

	XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			    x0- dc->dcOx, y- dc->dcOy, x1- x0, thick );
	}

    return;
    }

static void tedDrawVerticalBorder(	const BorderProperties *	bp,
					const DrawingContext *		dc,
					int				x,
					int				y0,
					int				y1 )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    int			wide;
    int			thick= tedBorderThick( &wide, bp, add );

    if  ( thick > 0 )
	{
	XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			    x- dc->dcOx, y0- dc->dcOy- 1, thick, y1- y0+ 1 );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw a text item.							*/
/*									*/
/*  1)  Tabs need not to be drawn.					*/
/*									*/
/************************************************************************/

static int tedDrawParticules(	const BufferItem *	bi,
				TextParticule *		tp,
				int			count,
				int			baseLine,
				DrawingState *		ds,
				const DrawingContext *	dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    unsigned char *	paraString= bi->biParaString;

    int			drawn;
    int			i;
    int			len;
    int			y;

    AppPhysicalFont *	apf;

    /*  1  */
    switch( tp->tpKind )
	{ 
	case DOCkindTAB:
	    if  ( tp->tpObjectNumber >= 0			&&
		  tp->tpObjectNumber < bi->biParaTabCount	)
		{
		TabStop *	ts= bi->biParaTabStops+ tp->tpObjectNumber;

		switch( ts->tsLeader )
		    {
		    static char	dot[]= { 1, 2 };

		    case DOCtlNONE:
			break;

		    case DOCtlDOTS:
			XSetLineAttributes( add->addDisplay, dc->dcGc,
				    1, LineOnOffDash, CapButt, JoinMiter );
			XSetDashes( add->addDisplay, dc->dcGc,
						    0, dot, sizeof( dot ) );

			XDrawLine( add->addDisplay, dc->dcWindow, dc->dcGc,
			    tp->tpX0- dc->dcOx, baseLine- dc->dcOy,
			    tp->tpX0+ tp->tpPixelsWide- dc->dcOx,
			    baseLine- dc->dcOy );

			break;

		    case DOCtlHYPH:
		    case DOCtlUNDERLINE:
		    case DOCtlTHICK:
		    case DOCtlEQUAL:
		    default:
			LDEB(ts->tsLeader);
		    }
		}

	    drawn= 1;
	    return drawn;

	case DOCkindTEXT:
	    break;

	case DOCkindFIELDSTART:
	case DOCkindFIELDEND:
	case DOCkindBKMKSTART:
	case DOCkindBKMKEND:
	case DOCkindXE:
	case DOCkindTC:
	    drawn= 1;
	    return drawn;

	case DOCkindOBJECT:
	    if  ( tedDrawObject( bi, tp, baseLine, dc ) )
		{ LDEB(1); return -1;	}
	    return 1;

	default:
	    LDEB(tp->tpKind); return -1;
	}

    drawn= 1;
    len= tp[drawn-1].tpStroff+ tp[drawn-1].tpStrlen- tp->tpStroff;

    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;
    y= baseLine;

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT )
	{ y -= ( 3* apf->apfFullSizePixels )/ 10; }

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT )
	{ y += ( 3* apf->apfFullSizePixels )/ 10; }

    if  ( apf->apfFontSet )
	{
	XmbDrawString( add->addDisplay, dc->dcWindow, apf->apfFontSet, dc->dcGc,
		tp->tpX0- dc->dcOx, y- dc->dcOy,
		(char *)paraString+ tp->tpStroff, len );
	}
    else{
	if  ( tp->tpPhysicalFont != ds->dsPhysicalFont )
	    {
	    XSetFont( add->addDisplay, dc->dcGc, apf->apfFontStruct->fid );
	    ds->dsPhysicalFont= tp->tpPhysicalFont;
	    }

	XDrawString( add->addDisplay, dc->dcWindow, dc->dcGc,
				    tp->tpX0- dc->dcOx, y- dc->dcOy,
				    (char *)paraString+ tp->tpStroff, len );
	}

    i= 0;
    while( i < drawn )
	{
	int	x0;
	int	x1;
	int	y0;
	int	h;

	if  ( ! tp[i].tpTextAttribute.taIsUnderlined )
	    { i++; continue;	}

	x1= x0= tp[i].tpX0;
	y0= baseLine+ apf->apfUnderLinePositionPixels;
	h= apf->apfUnderlineThicknessPixels;

	while( i < drawn && tp[i].tpTextAttribute.taIsUnderlined )
	    { x1= tp[i].tpX0+ tp[i].tpPixelsWide; i++;	}

	XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
				    x0- dc->dcOx, y0- dc->dcOy, x1- x0, h );
	}

    return drawn;
    }

static int tedDrawTextLine(	const BufferItem *		bi,
				int				line,
				TextLine *			tl,
				const BufferSelection *		bs,
				const DocumentRectangle *	drClip,
				const FormattingFrame *		ff,
				DrawingState *			ds,
				const DrawingContext *		dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    int			done;
    TextParticule *	tp;

    done= 0;
    tp= bi->biParaParticules+ tl->tlFirstParticule;

    while( done < tl->tlParticuleCount )
	{
	int		drawn;

	if  ( tp->tpTextAttribute.taInField != ds->dsLinkPixelSet )
	    {
	    Display *	display= add->addDisplay;

	    if  ( tp->tpTextAttribute.taInField )
		{ XSetForeground( display, dc->dcGc, dc->dcLinkPixel ); }
	    else{ XSetForeground( display, dc->dcGc, dc->dcForePixel ); }

	    ds->dsLinkPixelSet= tp->tpTextAttribute.taInField;
	    }

	drawn= tedDrawParticules( bi, tp, tl->tlParticuleCount- done,
							    tl->tlY, ds, dc );
	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; tp += drawn;
	}

    return 0;
    }

static int tedLineRectangle(	const BufferItem *		bi,
				int				line,
				TextLine *			tl,
				const BufferSelection *		bs,
				const DocumentRectangle *	drClip,
				const FormattingFrame *		ff,
				DocumentRectangle *		drRedraw )
    {
    BufferPosition	bpLineBegin;
    BufferPosition	bpLineEnd;

    DocumentRectangle	drLine;

    const int		mindLine= 1;

    tedLinePositions( &bpLineBegin, &bpLineEnd, bi, line, tl );

    drLine.drY0= tl->tlY0;
    drLine.drY1= tl->tlY1;

    if  ( docComparePositions( &(bs->bsBegin), &bpLineBegin, mindLine ) < 0 )
	{
	drLine.drX0= ff->ffX0TextLines;

	if  ( bi->biParaFirstIndentTwips < 0 )
	    { drLine.drX0= ff->ffX0FirstLine;	}
	}
    else{ drLine.drX0= bs->bsBegin.bpX;	}

    if  ( docComparePositions( &(bs->bsEnd), &bpLineEnd, mindLine ) > 0 )
	{ drLine.drX1= ff->ffX1TextLines;	}
    else{ drLine.drX1= bs->bsEnd.bpX;		}

    if  ( docIntersectRectangle( drRedraw, drClip, &drLine ) )
	{ return 1;	}
    else{ return 0;	}
    }

static int tedDrawTextReverse(	const BufferItem *		bi,
				int				line,
				TextLine *			tl,
				const BufferSelection *		bs,
				const DocumentRectangle *	drClip,
				const FormattingFrame *		ff,
				DrawingState *			ds,
				const DrawingContext *		dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;

    DocumentRectangle	drRedraw;

    if  ( tedLineRectangle( bi, line, tl, bs, drClip, ff, &drRedraw ) )
	{
	XRectangle	xr;

	xr.x= drRedraw.drX0- dc->dcOx;
	xr.y= drRedraw.drY0- dc->dcOy;
	xr.width= drRedraw.drX1- drRedraw.drX0;
	xr.height= drRedraw.drY1- drRedraw.drY0+ 1;

	XSetClipRectangles( add->addDisplay, dc->dcGc, 0, 0, &xr, 1, Unsorted );

	tedDrawTextLine( bi, line, tl, bs, drClip, ff, ds, dc );
	}

    return 0;
    }

static int tedHighlightBack(	const BufferItem *		bi,
				int				line,
				TextLine *			tl,
				const BufferSelection *		bs,
				const DocumentRectangle *	drClip,
				const FormattingFrame *		ff,
				DrawingState *			ds,
				const DrawingContext *		dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;

    DocumentRectangle	drRedraw;

    if  ( tedLineRectangle( bi, line, tl, bs, drClip, ff, &drRedraw ) )
	{
	XSetForeground( add->addDisplay, dc->dcGc, dc->dcBackPixel );

	XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
		drRedraw.drX0- dc->dcOx,
		drRedraw.drY0- dc->dcOy,
		drRedraw.drX1- drRedraw.drX0,
		drRedraw.drY1- drRedraw.drY0+ 1 );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw a single paragraph. The Graphical context has been prepared	*/
/*  by the caller.							*/
/*									*/
/*  7)	Do not print below the bottom of fixed height cells. ( Word	*/
/*	does not do so either ).					*/
/*									*/
/************************************************************************/

static int tedDrawParaItem(	Widget				w,
				const BufferItem *		bi,
				const BufferSelection *		bs,
				const AppDrawingData *		add,
				int				bottom,
				const DocumentRectangle *	drClip,
				DrawingState *			ds,
				const DrawingContext *		dc )
    {
    int			line;
    TextLine *		tl;

    FormattingFrame	ff;
    int			x0;

    if  ( bi->biY0 > drClip->drY1 )
	{ return 0;	}
    if  ( bi->biY1 < drClip->drY0 )
	{ return 0;	}

    tedParagraphFrame( &ff, add, bottom, bi );
    x0= ff.ffX0TextLines;
    if  ( bi->biParaFirstIndentPixels < 0 )
	{ x0= ff.ffX0FirstLine;	}

    if  ( ! bi->biParaInTable && bi->biParaStartsOnNewPage )
	{
	XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
		add->addPaperRect.drX0- dc->dcOx,
		bi->biY0+ add->addDocRect.drY0- dc->dcOy, 
		add->addPaperRect.drX1- add->addPaperRect.drX0, 1 );
	}

    if  ( dc->dcDrawTableBorders && bi->biParaTopBorder.bpIsSet )
	{
	XSetForeground( add->addDisplay, dc->dcGc, dc->dcForePixel );
	ds->dsLinkPixelSet= 0;

	tedDrawHorizontalBorder( &(bi->biParaTopBorder), dc, drClip, 0,
				x0, ff.ffX1TextLines,
				bi->biY0+ bi->biParaSpaceBeforePixels );
	}
    else{
	if  ( dc->dcDrawTableBorders && bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	prevBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent > 0 )
		{
		prevBi= bi->biParent->biGroupChildren[
					    bi->biNumberInParent- 1];
		}
	    if  ( ! prevBi				||
		  ! prevBi->biParaBoxBorder.bpIsSet	)
		{
		XSetForeground( add->addDisplay, dc->dcGc, dc->dcForePixel );
		ds->dsLinkPixelSet= 0;

		tedDrawHorizontalBorder( &(bi->biParaBoxBorder), dc, drClip, 0,
				x0, ff.ffX1TextLines,
				bi->biY0+ bi->biParaSpaceBeforePixels );
		}
	    }
	}

    tl= bi->biParaLines;
    for ( line= 0; line < bi->biParaLineCount; tl++, line++ )
	{
	if  ( tl->tlY0 > drClip->drY1 )
	    { break;	}
	if  ( tl->tlY1 < drClip->drY0 )
	    { continue;	}

	if  ( bs )
	    {
	    BufferPosition	bpLineBegin;
	    BufferPosition	bpLineEnd;
	    const int		mindLine= 1;

	    tedLinePositions( &bpLineBegin, &bpLineEnd, bi, line, tl );

	    if  ( docComparePositions( &bpLineBegin, &(bs->bsEnd),
							    mindLine ) > 0 )
		{ break;	}
	    if  ( docComparePositions( &bpLineEnd, &(bs->bsBegin),
							    mindLine ) < 0 )
		{ continue;	}
	    }

	/*  7  */
	if  ( bottom > 0 && tl->tlY1 > bottom )
	    { break; }

	(*dc->dcDrawTextLine)( bi, line, tl, bs, drClip, &ff, ds, dc );
	}

    if  ( dc->dcDrawTableBorders && bi->biParaBottomBorder.bpIsSet )
	{
	XSetForeground( add->addDisplay, dc->dcGc, dc->dcForePixel );
	ds->dsLinkPixelSet= 0;

	tedDrawHorizontalBorder( &(bi->biParaBottomBorder), dc, drClip, 1,
				x0, ff.ffX1TextLines,
				bi->biY1- bi->biParaSpaceAfterPixels );
	}
    else{
	if  ( dc->dcDrawTableBorders && bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	nextBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent <
				bi->biParent->biGroupChildCount- 1 )
		{
		nextBi= bi->biParent->biGroupChildren[
					    bi->biNumberInParent+ 1];
		}
	    if  ( ! nextBi				||
		  ! nextBi->biParaBoxBorder.bpIsSet	)
		{
		XSetForeground( add->addDisplay, dc->dcGc, dc->dcForePixel );
		ds->dsLinkPixelSet= 0;

		tedDrawHorizontalBorder( &(bi->biParaBoxBorder), dc, drClip, 1,
				x0, ff.ffX1TextLines,
				bi->biY1- bi->biParaSpaceAfterPixels );
		}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw a single row. The Graphical context has been prepared by the	*/
/*  caller.								*/
/*									*/
/************************************************************************/

static void tedDrawCellBorders(		int				x0,
					int				x1,
					int				y0,
					int				y1,
					const CellProperties *		cp,
					const RowProperties *		rp,
					const DrawingContext *		dc,
					const DocumentRectangle *	drClip )
    {
    tedDrawHorizontalBorder( &(cp->cpTopBorder), dc, drClip, 0,
							    x0, x1, y0 );
    tedDrawHorizontalBorder( &(cp->cpBottomBorder), dc, drClip, 0,
							    x0, x1, y1 );

    tedDrawVerticalBorder( &(cp->cpLeftBorder), dc, x0, y0, y1 );
    tedDrawVerticalBorder( &(cp->cpRightBorder), dc, x1, y0, y1 );

    return;
    }

static void tedDrawRowGrid(	const BufferItem *		rowBi,
				const AppDrawingData *		add,
				DrawingState *			ds,
				const DrawingContext *		dc )
    {
    const BufferItem *		sectBi= rowBi->biParent;
    const BufferItem *		prevRow= (const BufferItem *)0;
    const BufferItem *		nextRow= (const BufferItem *)0;

    int				col;
    const CellProperties *	cp;
    const CellProperties *	prevCp= (const CellProperties *)0;
    const CellProperties *	nextCp= (const CellProperties *)0;

    const int			x= add->addDocRect.drX0;
    int				x0;
    int				x1;

    if  ( rowBi->biNumberInParent > 0 )
	{
	prevRow= sectBi->biGroupChildren[rowBi->biNumberInParent- 1];

	if  ( ! prevRow->biRowHasTableParagraphs			||
	      ! docAlignedColumns( &(rowBi->biRowProperties),
					&(rowBi->biRowProperties) )	)
	    { prevRow= (const BufferItem *)0;	}
	else{ prevCp= prevRow->biRowCells;	}
	}

    if  ( rowBi->biNumberInParent < sectBi->biGroupChildCount- 1 )
	{
	nextRow= sectBi->biGroupChildren[rowBi->biNumberInParent+ 1];

	if  ( ! nextRow->biRowHasTableParagraphs			||
	      ! docAlignedColumns( &(rowBi->biRowProperties),
					&(rowBi->biRowProperties) )	)
	    { nextRow= (const BufferItem *)0;	}
	else{ nextCp= nextRow->biRowCells;	}
	}

    XSetForeground( add->addDisplay, dc->dcGc, dc->dcTablePixel );
    ds->dsLinkPixelSet= -1;

    x0= x+ rowBi->biRowLeftIndentPixels;
    cp= rowBi->biRowCells;

    if  ( ! cp->cpLeftBorder.bpIsSet )
	{
	XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, rowBi->biY0- dc->dcOy- 1,
			1, rowBi->biY1- rowBi->biY0+ 1 );
	}

    for ( col= 0; col < rowBi->biRowCellCount; prevCp++, nextCp++, cp++, col++ )
	{
	x1= x+ cp->cpRightBoundaryPixels;

	if  ( ! cp->cpTopBorder.bpIsSet				&&
	      ( ! prevRow || ! prevCp->cpBottomBorder.bpIsSet )	)
	    {
	    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, rowBi->biY0- dc->dcOy- 1, x1- x0, 1 );

	    }

	if  ( ! cp->cpBottomBorder.bpIsSet			&&
	      ( ! nextRow || ! nextCp->cpBottomBorder.bpIsSet )	)
	    {
	    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, rowBi->biY1- dc->dcOy, x1- x0, 1 );
	    }

	if  ( ! cp->cpRightBorder.bpIsSet		&&
	      ( col == rowBi->biRowCellCount -1	||
	        ! cp[1].cpLeftBorder.bpIsSet	)	)
	    {
	    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
				x1- dc->dcOx, rowBi->biY0- dc->dcOy- 1,
				1, rowBi->biY1- rowBi->biY0+ 1 );
	    }

	x0= x1;
	}

    XSetForeground( add->addDisplay, dc->dcGc, dc->dcForePixel );
    ds->dsLinkPixelSet= 0;

    return;
    }

static int tedDrawRowItem(	Widget				w,
				const BufferItem *		rowBi,
				const BufferSelection *		bs,
				const AppDrawingData *		add,
				const DocumentRectangle *	drClip,
				DrawingState *			ds,
				const DrawingContext *		dc )
    {
    int			col;
    const int		x= add->addDocRect.drX0;
    int			x0;
    int			x1;
    CellProperties *	cp;
    int			bottom= -1;

    if  ( dc->dcDrawTableGrid )
	{ tedDrawRowGrid( rowBi, add, ds, dc ); }

    if  ( rowBi->biRowHeightTwips < 0 )
	{ bottom= rowBi->biY0+ rowBi->biRowHeightPixels;	}

    x0= x+ rowBi->biRowLeftIndentPixels;
    cp= rowBi->biRowCells;
    for ( col= 0; col < rowBi->biGroupChildCount; cp++, col++ )
	{
	int		par;
	BufferItem *	cellBi= rowBi->biGroupChildren[col];

	if  ( bs )
	    {
	    if  ( docCompareItemPositions( cellBi, bs->bsBegin.bpBi ) < 0 )
		{ continue;	}

	    if  ( docCompareItemPositions( cellBi, bs->bsEnd.bpBi ) > 0	)
		{ break;	}
	    }

	x1= x+ cp->cpRightBoundaryPixels;

	XSetForeground( add->addDisplay, dc->dcGc, dc->dcForePixel );
	ds->dsLinkPixelSet= 0;

	tedDrawCellBorders( x0, x1, rowBi->biY0, rowBi->biY1,
				cp, &(rowBi->biRowProperties), dc, drClip );

	if  ( ( ! bs || bs->bsCol0 < 0 || col >= bs->bsCol0 )	&&
	      ( ! bs || bs->bsCol1 < 0 || col <= bs->bsCol1 )	)
	    {
	    for ( par= 0; par < cellBi->biGroupChildCount; par++ )
		{
		BufferItem *	para= cellBi->biGroupChildren[par];

		if  ( tedDrawParaItem( w, para, bs, add, bottom,
							    drClip, ds, dc ) )
		    { LDEB(1); return -1;	}
		}
	    }

	x0= x1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw a single item. The Graphical context has been prepared by the	*/
/*  caller.								*/
/*									*/
/************************************************************************/

static int tedDrawItem(	Widget				w,
			const BufferItem *		bi,
			const BufferSelection *		bs,
			const AppDrawingData *		add,
			const DocumentRectangle *	drClip,
			DrawingState *			ds,
			const DrawingContext *		dc )
    {
    BufferItem *	child;
    int			i;

    if  ( bi->biY0 > drClip->drY1 )
	{ return 0;	}
    if  ( bi->biY1 < drClip->drY0 )
	{ return 0;	}

    if  ( bs )
	{
	if  ( docCompareItemPositions( bi, bs->bsBegin.bpBi ) < 0 )
	    { return 0;	}

	if  ( docCompareItemPositions( bi, bs->bsEnd.bpBi ) > 0	)
	    { return 0;	}
	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevCELL:
	rowAsGroup:

	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		child= bi->biGroupChildren[i];

		if  ( tedDrawItem( w, child, bs, add, drClip, ds, dc ) )
		    { LDEB(i); return -1;	}
		}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    if  ( tedDrawRowItem( w, bi, bs, add, drClip, ds, dc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( tedDrawParaItem( w, bi, bs, add, -1, drClip, ds, dc ) )
		{ LDEB(1); return -1;	}
	    break;
	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw an I Bar.							*/
/*									*/
/************************************************************************/
#   define	IW	5

int tedDrawIBar(	const BufferPosition *	bp,
			int			ox,
			int			oy,
			Display *		display,
			Window			win,
			GC			gc		)
    {
    XFillRectangle( display, win, gc,
				bp->bpX- ox, bp->bpY0- oy,
				1, bp->bpY1- bp->bpY0 );

    /*
    XDrawLine( display, win, gc,
		    bp->bpX- ox, bp->bpY0- oy, bp->bpX- ox, bp->bpY1- oy- 1 );
    */

    return 0;
    }

int tedExposeIBar(	const EditDocument *	ed,
			const BufferPosition *	bp )
    {
    DocumentRectangle	drIBar;

    drIBar.drX0= bp->bpX- IW;
    drIBar.drX1= bp->bpX+ IW;
    drIBar.drY0= bp->bpY0;
    drIBar.drY1= bp->bpY1;

    tedExposeRectangle( ed, &drIBar, /*scrolledX,Y*/ 0,0 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw the eight blocks around a selected object.			*/
/*									*/
/************************************************************************/

#   define	BLOCK	10

static void tedDrawObjectBlocks(	int			x0,
					int			y,
					int			wide,
					int			high,
					Pixel			borderPixel,
					Pixel			bodyPixel,
					const DrawingContext *	dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;

    XSetForeground( add->addDisplay, dc->dcGc, bodyPixel );

    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, y- BLOCK- dc->dcOy,
			BLOCK, BLOCK );
    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide/2- BLOCK/2- dc->dcOx, y- BLOCK- dc->dcOy,
			BLOCK, BLOCK );
    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide- BLOCK- dc->dcOx, y- BLOCK- dc->dcOy,
			BLOCK, BLOCK );

    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, y- high/2- BLOCK/2 -dc->dcOy,
			BLOCK, BLOCK );
    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide- BLOCK- dc->dcOx, y- high/2- BLOCK/2 -dc->dcOy,
			BLOCK, BLOCK );

    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, y- high- dc->dcOy,
			BLOCK, BLOCK );
    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide/2- BLOCK/2- dc->dcOx,
			y- high- dc->dcOy,
			BLOCK, BLOCK );
    XFillRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide- BLOCK- dc->dcOx,
			y- high- dc->dcOy,
			BLOCK, BLOCK );

    XSetForeground( add->addDisplay, dc->dcGc, borderPixel );

    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, y- BLOCK- dc->dcOy,
			BLOCK-1, BLOCK-1 );
    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide/2- BLOCK/2- dc->dcOx, y- BLOCK- dc->dcOy,
			BLOCK-1, BLOCK-1 );
    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide- BLOCK- dc->dcOx, y- BLOCK- dc->dcOy,
			BLOCK-1, BLOCK-1 );

    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, y- high/2- BLOCK/2 -dc->dcOy,
			BLOCK-1, BLOCK-1 );
    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide- BLOCK- dc->dcOx, y- high/2- BLOCK/2 -dc->dcOy,
			BLOCK-1, BLOCK-1 );

    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, y- high- dc->dcOy,
			BLOCK-1, BLOCK-1 );
    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide/2- BLOCK/2- dc->dcOx,
			y- high- dc->dcOy,
			BLOCK-1, BLOCK-1 );
    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0 +wide- BLOCK- dc->dcOx,
			y- high- dc->dcOy,
			BLOCK-1, BLOCK-1 );

    XDrawRectangle( add->addDisplay, dc->dcWindow, dc->dcGc,
			x0- dc->dcOx, y- high- dc->dcOy, wide- 1, high- 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Expose handler for documents.					*/
/*									*/
/*  2)  Clear background.						*/
/*  2b) If the selected area overlaps with the exposed region, draw the	*/
/*	selection background.						*/
/*  3)  Draw text.							*/
/*  4)	Draw page separators.						*/
/*  5)  Draw I bar if necessary.					*/
/*									*/
/************************************************************************/

void tedRedrawRectangle(	Widget			w,
				TedDocument *		td,
				DocumentRectangle *	drClip,
				AppDrawingData *	add,
				AppColors *		ac,
				int			ox,
				int			oy,
				Pixel			selPixel,
				Pixel			linkFore,
				Pixel			tableFrame,
				Window			win,
				GC			gc )
    {
    Display *			display= XtDisplay( w );
    int				screen= DefaultScreen( display );
    Pixel			whitePixel= WhitePixel( display, screen );
    Pixel			forePixel= BlackPixel( display, screen );

    BufferDocument *		bd= td->tdDocument;

    DrawingContext		dc;
    DrawingState		ds;

    const BufferSelection *	bs= &(td->tdSelection);
    const BufferItem *		selItem= (const BufferItem *)0;

    dc.dcDrawingData= add;
    dc.dcWindow= win;
    dc.dcGc= gc;

    dc.dcOx= ox;
    dc.dcOy= oy;

    dc.dcForePixel= forePixel;
    dc.dcLinkPixel= linkFore;
    dc.dcTablePixel= tableFrame;

    dc.dcDrawTableGrid= td->tdDrawTableGrid;

#   if LOG_REDRAWS
    docLogRectangle( "REDRAW", drClip );
#   endif

    /*  2  */
    XSetForeground( display, gc, whitePixel );
    XFillRectangle( display, win, gc,
		    drClip->drX0- ox, drClip->drY0-oy,
		    drClip->drX1- drClip->drX0, drClip->drY1- drClip->drY0+ 1 );

    /*  2a  */
    if  ( tedHasSelection( td ) && ! tedHasIBarSelection( td ) )
	{
	selItem= &(bd->bdItem);

	if  ( drClip->drY0 <= bs->bsEnd.bpY1	&&
	      drClip->drY1 >= bs->bsBegin.bpY0	)
	    {
	    if  ( bs->bsBegin.bpBi == bs->bsEnd.bpBi	&&
		  ! bs->bsBegin.bpBi->biParaInTable	)
		{ selItem= bs->bsBegin.bpBi;	}

	    if  ( bs->bsBegin.bpBi->biParaInTable		&&
		  docSelectionInsideRow( bs )			)
		{ selItem= bs->bsEnd.bpBi->biParent->biParent; }
	    }
	}

    if  ( selItem )
	{
	ds.dsLinkPixelSet= -1;
	dc.dcBackPixel= selPixel;
	dc.dcDrawTextLine= tedHighlightBack;
	dc.dcDrawTableBorders= 0;

	if  ( tedDrawItem( w, selItem, bs, add, drClip, &ds, &dc ) )
	    { LDEB(1);	}
	}

    /*  3  */
    dc.dcForePixel= forePixel;
    dc.dcBackPixel= whitePixel;
    dc.dcDrawTextLine= tedDrawTextLine;
    dc.dcDrawTableBorders= 1;

    XSetForeground( display, gc, dc.dcForePixel );
    XSetBackground( display, gc, whitePixel );

    ds.dsLinkPixelSet= -1;
    ds.dsPhysicalFont= -1;

    if  ( tedDrawItem( w, (&bd->bdItem), (BufferSelection *)0, add, drClip,
								&ds, &dc ) )
	{ LDEB(1);	}

    if  ( selItem && ac->acDepth < 4 )
	{
	ds.dsLinkPixelSet= -1;
	dc.dcForePixel= whitePixel;
	dc.dcLinkPixel= whitePixel;
	dc.dcDrawTextLine= tedDrawTextReverse;
	dc.dcDrawTableBorders= 0;

	if  ( tedDrawItem( w, selItem, bs, add, drClip, &ds, &dc ) )
	    { LDEB(1);	}
	}

    XSetForeground( display, gc, dc.dcForePixel );

    {
    TextLine *			tl;

    /*  5  */
    if  ( tedHasIBarSelection( td ) )
	{
	if  ( drClip->drY0 <= td->tdSelection.bsBegin.bpY1		&&
	      drClip->drY1 >= td->tdSelection.bsBegin.bpY0		&&
	      ! td->tdShowIBarId			)
	    {
	    tedDrawIBar( &(td->tdSelection.bsBegin), ox, oy, display, win, gc );
	    }
	}
    else{
	BufferItem *		bi;
	TextParticule *		tp;
	InsertedObject *	io;

	if  ( td->tdObjectSelected					&&
	      ! tedGetObjectSelection( td, &bi, &tl, &tp, &io  )	&&
	       drClip->drY0 <= tl->tlY1					&&
	       drClip->drY1 >= tl->tlY0					)
	    {
	    if  ( io->ioDragWide > 0 )
		{
		tedDrawObjectBlocks( tp->tpX0,
				    tl->tlY+ io->ioDragHigh- io->ioPixelsHigh,
				    io->ioDragWide, io->ioDragHigh,
				    dc.dcForePixel, selPixel, &dc );
		}
	    else{
		tedDrawObjectBlocks( tp->tpX0, tl->tlY,
				    io->ioPixelsWide, io->ioPixelsHigh,
				    dc.dcForePixel, selPixel, &dc );
		}
	    }
	}
    }
	  
    return;
    }

/************************************************************************/
/*									*/
/*  Cause a rectangle to be redrawn.					*/
/*									*/
/*  Cause the smallest rectangle that contains the selection to be	*/
/*  redrawn.								*/
/*									*/
/************************************************************************/

void tedExposeRectangle(	const EditDocument *		ed,
				const DocumentRectangle *	drChanged,
				int				scrolledX,
				int				scrolledY )
    {
    Widget			w= ed->edDocumentWidget;
    Display *			display= XtDisplay( w );
    Window			win= XtWindow( w );

    DocumentRectangle		drExpose;
    DocumentRectangle		drScrolled;

    if  ( ! drChanged )
	{ XClearArea( display, win, 0, 0, 0, 0, True ); return; }

    drScrolled= *drChanged;
    drScrolled.drX0 -= scrolledX;
    drScrolled.drX1 -= scrolledX;
    drScrolled.drY0 -= scrolledY;
    drScrolled.drY1 -= scrolledY;

    if  ( scrolledX != 0 )
	{
	drScrolled.drY0= ed->edVisibleRect.drY0;
	drScrolled.drY1= ed->edVisibleRect.drY1;
	}

    if  ( scrolledY != 0 )
	{
	drScrolled.drX0= ed->edVisibleRect.drX0;
	drScrolled.drX1= ed->edVisibleRect.drX1;
	}

    docUnionRectangle( &drScrolled, &drScrolled, drChanged );

    drExpose.drX0= ed->edVisibleRect.drX0;
    drExpose.drY0= ed->edVisibleRect.drY0;
    drExpose.drX1= ed->edVisibleRect.drX1;
    drExpose.drY1= ed->edVisibleRect.drY1;

    if  ( docIntersectRectangle( &drExpose, &drExpose, &drScrolled ) )
	{
	int	ox= ed->edVisibleRect.drX0;
	int	oy= ed->edVisibleRect.drY0;

#	if LOG_REDRAWS
	docLogRectangle( "CLEAR!", &drExpose );
#	endif

	XClearArea( display, win,
		drExpose.drX0- ox,
		drExpose.drY0- oy,
		drExpose.drX1- drExpose.drX0+ 1,
		drExpose.drY1- drExpose.drY0+ 1, True );
	}
    }

void tedExposeSelection(	const EditDocument *	ed,
				const BufferSelection *	bs,
				int			scrolledX,
				int			scrolledY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);
    DocumentRectangle		drClip;

    tedSelectionRectangle( &drClip, add, bs );

    tedExposeRectangle( ed, &drClip, scrolledX, scrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Blinking cursor.							*/
/*									*/
/*  1)  Turned off when we are debugging exposures and redraws.		*/
/*									*/
/************************************************************************/

void tedUndrawIBar(	const EditDocument *	ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

#   if LOG_REDRAWS
    /*  1  */
#   else
    tedExposeIBar( ed, &(td->tdSelection.bsBegin) );
#   endif

    return;
    }

void tedRedrawIBar(	TedDocument *	td,
			int		ox,
			int		oy,
			Display *	display,
			Window		win,
			GC		gc )
    {
    tedDrawIBar( &(td->tdSelection.bsBegin), ox, oy, display, win, gc );

    return;
    }

/************************************************************************/
/*									*/
/*  Create the windows that are used to reshape selected objects.	*/
/*									*/
/************************************************************************/

void tedSetObjectWindows(	TedDocument *		td,
				BufferItem *		bi,
				TextLine *		tl,
				TextParticule *		tp,
				int			ox,
				int			oy,
				Display *		display,
				Window			win )
    {
    InsertedObject *		io= bi->biParaObjects+ tp->tpObjectNumber;
    int				wide= io->ioPixelsWide;
    int				high= io->ioPixelsHigh;

    XSetWindowAttributes	xswa;

    static Cursor		moveCursor;
    static Cursor		bottomCursor;
    static Cursor		rightCursor;
    static Cursor		cornerCursor;

    int				x0= tp->tpX0;
    int				y= tl->tlY;

    if  ( ! moveCursor )
	{
	moveCursor= XCreateFontCursor( display, XC_fleur );
	bottomCursor= XCreateFontCursor( display, XC_bottom_side );
	rightCursor= XCreateFontCursor( display, XC_right_side );
	cornerCursor= XCreateFontCursor( display, XC_bottom_right_corner );
	}

    if  ( ! td->tdObjectWindow )
	{
	xswa.cursor= moveCursor;
	td->tdObjectWindow= XCreateWindow( display, win,
			    x0- ox, y- high- oy,
			    wide, high,
			    0, CopyFromParent, InputOnly, CopyFromParent,
			    0L, &xswa );
			    /*
			    CWCursor, &xswa );
			    */

	xswa.cursor= bottomCursor;
	td->tdObjectBottomWindow= XCreateWindow( display, td->tdObjectWindow,
			    wide/ 2- BLOCK/ 2, high- BLOCK,
			    BLOCK, BLOCK,
			    0, CopyFromParent, CopyFromParent, CopyFromParent,
			    CWCursor, &xswa );

	xswa.cursor= rightCursor;
	td->tdObjectRightWindow= XCreateWindow( display, td->tdObjectWindow,
			    wide- BLOCK, high/ 2- BLOCK/ 2,
			    BLOCK, BLOCK,
			    0, CopyFromParent, CopyFromParent, CopyFromParent,
			    CWCursor, &xswa );

	xswa.cursor= cornerCursor;
	td->tdObjectCornerWindow= XCreateWindow( display, td->tdObjectWindow,
			    wide- BLOCK, high- BLOCK,
			    BLOCK, BLOCK,
			    0, CopyFromParent, CopyFromParent, CopyFromParent,
			    CWCursor, &xswa );

	XMapRaised( display, td->tdObjectBottomWindow );
	XMapRaised( display, td->tdObjectRightWindow );
	XMapRaised( display, td->tdObjectCornerWindow );
	}
    else{
	XMoveResizeWindow( display, td->tdObjectWindow,
				x0- ox, y- high- oy, wide, high );

	XMoveWindow( display, td->tdObjectBottomWindow,
				wide/ 2- BLOCK/ 2, high- BLOCK );
	XMoveWindow( display, td->tdObjectRightWindow,
				wide- BLOCK, high/ 2- BLOCK/ 2 );
	XMoveWindow( display, td->tdObjectCornerWindow,
				wide- BLOCK, high- BLOCK );
	}

    XMapRaised( display, td->tdObjectWindow );

    td->tdObjectSelected= 1;
    }
