/************************************************************************/
/*  Ted: Interface to the find tool.					*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>

#   include	<Xm/DialogS.h>
#   include	<X11/Xatom.h>
#   include	<X11/IntrinsicP.h>
#   include	<Xm/Form.h>
#   include	<Xm/Text.h>
#   include	<Xm/PanedWP.h>
#   include	<Xm/Label.h>
#   include	<Xm/List.h>

#   include	<Xm/Protocols.h>

#   include	<appUtil.h>

#   include	"tedApp.h"

#   include	<debugon.h>

/************************************************************************/
/*  User data for a link dialog.					*/
/************************************************************************/

typedef struct BookmarkContext
    {
    Widget		bcMarkTextWidget;
    Widget		bcMarkListWidget;

    Widget		bcSetBookmarkButton;
    Widget		bcGoToBookmarkButton;
    Widget		bcCopyBookmarkButton;
    Widget		bcRemoveBookmarkButton;
    Widget		bcCancelButton;

    char *		bcMarkText;
    char *		bcSetBookmarkText;
    char *		bcGoToBookmarkText;
    char *		bcCopyBookmarkText;
    char *		bcRemoveBookmarkText;
    char *		bcCancelText;

    char *		bcSavedMark;
	
    int			bcResponse;

    EditDocument *	bcDocument;
    } BookmarkContext;

/************************************************************************/
/*  Resource table for various texts.					*/
/************************************************************************/
# define xx(x)	x,x

static XtResource TED_BookmarkResourceTable[]=
    {
	{ xx("bookmarkBookmark"), XtRString, sizeof(char *),
		    offsetof(BookmarkContext,bcMarkText),
		    XtRString, "Bookmark" },

	{ xx("bookmarkSetBookmark"), XtRString, sizeof(char *),
		    offsetof(BookmarkContext,bcSetBookmarkText),
		    XtRString, "Set Bookmark" },
	{ xx("bookmarkGoToBookmark"), XtRString, sizeof(char *),
		    offsetof(BookmarkContext,bcGoToBookmarkText),
		    XtRString, "Go To Bookmark" },
	{ xx("bookmarkCopyBookmark"), XtRString, sizeof(char *),
		    offsetof(BookmarkContext,bcCopyBookmarkText),
		    XtRString, "Copy as Link" },
	{ xx("bookmarkRemoveBookmark"), XtRString, sizeof(char *),
		    offsetof(BookmarkContext,bcRemoveBookmarkText),
		    XtRString, "Remove Bookmark" },
	{ xx("bookmarkCancel"), XtRString, sizeof(char *),
		    offsetof(BookmarkContext,bcCancelText),
		    XtRString, "Cancel" },
    };

/************************************************************************/
/*									*/
/*  Generalised button callback.					*/
/*									*/
/*  NOTE that the NO,YES responses are abused for 'Remove,Set Bookmark'.	*/
/*									*/
/************************************************************************/
static void tedBookmarkDialogClosed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    BookmarkContext *		bc= (BookmarkContext *)voidlc;

    bc->bcResponse= AQDrespCANCEL; return;
    }

static void tedBookmarkCancelPushed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    BookmarkContext *		bc= (BookmarkContext *)voidlc;

    bc->bcResponse= AQDrespCANCEL; return;
    }

static void tedBookmarkSetPushed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    BookmarkContext *		bc= (BookmarkContext *)voidlc;
    char *			mark;

    mark= XmTextGetString( bc->bcMarkTextWidget );
    if  ( ! mark )
	{ XDEB(mark); bc->bcResponse= AQDrespFAILURE; return;	}

    if  ( tedSetBookmark( bc->bcDocument, mark ) )
	{ SDEB(mark); bc->bcResponse= AQDrespFAILURE; return;	}

    XtFree( mark );

    bc->bcResponse= AQDrespOK; return;
    }

static void tedBookmarkGotoPushed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    BookmarkContext *		bc= (BookmarkContext *)voidlc;
    char *			markName;

    markName= XmTextGetString( bc->bcMarkTextWidget );
    if  ( ! markName )
	{ XDEB(markName); bc->bcResponse= AQDrespFAILURE; return;	}

    if  ( markName[0] )
	{ tedGoToBookmark( bc->bcDocument, markName, strlen( markName ) ); }

    XtFree( markName );

    bc->bcResponse= AQDrespOK; return;
    }

static void tedBookmarkRemovePushed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	call_data )
    {
    BookmarkContext *		bc= (BookmarkContext *)voidlc;

    if  ( tedRemoveBookmark( bc->bcDocument ) )
	{ LDEB(1);	}

    bc->bcResponse= AQDrespOK; return;
    }

static void tedBookmarkCopyPushed(	Widget		button,
					XtPointer	voidlc,
					XtPointer	voidpbcs )
    {
    BookmarkContext *			bc= (BookmarkContext *)voidlc;
    char *				mark;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcs;

    mark= XmTextGetString( bc->bcMarkTextWidget );
    if  ( ! mark )
	{ XDEB(mark); bc->bcResponse= AQDrespFAILURE; return;	}

    if  ( mark[0] )
	{ tedCopyBookmarkAsLink( bc->bcDocument, mark, pbcs->event );	}

    XtFree( mark );

    bc->bcResponse= AQDrespOK; return;
    }

/************************************************************************/
/*  Make the form with buttons.						*/
/************************************************************************/

static Widget tedBookmarkMakeButtonForm(	BookmarkContext *	bc,
						Widget			parent,
						Widget			dialog )
    {
    Widget		buttonRow;

    buttonRow= appMakeButtonRow( parent, 5 );

    bc->bcSetBookmarkButton= appMakeRowButton( buttonRow,
			    bc->bcSetBookmarkText, tedBookmarkSetPushed,
			    (void *)bc, 0, True );

    bc->bcGoToBookmarkButton= appMakeRowButton( buttonRow,
			    bc->bcGoToBookmarkText, tedBookmarkGotoPushed,
			    (void *)bc, 1, False );

    bc->bcCopyBookmarkButton= appMakeRowButton( buttonRow,
			    bc->bcCopyBookmarkText, tedBookmarkCopyPushed,
			    (void *)bc, 2, False );

    bc->bcRemoveBookmarkButton= appMakeRowButton( buttonRow,
			    bc->bcRemoveBookmarkText, tedBookmarkRemovePushed,
			    (void *)bc, 3, False );

    bc->bcCancelButton= appMakeRowButton( buttonRow,
			    bc->bcCancelText, tedBookmarkCancelPushed,
			    (void *)bc, 4, False );

    XtVaSetValues( dialog,
			XmNdefaultButton,	bc->bcSetBookmarkButton,
			NULL );
    XtVaSetValues( dialog,
			XmNcancelButton,	bc->bcCancelButton,
			NULL );

    XtManageChild( buttonRow );

    return buttonRow;
    }

/************************************************************************/
/*  Make the framework of a Question Dialog.				*/
/************************************************************************/
static void tedFreeBookmarkContext(	Widget			dialog,
				XtPointer		voidlc,
				XtPointer		callData )
    { free( voidlc ); return;	}


static Widget tedMakeBookmarkDialog(	EditApplication *	ea,
					Widget			relative,
					Widget *		pPaned )
    {
    Widget		shell;
    Widget		dialog;
    Widget		paned;
    const int		withSeparator= 0;

    BookmarkContext *	bc= (BookmarkContext *)malloc( sizeof(BookmarkContext) );

    XtGetApplicationResources( ea->eaTopWidget, (void *)bc,
		TED_BookmarkResourceTable, XtNumber(TED_BookmarkResourceTable),
		NULL, 0 );

    bc->bcMarkTextWidget= (Widget)0;
    bc->bcMarkListWidget= (Widget)0;

    bc->bcSetBookmarkButton= (Widget)0;
    bc->bcGoToBookmarkButton= (Widget)0;
    bc->bcRemoveBookmarkButton= (Widget)0;
    bc->bcCopyBookmarkButton= (Widget)0;
    bc->bcCancelButton= (Widget)0;

    bc->bcSavedMark= (char *)0;

    appMakeVerticalDialog( &shell, &dialog,
			    &paned, ea, tedBookmarkDialogClosed, (void *)bc,
			    (Pixmap)0, withSeparator, "tedBookmark" );

    XtAddCallback( dialog, XmNdestroyCallback,
				    tedFreeBookmarkContext, (void *)bc );

    *pPaned= paned;

    return dialog;
    }

/************************************************************************/
/*									*/
/*  Make the 'Destination' half of the link dialog.			*/
/*									*/
/************************************************************************/

static void tedBookmarkDestinationChanged(	Widget		w,
						XtPointer	voidbc,
						XtPointer	call_data )
    {
    BookmarkContext *		bc= (BookmarkContext *)voidbc;

    char *			text;
    int				len;

    XtSetSensitive( bc->bcRemoveBookmarkButton, False );

    text= XmTextGetString( w );
    len= strlen( text );

    if  ( len > 0 )
	{
	EditDocument *		ed= bc->bcDocument;
	TedDocument *		td= (TedDocument *)ed->edPrivateData;
	BufferDocument *	bd= td->tdDocument;
	DocumentFieldList *	dfl= &(bd->bdBookmarks);

	if  ( ! docFindBookmarkField( dfl, (unsigned char *)text, len ) )
	    {
	    XtSetSensitive( bc->bcGoToBookmarkButton, False );
	    XtSetSensitive( bc->bcCopyBookmarkButton, False );
	    XtSetSensitive( bc->bcSetBookmarkButton, ! ed->edIsReadonly );
	    }
	else{
	    XtSetSensitive( bc->bcGoToBookmarkButton, True );
	    XtSetSensitive( bc->bcCopyBookmarkButton, True );
	    XtSetSensitive( bc->bcSetBookmarkButton, False );
	    }
	}
    else{
	XtSetSensitive( bc->bcGoToBookmarkButton, False );
	XtSetSensitive( bc->bcCopyBookmarkButton, False );
	XtSetSensitive( bc->bcSetBookmarkButton, False );
	}

    XtFree( text );

    return;
    }

static void tedBookmarkMakeDestination(	Widget *		pTextWidget,
					BookmarkContext *	bc,
					const char *		labelText,
					Widget			paned )
    {
    Widget		label;
    Widget		text;

    appMakeColumnLabel( &label, paned, labelText );
    appMakeColumnText( &text, paned, 0, True );

    XtAddCallback( text, XmNvalueChangedCallback,
				    tedBookmarkDestinationChanged, (void *)bc );

    *pTextWidget= text; return;
    }

static void tedBookmarkChosen(		Widget			w,
					XtPointer		voidbc,
					XtPointer		voidlcs )
    {
    XmListCallbackStruct *	lcs= (XmListCallbackStruct *)voidlcs;
    BookmarkContext *		bc= (BookmarkContext *)voidbc;
    EditDocument *		ed= bc->bcDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    DocumentFieldList *		dfl= &(bd->bdBookmarks);

    DocumentField *		df;

    if  ( lcs->item_position > dfl->dflFieldCount )
	{ LLDEB(lcs->item_position,dfl->dflFieldCount); return;	}

    df= dfl->dflFields+ lcs->item_position- 1;
    df= dfl->dflFields+ df->dfNumberInDocument;

    XmTextSetString( bc->bcMarkTextWidget,
				    (char *)df->dfInstructions.odBytes );

    XtSetSensitive( bc->bcGoToBookmarkButton, True );
    XtSetSensitive( bc->bcCopyBookmarkButton, True );
    XtSetSensitive( bc->bcSetBookmarkButton, False );
    XtSetSensitive( bc->bcRemoveBookmarkButton, False );

    return;
    }

static void tedBookmarkMakeList(	Widget *		pListWidget,
					BookmarkContext *	bc,
					Widget			paned )
    {
    Arg			al[20];
    int			ac= 0;

    Widget		list;

    ac= 0;
    XtSetArg( al[ac],	XmNallowResize,		True ); ac++;
    XtSetArg( al[ac],	XmNvisibleItemCount,	12 ); ac++;
    list= XmCreateScrolledList( paned, WIDGET_NAME, al, ac );

    XtAddCallback( list,
		XmNbrowseSelectionCallback, tedBookmarkChosen, (void *)bc );

    XtManageChild( list );

    *pListWidget= list; return;
    }

/************************************************************************/
/*									*/
/*  Run the 'Bookmark Dialog' for a certain document.			*/
/*									*/
/*  8)  Keep an indirection from the position in the list to the number	*/
/*	in the document, to make it easy to find the item in the	*/
/*	list callback. Abuse the dfNumberInDocument fields for this.	*/
/*									*/
/************************************************************************/
void tedRunBookmarkDialog(	EditApplication *	ea,
				EditDocument *		ed,
				Widget			option,
				int			nameProposed,
				const char *		markName,
				int			markSize )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    DocumentFieldList *		dfl= &(bd->bdBookmarks);

    BookmarkContext *		bc= (BookmarkContext *)0;

    Widget			relative= ed->edTopWidget;

    static Widget		linkDialog;

    if  ( ! linkDialog )
	{
	Widget				paned;
	Widget				bottomForm;

	linkDialog= tedMakeBookmarkDialog( ea, relative, &paned );

	XtVaGetValues( linkDialog,
			XmNuserData,	&bc,
			NULL );

	tedBookmarkMakeDestination( &(bc->bcMarkTextWidget),
						bc, bc->bcMarkText, paned );

	tedBookmarkMakeList( &(bc->bcMarkListWidget), bc, paned );

	bottomForm= tedBookmarkMakeButtonForm( bc, paned, linkDialog );

	XtManageChild( paned );
	XtManageChild( linkDialog );
	}
    else{
	XtVaGetValues( linkDialog,
			XmNuserData,	&bc,
			NULL );
	}

    if  ( relative )
	{
	appDialogRelative( relative, linkDialog );
	appSetRelativeCallback( relative, linkDialog );
	}

    appSetFocusCallback( linkDialog );

    bc->bcDocument= ed;

    if  ( bc->bcSavedMark )
	{ free( bc->bcSavedMark ); bc->bcSavedMark= (char *)0; }

    if  ( markSize > 0 )
	{
	char *			saved;

	saved= (char *)malloc( markSize+ 1 );

	if  ( ! saved )
	    { XDEB(saved);	}
	else{
	    strncpy( saved, markName, markSize )[markSize]= '\0';
	    XmTextSetString( bc->bcMarkTextWidget, saved );
	    bc->bcSavedMark= saved;
	    }

	XtSetSensitive( bc->bcRemoveBookmarkButton,
				    ! ed->edIsReadonly && ! nameProposed );

	if  ( ! docFindBookmarkField( dfl,
					(unsigned char *)markName, markSize ) )
	    {
	    XtSetSensitive( bc->bcGoToBookmarkButton, False );
	    XtSetSensitive( bc->bcCopyBookmarkButton, False );
	    }
	else{
	    XtSetSensitive( bc->bcGoToBookmarkButton, True );
	    XtSetSensitive( bc->bcCopyBookmarkButton, True );
	    }
	}
    else{
	XtSetSensitive( bc->bcRemoveBookmarkButton, False );
	XtSetSensitive( bc->bcGoToBookmarkButton, False );
	XtSetSensitive( bc->bcCopyBookmarkButton, False );

	XmTextSetString( bc->bcMarkTextWidget, "" );
	}

    if  ( ! nameProposed )
	{ XtSetSensitive( bc->bcSetBookmarkButton, False );	}

    {
    DocumentField *	df;
    int			i;
    int			pos= 0;

    XmListDeselectAllItems( bc->bcMarkListWidget );
    XmListDeleteAllItems( bc->bcMarkListWidget );

    df= dfl->dflFields;
    for ( i= 0; i < dfl->dflFieldCount; df++, i++ )
	{
	XmString		labelString;

	if  ( df->dfKind == DOCfkFREE			||
	      df->dfInstructions.odBytes[0] == '_'	)
	    { continue;		}

	labelString= XmStringCreateLocalized(
					(char *)df->dfInstructions.odBytes );

	XmListAddItemUnselected( bc->bcMarkListWidget,
						    labelString, pos+ 1 );
	XmStringFree( labelString );

	/*  8  */
	dfl->dflFields[pos++].dfNumberInDocument= i;
	}
    }

    XtManageChild( linkDialog );

    appSetShellTitle( XtParent( linkDialog ), option, ea->eaApplicationName );

    XmProcessTraversal( bc->bcMarkTextWidget, XmTRAVERSE_CURRENT );

    bc->bcResponse= AQDrespNONE;
    while( bc->bcResponse == AQDrespNONE )
	{ XtAppProcessEvent( ea->eaContext, XtIMAll ); }

    XtUnmanageChild( linkDialog );

    return;
    }
