(**
   A tree object displaying simple tree-like data.
**)

MODULE VOTree;

(*
    A tree object displaying simple tree-like data.
    Copyright (C) 1998  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       E   := VOEvent,
       F   := VOFrame,
       G   := VOGUIObject,
       O   := VOObject,
       S   := VOScroller,
       TM  := VOTreeModel,
       U   := VOUtil,
       V   := VOValue,

       str := Strings;

CONST
  selectedMsg* = 0;

TYPE
  Tree*     = POINTER TO TreeDesc;
  TreeDesc* = RECORD (G.GadgetDesc)
                font         : D.Font;

                frame,
                focFrame     : F.Frame;

                left,top,
                hTotal,
                vTotal,
                hVisible,
                vVisible     : V.ValueModel;

                hScroll,
                vScroll      : S.Scroller;

                model-       : TM.TreeModel;
                selected-    : TM.TreeItem;
                selectedPos  : LONGINT;

                iX,iY,
                iWidth,
                iHeight      : LONGINT;

                colHeight    : LONGINT;
               END;

  SelectedMsg*      = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc*  = RECORD (O.MessageDesc)
                        item* : TM.TreeItem;
                      END;

  PROCEDURE (t : Tree) Init*;

  BEGIN
    t.Init^;

    INCL(t.flags,G.canFocus); (* We can show a focus frame *)
    EXCL(t.flags,G.stdFocus); (* we do the displaying of the focus frame ourself *)

    t.SetBackground(D.tableBackgroundColor);

    t.frame:=NIL;
    t.focFrame:=NIL;

    NEW(t.left);
    t.left.Init;
    t.AttachModel(t.left);

    NEW(t.top);
    t.top.Init;
    t.AttachModel(t.top);

    NEW(t.hTotal);
    t.hTotal.Init;
    NEW(t.vTotal);
    t.vTotal.Init;

    NEW(t.hVisible);
    t.hVisible.Init;
    NEW(t.vVisible);
    t.vVisible.Init;

    t.left.SetLongint(1);
    t.top.SetLongint(1);

    t.hVisible.SetLongint(0);
    t.vVisible.SetLongint(0);

    t.hTotal.SetLongint(0);
    t.vTotal.SetLongint(0);

    NEW(t.hScroll);
    t.hScroll.Init;
    t.hScroll.SetFlags({G.horizontalFlex});
    t.hScroll.Set(FALSE);
    t.hScroll.SetModel(t.left,t.hVisible,t.hTotal);

    NEW(t.vScroll);
    t.vScroll.Init;
    t.vScroll.SetFlags({G.verticalFlex});
    t.vScroll.Set(TRUE);
    t.vScroll.SetModel(t.top,t.vVisible,t.vTotal);

    t.model:=NIL;
    t.selected:=NIL;
    t.selectedPos:=-1;
  END Init;

  PROCEDURE (t : Tree) CalcSize*(display : D.Display);

  BEGIN
    t.font:=display.GetFont(D.normalFont);

    NEW(t.frame);
    t.frame.Init;
    t.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    t.frame.SetInternalFrame(F.double3DIn);
    t.frame.CalcSize(display);

    NEW(t.focFrame);
    t.focFrame.Init;
    t.focFrame.SetFlags({G.horizontalFlex,G.verticalFlex});
    t.focFrame.SetInternalFrame(F.dottedFocus);
    t.focFrame.CalcSize(display);

    t.width:=t.frame.leftBorder+10*display.spaceWidth+t.frame.rightBorder;
    t.height:=t.frame.topBorder+10*display.spaceHeight+t.frame.bottomBorder;

    t.hScroll.CalcSize(display);
    t.vScroll.CalcSize(display);

    INC(t.width,t.vScroll.width);
    INC(t.height,t.hScroll.height);

    t.minWidth:=t.width;
    t.minHeight:=t.height;

    t.colHeight:=t.font.height+display.spaceHeight DIV 2;

    t.CalcSize^(display);
  END CalcSize;

  PROCEDURE (t : Tree) GetClickedEntryPos(y : LONGINT):LONGINT;

  VAR
    pos : LONGINT;

  BEGIN
    IF t.model=NIL THEN
      RETURN -1;
    END;

    pos:=(y-t.iY) DIV t.colHeight;
    IF (y-t.iY) MOD t.colHeight>0 THEN
      INC(pos);
    END;

    INC(pos,t.top.GetLongint()-1);

    IF (pos>=1) & (pos<=t.model.visible) THEN
      RETURN pos;
    ELSE
      RETURN -1;
    END;
  END GetClickedEntryPos;

  PROCEDURE (t : Tree) MakeVisible(y : LONGINT);

  BEGIN
    IF y<t.top.GetLongint() THEN
      t.top.SetLongint(y);
    ELSIF y>t.top.GetLongint()+t.vVisible.GetLongint()-1 THEN
      t.top.SetLongint(G.MaxLong(1,y-t.vVisible.GetLongint()));
    END;
  END MakeVisible;

  PROCEDURE (t : Tree) DrawItem(item : TM.TreeItem; VAR y : LONGINT; offset : LONGINT; goUp : BOOLEAN);

  VAR
    text   : U.Text;
    extent : D.FontExtentDesc;
    x,yPos,
    byPos  : LONGINT;

  BEGIN
    WHILE (item#NIL) & (y<=t.iHeight) DO

      text:=item.GetText();
      IF text#NIL THEN
        t.font.TextExtent(text^,str.Length(text^),{},extent);
        x:=t.iX+offset*3*t.display.spaceWidth;
        yPos:=t.iY+y;
        byPos:=yPos+(t.colHeight-t.font.height) DIV 2;
        IF item.HasChilds() THEN
          t.draw.PushForeground(D.tableTextColor);
          t.draw.DrawLine(x,byPos,x+2*t.display.spaceWidth,byPos);
          t.draw.DrawLine(x+2*t.display.spaceWidth,byPos,
                          x+2*t.display.spaceWidth,byPos+t.font.height-1);
          t.draw.DrawLine(x+2*t.display.spaceWidth,byPos+t.font.height-1,
                          x,byPos+t.font.height-1);
          t.draw.DrawLine(x,byPos+t.font.height-1,x,byPos);

          t.draw.DrawLine(x+t.display.spaceWidth DIV 2,byPos+t.font.height DIV 2,
                          x+2*t.display.spaceWidth-t.display.spaceWidth DIV 2,
                          byPos+t.font.height DIV 2);
          IF ~item.shown THEN
            t.draw.DrawLine(x+t.display.spaceWidth,byPos+t.font.height DIV 4,
                            x+t.display.spaceWidth,byPos+t.font.height-t.font.height DIV 4-1);
          END;

          t.draw.PopForeground;
        END;
        IF item=t.selected THEN
          t.selectedPos:=t.top.GetLongint()+y DIV t.colHeight;
          t.draw.PushForeground(D.fillColor);
          t.draw.FillRectangle(x+3*t.display.spaceWidth,
                               yPos+(t.colHeight-t.font.height) DIV 2,
                               extent.width,t.font.height);
          t.draw.PopForeground;
          t.draw.PushForeground(D.fillTextColor);
          IF t.HasFocus() THEN
            t.focFrame.Resize(extent.width,t.font.height);
            t.focFrame.Draw(x+3*t.display.spaceWidth,
                            yPos+(t.colHeight-t.font.height) DIV 2,
                            t.draw);
          END;
        ELSE
          t.draw.PushForeground(D.tableTextColor);
        END;
        t.draw.PushFont(D.normalFont,{});
        t.draw.DrawString(x-extent.lbearing+3*t.display.spaceWidth,
                          yPos+t.font.ascent+(t.colHeight-t.font.height) DIV 2,
                          text^,str.Length(text^));
        t.draw.PopFont;
        t.draw.PopForeground;
      END;

      IF item.HasChilds() & item.shown THEN
        INC(y,t.colHeight);
        t.DrawItem(item.firstChild,y,offset+1,FALSE);
      ELSE
        INC(y,t.colHeight);
      END;

      IF goUp THEN
        WHILE (item#NIL) & (item.next=NIL) DO
          item:=item.parent;
          DEC(offset);
        END;
        IF item#NIL THEN
          item:=item.next;
        END;
      ELSE
        item:=item.next;
      END;
    END;
  END DrawItem;

  PROCEDURE (t : Tree) DrawTree;

  VAR
    yPos,
    offset : LONGINT;
    top    : TM.TreeItem;

  BEGIN
    t.selectedPos:=-1;

    t.draw.PushForeground(t.background);
    t.draw.FillRectangle(t.iX,t.iY,t.iWidth,t.iHeight);
    t.draw.PopForeground;

    IF (t.model#NIL) & (t.model.top#NIL) THEN
      t.draw.InstallClip;
      t.draw.AddRegion(t.iX,t.iY,t.iWidth,t.iHeight);
      yPos:=0;
      top:=t.model.GetVisibleItem(t.top.GetLongint(),offset);
      t.DrawItem(top,yPos,offset,TRUE);
      t.draw.FreeLastClip;

      t.vTotal.SetLongint(t.model.visible);
      IF t.top.GetLongint()+t.iHeight DIV t.colHeight-1>t.model.visible THEN
        t.vVisible.SetLongint(t.model.visible-t.top.GetLongint()+1);
      ELSE
        t.vVisible.SetLongint(G.MinLong(t.iHeight DIV t.colHeight,t.model.visible));
      END;
    ELSE
      t.vTotal.SetLongint(0);
      t.vVisible.SetLongint(0);
    END;
  END DrawTree;

  PROCEDURE (t : Tree) HandleClick(event : E.MouseEvent);

  VAR
    pos,offset : LONGINT;
    item       : TM.TreeItem;
    selected   : SelectedMsg;

  BEGIN
    pos:=t.GetClickedEntryPos(event.y);
    IF pos>0 THEN
      item:=t.model.GetVisibleItem(pos,offset);

      IF (event.x>=t.iX+offset*3*t.display.spaceWidth)
       & (event.x<=t.iX+offset*3*t.display.spaceWidth+2*t.display.spaceWidth) THEN
        IF item.childs>0 THEN
          IF item.shown THEN
            item.HideChilds;
          ELSE
            item.ShowChilds;
          END;
        END;
      ELSIF (event.x>t.iX+offset*3*t.display.spaceWidth+2*t.display.spaceWidth)
          & (event.x<t.iX+t.iWidth) THEN
        IF item#t.selected THEN
          t.selected:=item;
          NEW(selected);
          selected.item:=t.selected;
          t.Send(selected,selectedMsg);

          t.DrawTree;
        END;
      END;
    END;
  END HandleClick;

  PROCEDURE (t : Tree) GetFocus*(event : E.Event):G.Object;

  VAR
    object : G.Object;

  BEGIN
    IF ~t.visible OR t.disabled OR (t.model=NIL) THEN
      RETURN NIL;
    END;

    object:=t.hScroll.GetFocus(event);
    IF object#NIL THEN
      RETURN object;
    END;

    object:=t.vScroll.GetFocus(event);
    IF object#NIL THEN
      RETURN object;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & t.PointIsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        t.HandleClick(event);
        RETURN t;
      END;
    ELSE
    END;

    RETURN NIL;
  END GetFocus;

  PROCEDURE (t : Tree) HandleKeys(event :  E.KeyEvent):BOOLEAN;

  VAR
    keysym   : LONGINT;
    item     : TM.TreeItem;
    oldPos   : LONGINT;
    selected : SelectedMsg;

  BEGIN
    keysym:=event.GetKey();
    CASE keysym OF
      E.left:
        IF t.selected#NIL THEN
          IF t.selected.HasChilds() & t.selected.shown THEN
            t.selected.HideChilds;
          END;
        END;
    | E.right:
        IF t.selected#NIL THEN
          IF t.selected.HasChilds() & ~t.selected.shown THEN
            t.selected.ShowChilds;
          END;
        END;
    | E.up:
        IF t.selected#NIL THEN
          oldPos:=t.selectedPos;
          item:=t.selected.Last();
          IF (item#NIL) & (item#t.selected) THEN
            t.selected:=item;
            NEW(selected);
            selected.item:=t.selected;
            t.Send(selected,selectedMsg);

            t.DrawTree;
            IF oldPos>0 THEN
              t.MakeVisible(oldPos-1);
            END;
          END;
        END;
    | E.down:
        IF t.selected#NIL THEN
          oldPos:=t.selectedPos;
          item:=t.selected.Next();
          IF (item#NIL) & (item#t.selected) THEN
            t.selected:=item;
            NEW(selected);
            selected.item:=t.selected;
            t.Send(selected,selectedMsg);

            t.DrawTree;
            IF oldPos>0 THEN
              t.MakeVisible(oldPos+1);
            END;
          END;
        END;
    | E.home:
    | E.end:
    ELSE
    END;
    RETURN FALSE;
  END HandleKeys;

  PROCEDURE (t : Tree) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    WITH
      event : E.MouseEvent DO
        IF (event.button=E.button1) & (event.type=E.mouseUp) THEN
          RETURN TRUE;
        END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (t : Tree) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  BEGIN
    IF event.type=E.keyDown THEN
      RETURN t.HandleKeys(event);
    ELSE
      RETURN FALSE;
    END;
  END HandleFocusEvent;

  PROCEDURE (t : Tree) ReInit;

  BEGIN
    t.left.SetLongint(1);
    t.top.SetLongint(1);

    IF t.visible THEN
      t.DrawTree;
    END;
  END ReInit;

  PROCEDURE (t : Tree) SetModel*(model : TM.TreeModel);

  BEGIN
    IF t.model#NIL THEN
      t.UnattachModel(t.model);
    END;

    t.selected:=NIL;

    t.model:=model;

    IF t.model#NIL THEN
      t.AttachModel(model);
      t.ReInit;
    END;
  END SetModel;

  PROCEDURE (t : Tree) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    t.Draw^(x,y,draw);

    t.frame.Resize(t.width,t.height);
    t.frame.Draw(t.x,t.y,draw);

    t.hScroll.Resize(t.width-t.frame.leftBorder-t.frame.rightBorder-t.vScroll.width,-1);
    t.vScroll.Resize(-1,t.height-t.frame.topBorder-t.frame.bottomBorder-t.hScroll.height);

    t.hScroll.Draw(t.x+t.frame.leftBorder,
                   t.y+t.height-t.frame.bottomBorder-t.hScroll.oHeight,draw);
    t.vScroll.Draw(t.x+t.width-t.frame.rightBorder-t.vScroll.oWidth,
                   t.y+t.frame.topBorder,draw);

    t.iX:=t.x+t.frame.leftBorder;
    t.iY:=t.y+t.frame.topBorder;
    t.iWidth:=t.width-t.frame.leftBorder-t.vScroll.oWidth-t.frame.rightBorder;
    t.iHeight:=t.height-t.frame.topBorder-t.hScroll.oHeight-t.frame.bottomBorder;

    t.DrawTree;
  END Draw;

  PROCEDURE (t : Tree) Hide*;

  BEGIN
    IF t.visible THEN
      t.hScroll.Hide;
      t.vScroll.Hide;
      t.DrawHide;
      t.Hide^;
    END;
  END Hide;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (t : Tree) DrawFocus*;

  BEGIN
    t.DrawTree;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (t : Tree) HideFocus*;

  BEGIN
    t.DrawTree;
  END HideFocus;

  PROCEDURE (t : Tree) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF model=t.model THEN
      IF msg#NIL THEN
        WITH msg : TM.ItemSwitchedMsg DO
          IF t.selected#NIL THEN
            t.selected:=t.selected.VisibleParent();
          END;
          IF t.top.GetLongint()>t.model.visible THEN
            t.top.SetLongint(t.model.visible);
          ELSE
            t.DrawTree;
          END;
        ELSE
          t.ReInit;
        END;
      ELSE
        t.ReInit;
      END;
    ELSIF t.visible THEN
      t.DrawTree;
    END;
  END Resync;

END VOTree.
