/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMBuiltinMethod.m,v 1.30 1998/01/05 01:12:20 tiggr Exp $  */

#define OTMBUILTINMETHOD_DECLARE_PRIVATE_METHODS

#import "OTMBuiltinMethod.h"
#import "OTMAnyRefType.h"
#import "OTMBasic.h"
#import "OTMMeta.h"
#import "OTMClass.h"
#import "OTMVariable.h"
#import "OTMExpr.h"
#import "OTMExtension.h"
#import "OTMInvocation.h"
#import "OTMTuple.h"
#import "OTMTypeTuple.h"

/* These constant strings are replaced by their unique conterparts (in
   `+init').  */
id <TLString> builtin_operator_name[] =
{
  @"-",
  @"~",
  @"!",

  @"*",
  @"/",
  @"%",
  @"+",
  @"-",

  @"<<",
  @">>",
  @">>>",

  @"&",
  @"|",
  @"^",

  @"<",
  @"<=",
  @"!=",
  @"==",
  @">=",
  @">",

  @"&&",
  @"||",
  @"->",

  nil
};

id <TLString> argument_names[] =
{
  @"arg-1",
  @"arg-2",
};

struct operator
{
  enum builtin_operator op;

  int return_type;

  int types[2];
};

/* Operand argument type, when identical to the first or second argument.  */
#define OAT_SAME1	(1 << BT_NUM)
#define OAT_SAME2	(2 << BT_NUM)
#define OAT_MASK	(3 << BT_NUM)

#define OAT_LONG	(BTS_INT | BTS_LONG)
#define OAT_FLOAT	BT_FLOAT_SET

/* These operators sort-of follow C semantics, which need not be
   desirable.  XXX Define the precise semantics!  */
struct operator operators[] =
{
  {BO_NEG,	BT_INT,		{BTS_INT,	0}},
  {BO_NEG,	BT_LONG,	{BTS_LONG,	0}},
  {BO_NEG,	BT_FLOAT,	{BTS_FLOAT,	0}},
  {BO_NEG,	BT_DOUBLE,	{BTS_DOUBLE,	0}},

  {BO_INV,	BT_BOOLEAN,	{BTS_BOOLEAN,	0}},
  {BO_INV,	BT_INT,		{BTS_INT,	0}},
  {BO_INV,	BT_LONG,	{BTS_LONG,	0}},

  {BO_NOT,	BT_BOOLEAN,	{BT_ANY_SET,	0}},

  {BO_MUL,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_MUL,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_MUL,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_MUL,	BT_LONG,	{BTS_LONG,	BTS_LONG}},
  {BO_MUL,	BT_FLOAT,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_MUL,	BT_DOUBLE,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_DIV,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_DIV,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_DIV,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_DIV,	BT_LONG,	{BTS_LONG,	BTS_LONG}},
  {BO_DIV,	BT_FLOAT,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_DIV,	BT_DOUBLE,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_MOD,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_MOD,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_MOD,	BT_INT,		{BTS_INT,	BTS_LONG}},
  {BO_MOD,	BT_LONG,	{BTS_LONG,	BTS_LONG}},

  {BO_ADD,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_ADD,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_ADD,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_ADD,	BT_LONG,	{BTS_LONG,	BTS_LONG}},
  {BO_ADD,	BT_FLOAT,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_ADD,	BT_DOUBLE,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_SUB,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_SUB,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_SUB,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_SUB,	BT_LONG,	{BTS_LONG,	BTS_LONG}},
  {BO_SUB,	BT_FLOAT,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_SUB,	BT_DOUBLE,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_SHL,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_SHL,	BT_LONG,	{BTS_LONG,	BTS_INT}},

  {BO_SHR,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_SHR,	BT_LONG,	{BTS_LONG,	BTS_INT}},

  {BO_LSR,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_LSR,	BT_LONG,	{BTS_LONG,	BTS_INT}},

  {BO_AND,	BT_BOOLEAN,	{BTS_BOOLEAN,	BTS_BOOLEAN}},
  {BO_AND,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_AND,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_AND,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_AND,	BT_LONG,	{BTS_LONG,	BTS_LONG}},

  {BO_OR,	BT_BOOLEAN,	{BTS_BOOLEAN,	BTS_BOOLEAN}},
  {BO_OR,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_OR,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_OR,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_OR,	BT_LONG,	{BTS_LONG,	BTS_LONG}},

  {BO_EOR,	BT_BOOLEAN,	{BTS_BOOLEAN,	BTS_BOOLEAN}},
  {BO_EOR,	BT_BYTE,	{BTS_BYTE,	BTS_BYTE}},
  {BO_EOR,	BT_CHAR,	{BTS_CHAR,	BTS_CHAR}},
  {BO_EOR,	BT_INT,		{BTS_INT,	BTS_INT}},
  {BO_EOR,	BT_LONG,	{BTS_LONG,	BTS_LONG}},

  {BO_LT,	BT_BOOLEAN,	{BTS_BYTE,	BTS_BYTE}},
  {BO_LT,	BT_BOOLEAN,	{BTS_CHAR,	BTS_CHAR}},
  {BO_LT,	BT_BOOLEAN,	{BTS_INT,	BTS_INT}},
  {BO_LT,	BT_BOOLEAN,	{BTS_LONG,	BTS_LONG}},
  {BO_LT,	BT_BOOLEAN,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_LT,	BT_BOOLEAN,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_LE,	BT_BOOLEAN,	{BTS_BYTE,	BTS_BYTE}},
  {BO_LE,	BT_BOOLEAN,	{BTS_CHAR,	BTS_CHAR}},
  {BO_LE,	BT_BOOLEAN,	{BTS_INT,	BTS_INT}},
  {BO_LE,	BT_BOOLEAN,	{BTS_LONG,	BTS_LONG}},
  {BO_LE,	BT_BOOLEAN,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_LE,	BT_BOOLEAN,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_NE,	BT_BOOLEAN,	{BT_ANY_SET,	OAT_SAME1}},

  {BO_EQ,	BT_BOOLEAN,	{BT_ANY_SET,	OAT_SAME1}},

  {BO_GE,	BT_BOOLEAN,	{BTS_BYTE,	BTS_BYTE}},
  {BO_GE,	BT_BOOLEAN,	{BTS_CHAR,	BTS_CHAR}},
  {BO_GE,	BT_BOOLEAN,	{BTS_INT,	BTS_INT}},
  {BO_GE,	BT_BOOLEAN,	{BTS_LONG,	BTS_LONG}},
  {BO_GE,	BT_BOOLEAN,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_GE,	BT_BOOLEAN,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_GT,	BT_BOOLEAN,	{BTS_BYTE,	BTS_BYTE}},
  {BO_GT,	BT_BOOLEAN,	{BTS_CHAR,	BTS_CHAR}},
  {BO_GT,	BT_BOOLEAN,	{BTS_INT,	BTS_INT}},
  {BO_GT,	BT_BOOLEAN,	{BTS_LONG,	BTS_LONG}},
  {BO_GT,	BT_BOOLEAN,	{BTS_FLOAT,	BTS_FLOAT}},
  {BO_GT,	BT_BOOLEAN,	{BTS_DOUBLE,	BTS_DOUBLE}},

  {BO_SC_AND,	BT_BOOLEAN,	{BTS_BOOLEAN,	BTS_BOOLEAN}},
  {BO_SC_OR,	BT_BOOLEAN,	{BTS_BOOLEAN,	BTS_BOOLEAN}},
  {BO_IMPLIES,	BT_BOOLEAN,	{BTS_BOOLEAN,	BTS_BOOLEAN}},

  {0, 0, {0, 0}}
};

@implementation OTMBuiltinMethod

+(void) init
{
  OTMExtension *e = [[ltt_class_any extensionNamed: nil] semantics];
  struct operator *oper;
  id args[2];

  if (!e)
    ABORT ();

  for (oper = operators; oper->return_type; oper++)
    {
      TLString *op_name = unique_identifier (builtin_operator_name[oper->op]);
      int arg0, arg1;

      builtin_operator_name[oper->op] = op_name;

      for (arg0 = BT_VOID; arg0 < BT_NUM; arg0++)
	if (oper->types[0] & (1 << arg0))
	  {
	    if (arg0 == BT_RECV_CLASS
		|| arg0 == BT_RECV_INST)
	      continue;

	    if (arg0 == BT_RECV)
	      args[0] = [CO_OTMVariable variableWithName: argument_names[0]
			 type: the_any_ref_type];
	    else
	      args[0] = [CO_OTMVariable variableWithName: argument_names[0]
			 type: basic_type[arg0]];

	    if (oper->types[1] == OAT_SAME1)
	      {
		args[1] = [CO_OTMVariable variableWithName: argument_names[1]
			   type: [args[0] type]];
		[e addMethod:
		 [CO_OTMBuiltinMethod methodWithExtension: e
		  nameTypes: CONS (CONS (op_name, CONS (args[0], nil)),
				   CONS (CONS (unique_identifier_colon,
					       CONS (args[1], nil)), nil))
		  returnType: basic_type[oper->return_type]
		  operator: oper->op]];
	      }
	    else if (!oper->types[1])
	      {
		/* Unary operator.  */
		args[1] = nil;
		[e addMethod: [CO_OTMBuiltinMethod methodWithExtension: e
			       nameTypes: CONS (CONS (op_name,
						      CONS (args[0], nil)), nil)
			       returnType: basic_type[oper->return_type]
			       operator: oper->op]];
	      }
	    else for (arg1 = BT_VOID; arg1 < BT_NUM; arg1++)
	      if (oper->types[1] & (1 << arg1))
		{
		  args[1] = [CO_OTMVariable variableWithName: argument_names[1]
			     type: basic_type[arg1]];
		  [e addMethod:
		   [CO_OTMBuiltinMethod methodWithExtension: e
		    nameTypes: CONS (CONS (op_name, CONS (args[0], nil)),
				     CONS (CONS (unique_identifier_colon,
						 CONS (args[1], nil)), nil))
		    returnType: basic_type[oper->return_type]
		    operator: oper->op]];
		}
	  }
    }
}

+(OTMBuiltinMethod *) methodWithExtension: (OTMExtension *) e
				nameTypes: (TLCons *) nt
			       returnType: (id) rt
				 operator: (enum builtin_operator) o
{
  return [[self gcAlloc] initWithExtension: e nameTypes: nt returnType: rt
	  operator: o];
}

-(BOOL) builtinp
{
  return YES;
}

-initWithExtension: (OTMExtension *) e
	 nameTypes: (TLCons *) nt
	returnType: (id) rt
	  operator: (enum builtin_operator) o
{
  op = o;

  return [super initWithExtension: e nameTypes: nt returnType: rt flatp: YES];
}

-(enum builtin_operator) operator
{
  return op;
}

-(id <TLString>) outputName
{
  if (!output_name)
    {
      if (op == BO_NEG)
	ASGN_IVAR (output_name, @"-");
      else
	output_name = internal_name;
    }

  return [super outputName];
}

-(id) resultOfInvocation: (OTMInvocation *) inv
{
  TLVector *args = [inv arguments];

  switch (op)
    {
    case BO_NEG:
    case BO_INV:
    case BO_NOT:
      return formac (nil, @"%@(%@)", builtin_operator_name[op],
		     [[args _elementAtIndex: 0] result]);

    case BO_MUL:
    case BO_DIV:
    case BO_MOD:
    case BO_ADD:
    case BO_SUB:
    case BO_SHL:
    case BO_SHR:
    case BO_AND:
    case BO_OR:
    case BO_EOR:
    case BO_LT:
    case BO_LE:
    case BO_GE:
    case BO_GT:
      return formac (nil, @"(%@) %@ (%@)", [[args _elementAtIndex: 0] result],
		     builtin_operator_name[op],
		     [[args _elementAtIndex: 1] result]);

    case BO_LSR:
      return formac (nil, @"(unsigned) %@ >> %@",
		     [[args _elementAtIndex: 0] result],
		     [[args _elementAtIndex: 1] result]);

    case BO_NE:
    case BO_EQ:
      {
	OTMExpr *lhs = [args _elementAtIndex: 0];
	OTMExpr *rhs = [args _elementAtIndex: 1];
	OTMType *lt = [lhs type], *rt = [rhs type];
	BOOL do_cast = [lt isObjectType] && [rt isObjectType] && lt != rt;

	return formac (nil, @"(%@) %@ (%@%@)", [lhs result],
		       builtin_operator_name[op],
		       do_cast ? @"(void *) " : @"", [rhs result]);
      }

    default:
      ABORT ();
    }
  return nil;
}

@end
