<copyright> Date object and Gregorian calendar manipulation methods.<br>
    Written by <a href="mailto:michael@thi.nl">Michael L.H. Brouwer</a>

    Copyright &copy; 1996-1998 Pieter J. Schoenmakers.<br>

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    Some of the algorithms used in this class are derived from software
    described in ``Calendrical Calculations'' by Nachum Dershowitz and
    Edward M. Reingold, Software--Practice and Experience, vol. 20, no. 9,
    pp. 899-928 and from ``Calendrical Calculations, II: Three Historical
    Calendars'' by Edward M. Reingold, Nachum Dershowitz, and Stewart
    M. Clamen, Software--Practice and Experience, vol. 23, no. 4.

    <id>$Id: Date.t,v 1.15 1998/05/18 15:21:16 tiggr Exp $</id>
    </copyright>

<doc> The Date class implements absolute times using doubles to represent
    the number of seconds passed since a certain reference date.
    Internally the time at which the Date class was initialized is used as
    reference.  As absolute reference the first instant of January 1, 2001
    is used.  All gregorian calculation functions use an absolute date
    which is the number of days since the Gregorian date December 31, 1
    BC.  </doc>
implementation class
Date: State
{
  <doc> For two dates to be considered equal they should be no further
      apart than {EPSILON}.  </doc>
  const EPSILON = 1d-6;
  const OFFSET_DISTANT_FUTURE = 1d+100;
  const OFFSET_DISTANT_PAST = -OFFSET_DISTANT_FUTURE;
  const SECONDS_PER_DAY = 86400.0d;

  <doc> The number of days from the imaginary Gregorian date Sunday, 31
      december 1 BC to our reference date (January 1 2001).  </doc>
  const ABSOLUTE_REFDATE = 730486;

  <doc> Some offset from the reference date relative to which all Date
      instances maintain their notion of time.  This is set in {load},
      ensuring a high accuracy of dates near the moment in time during
      which this program is running.  </doc>
  public static double relative_offset;

  <doc> A date in the very far future and a date in the very far past.
      </doc>
  public static Date distant_future, distant_past;
}

<doc> Return the number of seconds after {relative_offset} it is now.
    </doc>
extern double (now)
  relativeTimeIntervalSinceNow
post
  now > 0.0;

<doc> Return the number of seconds after the absolute reference date it is
    now.  This number is negative for dates before the first instant of
    January 1, 2001.  </doc>
extern double
  timeIntervalSinceReferenceDate;

<doc> Return the absolute date and the seconds passed in that day for a
    time interval since the reference date.  </doc>
(int, double)
  absoluteAndSecondsOfTimeInterval double ti
{
  int absolute;

  absolute = int (ti / SECONDS_PER_DAY);
  ti = ti - double (absolute) * SECONDS_PER_DAY;
  if (ti < 0.0d)
    {
      ti += SECONDS_PER_DAY;
      absolute--;
    }
  absolute += ABSOLUTE_REFDATE;
  = (absolute, ti);
}

<doc> The number of days elapsed between the Gregorian date 12/31/1 BC and
    {(year, month, day)}.  The Gregorian date Sunday, December 31, 1
    BC is imaginary.  </doc>
int
  absoluteFromGregorian (int, int, int) (year, month, day)
{
  int prior_years = year - 1;
  = [self dayNumber (year, month, day)]
    + 365 * prior_years + prior_years / 4 - prior_years / 100
    + prior_years / 400;
}

<doc> The number of days elapsed between the Gregorian date 1 BC December
    31 and DATE.  The `ISO year' corresponds approximately to the
    Gregorian year, but weeks start on Monday and end on Sunday.  The
    first week of the ISO year is the first such week in which at least 4
    days are in a year.  The ISO commercial DATE has the form {(year,
    week, day)} in which {week} is in the range 1..52 and {day} is in the
    range 0..6 (1 == Monday, 2 == Tuesday, ..., 0 == Sunday).  The
    Gregorian date Sunday, December 31, 1 BC is imaginary.  </doc>
int
  absoluteFromIso (int, int, int) (year, week, day)
{
  = ([self dayNameOnOrBefore (1, 3 + [self absoluteFromGregorian (year, 1, 1)])]
     + (7 * (week - 1)) + (day == 0 ? 6 : day - 1));
}

<doc> Returns the absolute date of the {day_name} on or before
    {absolute}.  {day_name}==0 means Sunday, {day_name}==1 means Monday,
    and so on.

    Note: Applying this function to {absolute}+6 gives us the {day_name}
    on or after an absolute day d.  Similarly, applying it to {absolute}+3
    gives the {day_name} nearest to {absolute}, applying it to
    {absolute}-1 gives the {day_name} previous to {absolute}, and applying
    it to {absolute}+7 gives the {day_name} following {absolute}.  </doc>
int
  dayNameOnOrBefore (int, int) (day_name, absolute)
{
  = absolute - (absolute - day_name) % 7;
}

<doc> Return the day number within the year of the date {(year, month,
    day)}.  For example, dayNumber (1, 1, 1987) returns the value 1,
    while dayNumber (12, 31, 1980) returns 366.  </doc>
int
  dayNumber (int, int, int) (year, month, day)
{
  int day_of_year = day + 31 * (month - 1);

  if (month > 2)
    {
      day_of_year -= (23 + 4 * month) / 10;
      if ([self isLeapYear year])
	day_of_year++;
    }

  = day_of_year;
}

<doc> Return the Gregorian day of the week for {absolute} where
    0==Sunday, 1==Monday, ..., 6==Saturday.  </doc>
int
  dayOfWeekOfAbsolute int absolute
{
  = absolute % 7;
}

<doc> Compute the list (month, day, year) corresponding to the absolute
    DATE.  The absolute date is the number of days elapsed since the
    (imaginary) Gregorian date Sunday, December 31, 1 BC.  </doc>
(int, int, int)
  gregorianFromAbsolute int date
{
  int d0 = date - 1,
    n400 = d0 / 146097,
    d1 = d0 % 146097,
    n100 = d1 / 36524,
    d2 = d1 % 36524,
    n4 = d2 / 1461,
    d3 = d2 % 1461,
    n1 = d3 / 365,
    day = 1 + d3 % 365,
    year = 400 * n400 + 100 * n100 + 4 * n4 + n1;

  if (n100 == 4 || n1 == 4)
     = (year, 12, 31);
  else
    {
      year++;
      int month = 1;

      for (;;)
	{
	  int mdays = [self lastDayOfMonth month year year];
	  if (day <= mdays)
	    break;
	  month++;
	  day -= mdays;
	}

      = (year, month, day);
    }
}

<doc> Return {TRUE} iff year is a Gregorian leap year.  </doc>
boolean
  isLeapYear int year
{
  = year % 4 == 0 && (year % 100 == 0 -> year % 400 == 0);
}

<doc> Compute the `ISO commercial date' corresponding to the
    {absolute}.  The ISO year corresponds approximately to the
    Gregorian year, but weeks start on Monday and end on Sunday.  The
    first week of the ISO year is the first such week in which at least 4
    days are in a year.  The ISO commercial date has the form (year week
    day) in which week is in the range 1..52 and day is in the range 0..6
    (1 = Monday, 2 = Tuesday, ..., 0 = Sunday).  The absolute date is the
    number of days elapsed since the (imaginary) Gregorian date Sunday,
    December 31, 1 BC.  </doc>
(int, int, int)
  isoFromAbsolute int absolute
{
  int year;
  int sum = 0;

  (year,,) = [self gregorianFromAbsolute absolute - 3];

  while (absolute >= [self absoluteFromIso (1 + year, 1, 1)])
    year++;

  = (year, absolute % 7,
     1 + (absolute - [self absoluteFromIso (year, 1, 1)]) / 7);
}

<doc> Return the last day of the month {month} of the year
    {year}.  </doc>
int
  lastDayOfMonth int month
	    year int year
{
  if (month == 4 || month == 6 || month == 9 || month == 11)
    = 30;
  else if (month == 2)
    if ([self isLeapYear year])
      = 29;
    else
      = 28;
  else
    = 31;
}

<doc> Perform class initialization.  </doc>
extern void
  load MutableArray arguments;

<doc> Return a date instance representing this moment.	</doc>
Date
  now
{
  = [[self alloc] init];
}

<doc> Return the absolute date and the seconds passed in that day for a
    time interval since the reference date.  </doc>
protected double
  relativeTimeIntervalOfAbsoluteAndSeconds (int, double) (absolute, seconds)
{
  = ((double (absolute - ABSOLUTE_REFDATE) * SECONDS_PER_DAY - relative_offset)
     + seconds);
}

<doc> Return the absolute date and the seconds passed in that day for a
    time interval since the reference date.  </doc>
double
  timeIntervalOfAbsoluteAndSeconds (int, double) (absolute, seconds)
{
  = double (absolute - ABSOLUTE_REFDATE) * SECONDS_PER_DAY + seconds;
}

end;

implementation instance
Date
{
  double relative_ti;
}

<doc> Returns -1 if the receiver is earlier than {other} 0 if the
    difference is smaller that {EPSILON} and 1 if the receiver is
    after {other}. </doc>
int
  compare id other
{
  double diff = relative_ti - [other relativeTimeInterval];

  = diff < 0.0 ? (EPSILON >= -diff ? 0 : 1)
	       : (EPSILON >= diff ? 0 : -1);
}

<doc> Return a new instance initialized at {ti} seconds after the
    receiver.  </doc>
id
  dateWithOffset double ti
{
  = [[isa alloc] init relative_ti + ti];
}

<doc> Return {other} if it is earlier than the receiver, return the
    receiver otherwise.  </doc>
Date
  earlierDate Date other
{
  = relative_ti <= [other relativeTimeInterval] ? self : other;
}

<doc> Return {TRUE} iff the receiver is within {EPSILON} seconds
    of {d}.  </doc>
boolean
  equals id d
{
  double diff = relative_ti - [d relativeTimeInterval];

  = EPSILON >= (diff < 0.0 ? -diff : diff);
}

<doc> Designated initializer.  </doc>
protected id
  init double d
{
  relative_ti = d;

  = self;
}

<doc> Initialize with the current time.  </doc>
id
  init
{
  = [self init [isa relativeTimeIntervalSinceNow]];
}

<doc> Initialize with {ti} seconds after the current time.  </doc>
id
  initWithTimeIntervalSinceNow double ti
{
  = [self init [isa relativeTimeIntervalSinceNow] + ti];
}

<doc> Initialize with {ti} seconds after the absolute reference date
    January 1, 2001.  </doc>
id
  initWithTimeIntervalSinceReferenceDate double ti
{
  = [self init ti - relative_offset];
}

<doc> Return {other} iff it is later than the receiver, return the
    receiver otherwise.  </doc>
Date
  laterDate Date other
{
  = relative_ti >= [other relativeTimeInterval] ? self : other;
}

<doc> Return the number of seconds after {relative_offset}  </doc>
protected double
  relativeTimeInterval
{
  = relative_ti;
}

<doc> Return the number of seconds passed since {d}.  This number is
    negative if the receiver is earlier than {d}.  </doc>
double
  timeIntervalSinceDate Date d
{
  = relative_ti - [d relativeTimeInterval];
}

<doc> Return the number of seconds passed since now.  This number is
    negative for dates before now.  </doc>
double
  timeIntervalSinceNow
{
  = relative_ti - [isa relativeTimeIntervalSinceNow];
}

<doc> Return the number of seconds passed since the absolute reference
    date.  This number is negative for all dates before the first instant
    of January 1 2001.  </doc>
double
  timeIntervalSinceReferenceDate
{
  = relative_ti + relative_offset;
}

<doc> Print this date in a human readable format, relative to GMT.  </doc>
OutputStream
  write OutputStream s
{
  int year, month, day;
  int absolute, seconds;
  double ti;

  (absolute, ti) = [isa absoluteAndSecondsOfTimeInterval
		    [self timeIntervalSinceReferenceDate]];

  (year, month, day) = [isa gregorianFromAbsolute absolute];
  seconds = int (ti);

  = [s print (year, "-", month, "-", day, " ", seconds / 3600, ":",
	      seconds / 60 % 60, ":", seconds % 60, " GMT")];
}

<doc> <h4>Encoding</h4> </doc>

void
  encodeUsingCoder Encoder coder
{
  if (![coder hasBeenCodedFor [Date self]])
    {
      [super encodeUsingCoder coder];
      [coder encode [self timeIntervalSinceReferenceDate]];
    }
}

void
  initWithCoder Decoder coder
{
  if (![coder hasBeenCodedFor [Date self]])
    {
      [super initWithCoder coder];
      relative_ti = [coder decode] - relative_offset;
    }
}

end;
