/* Implementation of TLNumber class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLNumber.m,v 1.2 1998/02/23 14:17:38 tiggr Exp $  */

#define TLNUMBER_DECLARE_PRIVATE_METHODS
#import "tl/support.h"
#import "tl/TLNumber.h"
#import "tl/TLSymbol.h"
#import <ctype.h>

@implementation TLNumber: TLObject

+(TLNumber *) numberWithChar: (char) value
{
  return ([[self gcAlloc] initWithChar: value]);
} /* +numberWithChar: */

+(TLNumber *) numberWithDouble: (double) value
{
  return ([[self gcAlloc] initWithDouble: value]);
} /* +numberWithDouble: */

+(TLNumber *) numberWithFloat: (float) value
{
  return ([[self gcAlloc] initWithFloat: value]);
} /* +numberWithFloat: */

+(TLNumber *) numberWithInt: (int) value
{
  return ([[self gcAlloc] initWithInt: value]);
} /* +numberWithInt: */

+(TLNumber *) numberWithLong: (long) value
{
  return ([[self gcAlloc] initWithLong: value]);
} /* +numberWithLong: */

+(TLNumber *) numberWithLongLong: (long long) value
{
  return ([[self gcAlloc] initWithLongLong: value]);
} /* +numberWithLongLong: */

+(TLNumber *) numberWithShort: (short) value
{
  return ([[self gcAlloc] initWithShort: value]);
} /* +numberWithShort: */

+(TLNumber *) numberWithUnsignedChar: (unsigned char) value
{
  return ([[self gcAlloc] initWithUnsignedChar: value]);
} /* +numberWithUnsignedChar: */

+(TLNumber *) numberWithUnsignedInt: (unsigned int) value
{
  return ([[self gcAlloc] initWithUnsignedInt: value]);
} /* +numberWithUnsignedInt: */

+(TLNumber *) numberWithUnsignedLong: (unsigned long) value
{
  return ([[self gcAlloc] initWithUnsignedLong: value]);
} /* +numberWithUnsignedLong: */

+(TLNumber *) numberWithUnsignedLongLong: (unsigned long long) value
{
  return ([[self gcAlloc] initWithUnsignedLongLong: value]);
} /* +numberWithUnsignedLongLong: */

+(TLNumber *) numberWithUnsignedShort: (unsigned short) value
{
  return ([[self gcAlloc] initWithUnsignedShort: value]);
} /* +numberWithUnsignedShort: */

-floatp
{
  return (encoding == 'f' || encoding == 'd' ? Qt : nil);
} /* -floatp */

-initWithDouble: (double) value
{
  encoding = *@encode (double);
  v.d = value;
  return (self);
} /* -initWithDouble: */

-initWithFloat: (float) value
{
  encoding = *@encode (float);
  v.f = value;
  return (self);
} /* -initWithFloat: */

-initWithChar: (char) value
{
  encoding = *@encode (char);
  v.c = value;
  return (self);
} /* -initWithChar: */

-initWithShort: (short) value
{
  encoding = *@encode (short);
  v.s = value;
  return (self);
} /* -initWithShort: */

-initWithInt: (int) value
{
  encoding = *@encode (int);
  v.i = value;
  return (self);
} /* -initWithInt: */

-initWithLong: (long) value
{
  encoding = *@encode (long);
  v.l = value;
  return (self);
} /* -initWithLong: */

-initWithLongLong: (long long) value
{
  encoding = *@encode (long long);
  v.q = value;
  return (self);
} /* -initWithLongLong: */

-initWithUnsignedChar: (unsigned char) value
{
  encoding = *@encode (unsigned char);
  v.c = value;
  return (self);
} /* -initWithUnsignedChar: */

-initWithUnsignedShort: (unsigned short) value
{
  encoding = *@encode (unsigned short);
  v.s = value;
  return (self);
} /* -initWithUnsignedShort: */

-initWithUnsignedInt: (unsigned int) value
{
  encoding = *@encode (unsigned int);
  v.i = value;
  return (self);
} /* -initWithUnsignedInt: */

-initWithUnsignedLong: (unsigned long) value
{
  encoding = *@encode (unsigned long);
  v.l = value;
  return (self);
} /* -initWithUnsignedLong: */

-initWithUnsignedLongLong: (unsigned long long) value
{
  encoding = *@encode (unsigned long long);
  v.q = value;
  return (self);
} /* -initWithUnsignedLongLong: */

-integerp
{
  return (encoding != 'f' && encoding != 'd' ? Qt : nil);
} /* -integerp */

-(void) print: (id <TLMutableStream>) stream quoted: (BOOL) qp
{
  switch (encoding)
    {
    case 'f': formac (stream, @"%f", v.f); break;
    case 'd': formac (stream, @"%lf", v.d); break;
    case 'c': case 'C': formac (stream, @"%c", v.c); break;
    case 's': formac (stream, @"%d", v.s); break;
    case 'i': formac (stream, @"%d", v.i); break;
    case 'l': formac (stream, @"%ld", v.l); break;
    case 'q': formac (stream, @"%qd", v.q); break;
    case 'S': formac (stream, @"%u", v.s); break;
    case 'I': formac (stream, @"%u", v.i); break;
    case 'L': formac (stream, @"%lu", v.l); break;
    case 'Q': formac (stream, @"%qu", v.q); break;
    default:
      formac (stream, @"%lx: unimplemented encoding `%c'", (long) self,
	      encoding);
      break;
    }
} /* -print:quoted: */

/******************** TLNumber protocol ********************/

-(char) encoding
{
  return (encoding);
} /* -encoding */

#define RETRIEVE_VALUE_FLOATS  \
      case 'f': return (v.f);			\
      case 'd': return (v.d);			\

#define RETRIEVE_VALUE_INTS  \
      case 'c': return (v.c);			\
      case 's': return (v.s);			\
      case 'i': return (v.i);			\
      case 'l': return (v.l);			\
      case 'q': return (v.q);			\
						\
      case 'C': return ((unsigned) v.c);	\
      case 'S': return ((unsigned) v.s);	\
      case 'I': return ((unsigned) v.i);	\
      case 'L': return ((unsigned) v.l);	\
      case 'Q': return ((unsigned) v.q);

#define RETRIEVE_VALUE(TYPE, NAME)  \
  -(TYPE) NAME ## Value					\
  {							\
    switch (encoding)					\
      {							\
      RETRIEVE_VALUE_FLOATS				\
      RETRIEVE_VALUE_INTS				\
      default:						\
	[self error: "bad encoding `%c'", encoding];	\
      }							\
    return (0);						\
  }

RETRIEVE_VALUE (double, double)
RETRIEVE_VALUE (float, float)
RETRIEVE_VALUE (char, char)
RETRIEVE_VALUE (short, short)
RETRIEVE_VALUE (int, int)
RETRIEVE_VALUE (long, long)
RETRIEVE_VALUE (long long, longLong)
RETRIEVE_VALUE (unsigned char, unsignedChar)
RETRIEVE_VALUE (unsigned short, unsignedShort)
RETRIEVE_VALUE (unsigned int, unsignedInt)
RETRIEVE_VALUE (unsigned long, unsignedLong)
RETRIEVE_VALUE (unsigned long long, unsignedLongLong)

-(double) floatPDoubleValue
{
  switch (encoding)
    {
      RETRIEVE_VALUE_FLOATS
    default:
      [self error: "not a floating point number (encoding=`%c')", encoding];
    }
  return (0);
} /* floatPDoubleValue */

-(float) floatPFloatValue
{
  switch (encoding)
    {
      RETRIEVE_VALUE_FLOATS
    default:
      [self error: "not a floating point number (encoding=`%c')", encoding];
    }
  return (0);
} /* floatPFloatValue */

#define RETRIEVE_INTEGER(TYPE, NAME)  \
  -(TYPE) integerP ## NAME ## Value					\
  {									\
    switch (encoding)							\
      {									\
      RETRIEVE_VALUE_INTS						\
      default:								\
	[self error: "not an int number (encoding=`%c')", encoding];	\
      }									\
    return (0);								\
  }

RETRIEVE_INTEGER (char, Char)
RETRIEVE_INTEGER (short, Short)
RETRIEVE_INTEGER (int, Int)
RETRIEVE_INTEGER (long, Long)
RETRIEVE_INTEGER (long long, LongLong)
RETRIEVE_INTEGER (unsigned char, UnsignedChar)
RETRIEVE_INTEGER (unsigned short, UnsignedShort)
RETRIEVE_INTEGER (unsigned int, UnsignedInt)
RETRIEVE_INTEGER (unsigned long, UnsignedLong)
RETRIEVE_INTEGER (unsigned long long, UnsignedLongLong)

/* Assign to LV our value, based on the encoding, which we know is not
   floating.  */
#define ASSIGN_LOCAL_VALUE  \
  switch (encoding)							\
    {									\
    case 'c': lv = v.c; break;						\
    case 's': lv = v.s; break;						\
    case 'i': lv = v.i; break;						\
    case 'l': lv = v.l; break;						\
    case 'q': lv = v.q; break;						\
    case 'C': lv = (unsigned) v.c; break;				\
    case 'S': lv = (unsigned) v.s; break;				\
    case 'I': lv = (unsigned) v.i; break;				\
    case 'L': lv = (unsigned) v.l; break;				\
    case 'Q': lv = (unsigned) v.q; break;				\
    default:								\
      [self error: "bad or unimplemented encoding `%c'", encoding];	\
      lv = 0;								\
    }

/* XXX Something tells me this could possibly be simpler.  */
#define COMPARE(NAME, OP)  \
  -(id) NAME (id <TLNumber>) value	        			     \
  {					        			     \
    char re;				        			     \
					        			     \
    /* Handle case in which we're floating.  */				     \
    if (encoding == 'd')		        			     \
      return (v.d OP [value doubleValue] ? Qt : nil);			     \
    if (encoding == 'f')		        			     \
      return ((double) v.f OP [value doubleValue] ? Qt : nil);		     \
					        			     \
    re = [value encoding];		        			     \
					        			     \
    /* Handle case in which the VALUE is floating and we aren't.  */	     \
    if (re == 'd' || re == 'f')		        			     \
      {					        			     \
	double lv, rv = [value doubleValue];				     \
					        			     \
	ASSIGN_LOCAL_VALUE		        			     \
	return (lv OP rv ? Qt : nil);					     \
      }					        			     \
					        			     \
    /* Handle unsigned long long case (expensive on some machines).  */	     \
    if (encoding == 'Q' || re == 'Q')	        			     \
      {					        			     \
	unsigned long long lv, rv = [value unsignedLongLongValue];	     \
					        			     \
	ASSIGN_LOCAL_VALUE		        			     \
	return (lv OP rv ? Qt : nil);					     \
      }					        			     \
					        			     \
    /* Handle signed long long case (expensive on some machines).  */	     \
    if (encoding == 'q' || re == 'q')	        			     \
      {					        			     \
	long long lv, rv = [value longLongValue];			     \
					        			     \
	ASSIGN_LOCAL_VALUE		        			     \
	return (lv OP rv ? Qt : nil);					     \
      }					        			     \
					        			     \
    /* Handle unsigned int case.  */	        			     \
    if (encoding == 'I' || re == 'I')	        			     \
      {					        			     \
	unsigned int lv, rv = [value unsignedIntValue];			     \
					        			     \
	ASSIGN_LOCAL_VALUE		        			     \
	return (lv OP rv ? Qt : nil);					     \
      }					        			     \
					        			     \
    /* The default case.  It takes time to get here.  XXX Maybe the encoding \
       should be a closed namespace, enabling the use of switch with a jump  \
       table?  */							     \
    {									     \
      int lv, rv = [value intValue];					     \
									     \
      ASSIGN_LOCAL_VALUE						     \
      return (lv OP rv ? Qt : nil);					     \
    }									     \
  }

COMPARE (lessThan:, <)
COMPARE (lessThanOrEqual:, <=)
COMPARE (notEqual:, !=)
COMPARE (equal:, ==)
COMPARE (greaterThanOrEqual:, >=)
COMPARE (greaterThan:, >)

/* XXX Something tells me this could possibly be simpler.  */
#define ARITH(NAME, OP)  \
  -(id) NAME (id <TLNumber>) value					       \
  {									       \
    char re = [value encoding];						       \
									       \
    /* Handle cases in which we're floating.  */			       \
    if (encoding == 'd')						       \
      return ([isa numberWithDouble: v.d OP [value doubleValue]]);	       \
    if (encoding == 'f')						       \
      if (re == 'd')							       \
	return ([isa numberWithDouble: (double) v.f OP [value doubleValue]]);  \
      else								       \
	return ([isa numberWithFloat: v.f OP [value floatValue]]);	       \
									       \
    /* Handle cases in which the VALUE is floating and we aren't.  */	       \
    if (re == 'd')							       \
      {									       \
	double lv, rv = [value doubleValue];				       \
									       \
	ASSIGN_LOCAL_VALUE						       \
	return ([isa numberWithDouble: lv OP rv]);			       \
      }									       \
    if (re == 'f')							       \
      {									       \
	float lv, rv = [value floatValue];				       \
									       \
	ASSIGN_LOCAL_VALUE						       \
	return ([isa numberWithFloat: lv OP rv]);			       \
      }									       \
									       \
    /* Handle unsigned long long case (expensive on some machines).  */	       \
    if (encoding == 'Q' || re == 'Q')					       \
      {									       \
	unsigned long long lv, rv = [value unsignedLongLongValue];	       \
									       \
	ASSIGN_LOCAL_VALUE						       \
	return ([isa numberWithUnsignedLongLong: lv OP rv]);		       \
      }									       \
									       \
    /* Handle signed long long case (expensive on some machines).  */	       \
    if (encoding == 'q' || re == 'q')					       \
      {									       \
	long long lv, rv = [value longLongValue];			       \
									       \
	ASSIGN_LOCAL_VALUE						       \
	return ([isa numberWithLongLong: lv OP rv]);			       \
      }									       \
									       \
    /* Handle unsigned int case.  */					       \
    if (encoding == 'I' || re == 'I')					       \
      {									       \
	unsigned int lv, rv = [value unsignedIntValue];			       \
									       \
	ASSIGN_LOCAL_VALUE						       \
	return ([isa numberWithUnsignedInt: lv OP rv]);		       \
      }									       \
									       \
    /* The default case.  Note that an operation on anything shorter than      \
       int still results in an int.  It takes time to get here.  XXX Maybe     \
       the encoding should be a closed namespace, enabling the use of switch   \
       with a jump table?  */						       \
    {									       \
      int lv, rv = [value intValue];					       \
									       \
      ASSIGN_LOCAL_VALUE						       \
      return ([isa numberWithInt: lv OP rv]);				       \
    }									       \
  }

ARITH (numberByAdd:, +)
ARITH (numberBySubtract:, -)
ARITH (numberByMultiply:, *)
ARITH (numberByDivide:, /)

/* XXX Something tells me this could possibly be simpler.  */
#define INTEGER_ARITH(NAME, OP)  \
  -(id) NAME (id <TLNumber>) value					     \
  {									     \
    char re = [value encoding];						     \
									     \
    if (encoding == 'd' || encoding == 'f' || re == 'd' || re == 'f')	     \
      [self error: "bad type (encoding=(`%c', `%c')) for operation `%s'",    \
       encoding, re, sel_get_name (_cmd)];				     \
									     \
    /* Handle unsigned long long case (expensive on some machines).  */	     \
    if (encoding == 'Q' || re == 'Q')					     \
      {									     \
	unsigned long long lv, rv = [value unsignedLongLongValue];	     \
									     \
	ASSIGN_LOCAL_VALUE						     \
	return ([isa numberWithUnsignedLongLong: lv OP rv]);		     \
      }									     \
									     \
    /* Handle signed long long case (expensive on some machines).  */	     \
    if (encoding == 'q' || re == 'q')					     \
      {									     \
	long long lv, rv = [value longLongValue];			     \
									     \
	ASSIGN_LOCAL_VALUE						     \
	return ([isa numberWithLongLong: lv OP rv]);			     \
      }									     \
									     \
    /* Handle unsigned int case.  */					     \
    if (encoding == 'I' || re == 'I')					     \
      {									     \
	unsigned int lv, rv = [value unsignedIntValue];			     \
									     \
	ASSIGN_LOCAL_VALUE						     \
	return ([isa numberWithUnsignedInt: lv OP rv]);			     \
      }									     \
									     \
    /* The default case.  Note that an operation on anything shorter than    \
       int still results in an int.  It takes time to get here.  XXX Maybe   \
       the encoding should be a closed namespace, enabling the use of switch \
       with a jump table?  */						     \
    {									     \
      int lv, rv = [value intValue];					     \
									     \
      ASSIGN_LOCAL_VALUE						     \
      return ([isa numberWithInt: lv OP rv]);				     \
    }									     \
  }

INTEGER_ARITH (numberByModulo:, %)
INTEGER_ARITH (numberByAnd:, &)
INTEGER_ARITH (numberByOr:, |)
INTEGER_ARITH (numberByXor:, ^)

/* XXX Something tells me this could possibly be simpler.  */
#define INT_ARITH(NAME, OP)  \
  -(id) NAME (int) value						     \
  {									     \
    switch (encoding)							     \
      {									     \
      case 'd':								     \
	return ([isa numberWithDouble: v.d OP value]);			     \
      case 'f':								     \
	return ([isa numberWithFloat: v.f OP value]);			     \
      case 'c':								     \
	return ([isa numberWithInt: v.c OP value]);			     \
      case 's':								     \
	return ([isa numberWithInt: v.s OP value]);			     \
      case 'i':								     \
	return ([isa numberWithInt: v.i OP value]);			     \
      case 'l':								     \
	return ([isa numberWithLong: v.l OP value]);			     \
      case 'q':								     \
	return ([isa numberWithLongLong: v.q OP value]);		     \
      case 'C':								     \
	return ([isa numberWithUnsignedInt: (unsigned) v.c OP value]);	     \
      case 'S':								     \
	return ([isa numberWithUnsignedInt: (unsigned) v.s OP value]);	     \
      case 'I':								     \
	return ([isa numberWithUnsignedInt: (unsigned) v.i OP value]);	     \
      case 'L':								     \
	return ([isa numberWithUnsignedLong: (unsigned) v.l OP value]);      \
      case 'Q':								     \
	return ([isa numberWithUnsignedLongLong: (unsigned) v.q OP value]);  \
      default:								     \
	[self error: "bad or unimplemented encoding `%c'", encoding];	     \
      }									     \
    return (nil);							     \
  }

INT_ARITH (numberByAddingInt:, +)
INT_ARITH (numberBySubtractingInt:, -)
INT_ARITH (numberByMultiplyingByInt:, *)
INT_ARITH (numberByDividingByInt:, /)

/* XXX Something tells me this could possibly be simpler.  */
#define INTEGER_INT_ARITH(NAME, OP)  \
  -(id) NAME (int) value						     \
  {									     \
    switch (encoding)							     \
      {									     \
      case 'c':								     \
	return ([isa numberWithInt: v.c OP value]);			     \
      case 's':								     \
	return ([isa numberWithInt: v.s OP value]);			     \
      case 'i':								     \
	return ([isa numberWithInt: v.i OP value]);			     \
      case 'l':								     \
	return ([isa numberWithLong: v.l OP value]);			     \
      case 'q':								     \
	return ([isa numberWithLongLong: v.q OP value]);		     \
      case 'C':								     \
	return ([isa numberWithUnsignedInt: (unsigned) v.c OP value]);	     \
      case 'S':								     \
	return ([isa numberWithUnsignedInt: (unsigned) v.s OP value]);	     \
      case 'I':								     \
	return ([isa numberWithUnsignedInt: (unsigned) v.i OP value]);	     \
      case 'L':								     \
	return ([isa numberWithUnsignedLong: (unsigned) v.l OP value]);      \
      case 'Q':								     \
	return ([isa numberWithUnsignedLongLong: (unsigned) v.q OP value]);  \
      default:								     \
	[self error: "bad encoding `%c' for integer int arith", encoding];   \
      }									     \
    return (nil);							     \
  }

INTEGER_INT_ARITH (numberByModuloByInt:, %)
INTEGER_INT_ARITH (numberByAndWithInt:, &)
INTEGER_INT_ARITH (numberByOrWithInt:, |)
INTEGER_INT_ARITH (numberByXorWithInt:, ^)

@end
