/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/* FLASK */

#include "policydb.h"
#include "services.h"
#include "sidtab.h"
#include "extension.h"

sidtab_t        sidtab;
policydb_t      policydb;
extavtab_t      extavtab;
exttrtab_t      exttrtab;
int             ss_initialized = 0;

static unsigned int latest_granting = 0;


static inline ss_root_context_t *
root_ancestor(security_id_t sid)
{
	sid_datum_t    *sid_datum;
	ss_child_context_t *child;


	sid_datum = sidtab_search(&sidtab, sid);
	while (sid_datum && !sid_datum->context.isroot) {
		child = &CCONTEXT(&sid_datum->context);
		sid_datum = sidtab_search(&sidtab, child->parent);
	}

	if (!sid_datum)
		return 0;

	return &RCONTEXT(&sid_datum->context);
}


static int
constraint_expr_eval(ss_root_context_t * scontext,
		     ss_root_context_t * tcontext,
		     constraint_expr_t * expr)
{
	role_datum_t   *srole, *trole;

	switch (expr->expr_type) {
	case CONSTRAINT_EXPR_TYPE_NOT:
		return !constraint_expr_eval(scontext, tcontext, expr->left);
	case CONSTRAINT_EXPR_TYPE_AND:
		return constraint_expr_eval(scontext, tcontext, expr->left) &&
			constraint_expr_eval(scontext, tcontext, expr->right);
	case CONSTRAINT_EXPR_TYPE_OR:
		return constraint_expr_eval(scontext, tcontext, expr->left) ||
			constraint_expr_eval(scontext, tcontext, expr->right);
	case CONSTRAINT_EXPR_TYPE_SAMEUSER:
		return (scontext->user == tcontext->user);
	case CONSTRAINT_EXPR_TYPE_TYPE_SOURCE:
		return ebitmap_get_bit(&expr->bitmap, scontext->type - 1);
	case CONSTRAINT_EXPR_TYPE_TYPE_TARGET:
		return ebitmap_get_bit(&expr->bitmap, tcontext->type - 1);
	case CONSTRAINT_EXPR_TYPE_ROLE_SOURCE:
		return ebitmap_get_bit(&expr->bitmap, scontext->role - 1);
	case CONSTRAINT_EXPR_TYPE_ROLE_TARGET:
		return ebitmap_get_bit(&expr->bitmap, tcontext->role - 1);
	case CONSTRAINT_EXPR_TYPE_ROLE_RELATION:
		srole = policydb.role_val_to_struct[scontext->role - 1];
		trole = policydb.role_val_to_struct[tcontext->role - 1];
		switch (expr->relation) {
		case CONSTRAINT_EXPR_VALUE_RELATION_DOM:
			return (scontext->role != tcontext->role) &&
				ebitmap_get_bit(&srole->dominates,
						tcontext->role - 1);
		case CONSTRAINT_EXPR_VALUE_RELATION_DOMBY:
			return (scontext->role != tcontext->role) &&
				ebitmap_get_bit(&trole->dominates,
						scontext->role - 1);
		case CONSTRAINT_EXPR_VALUE_RELATION_EQ:
			return (scontext->role == tcontext->role);
		case CONSTRAINT_EXPR_VALUE_RELATION_INCOMP:
			return (scontext->role != tcontext->role) &&
				!ebitmap_get_bit(&srole->dominates,
						 tcontext->role - 1) &&
				!ebitmap_get_bit(&trole->dominates,
						 scontext->role - 1);
		}
		break;
	}

	return FALSE;
}


int 
security_compute_av(
		    security_id_t sourcesid,	
		    security_id_t targetsid,	
		    security_class_t targetclass, 
		    access_vector_t requested,	
		    access_vector_t * allowed,	
		    access_vector_t * decided,	
#ifdef CONFIG_FLASK_AUDIT
		    access_vector_t * auditallow, 
		    access_vector_t * auditdeny, 
#endif
#ifdef CONFIG_FLASK_NOTIFY
		    access_vector_t * notify,	
#endif
		    unsigned int *seqno)	
{
	sid_datum_t    *ssid_datum, *tsid_datum;
	ss_root_context_t *scontext = 0, *tcontext = 0;
	ss_child_context_t *child_scontext = 0, *child_tcontext = 0;
	extav_key_t     avkey;
	extav_datum_t  *avdatum;
	access_vector_t tallowed, tdecided, tauditallow, tauditdeny, tnotify;
	unsigned int    rc;


	*seqno = latest_granting;

	if (!ss_initialized) {
		*allowed = requested;
		*decided = requested;
#ifdef CONFIG_FLASK_AUDIT
		*auditallow = 0;
		*auditdeny = 0xffffffff;
#endif
#ifdef CONFIG_FLASK_NOTIFY
		*notify = 0;
#endif
		return 0;
	}
	ssid_datum = sidtab_search(&sidtab, sourcesid);
	tsid_datum = sidtab_search(&sidtab, targetsid);
	if (!ssid_datum || !tsid_datum) {
		return -EINVAL;
	}
	if (ssid_datum->context.isroot && tsid_datum->context.isroot) {
		constraint_node_t *constraint;
		avtab_key_t     avkey;
		avtab_datum_t  *avdatum;
		avdef_datum_t  *avdefdatum;
		unsigned int    l2l, h2h;

		scontext = &RCONTEXT(&ssid_datum->context);
		tcontext = &RCONTEXT(&tsid_datum->context);

		avdefdatum = avdeftab_search(policydb.avdefs, targetclass);
		if (!avdefdatum) {
			return -EINVAL;
		}

		*decided = 0xffffffff;

		switch (policydb.defallow) {
		case TE_DEFAULT_NONE:
			*allowed = 0;
			break;
		case TE_DEFAULT_ALL:
			*allowed = 0xffffffff;
			break;
		case TE_DEFAULT_SELF:
			if (scontext->type == tcontext->type)
				*allowed = 0xffffffff;
			else
				*allowed = 0;
			break;
		}

#ifdef CONFIG_FLASK_AUDIT
		switch (policydb.defauditallow) {
		case TE_DEFAULT_NONE:
			*auditallow = 0;
			break;
		case TE_DEFAULT_ALL:
			*auditallow = 0xffffffff;
			break;
		case TE_DEFAULT_SELF:
			if (scontext->type == tcontext->type)
				*auditallow = 0;
			else
				*auditallow = 0xffffffff;
			break;
		}
		switch (policydb.defauditdeny) {
		case TE_DEFAULT_NONE:
			*auditdeny = 0;
			break;
		case TE_DEFAULT_ALL:
			*auditdeny = 0xffffffff;
			break;
		case TE_DEFAULT_SELF:
			if (scontext->type == tcontext->type)
				*auditdeny = 0;
			else
				*auditdeny = 0xffffffff;
			break;
		}
#endif
#ifdef CONFIG_FLASK_NOTIFY
		switch (policydb.defnotify) {
		case TE_DEFAULT_NONE:
			*notify = 0;
			break;
		case TE_DEFAULT_ALL:
			*notify = 0xffffffff;
			break;
		case TE_DEFAULT_SELF:
			if (scontext->type == tcontext->type)
				*notify = 0;
			else
				*notify = 0xffffffff;
			break;
		}
#endif

		avkey.source_type = scontext->type;
		avkey.target_type = tcontext->type;
		avkey.target_class = targetclass;

		avdatum = avtab_search(policydb.rules, &avkey);
		if (avdatum) {
			if (avdatum->specified & AVTAB_ALLOWED)
				*allowed = avdatum->allowed;
#ifdef CONFIG_FLASK_AUDIT
			if (avdatum->specified & AVTAB_AUDITALLOW)
				*auditallow = avdatum->auditallow;
			if (avdatum->specified & AVTAB_AUDITDENY)
				*auditdeny = avdatum->auditdeny;
#endif
#ifdef CONFIG_FLASK_NOTIFY
			if (avdatum->specified & AVTAB_NOTIFY)
				*notify = avdatum->notify;
#endif
		}

		l2l = mls_level_relation(scontext->range.low,
					 tcontext->range.low);

		h2h = mls_level_relation(scontext->range.high,
					 tcontext->range.high);

		if (h2h != MLS_RELATION_EQ) {
			if (h2h != MLS_RELATION_DOM) {
				/* read(s,t) = (s.high >= t.high) = False */
				*allowed = (*allowed) & ~(avdefdatum->mlsperms.read);
			}
			if (h2h != MLS_RELATION_DOMBY) {
				/* readby(s,t) = read(t,s) = False */
				*allowed = (*allowed) & ~(avdefdatum->mlsperms.readby);
			}
		}
		if ((l2l != MLS_RELATION_DOMBY && l2l != MLS_RELATION_EQ) ||
		    ((!mls_level_eq(tcontext->range.low,
				    tcontext->range.high)) &&
		     (h2h != MLS_RELATION_DOM && h2h != MLS_RELATION_EQ))) {
			/*
			 * write(s,t) = ((s.low <= t.low = t.high) or (s.low
			 * <= t.low <= t.high <= s.high)) = False
			 */
			*allowed = (*allowed) & ~(avdefdatum->mlsperms.write);
		}
		if ((l2l != MLS_RELATION_DOM && l2l != MLS_RELATION_EQ) ||
		    ((!mls_level_eq(scontext->range.low,
				    scontext->range.high)) &&
		   (h2h != MLS_RELATION_DOMBY && h2h != MLS_RELATION_EQ))) {
			/* writeby(s,t) = write(t,s) = False */
			*allowed = (*allowed) & ~(avdefdatum->mlsperms.writeby);
		}

		constraint = avdefdatum->constraints;
		while (constraint) {
			if ((constraint->permissions & (*allowed)) &&
			    !constraint_expr_eval(scontext, tcontext, constraint->expr)) {
				*allowed = (*allowed) & ~(constraint->permissions);
			}
			constraint = constraint->next;
		}

		return 0;
	}

	*allowed = 0;
	*decided = 0xffffffff;
	avkey.source_sid = sourcesid;
	avkey.target_sid = targetsid;
	avkey.target_class = targetclass;

	avdatum = (extav_datum_t *) hashtab_search(extavtab.table,
						   (hashtab_key_t) & avkey);
	if (avdatum) {
		*allowed = avdatum->allowed;
		return 0;
	}

	if (!ssid_datum->context.isroot)
		child_scontext = &CCONTEXT(&ssid_datum->context);
	if (!tsid_datum->context.isroot)
		child_tcontext = &CCONTEXT(&tsid_datum->context);

	if ((child_tcontext && child_tcontext->parent == sourcesid) ||
	    (child_scontext && child_scontext->parent == targetsid) ||
	    (child_scontext && child_tcontext && child_scontext->parent == child_tcontext->parent)) {
		return 0;
	}

	if (!child_tcontext ||
	    (child_scontext && (child_scontext->depth >= child_tcontext->depth))) {
		rc = security_compute_av(sourcesid, child_scontext->parent,
					 targetclass,
					 requested, allowed,
					 decided,
#ifdef CONFIG_FLASK_AUDIT
					 auditallow,
					 auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
					 notify,
#endif
					 seqno);

		if (rc)
			return rc;

		rc = security_compute_av(child_scontext->parent, targetsid,
					 targetclass,
					 requested, &tallowed,
					 &tdecided,
#ifdef CONFIG_FLASK_AUDIT
					 &tauditallow,
					 &tauditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
					 &tnotify,
#endif
					 seqno);
		if (rc)
			return rc;
	} else {
		rc = security_compute_av(child_tcontext->parent, targetsid,
					 targetclass,
					 requested, allowed,
					 decided,
#ifdef CONFIG_FLASK_AUDIT
					 auditallow,
					 auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
					 notify,
#endif
					 seqno);
		if (rc)
			return rc;

		rc = security_compute_av(sourcesid, child_tcontext->parent,
					 targetclass,
					 requested, &tallowed,
					 &tdecided,
#ifdef CONFIG_FLASK_AUDIT
					 &tauditallow,
					 &tauditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
					 &tnotify,
#endif
					 seqno);
		if (rc)
			return rc;
	}

	*allowed = *allowed & tallowed;
	*decided = *decided & tdecided;
#ifdef CONFIG_FLASK_AUDIT
	*auditallow = *auditallow & tauditallow;
	*auditdeny = *auditdeny & tauditdeny;
#endif
#ifdef CONFIG_FLASK_NOTIFY
	*notify = *notify & tnotify;
#endif

	return 0;
}


#include <flask/initial_sid_to_string.h>

int 
security_sid_to_context(
			security_id_t sid,	
			security_context_t * scontext,
			unsigned int *scontext_len)
{
	ss_root_context_t *context;
	char           *scontextp;
	int             i, rc, low_eq_high;


	*scontext = 0;
	*scontext_len = 0;

	if (!ss_initialized) {
		if (sid <= OSKIT_SECINITSID_NUM) {
			char           *scontextp;

			*scontext_len = strlen(initial_sid_to_string[sid]) + 1;
			scontextp = malloc(*scontext_len);
			strcpy(scontextp, initial_sid_to_string[sid]);
			*scontext = (security_context_t) scontextp;
			return 0;
		}
		printf("sid_to_context: called before initialization\n");
		exit(1);
	}
	context = root_ancestor(sid);
	if (!context) {
		rc = -EINVAL;
		goto out;
	}
	low_eq_high = (mls_level_relation(context->range.low,
					  context->range.high)
		       == MLS_RELATION_EQ);

	/* Compute the size of the context. */
	*scontext_len += strlen(policydb.user_val_to_name[context->user - 1]) + 1;
	*scontext_len += strlen(policydb.role_val_to_name[context->role - 1]) + 1;
	*scontext_len += strlen(policydb.type_val_to_name[context->type - 1]) + 1;
	*scontext_len += strlen(policydb.sens_val_to_name[context->range.low.sens - 1]) + 1;

	for (i = 1; i <= ebitmap_length(&context->range.low.cat); i++)
		if (ebitmap_get_bit(&context->range.low.cat, i - 1))
			*scontext_len += strlen(policydb.cat_val_to_name[i - 1]) + 1;

	if (!low_eq_high) {
		*scontext_len += strlen(policydb.sens_val_to_name[context->range.high.sens - 1]) + 1;
		for (i = 1; i <= ebitmap_length(&context->range.high.cat); i++)
			if (ebitmap_get_bit(&context->range.high.cat, i - 1))
				*scontext_len += strlen(policydb.cat_val_to_name[i - 1]) + 1;
	}

	/* Allocate space for the context; caller must free this space. */
	scontextp = (char *) MALLOC(*scontext_len);
	if (!scontextp) {
		return -ENOMEM;
	}
	*scontext = (security_context_t) scontextp;

	/*
	 * Copy the user name, role name, type name and 
	 * low level into the security context.
	 */
	sprintf(scontextp, "%s:%s:%s:",
		policydb.user_val_to_name[context->user - 1],
		policydb.role_val_to_name[context->role - 1],
		policydb.type_val_to_name[context->type - 1]);
	scontextp += strlen(policydb.user_val_to_name[context->user - 1]) + 1 + strlen(policydb.role_val_to_name[context->role - 1]) + 1 + strlen(policydb.type_val_to_name[context->type - 1]) + 1;
	strcpy(scontextp, policydb.sens_val_to_name[context->range.low.sens - 1]);
	scontextp += strlen(policydb.sens_val_to_name[context->range.low.sens - 1]);
	*scontextp = ':';
	scontextp++;
	for (i = 1; i <= ebitmap_length(&context->range.low.cat); i++)
		if (ebitmap_get_bit(&context->range.low.cat, i - 1)) {
			strcpy(scontextp, policydb.cat_val_to_name[i - 1]);
			scontextp += strlen(policydb.cat_val_to_name[i - 1]);
			*scontextp = ',';
			scontextp++;
		}
	if (!low_eq_high) {
		/* Copy the high level into the security context */
		scontextp--;
		sprintf(scontextp, "-%s", policydb.sens_val_to_name[context->range.high.sens - 1]);
		scontextp += strlen(policydb.sens_val_to_name[context->range.high.sens - 1]) + 1;
		*scontextp = ':';
		scontextp++;
		for (i = 1; i <= ebitmap_length(&context->range.high.cat); i++)
			if (ebitmap_get_bit(&context->range.high.cat, i - 1)) {
				strcpy(scontextp, policydb.cat_val_to_name[i - 1]);
				scontextp += strlen(policydb.cat_val_to_name[i - 1]);
				*scontextp = ',';
				scontextp++;
			}
	}
	scontextp--;
	*scontextp = 0;

	rc = 0;

out:
	return rc;
}


int 
security_sid_to_econtext(
			 security_id_t sid,	
			 security_context_t * scontext,
			 unsigned int *scontext_len,
			 security_econtext_t * econtext,
			 unsigned int *econtext_len)
{
	ss_child_context_t *child = 0;
	sid_datum_t    *siddatum;
	char           *oldecontextp, *econtextp;
	int             rc;


	*econtext = 0;
	*econtext_len = 0;

	siddatum = sidtab_search(&sidtab, sid);
	if (!siddatum) {
		rc = -EINVAL;
		goto out;
	}
	while (!siddatum->context.isroot) {
	        /* Compute the size of the extension context. */
		child = &CCONTEXT(&siddatum->context);
		siddatum = sidtab_search(&sidtab, child->parent);
		if (!siddatum) {
			rc = -EINVAL;
			goto out;
		}
		*econtext_len += strlen(siddatum->child_val_to_name[child->value - 1]) + 1;
	}

	if (*econtext_len) {
		/*
		 * Allocate space for the extension context; 
		 * the caller must free.
		 */
		econtextp = (char *) MALLOC(*econtext_len);
		if (!econtextp) {
			rc = -ENOMEM;
			goto out;
		}
		*econtext = (security_econtext_t) econtextp;
		econtextp += *econtext_len - 1;

		/* Copy the child names into the extension context. */

		siddatum = sidtab_search(&sidtab, sid);
		if (!siddatum) {
			rc = -EINVAL;
			FREE(*econtext, *econtext_len);
			goto out;
		}
		while (!siddatum->context.isroot) {
			child = &CCONTEXT(&siddatum->context);

			siddatum = sidtab_search(&sidtab, child->parent);
			if (!siddatum) {
				rc = -EINVAL;
				FREE(*econtext, *econtext_len);
				goto out;
			}
			oldecontextp = econtextp;
			econtextp -= strlen(siddatum->child_val_to_name[child->value - 1]);
			strcpy(econtextp, siddatum->child_val_to_name[child->value - 1]);
			econtextp--;
			*oldecontextp = ':';
		}

		econtextp = *econtext + *econtext_len - 1;
		*econtextp = 0;
	} else {
		econtextp = (char *) MALLOC(1);
		if (!econtextp) {
			rc = -ENOMEM;
			goto out;
		}
		*econtextp = 0;
		*econtext = (security_econtext_t) econtextp;
	}

	if (child)
		rc = security_sid_to_context(child->parent, 
					     scontext, scontext_len);
	else
		rc = security_sid_to_context(sid, scontext, scontext_len);
	if (rc) {
		if (*econtext_len)
			FREE(*econtext, *econtext_len);
		goto out;
	}
out:
	return rc;
}


static int
userrangematch(queue_element_t e, void *datap)
{
	mls_range_t    *range1, *range2;

	range1 = (mls_range_t *) e;
	range2 = (mls_range_t *) datap;

	if (mls_range_contains(*range1, *range2))
		return 1;

	return 0;
}

int
ss_root_context_valid(ss_root_context_t * context,
		      policydb_t * p,
		      int *rc)
{
	role_datum_t   *role;
	type_datum_t   *typdatum;
	level_datum_t  *levdatum;
	user_datum_t   *usrdatum;
	unsigned int    relation;
	int             i, ret;


	*rc = -EINVAL;

	/*
	 * MLS range validity checks: High must dominate low, low level must
	 * be valid (category set <-> sensitivity check), and high level must
	 * be valid (category set <-> sensitivity check)
	 */

	relation = mls_level_relation(context->range.high,
				      context->range.low);
	if (!(relation & (MLS_RELATION_DOM | MLS_RELATION_EQ)))
		/* High does not dominate low. */
		return FALSE;

	levdatum = (level_datum_t *) hashtab_search(p->levels->table,
			  p->sens_val_to_name[context->range.low.sens - 1]);
	if (!levdatum)
		return FALSE;

	for (i = 1; i <= ebitmap_length(&context->range.low.cat); i++) {
		if (ebitmap_get_bit(&context->range.low.cat, i - 1)) {
			if (!ebitmap_get_bit(&levdatum->level->cat, i - 1))
				/*
				 * Category may not be associated with
				 * sensitivity in low level.
				 */
				return FALSE;
		}
	}

	levdatum = (level_datum_t *) hashtab_search(p->levels->table,
			 p->sens_val_to_name[context->range.high.sens - 1]);
	if (!levdatum)
		return FALSE;

	for (i = 1; i <= ebitmap_length(&context->range.high.cat); i++) {
		if (ebitmap_get_bit(&context->range.high.cat, i - 1)) {
			if (!ebitmap_get_bit(&levdatum->level->cat, i - 1))
				/*
				 * Category may not be associated with
				 * sensitivity in high level.
				 */
				return FALSE;
		}
	}

	/*
	 * Role validity checks:  Role must be authorized for the type.
	 */
	role = p->role_val_to_struct[context->role - 1];
	typdatum = (type_datum_t *) hashtab_search(p->types->table,
				    p->type_val_to_name[context->type - 1]);
	if (!typdatum)
		return FALSE;

	if (!ebitmap_get_bit(&role->types,
			     context->type - 1))
		/* role may not be associated with type */
		return FALSE;

	/*
	 * User validity checks: User must be authorized for the role
	 * and for the MLS range.
	 */
	usrdatum = p->user_val_to_struct[context->user - 1];
	if (!usrdatum)
		return FALSE;

	if (!ebitmap_get_bit(&usrdatum->roles,
			     context->role - 1))
		/* user may not be associated with role */
		return FALSE;

	ret = queue_map(usrdatum->ranges,
			userrangematch,
			&(context->range));
	if (!ret)
		/* user may not be associated with range */
		return FALSE;

	*rc = 0;

	return TRUE;
}


int 
security_context_to_sid(
			security_context_t scontext,
			unsigned int scontext_len,
			security_id_t * sid)
{
	security_context_t scontext2;
	ss_context_t    context;
	ss_root_context_t *root;
	role_datum_t   *role;
	type_datum_t   *typdatum;
	level_datum_t  *levdatum;
	cat_datum_t    *catdatum;
	user_datum_t   *usrdatum;
	char           *scontextp, *p, delim;
	int             rc;

	if (!ss_initialized) {
		int             i;

		for (i = 1; i < OSKIT_SECINITSID_NUM; i++) {
			if (!strcmp(initial_sid_to_string[i], scontext)) {
				*sid = i;
				return 0;
			}
		}

		printf("context_to_sid: called before initialization\n");
		exit(1);
	}
	*sid = SECSID_NULL;

	scontext2 = malloc(scontext_len);
	if (!scontext2) {
		return -ENOMEM;
	}
	memcpy(scontext2, scontext, scontext_len);

	context_init(&context, TRUE);
	root = &RCONTEXT(&context);
	*sid = SECSID_NULL;

	/* Parse the security context. */

	rc = -EINVAL;
	scontextp = (char *) scontext2;
	if (scontextp[scontext_len - 1])
		/* Security context is not null-terminated. */
		goto out;

	/* Extract the user. */
	p = scontextp;
	while (*p && *p != ':')
		p++;

	if (*p == 0)
		goto out;

	*p++ = 0;

	usrdatum = (user_datum_t *) hashtab_search(policydb.users->table,
						 (hashtab_key_t) scontextp);
	if (!usrdatum)
		goto out;

	root->user = usrdatum->value;

	/* Extract role. */
	scontextp = p;
	while (*p && *p != ':')
		p++;

	if (*p == 0)
		goto out;

	*p++ = 0;

	role = (role_datum_t *) hashtab_search(policydb.roles->table,
					       (hashtab_key_t) scontextp);
	if (!role)
		goto out;
	root->role = role->value;

	/* Extract type. */
	scontextp = p;
	while (*p && *p != ':')
		p++;

	if (*p == 0)
		goto out;

	*p++ = 0;

	typdatum = (type_datum_t *) hashtab_search(policydb.types->table,
						 (hashtab_key_t) scontextp);

	if (!typdatum)
		goto out;

	root->type = typdatum->info->value;

	/* Extract low sensitivity. */
	scontextp = p;
	while (*p && *p != ':' && *p != '-')
		p++;

	delim = *p;
	if (delim != 0)
		*p++ = 0;

	levdatum = (level_datum_t *) hashtab_search(policydb.levels->table,
						 (hashtab_key_t) scontextp);

	if (!levdatum)
		goto out;

	root->range.low.sens = levdatum->level->sens;

	if (delim == ':') {
		/* Extract low category set. */
		while (1) {
			scontextp = p;
			while (*p && *p != ',' && *p != '-')
				p++;

			delim = *p;
			if (delim != 0)
				*p++ = 0;

			catdatum = (cat_datum_t *) hashtab_search(policydb.cats->table,
						 (hashtab_key_t) scontextp);

			if (!catdatum)
				goto out;

			if (!ebitmap_set_bit(&root->range.low.cat,
					     catdatum->value - 1, TRUE)) {
				rc = -ENOMEM;
				goto out;
			}
			if (delim != ',')
				break;
		}
	}
	if (delim == '-') {
		/* Extract high sensitivity. */
		scontextp = p;
		while (*p && *p != ':')
			p++;

		delim = *p;
		if (delim != 0)
			*p++ = 0;

		levdatum = (level_datum_t *) hashtab_search(policydb.levels->table,
						 (hashtab_key_t) scontextp);
		if (!levdatum)
			goto out;

		root->range.high.sens = levdatum->level->sens;

		if (delim == ':') {
			/* Extract high category set. */
			while (1) {
				scontextp = p;
				while (*p && *p != ',')
					p++;

				delim = *p;
				if (delim != 0)
					*p++ = 0;

				catdatum = (cat_datum_t *) hashtab_search(policydb.cats->table,
						 (hashtab_key_t) scontextp);
				if (!catdatum)
					goto out;

				if (!ebitmap_set_bit(&root->range.high.cat,
					       catdatum->value - 1, TRUE)) {
					rc = -ENOMEM;
					goto out;
				}
				if (delim != ',')
					break;
			}
		}
	} else {
		root->range.high.sens = root->range.low.sens;
		ebitmap_cpy(&root->range.high.cat, &root->range.low.cat);
	}

	/* Check the validity of the new context. */
	if (!ss_root_context_valid(root, &policydb, &rc))
		goto out;

	/* Obtain the new sid. */
	if (sidtab_context_to_sid(&sidtab, &context, sid)) {
		rc = -ENOMEM;
		goto out;
	}
	rc = 0;

out:
	context_destroy(&context);
	free(scontext2);
	return rc;
}


int 
security_econtext_to_sid(
			 security_context_t scontext,	
			 unsigned int scontext_len,	
			 security_econtext_t econtext,	
			 unsigned int econtext_len,	
			 security_id_t * sid)
{
	security_econtext_t econtext2;
	sid_datum_t    *siddatum;
	child_datum_t  *childdatum;
	char           *econtextp, *p, delim;
	int             rc;

	rc = security_context_to_sid(scontext, scontext_len, sid);
	if (rc)
		return rc;

	econtext2 = malloc(econtext_len);
	if (!econtext2) {
		return -ENOMEM;
	}
	memcpy(econtext2, econtext, econtext_len);

	if (econtext_len && econtext[0]) {
		rc = -EINVAL;
		siddatum = sidtab_search(&sidtab, *sid);
		if (!siddatum)
			goto out;

		econtextp = (char *) econtext2;
		if (econtextp[econtext_len - 1])
			/* Extension context is not null-terminated. */
			goto out;

		p = econtextp;
		while (1) {
			econtextp = p;
			while (*p && *p != ':')
				p++;

			delim = *p;
			if (delim != 0)
				*p++ = 0;

			childdatum = (child_datum_t *)
				hashtab_search(siddatum->children.table,
					       (hashtab_key_t) econtextp);
			if (!childdatum) {
				goto out;
			}
			*sid = childdatum->sid;

			siddatum = sidtab_search(&sidtab, *sid);
			if (!siddatum) {
				goto out;
			}
			if (delim != ':')
				break;
		}
	}
	rc = 0;

out:
	free(econtext2);
	return rc;
}


int 
security_transition_sid(
			security_id_t ssid,	
			security_id_t tsid,	
			security_class_t tclass,
			security_id_t * out_sid)
{
	ss_context_t    trancontext;
	ss_root_context_t *scontext, *tcontext, *root;
	tr_key_t        trkey;
	tr_datum_t     *trdatum;
	exttr_key_t     exttrkey;
	exttr_datum_t  *exttrdatum;
	int             rc = 0;


	if (!ss_initialized) {
		switch (tclass) {
		case OSKIT_SECCLASS_PROCESS:
			*out_sid = ssid;
			break;
		case OSKIT_SECCLASS_SOCK_FILE:
		case OSKIT_SECCLASS_LNK_FILE:
		case OSKIT_SECCLASS_FILE:
		case OSKIT_SECCLASS_BLK_FILE:
		case OSKIT_SECCLASS_DIR:
		case OSKIT_SECCLASS_CHR_FILE:
		case OSKIT_SECCLASS_FIFO_FILE:
			*out_sid = tsid;
			break;
		default:
			*out_sid = ssid;
			break;
		}
		return 0;
	}

	exttrkey.source_sid = ssid;
	exttrkey.target_sid = tsid;
	exttrkey.object_class = tclass;
	exttrdatum = (exttr_datum_t *) hashtab_search(exttrtab.table,
						(hashtab_key_t) & exttrkey);
	if (exttrdatum) {
		*out_sid = *exttrdatum;
		goto out;
	}

	scontext = root_ancestor(ssid);
	if (!scontext) {
		rc = -EINVAL;
		goto out;
	}
	tcontext = root_ancestor(tsid);
	if (!tcontext) {
		rc = -EINVAL;
		goto out;
	}
	trkey.source_type = scontext->type;
	trkey.target_type = tcontext->type;
	trkey.object_class = tclass;
	trdatum = (tr_datum_t *) hashtab_search(policydb.transitions->table,
						(hashtab_key_t) & trkey);

	context_init(&trancontext, TRUE);
	root = &RCONTEXT(&trancontext);
	switch (tclass) {
	case OSKIT_SECCLASS_PROCESS:
		/* Inherit attribute default values from the process. */
		if (!trdatum) {
			/* No change in process SID */
			*out_sid = ssid;
			goto out;
		}
		if (!root_context_cpy(root, scontext)) {
			rc = -ENOMEM;
			goto out;
		}
		break;
	case OSKIT_SECCLASS_SOCK_FILE:
	case OSKIT_SECCLASS_LNK_FILE:
	case OSKIT_SECCLASS_FILE:
	case OSKIT_SECCLASS_BLK_FILE:
	case OSKIT_SECCLASS_DIR:
	case OSKIT_SECCLASS_CHR_FILE:
	case OSKIT_SECCLASS_FIFO_FILE:
		/*
		 * Inherit most attribute default values from the parent
		 * directory.
		 */
		if (!trdatum && (scontext->user == tcontext->user)) {
			/* Simply use the parent directory SID. */
			*out_sid = tsid;
			goto out;
		}
		if (!root_context_cpy(root, tcontext)) {
			rc = -ENOMEM;
			goto out;
		}
		/* Inherit the user identity from the creating process. */
		root->user = scontext->user;
		break;
	default:
		rc = -EINVAL;
		goto out;
	}

	if (trdatum) {
		/* Inherit type value from the type transition rule. */
		root->type = trdatum->new_type;
	}
	/* Check the validity of the transition context. */
	if (!ss_root_context_valid(root, &policydb, &rc)) {
		context_destroy(&trancontext);
		/*
		 * The transition context is invalid.  Fall back to
		 * one of the existing SIDs based on the security class
		 * involved.  
		 */
		printf("security_transition_sid(%ld,%ld,%d):  transition yielded an invalid security context\n", ssid, tsid, tclass);
		switch (tclass) {
		case OSKIT_SECCLASS_PROCESS:
			*out_sid = ssid;
			break;
		case OSKIT_SECCLASS_SOCK_FILE:
		case OSKIT_SECCLASS_LNK_FILE:
		case OSKIT_SECCLASS_FILE:
		case OSKIT_SECCLASS_BLK_FILE:
		case OSKIT_SECCLASS_DIR:
		case OSKIT_SECCLASS_CHR_FILE:
		case OSKIT_SECCLASS_FIFO_FILE:
			*out_sid = tsid;
			break;
		default:
			*out_sid = ssid;
			break;
		}
		rc = 0;
		goto out;
	}
	/* Obtain the sid for the transition context. */
	if (sidtab_context_to_sid(&sidtab, &trancontext, out_sid)) {
		context_destroy(&trancontext);
		rc = -ENOMEM;
		goto out;
	}
	context_destroy(&trancontext);
	goto out;

out:
	return rc;
}


int
security_member_sid(security_id_t ssid,	
		    security_id_t tsid,	
		    security_class_t tclass,
		    security_id_t * out_sid)
{
	ss_root_context_t *scontext, *tcontext;
	ss_context_t    newcontext;
	ss_root_context_t *root;
	int             rc;


	if (!ss_initialized) {
		*out_sid = ssid;
		return 0;
	}
	scontext = root_ancestor(ssid);
	if (!scontext) {
		rc = -EINVAL;
		goto out;
	}
	tcontext = root_ancestor(tsid);
	if (!tcontext) {
		rc = -EINVAL;
		goto out;
	}
	/*
	 * Define a new security context, with the MLS range inherited from
	 * the source SID, but all other attributes inherited from the target
	 * SID.
	 */
	context_init(&newcontext, TRUE);
	root = &RCONTEXT(&newcontext);

	root->range.low.sens = scontext->range.low.sens;
	if (!ebitmap_cpy(&root->range.low.cat, &scontext->range.low.cat)) {
		rc = -ENOMEM;
		goto out;
	}
	root->range.high.sens = scontext->range.high.sens;
	if (!ebitmap_cpy(&root->range.high.cat, &root->range.high.cat)) {
		ebitmap_destroy(&root->range.low.cat);
		rc = -ENOMEM;
		goto out;
	}
	root->role = tcontext->role;
	root->type = tcontext->type;
	root->user = tcontext->user;

	/* Check the validity of the new context. */
	if (!ss_root_context_valid(root, &policydb, &rc)) {
		context_destroy(&newcontext);
		goto out;
	}
	/* Obtain the sid for the new context. */
	if (sidtab_context_to_sid(&sidtab, &newcontext, out_sid)) {
		context_destroy(&newcontext);
		rc = -ENOMEM;
		goto out;
	}
	context_destroy(&newcontext);

	rc = 0;

out:
	return rc;
}


static int
validate_perm(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	hashtab_t       h;
	perm_datum_t   *perdatum, *perdatum2;


	h = (hashtab_t) p;
	perdatum = (perm_datum_t *) datum;

	perdatum2 = (perm_datum_t *) hashtab_search(h, key);
	if (!perdatum2) {
		printf("ss:  permission %s disappeared", key);
		return -1;
	}
	if (perdatum->value != perdatum2->value) {
		printf("ss:  the value of permission %s changed", key);
		return -1;
	}
	return 0;
}

static int
validate_class(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	policydb_t     *newp;
	class_datum_t  *cladatum, *cladatum2;

	newp = (policydb_t *) p;
	cladatum = (class_datum_t *) datum;

	cladatum2 = (class_datum_t *) hashtab_search(newp->classes->table, key);
	if (!cladatum2) {
		printf("ss:  class %s disappeared\n", key);
		return -1;
	}
	if (cladatum->value != cladatum2->value) {
		printf("ss:  the value of class %s changed\n", key);
		return -1;
	}
	return 0;
}


static int
validate_avdef(unsigned int key, avdef_datum_t *avddatum, void *p)
{
	policydb_t     *newp;
	avdef_datum_t  *avddatum2;

	newp = (policydb_t *) p;

	avddatum2 = avdeftab_search(newp->avdefs, key);
	if (!avddatum2) {
		printf("ss:  access vector definition for %d disappeared\n", key);
		return -1;
	}
	if ((avddatum->common && !avddatum2->common) ||
	    (!avddatum->common && avddatum2->common)) {
		printf("ss:  the inherits clause for the access vector definition for %d changed\n", key);
		return -1;
	}
	if (avddatum->common) {
		if (hashtab_map(avddatum->common->table, validate_perm,
				avddatum2->common->table)) {
			printf(" in access vector definition (%d)\n", key);
			return -1;
		}
	}
	if (hashtab_map(avddatum->private.table, validate_perm,
			avddatum2->private.table)) {
		printf(" in access vector definition (%d)\n", key);
		return -1;
	}
	return 0;
}


typedef struct {
	policydb_t     *oldp;
	policydb_t     *newp;
}               root_context_convert_args_t;

static int
root_context_convert(security_id_t key,
		     sid_datum_t *siddatum,
		     void *p)
{
	root_context_convert_args_t *args;
	ss_root_context_t *root;
	role_datum_t   *role;
	type_datum_t   *typdatum;
	level_datum_t  *levdatum;
	cat_datum_t    *catdatum;
	user_datum_t   *usrdatum;
	ebitmap_t       bitmap;
	int             i, rc;


	args = (root_context_convert_args_t *) p;
	ebitmap_init(&bitmap);

	if (!siddatum->context.isroot)
		return 0;

	root = &RCONTEXT(&siddatum->context);

	rc = -EINVAL;

	/* Convert the user. */
	usrdatum = (user_datum_t *) hashtab_search(args->newp->users->table,
			      args->oldp->user_val_to_name[root->user - 1]);

	if (!usrdatum) {
		goto bad;
	}
	root->user = usrdatum->value;

	/* Convert the role. */
	role = (role_datum_t *) hashtab_search(args->newp->roles->table,
			      args->oldp->role_val_to_name[root->role - 1]);
	root->role = role->value;

	/* Convert the type. */
	typdatum = (type_datum_t *)
		hashtab_search(args->newp->types->table,
			       args->oldp->type_val_to_name[root->type - 1]);
	if (!typdatum) {
		goto bad;
	}
	root->type = typdatum->info->value;

	/* Convert the low sensitivity. */
	levdatum = (level_datum_t *) hashtab_search(args->newp->levels->table,
		    args->oldp->sens_val_to_name[root->range.low.sens - 1]);

	if (!levdatum) {
		goto bad;
	}
	root->range.low.sens = levdatum->level->sens;

	/* Convert the low category set. */
	for (i = 1; i <= ebitmap_length(&root->range.low.cat); i++) {
		if (ebitmap_get_bit(&root->range.low.cat, i - 1)) {
			catdatum = (cat_datum_t *) hashtab_search(args->newp->cats->table,
					args->oldp->cat_val_to_name[i - 1]);

			if (!catdatum) {
				goto bad;
			}
			if (!ebitmap_set_bit(&bitmap, catdatum->value - 1, TRUE)) {
				rc = -ENOMEM;
				goto bad;
			}
		}
	}
	ebitmap_destroy(&root->range.low.cat);
	root->range.low.cat = bitmap;

	/* Convert the high sensitivity. */
	levdatum = (level_datum_t *) hashtab_search(args->newp->levels->table,
		   args->oldp->sens_val_to_name[root->range.high.sens - 1]);

	if (!levdatum) {
		goto bad;
	}
	root->range.high.sens = levdatum->level->sens;

	/* Convert the high category set. */
	ebitmap_init(&bitmap);
	for (i = 1; i <= ebitmap_length(&root->range.high.cat); i++) {
		if (ebitmap_get_bit(&root->range.high.cat, i - 1)) {
			catdatum = (cat_datum_t *) hashtab_search(args->newp->cats->table,
					args->oldp->cat_val_to_name[i - 1]);

			if (!catdatum) {
				goto bad;
			}
			if (!ebitmap_set_bit(&bitmap, catdatum->value - 1, TRUE)) {
				rc = -ENOMEM;
				goto bad;
			}
		}
	}
	ebitmap_destroy(&root->range.high.cat);
	root->range.high.cat = bitmap;

	/* Check the validity of the new context. */
	if (!ss_root_context_valid(root, args->newp, &rc))
		goto bad;

	return 0;

bad:
	ebitmap_destroy(&bitmap);

#if VERBOSE
	printf("ss:  Warning!  Invalidating SID %ld", key);
	switch (rc) {
	case -EINVAL:
		printf(" (context is no longer valid)\n");
		break;
	case -ENOMEM:
		printf(" (out of memory)\n");
		break;
	}
#endif				/* VERBOSE */

	return -1;
}


static void
destroy_extav(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	extav_key_t    *avkey;

	avkey = (extav_key_t *) key;

#if VERBOSE
	printf("ss:  removing permission rule for source (%ld) target (%ld, %d)\n",
	       avkey->source_sid, avkey->target_sid,
	       avkey->target_class);
#endif				/* VERBOSE */

	FREE(key, sizeof(extav_key_t));
	FREE(datum, sizeof(extav_datum_t));
}


static int
root_update_extav(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	extav_key_t    *avkey;
	extav_datum_t  *avdatum;
	access_vector_t allowed, decided, auditallow, auditdeny, notify;
	int             rc;
	unsigned int    seqno;

	avkey = (extav_key_t *) key;
	avdatum = (extav_datum_t *) datum;

	rc = security_compute_av(avkey->source_sid,
				 avkey->target_sid,
				 avkey->target_class,
				 avdatum->allowed,
				 &allowed,
				 &decided,
#ifdef CONFIG_FLASK_AUDIT
				 &auditallow,
				 &auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
				 &notify,
#endif
				 &seqno);
	if (rc) {
		return -1;
	}
#if VERBOSE
	if ((avdatum->allowed & allowed) != avdatum->allowed) {
		printf("ss:  reducing permissions for source (%ld) target (%ld,%d) from 0x%lx to 0x%lx\n",
		       avkey->source_sid,
		       avkey->target_sid,
		       avkey->target_class,
		       avdatum->allowed,
		       (avdatum->allowed & allowed));
	}
#endif				/* VERBOSE */

	avdatum->allowed &= allowed;
	return 0;
}


static void
destroy_exttr(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	exttr_key_t    *trkey;
	exttr_datum_t  *trdatum;

	trkey = (exttr_key_t *) key;
	trdatum = (exttr_datum_t *) datum;

#if VERBOSE
	printf("ss:  removing transition rule from %ld on %ld to %ld\n",
	       trkey->source_sid, trkey->target_sid, *trdatum);
#endif				/* VERBOSE */

	FREE(key, sizeof(exttr_key_t));
	FREE(datum, sizeof(exttr_datum_t));
}


static int
update_exttr(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	sid_datum_t    *sid_datum;
	exttr_key_t    *trkey;
	exttr_datum_t  *trdatum;

	trkey = (exttr_key_t *) key;
	trdatum = (exttr_datum_t *) datum;

	sid_datum = sidtab_search(&sidtab, trkey->source_sid);
	if (!sid_datum) {
		return -1;
	}
	sid_datum = sidtab_search(&sidtab, trkey->target_sid);
	if (!sid_datum) {
		return -1;
	}
	sid_datum = sidtab_search(&sidtab, *trdatum);
	if (!sid_datum) {
		return -1;
	}
	return 0;
}


extern char    *policyfile;
#ifdef __KERNEL__
#define yyparse   oskit_security_yyparse
#define yyrestart oskit_security_yyrestart
#endif
extern int      yyparse(void);
extern void     yyrestart(FILE *);

int
security_load_policy(FILE *fp)
{
	policydb_t      newpolicydb;
	root_context_convert_args_t args;
	char           *file;
	int             rc;

	printf("ss:  loading policy configuration\n");

	yyrestart(fp);
	if (policydb_init(&newpolicydb)) {
		printf("ss:  out of memory\n");
		policydb_destroy(&newpolicydb);
		rc = -ENOMEM;
		goto out;
	}
	id_queue = queue_create();
	if (!id_queue) {
		printf("ss:  out of memory\n");
		policydb_destroy(&newpolicydb);
		rc = -ENOMEM;
		goto out;
	}
	policydbp = &newpolicydb;
	policydb_errors = 0;
	if (yyparse() || policydb_errors) {
		printf("ss:  error(s) encountered while parsing configuration\n");
		queue_destroy(id_queue);
		policydb_destroy(&newpolicydb);
		rc = -EINVAL;
		goto out;
	}
	queue_destroy(id_queue);

	if (policydb_transform(&newpolicydb)) {
		printf("ss:  out of memory\n");
		policydb_destroy(&newpolicydb);
		rc = -ENOMEM;
		goto out;
	}

	printf("ss:  %d users, %d roles, %d types, %d levels, %d transitions\n",
	       newpolicydb.users->nprim, newpolicydb.roles->nprim,
	       newpolicydb.types->nprim, newpolicydb.nlevels,
	       newpolicydb.transitions->nel);

	printf("ss:  %d access vectors (%d constrained), %d access rules (%d constrained)\n",
	       newpolicydb.avdefs->nel, newpolicydb.avdefs->ncons,
	       newpolicydb.rules->nel, newpolicydb.rules->ncons);

	avtab_hash_eval(newpolicydb.rules, "ss", "rules");

	/* Verify that the existing classes did not change. */
	if (hashtab_map(policydb.classes->table, validate_class, &newpolicydb)) {
		printf("ss:  the definition of an existing class changed\n");
		policydb_destroy(&newpolicydb);
		rc = -EINVAL;
		goto out;
	}
	/* Verify that the existing permissions did not change. */
	if (avdeftab_map(policydb.avdefs, validate_avdef, &newpolicydb)) {
		printf("ss:  the definition of an existing permission changed\n");
		policydb_destroy(&newpolicydb);
		rc = -EINVAL;
		goto out;
	}
	/* Convert the internal representations of contexts. */
	args.oldp = &policydb;
	args.newp = &newpolicydb;
	sidtab_map_remove_on_error(&sidtab,
				   root_context_convert,
				   &args);

	/* Copy the new policy into place. */
	policydb_destroy(&policydb);
	policydb = newpolicydb;

	/* Propagate changes to the extension access control policy */
	hashtab_map_remove_on_error(extavtab.table, root_update_extav,
				    destroy_extav, 0);
	hashtab_map_remove_on_error(exttrtab.table, update_exttr,
				    destroy_exttr, 0);

#ifdef __KERNEL__
	printf("ss:  resetting access vector caches\n");
	avc_ss_reset(++latest_granting);
#endif

	rc = 0;

out:
	return rc;
}


static int
child_ancestor(parent_info_t * parent,
	       security_id_t sid,
	       security_id_t * child_sid)
{
	sid_datum_t    *sid_datum;
	ss_child_context_t *child;


	sid_datum = sidtab_search(&sidtab, sid);
	while (sid_datum && !sid_datum->context.isroot) {
		child = &CCONTEXT(&sid_datum->context);

		if (child->parent == parent->sid) {
			*child_sid = sid;
			return 0;
		}
		if (!parent->datum->context.isroot &&
		  child->depth <= CCONTEXT(&parent->datum->context).depth) {
			*child_sid = SECSID_NULL;
			return 0;
		}
		sid = child->parent;
		sid_datum = sidtab_search(&sidtab, sid);
	}

	if (!sid_datum) {
		*child_sid = SECSID_NULL;
		return -1;
	}
	*child_sid = SECSID_NULL;
	return 0;
}


static int
ext_update_extav(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	extav_key_t    *avkey;
	extav_datum_t  *avdatum;
	parent_info_t  *parent;
	security_id_t   child_ssid, child_tsid;
	access_vector_t allowed, decided, auditallow, auditdeny, notify;
	int             rc, seqno;


	avkey = (extav_key_t *) key;
	avdatum = (extav_datum_t *) datum;
	parent = (parent_info_t *) p;

	if (child_ancestor(parent, avkey->source_sid, &child_ssid))
		return -1;

	if (child_ancestor(parent, avkey->target_sid, &child_tsid))
		return -1;

	if (child_ssid == SECSID_NULL && child_tsid == SECSID_NULL) {
		/*
		 * Neither the source nor the target are descendants of
		 * parent. Hence, any changes to the extension policy for
		 * this parent are irrelevant to this rule.
		 */
		return 0;
	}
	if ((child_ssid == avkey->source_sid && avkey->target_sid == parent->sid) ||
	    (child_tsid == avkey->target_sid && avkey->source_sid == parent->sid) ||
	    (child_ssid == avkey->source_sid && child_tsid == avkey->target_sid)) {
		/*
		 * Child<->parent, parent<->child, and child<->child rules
		 * are limited by the parent<->parent rule.  However, the
		 * parent<->parent rule could not have changed during a
		 * security_load_extension on the parent, so these rules are
		 * fine.
		 */
		return 0;
	}
	if (child_ssid != SECSID_NULL && child_tsid != SECSID_NULL) {
		/*
		 * Both source and target are descendants of parent. Ensure
		 * that the rule is limited by the rule for the child
		 * ancestors of the source and target SIDs.
		 */
		rc = security_compute_av(child_ssid,
					 child_tsid,
					 avkey->target_class,
					 avdatum->allowed,
					 &allowed,
					 &decided,
#ifdef CONFIG_FLASK_AUDIT
					 &auditallow,
					 &auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
					 &notify,
#endif
					 &seqno);
	} else if (child_ssid == SECSID_NULL && child_tsid != SECSID_NULL) {
		/*
		 * Only the target is a descendant of parent.  Ensure that
		 * the rule is limited by the rule between then parent and
		 * child ancestor of the target SID.
		 */
		rc = security_compute_av(parent->sid,
					 child_tsid,
					 avkey->target_class,
					 avdatum->allowed,
					 &allowed,
					 &decided,
#ifdef CONFIG_FLASK_AUDIT
					 &auditallow,
					 &auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
					 &notify,
#endif
					 &seqno);
	} else {
		/*
		 * Only the source is a descendant of parent.  Ensure that
		 * the rule is limited by the rule between the child ancestor
		 * of the source SID and the parent.
		 */
		rc = security_compute_av(child_ssid,
					 parent->sid,
					 avkey->target_class,
					 avdatum->allowed,
					 &allowed,
					 &decided,
#ifdef CONFIG_FLASK_AUDIT
					 &auditallow,
					 &auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
					 &notify,
#endif
					 &seqno);
	}

	if (rc) {
		return -1;
	}
#if VERBOSE
	if ((avdatum->allowed & allowed) != avdatum->allowed) {
		printf("ss:  reducing permissions for source (%ld) target (%ld,%d) from 0x%lx to 0x%lx\n",
		       avkey->source_sid,
		       avkey->target_sid,
		       avkey->target_class,
		       avdatum->allowed,
		       (avdatum->allowed & allowed));
	}
#endif				/* VERBOSE */

	avdatum->allowed &= allowed;

	return 0;
}

extern int      extparse(void);
extern void     extrestart(FILE *);

int
security_load_extension(security_id_t parent,	
			FILE *fp)
{
	int             rc;

	parent_info.sid = parent;
	parent_info.datum = sidtab_search(&sidtab, parent);
	if (!parent_info.datum) {
		rc = -EINVAL;
		goto out;
	}
	printf("ss:  loading policy extension\n");

	extrestart(fp);

	id_queue = queue_create();
	if (!id_queue) {
		printf("ss:  out of memory\n");
		rc = -ENOMEM;
		goto out;
	}
	policydbp = &policydb;
	extension_errors = 0;
	if (extparse() || extension_errors) {
		printf("ss:  error(s) encountered while parsing extension\n");
		queue_destroy(id_queue);
		rc = -EINVAL;
		goto out;
	}
	queue_destroy(id_queue);

	/*
	 * Propagate changes to extension access control policy rules which
	 * involve descendants of any child of the parent.
	 */
	hashtab_map_remove_on_error(extavtab.table, ext_update_extav,
				    destroy_extav, &parent_info);
	hashtab_map_remove_on_error(exttrtab.table, update_exttr,
				    destroy_exttr, 0);

#ifdef __KERNEL__
	printf("ss:  resetting access vector caches\n");
	avc_ss_reset(++latest_granting);
#endif

	rc = 0;

out:
	return rc;
}

/* FLASK */
