/* Modified from FreeBSD 2.1 sys/i386/boot/netboot */
/***********************************************************************

Remote Procedure Call Support Routines

Author: Martin Renters
  Date: Oct/1994

***********************************************************************/

#include <oskit/c/arpa/inet.h>
#include <oskit/c/string.h>
#include <oskit/c/stdio.h>

#include "netboot.h"
#include "driver.h"
#include "ether.h"
#include "udp.h"
#include "await.h"
#include "timer.h"

#include "rpc.h"

#undef min
#define min(a,b)	((a) < (b)? (a) : (b))

#define BUFMAX 16

struct rpc_buf read_cache[BUFMAX];
int read_cache_size = 0;

int rpc_id;

void
rpc_init()
{
	int i;
	rpc_id = currticks();
	for (i = 0; i < BUFMAX; i++) {
		read_cache[i].valid = 0;
		read_cache[i].offset = 0;
		read_cache[i].len = 0;
	}
}

/***************************************************************************

RPC_LOOKUP:  Lookup RPC Port numbers

***************************************************************************/
int
rpc_lookup(int addr, int prog, int ver)
{
	struct rpc_t buf, *rpc;
	char *rpcptr;
	int retries = MAX_RPC_RETRIES;
	rpcptr = netsprintf(buf.u.data,"%L%L%L%L%L%L%L%L%L%L%L%L%L%L",
		rpc_id, MSG_CALL, 2, PROG_PORTMAP, 2, PORTMAP_LOOKUP,
		0, 0, 0, 0, prog, ver, IP_UDP, 0);
	while(retries--) {
		udp_transmit(arptable[addr].ipaddr, RPC_SOCKET,
			SUNRPC, rpcptr - (char *)&buf, &buf);
		if (await_reply(AWAIT_RPC, rpc_id, NULL)) {
			rpc = (struct rpc_t *)&packet[ETHER_HDR_SIZE];
			if (rpc->u.reply.rstatus == rpc->u.reply.verifier ==
				rpc->u.reply.astatus == 0)
				return(ntohl(rpc->u.reply.data[0]));
			else {
				printf("%s\n", rpc_strerror(rpc));
				return(-1);
			}
		}
	}
	return(-1);
}

/***************************************************************************

NFS_MOUNT:  Mount an NFS Filesystem

***************************************************************************/
int
nfs_mount(int server, int port, char *path, char *fh)
{
	struct	rpc_t buf, *rpc;
	char	*rpcptr;
	int retries = MAX_RPC_RETRIES;
	rpcptr = netsprintf(buf.u.data,"%L%L%L%L%L%L%L%L%L%S%L%L%L%L%L%L%L%S",
		rpc_id, MSG_CALL, 2, PROG_MOUNT, 1, MOUNT_ADDENTRY,
		1, hostnamelen + 28,0,hostname,0,0,2,0,0,0,0,
		path);
	while(retries--) {
		udp_transmit(arptable[server].ipaddr, RPC_SOCKET,
			port, rpcptr - (char *)&buf, &buf);
		if (await_reply(AWAIT_RPC, rpc_id, NULL)) {
			rpc = (struct rpc_t *)&packet[ETHER_HDR_SIZE];
			if (rpc->u.reply.rstatus || rpc->u.reply.verifier ||
				rpc->u.reply.astatus || rpc->u.reply.data[0]) {
				printf("%s\n", rpc_strerror(rpc));
				return(-(ntohl(rpc->u.reply.data[0])));
			} else {
				memcpy(fh, &rpc->u.reply.data[1], 32);
				return(0);
			}
		}
	}
	return(-1);
}


/***************************************************************************

NFS_LOOKUP:  Lookup Pathname

***************************************************************************/
int
nfs_lookup(int server, int port, char *fh, char *path, char *file_fh)
{
	struct	rpc_t buf, *rpc;
	char	*rpcptr;
	int retries = MAX_RPC_RETRIES;
	rpcptr = netsprintf(buf.u.data,"%L%L%L%L%L%L%L%L%L%S%L%L%L%L%L%L%L%M%S",
		rpc_id, MSG_CALL, 2, PROG_NFS, 2, NFS_LOOKUP,
		1, hostnamelen + 28,0,hostname,0,0,2,0,0,0,0,
		32, fh, path);
	while(retries--) {
		udp_transmit(arptable[server].ipaddr, RPC_SOCKET,
			port, rpcptr - (char *)&buf, &buf);
		if (await_reply(AWAIT_RPC, rpc_id, NULL)) {
			rpc = (struct rpc_t *)&packet[ETHER_HDR_SIZE];
			if (rpc->u.reply.rstatus || rpc->u.reply.verifier ||
				rpc->u.reply.astatus || rpc->u.reply.data[0]) {
				printf("%s\n", rpc_strerror(rpc));
				return(-(ntohl(rpc->u.reply.data[0])));
			} else {
				memcpy(file_fh, &rpc->u.reply.data[1], 32);
				return(0);
			}
		}
	}
	return(-1);
}

/***************************************************************************

CHECK_READ_CACHE:  Determine if a block is in the read cache

***************************************************************************/

int
check_read_cache(int offset, int len, char *buffer)
{
	int i;
	struct rpc_buf *cache;
	int rlen = 0;

	for (i = 0; i < BUFMAX; i++) {
		cache = &(read_cache[i]);
		if (cache->valid &&
		    cache->offset == offset &&
		    cache->len    >= len) {
			memcpy(buffer, cache->buffer, len);
			rlen = len;
		}
	}
	return rlen;
}

/***************************************************************************

NFS_READ:  Read File

***************************************************************************/
int
nfs_read(int server, int port, char *fh, int offset, int len, char *buffer,
	 int readahead)
{
	struct	rpc_t buf, *rpc;
	char	*rpcptr;
	int	retries = MAX_RPC_RETRIES;
	int	rlen;
	int     i, curoff, pktcount = 0;
	int     saved_rpc_id;
	int     rid;
	struct rpc_buf *cache;

	rlen = check_read_cache(offset, len, buffer);
	if (rlen) {
		return rlen;
	}

	/*
	 * Since the cache is purely readahead, fill 'er up if we
	 * miss.  It's not going to be as efficient as it could
	 * in the face of packet loss, but we're not doing
	 * this in a particularly smart way anyway. :)
	 */

	readahead = min(readahead, BUFMAX);
	saved_rpc_id = rpc_id;

	if (readahead < 1) readahead = 1;

	curoff = offset;
	while (readahead) {
		cache = &(read_cache[rpc_id - saved_rpc_id]);
		cache->valid = 0;
		cache->offset = curoff;
		cache->len = len;
		cache->id  = rpc_id;

		rpcptr = netsprintf(buf.u.data,
		        "%L%L%L%L%L%L%L%L%L%S%L%L%L%L%L%L%L%M%L%L%L",
			rpc_id, MSG_CALL, 2, PROG_NFS, 2, NFS_READ,
		        1, hostnamelen + 28,0,hostname,0,0,2,0,0,0,0,
		        32, fh, curoff, len, 0);
		curoff += len;
		readahead--;
		pktcount++;
		rpc_id++;
		udp_transmit(arptable[server].ipaddr, RPC_SOCKET,
			port, rpcptr - (char *)&buf, &buf);
	}

	while (pktcount) {
		if (await_reply(AWAIT_RPC, 0, NULL)) {
			rpc = (struct rpc_t *)&packet[ETHER_HDR_SIZE];
			if (rpc->u.reply.rstatus || rpc->u.reply.verifier ||
				rpc->u.reply.astatus || rpc->u.reply.data[0]) {
				printf("%s\n", rpc_strerror(rpc));
				return(-(ntohl(rpc->u.reply.data[0])));
			} else {
				rid = ntohl(rpc->u.reply.id);
				cache = NULL;
				for (i = 0; i < BUFMAX; i++) {
					if (read_cache[i].id == rid) {
						cache = &(read_cache[i]);
					}
				}
				if (cache == NULL) {
#if 0
					printf("unexpected reply id 0x%lx\n",
					       rid);
#endif
					continue;
				}
				rlen = ntohl(rpc->u.reply.data[18]);
				if (len < rlen) rlen = len;
				if (len > rlen) netprintf("short read\r\n");
				memcpy(cache->buffer, &rpc->u.reply.data[19],
				       rlen);
				cache->valid = 1;
			}
		}
		pktcount--;
	}

	rlen = check_read_cache(offset, len, buffer);
	if (rlen) {
		return rlen;
	}

	/*
	 * Okay - fall back to normal behavior.  We missed the
	 * packet somehow.
	 */

	printf("nfs_read: lost packet\n");

	while(retries--) {
		rpcptr = netsprintf(buf.u.data,
		        "%L%L%L%L%L%L%L%L%L%S%L%L%L%L%L%L%L%M%L%L%L",
			rpc_id, MSG_CALL, 2, PROG_NFS, 2, NFS_READ,
		        1, hostnamelen + 28,0,hostname,0,0,2,0,0,0,0,
		        32, fh, offset, len, 0);
		udp_transmit(arptable[server].ipaddr, RPC_SOCKET,
			port, rpcptr - (char *)&buf, &buf);
		if (await_reply(AWAIT_RPC, rpc_id, NULL)) {
			rpc = (struct rpc_t *)&packet[ETHER_HDR_SIZE];
			if (rpc->u.reply.rstatus || rpc->u.reply.verifier ||
				rpc->u.reply.astatus || rpc->u.reply.data[0]) {
				printf("%s\n", rpc_strerror(rpc));
				return(-(ntohl(rpc->u.reply.data[0])));
			} else {
				rlen = ntohl(rpc->u.reply.data[18]);
				if (len < rlen) rlen = len;
				if (len > rlen) netprintf("short read\r\n");
				memcpy(buffer, &rpc->u.reply.data[19], rlen);

				return(rlen);
			}
		}
	}
	return(-1);
}

char *
rpc_strerror(struct rpc_t *rpc)
{
	static char buf[64];

	sprintf(buf, "RPC Error: (%d,%d,%d)",
		ntohl(rpc->u.reply.rstatus),
		ntohl(rpc->u.reply.verifier),
		ntohl(rpc->u.reply.astatus));
	return buf;
}

char *
nfs_strerror(int err)
{
	static char buf[64];

	switch (-err) {
	case NFSERR_PERM:	return "Not owner";
	case NFSERR_NOENT:	return "No such file or directory";
	case NFSERR_ACCES:	return "Permission denied";
	default:
		sprintf(buf, "Unknown NFS error %d", -err);
		return buf;
	}
}
