/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include "Condition.h"
#include "Bytecode.h"
#include "RCX_Cmd.h"
#include "Clause.h"


// these are VERY dependent on RCX_Constants

static int sReversed[] = {
	kGreaterOrEqual, kLessOrEqual, kNotEqualTo, kEqualTo, kLessThan, kGreaterThan
};

static int sInverted[] = {
	kGreaterThan, kLessThan, kEqualTo, kNotEqualTo, kLessOrEqual, kGreaterOrEqual
};

static RCX_Relation sRCX_Codes[] = {
	kRCX_LessOrEqual, kRCX_GreaterOrEqual, kRCX_NotEqualTo, kRCX_EqualTo 
};


#define TEST_MASK	(~(TYPEMASK(kRCX_RandomType) + \
						TYPEMASK(kRCX_ProgramType) + \
						TYPEMASK(kRCX_AGCType)))



Condition::~Condition()
{
}


void Condition::EmitInv(Bytecode &b, int label)
{
	Invert();
	Emit(b, label);
	Invert();
}


TestCond::TestCond(Clause *left, int relation, Clause *right) :
	fLeft(left),
	fRight(right),
	fRelation(relation)
{
}


TestCond::~TestCond()
{
	delete fLeft;
	delete fRight;
}


bool TestCond::Evaluate(bool &v) const
{
	int v1, v2;

	if (!fLeft->GetExpr()->Evaluate(v1)) return false;
	if (!fRight->GetExpr()->Evaluate(v2)) return false;
	
	switch(fRelation)
	{
		case kLessOrEqual:
			v = (v1 <= v2);
			break;
		case kGreaterOrEqual:
			v = (v1 >= v2);
			break;
		case kNotEqualTo:
			v = (v1 != v2);
			break;
		case kEqualTo:
			v = (v1 == v2);
			break;
		case kGreaterThan:
			v = (v1 > v2);
			break;
		case kLessThan:
			v = (v1 < v2);
			break;
	}
	
	return true;
}


void TestCond::Invert()
{
	fRelation = sInverted[fRelation];
}


void TestCond::Emit(Bytecode &b, int label)
{
	RCX_Cmd cmd;
	int v;
	
	// if value 2 is constant, swap with value 1
	if (fRight->GetExpr()->Evaluate(v))
	{
		Clause *t = fLeft;
		fLeft = fRight;
		fRight = t;
		fRelation = sReversed[fRelation];
	}

	RCX_Value ea1 = fLeft->EmitConstrained(b, TEST_MASK);
	RCX_Value ea2 = fRight->EmitConstrained(b, TEST_MASK);
	
	switch(fRelation)
	{
		case kLessOrEqual:
		case kGreaterOrEqual:
		case kNotEqualTo:
		case kEqualTo:
			b.AddTest(ea1, sRCX_Codes[fRelation], ea2, label);
			break;
		case kGreaterThan:
		case kLessThan:
			if (RCX_VALUE_TYPE(ea1) == kRCX_ConstantType)
			{
				RCX_Relation rel;
				short adjust;
				short limit;
				short c;
				
				if (fRelation == kGreaterThan)
				{
					rel = kRCX_GreaterOrEqual;
					adjust = -1;
					limit = -32768;
				}
				else
				{
					rel = kRCX_LessOrEqual;
					adjust = 1;
					limit = 32767;
				}

				c = RCX_VALUE_DATA(ea1);
				
				// check for impossible range
				if (c == limit) break;
				
				// test for adjusted range
				b.AddTest(RCX_VALUE(kRCX_ConstantType, c+adjust), rel, ea2, label);
			}
			else
			{
				cmd.MakeTest(ea1, sRCX_Codes[sInverted[fRelation]], ea2, 11);
				b.Add(cmd);
				b.AddJump(label);
			}
			break;
	}
	
	b.ReleaseTempEA(ea1);
	b.ReleaseTempEA(ea2);
}


TestCond* TestCond::Clone(Mapping *b) const
{
	return new TestCond(fLeft->Clone(b), fRelation, fRight->Clone(b));
}


bool ConstCond::Evaluate(bool &v) const
{
	v = fValue;
	return true;
}


void ConstCond::Invert()
{
	fValue = !fValue;
}


void ConstCond::Emit(Bytecode &b, int label)
{
	if (fValue)
		b.AddJump(label);
}


ConstCond* ConstCond::Clone(Mapping *) const
{
	return new ConstCond(fValue);
}


CompCond::CompCond(Condition *c1, CompCondType type, Condition *c2)
{
	fC1 = c1;
	fType = type;
	fC2 = c2;
}


CompCond::~CompCond()
{
	delete fC1;
	delete fC2;
}


bool CompCond::Evaluate(bool &v) const
{
	bool v1, v2;
	bool d1, d2;
	bool dom;


	dom = (fType==kCompOr) ? true : false;	
	
	d1 = fC1->Evaluate(v1);
	d2 = fC2->Evaluate(v2);

	if ((d1 && (v1 == dom)) ||
		(d2 && (v2 == dom)))
	{
		v = dom;
		return true;
	}
	
	if (d1 && d2)
	{
		v = !dom;
		return true;
	}
	
	return false;
}


void CompCond::Invert()
{
	fType = (fType == kCompOr) ? kCompAnd : kCompOr;
	fC1->Invert();
	fC2->Invert();
}


void CompCond::Emit(Bytecode &b, int label)
{
	if (fType == kCompOr)
	{
		/*
			test c1 -> label
			test c2 -> label
		*/

		fC1->Emit(b, label);
		fC2->Emit(b, label);
	}
	else
	{
		/*
			test !c1 -> out
			test c2->label
		out:
		*/
		int out = b.NewLabel();
		
		fC1->EmitInv(b, out);
		fC2->Emit(b, label);
		b.SetLabel(out);
	}
}

CompCond* CompCond::Clone(Mapping *b) const
{
	return new CompCond(fC1->Clone(b), fType, fC2->Clone(b));
}

