#!/usr/bin/wish
# This Tcl/Tk script implements a simple text editor.
#
# For the public domain.
# Originally generated by...
#
#         D. Richard Hipp
#         Hipp, Wyrick & Company, Inc.
#         6200 Maple Cove Lane
#         Charlotte, NC 28269
#         704.948.4565
#         drh@vnet.net
#
option add *highlightThickness 0

# The next pair of functions implement a dialog box that tells about
# this program.  The dialog box is the one that appears when the user
# selects Help/AboutThisProgram from the main menu.
#
proc AboutMessage {w text} {
  label $w.title -text TkEdit \
    -font -adobe-times-bold-i-normal--24-240-75-75-p-128-iso8859-1
  pack $w.title -side top -pady 15
  message $w.subtitle -width 10c -justify center \
    -font -adobe-times-bold-i-normal-*-14-140-75-75-p-77-iso8859-1 \
    -text "An ASCII text editor built upon\nthe Tcl/Tk Text widget"
  pack $w.subtitle -side top -pady 10 -padx 15
  message $w.msg -width 10c -text "
By D. Richard Hipp
Hipp, Wyrick & Company, Inc.
6200 Maple Cove Lane
Charlotte, NC 28269
704-948-4565
drh@tobit.vnet.net" \
    -font -adobe-times-medium-r-normal-*-12-120-75-75-p-64-iso8859-1
  pack $w.msg -padx 15 -anchor w
  label $w.spacer -text { }
  pack $w.spacer
}
proc HelpAboutThisProgram {} {
  ModalDialogBox . {About} AboutMessage {} {{Dismiss {}}}
}

# Get the name of the file to be edited.
#
if {$argc>1} {
  # If more than one filename is listed on the command line, invoke
  # a separate editor for each one.
  foreach i $argv {
    catch {exec $argv0 $i &}
  }
  exit 0
}
if {$argc==1} {
  # This is the case where a file is specified on the command line.
  # Bring up the editor on that single file
  #
  set path $argv
  if {"[string index $path 0]"=="/"} {
    set full_path $path
  } else {
    set full_path [pwd]/$path
  }
  set FullPath {}

  # See if any other editor is currently working on this file.  If it is,
  # then deiconify and raise that editor and exit, because we don't need
  # more than a single editor running on each file.
  #
  wm withdraw .
  foreach other [winfo interps] {
    if {![string match tkedit* $other]} continue
    catch {
      if {[send $other {set tkedit_FullPath}]==$full_path} {
        send $other {wm deiconify .; raise .}
        exit 0
      }
    }
  }

  set FullPath $full_path
  set tkedit_FullPath $full_path
  wm deiconify .
} else {
  # This is the case where no filename is given.  Simply start a new
  # editor.
  #
  set FullPath {}
  set tkedit_FullPath {}
}

# Make sure the name doesn't correspond to a directory.  It doesn't have
# to be a file which exists, but it can't be a directory.
#
if {"$FullPath"!="" && [file exists $FullPath] && ![file readable $FullPath]} {
  puts stderr "tkedit: Can't open $FullPath for reading"
  exit 1
}

# Construct the editor
#
if {"$FullPath"!=""} {
  wm title . [file tail $FullPath]
  wm iconname . [file tail $FullPath]
} else {
  wm title . <Unnamed>
  wm iconname . <Unnamed>
}
wm minsize . 320 240
wm protocol . WM_DELETE_WINDOW FileQuit

# Construct the menu bar across the top of the application
#
frame .mbar -bd 2 -relief raised
pack .mbar -side top -fill x
proc MakeMenuButton i {
  set name [string toupper [string range $i 0 0]][string range $i 1 end]
  menubutton .mbar.$i -text $name -underline 0 -menu .mbar.$i.menu -pady 0
  pack .mbar.$i -side left -padx 8
}
MakeMenuButton file
MakeMenuButton edit
MakeMenuButton view
if {[catch {source [glob -nocomplain ~/.apps]}]==0} {
  MakeMenuButton tools
  bind .mbar.tools <Button-1> FillToolsMenu
}
MakeMenuButton help
pack .mbar -side top -fill x

menu .mbar.file.menu
.mbar.file.menu add command -label Revert -command FileRevert -underline 0
.mbar.file.menu add command -label Open... -command FileOpen -underline 0
.mbar.file.menu add command -label New -command FileNew -underline 0
.mbar.file.menu add command -label Save -command FileSave -underline 0
.mbar.file.menu add command -label {Save as...} -command FileSaveAs -underline 5
.mbar.file.menu add command -label Shell -command ShellOut -underline 1
#.mbar.file.menu add command -label Print... -command FilePrint -underline 0
.mbar.file.menu add separator
.mbar.file.menu add command -label Quit -command FileQuit -underline 0

menu .mbar.edit.menu
.mbar.edit.menu add command -label Cut -command EditCut -underline 2 \
   -accelerator Ctrl-x
.mbar.edit.menu add command -label Copy -command EditCopy -underline 0 \
   -accelerator Ctrl-c
.mbar.edit.menu add command -label Paste -command EditPaste -underline 0 \
   -accelerator Ctrl-v
.mbar.edit.menu add command -label {Paste Quoted} -command EditPasteQuoted \
   -underline 6
.mbar.edit.menu add command -label Delete -command EditDelete -underline 0
.mbar.edit.menu add separator
.mbar.edit.menu add command -label Search... -command EditSearch -underline 0 \
   -accelerator Ctrl-s

menu .mbar.view.menu
.mbar.view.menu add cascade -label {Font Size} -menu .mbar.view.menu.fontsize \
   -underline 0
.mbar.view.menu add cascade -label Height -menu .mbar.view.menu.height \
   -underline 0
.mbar.view.menu add cascade -label Width -menu .mbar.view.menu.width \
   -underline 0
.mbar.view.menu add cascade -label Tabs -menu .mbar.view.menu.tabs \
   -underline 0
.mbar.view.menu add cascade -label Wrap -menu .mbar.view.menu.wrap -underline 1
.mbar.view.menu add command -label SnapShot -command SnapShot -underline 0
menu .mbar.view.menu.fontsize
foreach i {
  {Tiny -schumacher-clean-medium-r-normal--6-60-75-75-c-40-iso8859-1 0}
  {Small -schumacher-clean-medium-r-normal--8-80-75-75-c-50-iso8859-1 1}
  {Short -schumacher-clean-medium-r-normal-*-10-100-75-75-c-60-iso8859-1 0}
  {Normal -misc-fixed-medium-r-semicondensed-*-13-120-75-75-c-60-iso8859-1 0}
  {Big -misc-fixed-medium-r-normal--14-130-75-75-c-70-iso8859-1 0}
  {Huge -misc-fixed-medium-r-normal--20-200-75-75-c-100-iso8859-1 0}
} {
  .mbar.view.menu.fontsize add radiobutton \
     -label [lindex $i 0] -variable Font -value [lindex $i 1] \
     -command "ChangeFont [lindex $i 1]" -underline [lindex $i 2]
}
set Font -misc-fixed-medium-r-semicondensed-*-13-120-75-75-c-60-iso8859-1
menu .mbar.view.menu.height
foreach i {16 20 24 30 36 40 48 58 72} {
  .mbar.view.menu.height add radiobutton -label "$i lines" \
    -variable Height -value $i -command "ChangeHeight $i"
}
set Height 24
menu .mbar.view.menu.width
foreach i {40 64 80 92 112 124 140 160} {
  .mbar.view.menu.width add radiobutton -label "$i chars" \
    -variable Width -value $i -command "ChangeWidth $i"
}
set Width 80
menu .mbar.view.menu.tabs
foreach i {2 3 4 8} {
  .mbar.view.menu.tabs add radiobutton -label "$i chars" \
    -variable Tabs -value $i -command "ChangeTab $i"
}
set Tabs 8
menu .mbar.view.menu.wrap
.mbar.view.menu.wrap add radiobutton -label Word -underline 0 \
  -variable Wrap -value word -command {ChangeWrap word}
.mbar.view.menu.wrap add radiobutton -label Character -underline 0 \
  -variable Wrap -value char -command {ChangeWrap char}
.mbar.view.menu.wrap add radiobutton -label None -underline 0 \
  -variable Wrap -value none -command {ChangeWrap none}
set Wrap char

menu .mbar.tools.menu
proc FillToolsMenu {} {
  .mbar.tools.menu delete 0 end
  catch {source [glob ~/.apps]}
  if {![info exists application]} return
  foreach appname [lsort [array names application]] {
    .mbar.tools.menu add command -label $appname \
       -command "catch {$application($appname)}"
  }
}


menu .mbar.help.menu
.mbar.help.menu add command -label {About this program...} \
  -command HelpAboutThisProgram

# Put a speed bar directly below the menu bar
#
# To change the contents of the speedbar, just add (or remove) entrys
# to the list on the 2nd argument to the "foreach" statement below.  The
# first field of each entry is the text to go in the speed button, the
# second field is the procedure to call when the speed button is pressed.
#
frame .sbar -bd 1  -relief raised
pack .sbar -side top -fill x
set i 0
foreach btn {
  {Quit FileQuit 0}
  {Save FileSave 0}
  {Copy EditCopy 0}
  {Paste EditPaste 0}
  {Shell ShellOut 1}
  {Search EditSearch 3}
  {{24} {ChangeHeight 24} 0}
  {{36} {ChangeHeight 36} 0}
  {{48} {ChangeHeight 48} 0}
  {{58} {ChangeHeight 58} 0}
  {Small {ChangeFont -schumacher-clean-medium-r-normal--8-80-75-75-c-50-iso8859-1} 1}
  {Short {ChangeFont -schumacher-clean-medium-r-normal-*-10-100-75-75-c-60-iso8859-1} 4}
  {Normal {ChangeFont -misc-fixed-medium-r-semicondensed-*-13-120-75-75-c-60-iso8859-1} 0}
     
} {
  incr i
  button .sbar.b$i -text [lindex $btn 0] \
    -font -adobe-helvetica-medium-r-normal-*-8-80-75-75-p-46-iso8859-1 \
    -command [lindex $btn 1] -underline [lindex $btn 2]\
    -padx 1 -pady 0
  pack .sbar.b$i -side left -padx 1 -pady 1
}
label .sbar.where -pady 0 \
  -font -adobe-helvetica-medium-r-normal-*-10-100-75-75-p-56-iso8859-1
pack .sbar.where -side right -pady 0 -anchor e

# The ".mesg" frame is used to display information about the current
# file and our position within that file, at the bottom of the screen.
#
frame .mesg -bd 1 -relief raised
pack .mesg -side bottom -fill x
label .mesg.name -text $FullPath -font $Font
pack .mesg.name -side left
frame .mesg.spacer
pack .mesg.spacer -side left -fill x -expand 1
label .mesg.line -width 20 -font $Font
pack .mesg.line -side left

# The main text widget with its scroll bar.
#
scrollbar .sb -orient vertical -command {.t yview} \
  -highlightthickness 0 -bd 1 -relief raised
pack .sb -side right -fill y
text .t -font $Font -width $Width -height $Height \
   -yscrollcommand {ChangeWhere; .sb set} \
   -wrap $Wrap -bd 1 -relief raised -padx 3 -pady 2 \
   -exportselection 0
bindtags .t .t
pack .t -fill both -expand 1
.t insert end {}
#update
#update idletasks

# Change the indicator showing the top line of the screen
#
proc ChangeWhere {} {
  scan [.t index @0,0] %d w
  .sbar.where config -text "top line: $w"
}

# Change the height of the editor.  The argument is the number of rows
# of text to display at one time.
#
proc ChangeHeight newheight {
  global Width Height
  .t config -height $newheight -width $Width
  set Height $newheight
  ResizeRoot
}

# Change the width of the editor.  The argument is the number of columns
# of text to display.
#
proc ChangeWidth newwidth {
  global Height Width
  .t config -height $Height -width $newwidth
  set Width $newwidth
  ResizeRoot
}

# Change the size of the root window (.) whenever the text widget
# is resized.
#
proc ResizeRoot {} {
  update idletasks
  set w [winfo reqwidth .]
  set h [winfo reqheight .]
  wm geometry . ${w}x$h
}

# Set the tab spacing
#
proc ChangeTab spacing {
  global Width Tabs
  set sp [expr int($spacing*([winfo reqwidth .t]/$Width))]
  .t config -tabs $sp
  set Tabs $spacing
}

# Change the word-wrap mode for the editor.  The argument should be
# one of:   word   char   none
#
proc ChangeWrap newmode {
  .t config -wrap $newmode
}

# Change the font used by the editor.  The argument is the name of the
# font to use.  For best results, use a constant-width font.
#
proc ChangeFont newfont {
  global Font
  .t config -font $newfont
  .mesg.name config -font $newfont
  .mesg.line config -font $newfont
  set Font $newfont
  ResizeRoot
}

# Attempt to launch a shell in the same directory as the file currently
# being edited.  We try to launch tkterm first, but if that fails we
# try xterm as a backup.
#
proc ShellOut {} {
  global FullPath
  cd [file dir $FullPath]
  if {[catch {exec tkterm &}]} {
    catch {exec xterm &}
  }
}

# The procedure defined below implements a generic dialog box.  The
# arguments are as follows:
#
#   position      The new dialog box is centered over the window given
#                 by this argument
#
#   title         This is the title for the dialog box
#
#   build         This procedure is called to construct the top-most
#                 panel of the dialog box.  The first argument to the
#                 procedure is the name of the frame widget which
#                 is the top panel.  Subsequent arguments are given by
#                 the "buildargs" parameter.
#
#   buildargs     This is arguments to the "build" command which come
#                 after the name of the top panel widget.
#
#   btns          This is a list of button descriptions.  Each button
#                 description consists of the name of the button and
#                 some text to be displayed beside that button.
#
# The procedure builds a model dialog box and waits for a button to be
# pressed.  When a button is pressed, the dialog box goes away and the
# procedure returns an integer which is the index of the selected button.
# The first button is numbered 0.
#
proc ModalDialogBox {pos title build buildargs btns} {
  global dialog_button
  if [winfo exists .d] {destroy .d}
  toplevel .d -class Dialog
  wm title .d $title
  wm iconname .d Dialog
  frame .d.msg -relief raised -bd 1
  $build .d.msg $buildargs
  pack .d.msg -side top -fill both -expand 1
  set cnt -1
  foreach btn $btns {
    incr cnt
    set btnname [lindex $btn 0]
    set btntext [lindex $btn 1]
    frame .d.x$cnt -relief raised -bd 1
    if [llength $btn]==3 {
      set cmd "[lindex $btn 2] .d; set dialog_button $cnt"
    } else {
      set cmd "set dialog_button $cnt"
    }
    button .d.x$cnt.btn -text $btnname -command $cmd -width 9 -underline 0
    pack .d.x$cnt.btn -side left -padx 5 -pady 5
    message .d.x$cnt.msg -text $btntext -width 10c
    pack .d.x$cnt.msg -anchor w -padx 5 -pady 5
    pack .d.x$cnt -side top -fill x
  }
  wm withdraw .d
  update idletasks
  set x [expr [winfo rootx $pos] + ([winfo width $pos]-[winfo reqwidth .d])/2]
  set y [expr [winfo rooty $pos] + ([winfo height $pos]-[winfo reqheight .d])/2]
  wm geometry .d +$x+$y
  wm deiconify .d
  set old_focus [focus]
  focus .d
  grab set .d
  bind .d <ButtonPress> {
    if {![string match .d.* %W]} {
      bell
      wm withdraw .d
      wm deiconify .d
    }
  }
  bind .d <KeyPress> {tkTraverseToMenu %W %A}
  tkwait variable dialog_button
  grab release .d
  focus $old_focus
  destroy .d
  return $dialog_button
}

#
# The following procedures are used to construct a dialog box header which
# contains an icon and a message.  The difference is in the icon.
#
proc QuestionMessage {w text} {
  label $w.icon -bitmap info
  pack $w.icon -side left -padx 15 -pady 15
  message $w.msg -text $text -width 10c
  pack $w.msg -padx 15 -pady 15 -anchor w
}
proc WarningMessage {w text} {
  label $w.icon -bitmap warning
  pack $w.icon -side left -padx 15 -pady 15
  message $w.msg -text $text -width 10c
  pack $w.msg -padx 15 -pady 15 -anchor w
}

#
# The following procedure creates a search dialog box.  Or, if the dialog
# box already exists, it raises and deiconifies it.
#
set SearchData(type) exact
set SearchData(from) {}
set SearchData(to) {}
proc EditSearch {} {
  global FullPath SearchData
  if [winfo exists .srch] {
    wm deiconify .srch
    raise .srch
    focus .srch.pat.e
    return
  }
  toplevel .srch -class Dialog
  wm title .srch [file tail $FullPath]
  wm iconname .srch [file tail $FullPath]
  wm withdraw .srch
  frame .srch.btn
  pack .srch.btn -side bottom -padx 5 -pady 8 -fill x
  foreach b {
    {close    Close          0  {focus .t; destroy .srch}}
    {prev     {Prev (F4)}    0  FindPrev}
    {next     {Next (F3)}    0  FindNext}
    {replace  {Change (F8)}  1  Replace}
    {all      {Change All}   7  ReplaceAll}
  } {
    button .srch.btn.[lindex $b 0] -text [lindex $b 1] -command [lindex $b 3] \
       -pady 1 -pady 2 -underline [lindex $b 2]
    pack .srch.btn.[lindex $b 0] -side left -padx 2 -expand 1
  }
  frame .srch.ctrl
  pack .srch.ctrl -side right -fill y
  radiobutton .srch.ctrl.b1 -text Exact -variable SearchData(type) \
     -value exact -pady 0 -underline 0
  radiobutton .srch.ctrl.b2 -text {No Case} -variable SearchData(type) \
     -value nocase -pady 0 -underline 1
  radiobutton .srch.ctrl.b3 -text {Reg Exp} -variable SearchData(type) \
     -value regexp -pady 0 -underline 5
  pack .srch.ctrl.b1 .srch.ctrl.b2 .srch.ctrl.b3 -side top -pady 0 -padx 5 \
     -anchor w
  frame .srch.pat
  pack .srch.pat -side top -padx 5 -pady 5 -fill x
  label .srch.pat.l -text {Search for:} -width 10 -anchor e
  pack .srch.pat.l -side left
  entry .srch.pat.e -width 40 -bd 2 -relief sunken \
     -textvariable SearchData(from)
  pack .srch.pat.e -side left -fill x
  bind .srch.pat.e <Return> FindNext
  frame .srch.to
  pack .srch.to -side top -padx 5 -pady 5 -fill x
  label .srch.to.l -text {Change to:} -width 10 -anchor e
  pack .srch.to.l -side left
  entry .srch.to.e -width 40 -bd 2 -relief sunken \
     -textvariable SearchData(to)
  pack .srch.to.e -side left -fill x
  bind .srch.pat.e <Return> FindNext
  update idletasks
  set y [expr {[winfo rooty .] - [winfo reqheight .srch] - 40}]
  if {$y<0} {set y 0}
  wm geometry .srch +[winfo rootx .]+$y
  wm deiconify .srch
  focus .srch.pat.e
  bind .srch <F3> FindNext
  bind .srch <F4> FindPrev
  bind .srch <F8> Replace
}

#
# Using the pattern string located in the entry widget .srch.pat.e,
# find the next occurance of the pattern in the text widget.
#
proc FindNext {} {
  global SearchData
  set type -$SearchData(type)
  set pattern $SearchData(from)
  set count 0
  set w [.t search -count count $type -- $pattern {insert +1 chars}]
  if {"$w"==""} {
    set w [.t search -count count $type -- $pattern 1.0 insert]
  }
  if {"$w"!=""} {
    .t mark set insert $w
    textFixup
    textSel insert "insert +$count chars"
  }
  focus .t
}
proc FindPrev {} {
  global SearchData
  set type -$SearchData(type)
  set pattern $SearchData(from)
  set count 0
  set w [.t search -count count -backwards $type -- $pattern {insert -1 chars}]
  if {"$w"==""} {
    set w [.t search -count count -backwards $type -- $pattern end insert]
  }
  if {"$w"!=""} {
    .t mark set insert $w
    textFixup
    textSel insert "insert +$count chars"
  }
  focus .t
}

#
# If the text at the insertion cursor matches the pattern in the
# search dialog, then replace the pattern with the substitution string
# and find the next occurance of the pattern.
#
proc Replace {} {
  global Dirty SearchData
  set pattern $SearchData(from)
  set new $SearchData(to)
  set mode -$SearchData(type)
  set count 0
  set w [.t search $mode -count count -- $pattern insert {insert +1 char}]
  if {"$w"!="" && $count>0} {
    .t delete insert "insert +$count chars"
    .t insert insert $new
    set Dirty 1
    FindNext
    return 1
  } else {
    return 0
  }
}
proc ReplaceAll {} {
  FindNext
  while {[Replace]} {}
}

############################################################################
# The following code implements the file selection dialog box
#

#
# These variables are used internally by the GetFileName procedure.
# They should not be accessed or changed by any outside procedure.
#
set GFN(path) [pwd]           ;# The directory we are currently reading
set GFN(name)   {}            ;# Name of a selected file
set GFN(result) {}            ;# Write to this to trigger end-of-file
set GFN(filter) {*}           ;# Filter filenames using this pattern
set GFN(entry)  {}            ;# Holds complete text of the entry widget
set GFN(side)   {dir}         ;# Either "dir" or "file"
set GFN(index)  0             ;# Index of the highlighted file or directory.

#
# This is the main routine -- the only routine which should be accessed
# or used from outside this file.  This routine causes a file chooser
# dialog box to appear centered in the window "pos"  The title of the
# dialog box will be "title".  File names will be filtered using "filter".
#
# The procedure will return either the name of a selected file or {} if
# the cancel button is pressed.  The name returned does not necessary
# refer to a file that exists -- the user may have typed in a bogus
# name
#
proc GetFileName {pos title filter} {
  global GFN

  set GFN(filter) $filter
  if [winfo exists .d] {destroy .d}
  toplevel .d -class Dialog
  wm protocol .d WM_DELETE_WINDOW {set GFN(result) {}}
  wm title .d $title
  wm iconname .d Dialog
  wm withdraw .d
  frame .d.f2
  pack .d.f2 -side top -padx 2 -pady 2 -fill x
  label .d.f2.l -text Name
  pack .d.f2.l -side left
  entry .d.f2.e -bd 2 -relief sunken -textvariable GFN(entry)
  bind .d.f2.e <Return> {GFN_ReturnPressedInEntry}
  pack .d.f2.e -side left -expand 1 -fill x
  set old_focus [focus]
  focus .d.f2.e
  frame .d.f3
  listbox .d.f3.ll -yscrollcommand {.d.f3.sl set} -bd 2 -relief sunken \
     -exportselection 0 -takefocus 0
  bind .d.f3.ll <1> {GFN_SingleClickFile %y}
  # bind .d.f3.ll <Double-1> {set GFN(result) $GFN(path)/$GFN(name)}
  bind .d.f3.ll <Double-1> GFN_ReturnPressedInEntry
  bindtags .d.f3.ll .d.f3.ll
  scrollbar .d.f3.sl -command {.d.f3.ll yview} -orient vertical -takefocus 0
  frame .d.f3.spacer -width 5
  listbox .d.f3.lr -yscrollcommand {.d.f3.sr set} -bd 2 -relief sunken \
     -exportselection 0 -takefocus 0
  bind .d.f3.lr <1> {GFN_SingleClickDir %y}
  # bind .d.f3.lr <Double-1> {GFN_DoubleClickRightListbox %y}
  bind .d.f3.lr <Double-1> GFN_ReturnPressedInEntry
  bindtags .d.f3.lr .d.f3.lr
  scrollbar .d.f3.sr -command {.d.f3.lr yview} -orient vertical -takefocus 0
  pack .d.f3.lr .d.f3.sr .d.f3.spacer .d.f3.ll .d.f3.sl \
     -side left -fill y
  pack .d.f3 -padx 2 -pady 5
  frame .d.f4
  button .d.f4.ok -width 6 -text Ok -command {set GFN(result) [.d.f2.e get]} \
     -underline 0 -takefocus 0
  button .d.f4.cancel -text Cancel -command {set GFN(result) {}} \
     -underline 0 -takefocus 0
  pack .d.f4.ok .d.f4.cancel -expand 1 -side left -pady 5
  pack .d.f4 -fill x
  update idletasks
  GFN_FillListboxes
  set x [expr [winfo rootx $pos] + ([winfo width $pos]-[winfo reqwidth .d])/2]
  set y [expr [winfo rooty $pos] + ([winfo height $pos]-[winfo reqheight .d])/2]
  wm geometry .d +$x+$y
  wm deiconify .d
  grab set .d
  bind .d <Any-ButtonPress> {
    if {![string match .d.* %W]} {bell; MakeWindowVisible .d}
  }
  bind .d <Alt-KeyPress> {tkTraverseToMenu %W %A}
  bind .d <Up> {GFN_UpDown -1}
  bind .d <Down> {GFN_UpDown 1}
  bind .d <Prior> {GFN_UpDown -10}
  bind .d <Next> {GFN_UpDown 10}
  bind .d <Any-Tab> GFN_LeftRight
  set GFN(result) no
  update
  tkEntrySeeInsert .d.f2.e
  tkwait variable GFN(result)
  grab release .d
  focus $old_focus
  destroy .d
  return $GFN(result)
}

#
# Make the named window visible
#
proc MakeWindowVisible w {
  wm iconify $w
  update
  wm deiconify $w
  update
}

#
# This routine is called whenever the contents of the entry box change.
# This routine will change the status of the "Ok" button appropriately.
# The "OK" button is disabled is the text in the entry box contains
# wildcards, and is enabled if the text contains no wildcards.
#
proc GFN_EntryChange {args} {
  if {![winfo exists .d.f4.ok]} return
  global GFN
  set basename [file tail $GFN(entry)]
  if {[regexp {[*?[]} $basename] || [file isdir $GFN(entry)]} {
    .d.f4.ok config -state disabled
  } else {
    .d.f4.ok config -state normal
  }
}
trace variable GFN(entry) w GFN_EntryChange

#
# This routine is called to refill the listboxes.  The left listbox
# (.d.f3.ll) gets the names of all files in the directory $path
# which match the filter given by $filter.  The right listbox (.d.f3.lr)
# gets the name of every directory in the directory $path, plus the
# special directories "." and "..".  The text in the entry box is also
# updated.
#
proc GFN_FillListboxes {} {
  global GFN
  .d config -cursor watch
  update
  .d.f3.lr delete 0 end
  .d.f3.ll delete 0 end
  .d.f3.lr insert end .
  if {"$GFN(path)"!=""} {.d.f3.lr insert end ..}
  GFN_ChangeEntryText $GFN(path)/$GFN(filter)
  if {[file isdir /$GFN(path)] && [file readable /$GFN(path)]} {
    foreach fn [lsort [glob -nocomplain $GFN(path)/*]] {
      set tail [file tail $fn]
      if [file isdir $fn] {
        .d.f3.lr insert end $tail
      } elseif [string match $GFN(filter) $tail] {
        .d.f3.ll insert end $tail
      }
    }
  }
  if {"$GFN(side)"==""} {
    set GFN(side) file
  }
  set GFN(index) 0
  .d config -cursor {}
  GFN_UpDown 0
}

#
# This routine is called to change the text in the entry box.
#
proc GFN_ChangeEntryText {text} {
  if {[regsub -all ///* $text / new]} {
    set text $new
  }
  .d.f2.e delete 0 end
  .d.f2.e insert 0 $text
  .d.f2.e icursor end
  if {[winfo ismapped .d.f2.e]} {tkEntrySeeInsert .d.f2.e}
}

#
# This routine is called whenever the user single-clicks any of the
# directories names
#
proc GFN_SingleClickDir {y} {
  global GFN
  set GFN(side) dir
  set GFN(index) [.d.f3.lr nearest $y]
  GFN_UpDown 0
}

#
# This routine is called whenever the user single-clicks any of the
# files named in the right listbox.  The effect is to set the variable
# GFN(name) to the file name so that the Ok button will know how to
# set the result if and when it is pressed
#
proc GFN_SingleClickFile {y} {
  global GFN
  set GFN(side) file
  set GFN(index) [.d.f3.ll nearest $y]
  GFN_UpDown 0
}

#
# Move the selected entry up or down by the amount $cnt
#
proc GFN_UpDown {cnt} {
  global GFN
  if {"$GFN(index)"==""} return
  if {"$GFN(side)"=="dir"} {
    set w .d.f3.lr
  } else {
    set w .d.f3.ll
  }
  incr GFN(index) $cnt
  if {$GFN(index)<0} {
    set GFN(index) 0
  } elseif {$GFN(index)>=[$w index end]} {
    set GFN(index) [expr [$w index end]-1]
  }
  .d.f3.ll select clear 0 end
  .d.f3.lr select clear 0 end
  $w select set $GFN(index)
  $w see $GFN(index)
  set GFN(name) [$w get $GFN(index)]
  GFN_ChangeEntryText $GFN(path)/$GFN(name)
}

#
# Move the cursor from the Left box to the Right box or back
# again.
#
proc GFN_LeftRight {} {
  global GFN
  if {"$GFN(index)"==""} {
    set GFN(index) 0
    set GFN(side) dir
  } elseif {"$GFN(side)"=="dir"} {
    set GFN(side) file
  } else {
    set GFN(side) dir
  }
  GFN_UpDown 0
}

#
# This routine is called whenever the user presses Return while in
# the GFN dialog box.
#
proc GFN_ReturnPressedInEntry {} {
  global GFN
  set fn [.d.f2.e get]
  if {"$fn"==""} {
    if {"$GFN(path)"=="/" || "$GFN(path)"==""} return
    set GFN(path) /
    GFN_FillListboxes
    return
  }
  set basename [file tail $fn]
  if [regexp {[*?[]} $basename] {
    if {"$GFN(path)"=="[file dirname $fn]"} return
    set GFN(filter) $basename
    set GFN(path) [file dirname $fn]
    GFN_FillListboxes
  } elseif {"$basename"=="."} {
    # do nothing
  } elseif {"$basename"==".."} {
    set GFN(path) [file dirname [file dirname $fn]]
    GFN_FillListboxes
  } elseif [file isdir $fn] {
    set GFN(path) $fn
    GFN_FillListboxes
  } else {
    set GFN(result) $fn
  }
  if {"$GFN(path)"=="/"} {set GFN(path) {}}
}

# Here ends the code for implementing the file dialog box
##############################################################################

#
# The following function loads the text of a file into the text widget.
#
proc FileLoad {filename} {
  . config -cursor watch
  .t delete 1.0 end
  if [file readable $filename] {
    set f [open $filename]
    .t insert end [read $f]
    close $f
  }
  focus .t
  .t mark set insert 1.0
  textFixup
  . config -cursor {}
  global Dirty
  set Dirty 0
}

#
# This routine is called in response to the "File/Quit" menu action.
#
proc FileQuit {} {
  global Dirty FullPath
  while {$Dirty} {
    set response [ModalDialogBox . {Save Changes?} QuestionMessage \
      "You have made changes to the file \"$FullPath\". Do you want to save those changes before exiting?" {
         {Save     {Save the changes to the file before exiting}}
         {Discard  {Don't save the changes, but exit anyway}}
         {Cancel   {Don't exit.  Return to the editor}}
      }]
    if $response==2 return
    if $response==0 FileSave
    if $response==1 {set Dirty 0}
  }
  destroy .
}

#
# Write the editor contents into the file FullPath.
#
# This routine is never called unless we are sure that the file has
# write permission.
#
proc FileWrite {} {
  global FullPath Dirty
  set f [open $FullPath w]
  puts $f [string trimright [.t get 1.0 end]]
  close $f
  set Dirty 0
}

#
# This routine is called in order to save the editor contents to its
# default filename.
#
proc FileSave {} {
  global FullPath Dirty
  if {"$FullPath"==""} {
    FileSaveAs
    return
  }
  if {[file exists $FullPath] && ![file writable $FullPath]} {
    set response [ModalDialogBox . {Can't Write} WarningMessage \
      "The file \"$FullPath\" is not writable.  What do you want to do?" {
      {{Save As...}  {Save the editor contents to a different filename}}
      {{Try Again}   {Attempt to change permissions to allow the file to be written}}
      {{Cancel}      {Don't save the contents of the editor}}
    }]
    if {$response==2} return
    if {$response==0} {FileSaveAs; break}
    catch {exec chmod +w $FullPath}
    if {![file writable $FullPath]} {
      set response [ModalDialogbox . {Still Can't Write} WarningMessage \
        "Unable to enable write permission for the file \"$FullPath\".  What now?" {
        {{Save As...}  {Save the editor contents to a different filename}}
        {{Cancel}      {Don't save the contents of the editor}}
      }]
      if {$response==1} return
      FileSaveAs
    } else {
      FileWrite
    }
  } else {
    FileWrite
  }
}

#
# This routine is called query the user for a new filename in which
# to write the contents of a file.
#
proc FileSaveAs {} {
  global FullPath
  set loop 1
  while {$loop} {
    set loop 0
    set newname [GetFileName . {Save As} {*}]
    if {"$newname"==""} return
    if {[file writable $newname]} {
      set response [ModalDialogBox . {File Exists} WarningMessage \
        "The file \"$newname\" already exists.  What do you want to do?" {
          {Overwrite  {Replace the existing contents of the file with the current contents of the editor}}
          {Rename     {Choose a new name to which to write the file}}
          {Cancel     {Don't save the editor contents}}
        }]
      if $response==1 {set loop 1; continue}
      if $response==2 return
      if $response==0 break
    }   
  }
  set FullPath $newname; 
  .mesg.name config -text $FullPath
  wm title . [file tail $FullPath]
  wm iconname . [file tail $FullPath]
  FileSave
}

#
# Create a new file editor
#
proc FileNew {} {
  global argv0
  catch "exec $argv0 &"
}

#
# Open a new file
#
proc FileOpen {} {
  set newname [GetFileName . {Open} {*}]
  if {"$newname"!=""} {
    global argv0
    catch "exec $argv0 $newname &"
  }
}

#
# This routine is called in response to the File/Revert menu selection
#
proc FileRevert {} {
  global FullPath Dirty
  set oldname $FullPath
  while {$Dirty} {
    set response [ModalDialogBox . {Undo Changes?} WarningMessage \
      "The contents of the editor have been modified. If you reload the editor you will lose your changes!" {
        {Revert  {Go ahead and overwrite the changes in the editor}}
        {Save    {Save editor contents to a different file before reloading}}
        {Cancel  {Don't reload the editor}}
      }]
    if $response==0 break
    if $response==1 FileSaveAs
    if $response==2 return
  }
  set FullPath $oldname
  .mesg.name config -text $FullPath
  FileLoad $FullPath
}

#
# This routine is called to compute and display the line information
# in the message frame.  It should be called whenever the insertion
# cursor moves.
#
proc ComputeLine {} {
  scan [.t index insert] {%d.%*d} now
  scan [.t index end] {%d.%*d} last
  incr last -1
  .mesg.line config -text "line $now of $last"
}

# A routine for dispensing the selection.  The selection is always owned
# by the window ".".  Its value is stored in the variable "Selection"
#
set Selection {}
selection handle . RetrieveSelection
proc RetrieveSelection {offset max} {
  global Selection
  return [string range $Selection $offset [expr {$offset+$max}]]
}

# This routine is called whenever "." owns the selection but another
# window claims ownership.
#
proc LoseSelection {} {
  global Selection
  set Selection {}
}

# Copy the text selected in the text widget into the Selection variable,
# then claim ownership of the selection.
#
proc EditCopy {} {
  global Selection
  catch {
    set Selection [.t get sel.first sel.last]
    selection own . LoseSelection
  }
}
proc EditCut {} {
  EditCopy
  EditDelete
}
proc EditPaste {} {
  catch {
    .t insert insert [selection get]
    textChange
  }
}
proc EditPasteQuoted {} {
  catch {
    set x [selection get]
    regsub -all "\n" $x "\n> " x
    .t insert insert "> $x"
    textChange
  }
}
proc EditDelete {} {
  catch {
    .t delete sel.first sel.last
    textChange
  }
}

#
# Create a dialog holding a snap-shot of the currently selected text.
#
proc SnapShot {} {
  set data [.t get @0,0 "@0,[winfo height .t] lineend"]
  global FullPath
  set n 0
  while {[winfo exists .ss$n]} {incr n}
  set w .ss$n
  toplevel $w
  set title "Snapshot from \"[file tail $FullPath]\""
  wm title $w $title
  wm iconname $w $title
  button $w.close -command "destroy $w" -text "Close"
  pack $w.close -side bottom -pady 4
  text $w.t -bd 1 -relief raised \
    -font [.t cget -font] \
    -height [.t cget -height] \
    -width [.t cget -width] \
    -tabs [.t cget -tabs]
  pack $w.t -side left -fill both -expand 1
  $w.t insert end $data
  $w.t config -state disabled
}

# The default bindings for the text widget are woefully inadequate.
# The following section implements improved bindings.
#
bind .t <KeyPress> {if {"%A"!=""} {.t insert insert %A; textChange}}
bind .t <Alt-KeyPress> {tkTraverseToMenu %W %A}
bind .t <F3> FindNext
bind .t <F4> FindPrev
bind .t <F8> Replace
bind .t <Return> {.t insert insert \n; textChange}
bind .t <Delete> {.t delete {insert -1 char}; textChange}
#bind .t <Delete> {.t delete insert; textChange}
bind .t <BackSpace> {.t delete {insert -1 char}; textChange}
bind .t <Control-h> {.t delete {insert -1 char}; textChange}
bind .t <Control-x> EditCut
bind .t <Control-c> EditCopy
bind .t <Control-v> EditPaste
bind .t <Control-u> {}
bind .t <Control-d> {.t delete insert; textChange}
bind .t <Control-w> {}
bind .t <Control-k> {.t delete insert {insert lineend}; textChange}
bind .t <Control-p> {textAnchor {insert -1 line}}
bind .t <Control-a> {textAnchor {insert linestart}}
bind .t <Control-e> {textAnchor {insert lineend}}
bind .t <Control-n> {textAnchor {insert +1 line}}
bind .t <Control-b> {textAnchor {insert -1 char}}
bind .t <Control-f> {textAnchor {insert +1 char}}
bind .t <Control-s> {EditSearch}
bind .t <Left> {textAnchor {insert -1 char}}
bind .t <Right> {textAnchor {insert +1 char}}
bind .t <Home> {textAnchor {insert linestart}}
bind .t <End> {textAnchor {insert lineend}}
bind .t <Up> {textAnchor {insert -1 line}}
bind .t <Down> {textAnchor {insert +1 line}}
bind .t <Prior> {textAnchor "insert -$Height lines"}
bind .t <Next> {textAnchor "insert +$Height lines"}
bind .t <Shift-Left> {textInsert {insert -1 char}; textSel tk_anchor insert}
bind .t <Shift-Right> {textInsert {insert +1 char}; textSel tk_anchor insert}
bind .t <Shift-Home> {textInsert {insert linestart}; textSel tk_anchor insert}
bind .t <Shift-End> {textInsert {insert lineend}; textSel tk_anchor insert}
bind .t <Shift-Up> {textInsert {insert -1 line}; textSel tk_anchor insert}
bind .t <Shift-Down> {textInsert {insert +1 line}; textSel tk_anchor insert}
bind .t <Shift-Prior> {textInsert "insert -$Height lines";
  textSel tk_anchor insert}
bind .t <Control-Left> {textInsert {insert -1 char}; textSel tk_anchor insert}
bind .t <Control-Right> {textInsert {insert +1 char}; textSel tk_anchor insert}
bind .t <Control-Home> {textInsert {insert linestart}; textSel tk_anchor insert}
bind .t <Control-End> {textInsert {insert lineend}; textSel tk_anchor insert}
bind .t <Control-Up> {textInsert {insert -1 line}; textSel tk_anchor insert}
bind .t <Control-Down> {textInsert {insert +1 line}; textSel tk_anchor insert}
bind .t <Control-Prior> {textInsert "insert -$Height lines";
  textSel tk_anchor insert}
bind .t <Control-Next> {
  textInsert "insert +$Height lines"
  textSel tk_anchor insert
}
bind .t <1> {.t tag remove sel 0.0 end; textAnchor @%x,%y; focus .t}
#bind .t <B1-Motion> {textInsert @%x,%y; textSel tk_anchor insert}
bind .t <B1-Motion> {B1Motion %x %y}
bind .t <ButtonRelease-1> MotorOff
bind .t <Double-1> {textSel {insert wordstart} {insert wordend}}
bind .t <Triple-1> {textSel {insert linestart} {insert lineend}}
bind .t <Shift-1> {textInsert @%x,%y; focus .t; textSel tk_anchor insert}
bind .t <Control-B1-Motion> {textSel tk_anchor @%x,%y}
bind .t <2> {.t scan mark 0 %y}
bind .t <B2-Motion> {.t scan dragto 0 %y}

#
# The next group of functions are used by the bindings.
#
proc textChange {} {textFixup; global Dirty; set Dirty 1}
#proc textFixup {} {.t yview -pickplace insert; ComputeLine }
proc textFixup {} {.t see insert; ComputeLine }
proc textSel {a b} {
  if [.t compare $a < $b] {
    .t tag remove sel 0.0 $a
    .t tag add sel $a $b
    .t tag remove sel $b end
  } else {
    .t tag remove sel 0.0 $b
    .t tag add sel $b $a
    .t tag remove sel $a end
  }
}
proc textInsert {w} {.t mark set insert $w; textFixup}
proc textAnchor {w} {.t mark set tk_anchor $w; textInsert $w}

#
# This routine does auto-motoring
#
proc B1Motion {x y} {
  set m @$x,$y
  if {$y<0} {
    set d [expr int($y/20)]
    MotorOn $d
    set m "$m $d line"
  } elseif {$y>[winfo height .t]} {
    set d +[expr int(($y-[winfo height .t])/20)+1]
    MotorOn $d
    set m "$m $d line"
  } else {
    MotorOff
    textInsert $m
    textSel tk_anchor insert
  }
}
proc MotorOn d {
  global MotorDir
  set old $MotorDir
  set MotorDir $d
  if {$old==0} {
    Motor
  }
}
proc Motor {} {
  global MotorDir
  if {$MotorDir==0} return;
  textInsert "insert $MotorDir line"
  textSel tk_anchor insert
  after 100 Motor
}
proc MotorOff {} {
  global MotorDir
  set MotorDir 0
}

# tkMenuFind --
# This procedure searches the entire window hierarchy under w for
# a menubutton that isn't disabled and whose underlined character
# is "char".  It returns the name of that window, if found, or an
# empty string if no matching window was found.  If "char" is an
# empty string then the procedure returns the name of the first
# menubutton found that isn't disabled.
#
# If a third argument is provided, it is used as a classname pattern
# for the window to search for.  Be default, this pattern is
# MenuButton, meaning that this routine will find only menubuttons.
# But if you change the class pattern to "*utton", the routine will
# find the first button of any type.
#
# Arguments:
# w -				Name of window where key was typed.
# char -			Underlined character to search for;
#				may be either upper or lower case, and
#				will match either upper or lower case.

proc tkMenuFind {w char {pattern Menubutton}} {
    global tkPriv
    set char [string tolower $char]
    set action [format {
        %s {
            set char2 [string index [$child cget -text] \
		[$child cget -underline]]
	    if {([string compare $char [string tolower $char2]] == 0)
		|| ($char == "")} {
  	        if {[$child cget -state] != "disabled"} {
		    return $child
		}
	    }
	}

	default {
	    set match [tkMenuFind $child $char %s]
	    if {$match != ""} {
		return $match
	    }
	}
    } $pattern $pattern]

    foreach child [winfo child $w] {
	switch -glob [winfo class $child] $action
    }
    return {}
}

# tkTraverseToMenu --
# This procedure implements keyboard traversal of menus.  Given an
# ASCII character "char", it looks for a menubutton with that character
# underlined.  If one is found, it posts the menubutton's menu.
#
# The routine will also look for buttons to invoke.  If a button is
# found that contains the given character, then that button is invoked.tkp
#
# Arguments:
# w -				Window in which the key was typed (selects
#				a toplevel window).
# char -			Character that selects a menu.  The case
#				is ignored.  If an empty string, nothing
#				happens.

proc tkTraverseToMenu {w char} {
    if ![winfo exists $w] return
    global tkPriv
    if {$char == ""} {
	return
    }
    while {[winfo class $w] == "Menu"} {
	if {$tkPriv(postedMb) == ""} {
	    return
	}
	set w [winfo parent $w]
    }
    set w [tkMenuFind [winfo toplevel $w] $char *utton]
    if {$w != ""} {
        switch [winfo class $w] {
            Menubutton {
   	        tkMbPost $w
	        tkMenuFirstEntry [$w cget -menu]
            }

            Button {
                tkButtonInvoke $w
            }

            Checkbutton {
                tkCheckRadioInvoke $w
            }

            Radiobutton {
                tkCheckRadioInvoke $w
            }
        }
    }
}

#
# Finally, load the file into the editor and begin editing
#
FileLoad $FullPath
.t mark set tk_anchor 1.0
