(**************************************************************************)
(*                                lexer.ml                                *)
(**************************************************************************)

type token = 
    Kwd   of string
  | Ident of string
  | Special of string
  | String of string
  | Char  of char
  | CB
  | CE

(* beginning of the "classical" lexer *)
let add_in_buff, get_buff =
  let buff = ref (String.create 80) in (
  (* add_in_buff *) (
    fun i x ->
      let len = String.length !buff in
      if i >= len then (buff := !buff ^ (String.create len); ());
      String.set !buff i x;
      succ i
  ),
  (* get_buff *) (
    fun len ->
      String.sub !buff 0 len
  ) )

let keywords = 
  ["Axiom"; "Case"; "Cases"; "Chapter"; "CoFixpoint";
   "CoInductive"; "Defined"; "Definition";
   "End"; "Export"; "Fix";"Fixpoint"; "Global"; "Grammar"; "Hint";
   "Hypothesis"; "Immediate"; "Inductive"; "Infix"; "Lemma"; "Load";
   "Local"; "Match"; "Mutual"; "Parameter"; "Print"; "Proof"; "Qed";
   "Record"; "Recursive"; "Remark"; "Require"; "Save"; "Scheme";
   "Section"; "Show"; "Syntactic"; "Syntax"; "Theorem"; "Variable";
   "Variables";
   "array"; "assert"; "begin"; "do"; "done"; "else"; "end"; "if";
   "in"; "invariant"; "let"; "of"; "ref"; "state"; "then"; "variant";
   "while"; "with"]

let rec ident len = parser
    [< ''a'..'z' | 'A'..'Z' | '0'..'9' | '\'' | '_' | '$' | '@' as c; s >] ->
      ident (add_in_buff len c) s

  | [<  >] -> let str = get_buff len in
              if List.mem str keywords then Kwd str else Ident str

let rec string len = parser
    [< ''"' (*"\""*) >] -> String (get_buff len)
  | [< ''\\'; 'c; s >] -> string (add_in_buff len c) s
  | [< 'c; s >] -> string (add_in_buff len c) s
  | [< _ = Stream.empty >] -> failwith "string not terminated"

let next_token = parser
    [< ''a'..'z' | 'A'..'Z' | '_' | '$' as c ;
       i = (ident (add_in_buff 0 c))        >] -> i

  | [< ''\\' ; t = (parser [< ''/' >] -> Special "\\/"
                         | [< >]      -> Char '\\') >] -> t 

  | [< ''/' ; t = (parser [< ''\\' >] -> Special "/\\"
                        | [< >]       -> Char '/') >] -> t 

  | [< ''<' ; t = (parser
		     [< ''-' ; t0 = (parser [< ''>' >] -> Special "<->"
				          | [< >] -> Special "<-") >] -> t0
		   | [< ''=' >]      -> Special "<="
		   | [< >]           -> Char '<') >] -> t
 
  | [< ''>' ; t = (parser
		     [< ''=' >]      -> Special ">="
		   | [< >]           -> Char '>') >] -> t
 
  | [< ''-' ; t = (parser [< ''>' >] -> Special "->"
                        | [< >]      -> Char '-') >] -> t
 
  | [< ''=' ; t = (parser [< ''>' >] -> Special "=>"
                        | [< >]      -> Char '=') >] -> t
 
  | [< ''(' ; t = (parser [< ''*' >] -> CB
                        | [< >]      -> Char '(') >] -> t
 
  | [< ''*' ; t = (parser [< '')' >] -> CE
                        | [< >]      -> Char '*') >] -> t

  | [< ''"' ; t = string 0 >] -> t
 
  | [< 'c >] -> Char c

let lexer s =
  Stream.from (fun _ -> 
    try  let tok = next_token s in Some tok
    with Stream.Failure -> None)

let parse_line s =
  lexer (Stream.of_string s)


(* $Id: lexer.ml,v 1.8 1999/06/24 07:03:13 loiseleu Exp $ *)


