(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                ccidpc.ml                                 *)
(****************************************************************************)

open Dpctypes;;

open Names;;
open Pp;;
open Std;;
open More_util;;
open Generic;;
open Term;;
open Environ;;
open Termenv;;
open Himsg;;
open Tacmach;;
open Reduction;;
open Clenv;;
open Tactics;;
open Trad;;
open Pattern;;
open Tacticals;;
open Tactics;;

let mmk = make_module_marker ["#Prelude.obj"];;

let and_pattern   = put_pat mmk "(and ? ?)";;
let or_pattern    = put_pat mmk "(or ? ?)";;
let not_pattern   = put_pat mmk "(not ?)";;
let exist_pattern = put_pat mmk "(ex ? ?)";;

let and_head      = put_pat mmk "and";;
let or_head       = put_pat mmk "or";;
let not_head      = put_pat mmk "not";;
let exist_head    = put_pat mmk "ex";;

let is_imp_term t =
    match t with
    DOP2(Prod,_,DLAM(_,b)) -> (not((dependent (Rel 1) b)))
  | _ -> false
;;

let is_forall_term t =
    match whd_betaiota t with
    DOP2(Prod,_,DLAM(_,b)) -> dependent (Rel 1) b
  | _ -> false
;;

let ctr = ref 0;;
let gen_ident id = make_ident (atompart_of_id id) (incr ctr;!ctr);;
let gen_name a na = 
    let (Name id) = (Environ.named_hd a na)
    in gen_ident id;;

let dpc_of_cci_term lid = 
 let rec tradrec cciterm =
    let (hd,args) = whd_betaiota_stack cciterm [] in
    let dpcargs = List.map tradrec args
    in (match hd with
        VAR id -> if dpcargs=[] then Var id 
                  else if List.mem id lid 
                       then error "dpc_of_cci_term (not first order)"
                       else App (Var id :: dpcargs)

      | ( DOPN(Const _,_) | DOPN(MutInd _,_) | DOPN(MutConstruct _,_)) as t 
      	   -> if dpcargs=[] then Glob (CN t)
	                    else App (Glob (CN t) :: dpcargs)

      | _ -> errorlabstrm "dpc_of_cci_term"
                [< 'sTR"Not a first order formula" >] )
 in tradrec
    
;;


let cci_of_dpc_term tradsign sign = 
 let rec tradrec = function
    Var id -> if mem_sign tradsign id then VAR(snd(lookup_sign id tradsign))
              else VAR id
  | Glob (ID id) -> VAR id
  | Glob (CN t)  -> t
  | App (t::dpcargs) ->
      let t' = tradrec t in
      applist(t', List.map tradrec dpcargs)
 in tradrec
    
;;

let dpc_of_cci_fmla gls cciterm = 
 let rec tradrec lid cciterm =
    if matches gls cciterm and_pattern then
        let [a;b] = dest_match gls cciterm and_pattern
        in Conj(tradrec lid a,tradrec lid b)
    else if matches gls cciterm or_pattern then
        let [a;b] = dest_match gls cciterm or_pattern
        in Disj(tradrec lid a,tradrec lid b)
    else if matches gls cciterm not_pattern then
        let [a] = dest_match gls cciterm not_pattern
        in Neg(tradrec lid a)
    else if matches gls cciterm exist_pattern then
        let [_;b] = dest_match gls cciterm exist_pattern
        in (match b with
            DOP2(Lambda,a,(DLAM(na,_) as b)) ->
            let id = gen_name a na
            in Exists(id,tradrec (id::lid) (sAPP b (VAR id)))
          | _ -> error "dpc_of_cci_fmla (exists)")
    else if is_imp_term cciterm then
        let (DOP2(Prod,a,DLAM(_,b))) = cciterm
        in Imp(tradrec lid a,tradrec lid (pop b))
    else if is_forall_term cciterm then
        let (DOP2(Prod,a,(DLAM(na,_) as b))) = cciterm in
        let id = gen_name a na
        in ForAll(id,tradrec (id::lid) (sAPP b (VAR id)))
    else let (hd,args) = whd_betaiota_stack cciterm [] in
         let dpcargs = List.map (dpc_of_cci_term lid) args
         in (match hd with
             VAR id -> if List.mem id lid 
                   then errorlabstrm "dpc_of_cci_fmla"
                          [< 'sTR"Quantification over a predicate" >]
                   else Atom((ID id,0),dpcargs)

           | (DOPN(MutInd _,_) | DOPN(MutConstruct _,_) | DOPN(Const _,_)) as t
      	        -> Atom( (CN t,0) , dpcargs)

           | _ -> errorlabstrm "dpc_of_cci_flma"
                     [< pTERMINENV(gLOB(pf_hyps gls),hd); 'sPC ;
                        'sTR"is not an atomic predicate" >] )
 in tradrec [] (whd_betaiota cciterm)
    
;;

let rec alpha_term bl1 bl2 p_0 p_1 =
  match p_0,p_1 with
      ((Var id1), (Var id2)) ->
      	if not (List.mem id1 bl1) then
	  id1=id2
      	else
	  index id1 bl1 = index id2 bl2
    | ((Glob t1), (Glob t2)) -> t1=t2
    | ((App al1), (App al2)) ->
      	for_all2eq (alpha_term bl1 bl2) al1 al2
    | (_, _) -> false
;;

let forAllI gls =
    if is_forall_term (pf_concl gls) then
        Tactics.intro gls
    else tclFAIL gls;;

let forAllE id t gls =
    match pf_whd_betadeltaiota gls (pf_type_of gls (VAR id)) with
    DOP2(Prod,a,(DLAM _ as b)) as t_0 ->
    if is_forall_term t_0 then
        ((tclTHENS
         (cut_intro (sAPP b t)) ([tclIDTAC;
                exact (applist(VAR id,[t]))]))) gls
    else tclFAIL gls
;;  

let existE id gls =
    if matches gls (snd(lookup_sign id (pf_untyped_hyps gls))) exist_pattern then
        ((tclTHEN (simplest_elim (VAR id))
         (tclDO 2 (dImp None)))) gls
    else tclFAIL gls
;;

let negE id gls = 
    if matches gls (snd(lookup_sign id (pf_untyped_hyps gls))) not_pattern then
        (simplest_elim (VAR id)) gls
    else tclFAIL gls
;;

let exist_intro_head = put_pat mmk "ex_intro";;

let existI t gls =
    let (wc,kONT) = startWalk gls in
    let clause = mk_clenv_hnf_constr_type_of wc (get_pat exist_intro_head) in
    let clause' = clenv_constrain_missing_args [t] clause
    in res_pf kONT clause' gls
;;

let rec alpha_fmla bl1 bl2 p_0 p_1 =
  match p_0,p_1 with
      ((Atom((cn1,_),al1)), (Atom((cn2,_),al2))) ->
    	cn1=cn2 & for_all2eq (alpha_term bl1 bl2) al1 al2
    | ((Conj(a1,b1)),(Conj(a2,b2))) ->
    	alpha_fmla bl1 bl2 a1 a2 & alpha_fmla bl1 bl2 b1 b2

    | ((Disj(a1,b1)),(Disj(a2,b2))) ->
    	alpha_fmla bl1 bl2 a1 a2 & alpha_fmla bl1 bl2 b1 b2

    | ((Imp(a1,b1)),(Imp(a2,b2))) ->
    	alpha_fmla bl1 bl2 a1 a2 & alpha_fmla bl1 bl2 b1 b2

    | ((Neg(a1)), (Neg(a2))) -> alpha_fmla bl1 bl2 a1 a2

    | ((ForAll(x1,a1)), (ForAll(x2,a2))) ->
    	alpha_fmla (x1::bl1) (x2::bl2) a1 a2

    | ((Exists(x1,a1)), (Exists(x2,a2))) ->
    	alpha_fmla (x1::bl1) (x2::bl2) a1 a2

    | (_, _) -> false;;

let alpha_eq (kspine,m) (jspine,n) = alpha_fmla kspine jspine m n;;

let first_pred p = 
 let rec firstrec = function
    [] -> error "first_pred"
  | h::t -> if p h then h else firstrec t
 in firstrec
    
;;

let find_fmla_left (kspine,f) (jspine,gl) =
    let sign = pf_untyped_hyps gl in
    let ids = ids_of_sign sign
    in first_pred 
         (fun id -> ( try alpha_eq (kspine,f)
                        (jspine,dpc_of_cci_fmla gl (snd(lookup_sign id sign)))
             with _ -> false) 
         ) ids
;;

let onNthClause tac n gls =
    let cls = nth_clause n gls
    in onClause tac cls gls
;;

let elimTypeFalse gls =
  (elim_type (pf_global gls (id_of_string "False"))) gls
;;

let rec tradpf kspine jspine dpcpf gls =
    match dpcpf with

    Proof2(_,_,Axiom2 f) ->
    let id = find_fmla_left (kspine,f) (jspine,gls)
    in exact (VAR id) gls

  | Proof2(_,_,LWeakening2(_,pf)) -> trad kspine jspine pf gls

  | Proof2(_,_,RWeakening2(_,pf)) ->
     ( (tclTHEN (elimTypeFalse) (tradpf kspine jspine pf)) ) gls

  | Proof2(_,_,RConj2(f1,pf1,f2,pf2)) ->
    ((tclTHENS (dAnd None) ([trad kspine jspine pf1;
                      trad kspine jspine pf2]))) gls

  | Proof2(_,_,LConj2(f1,f2,pf)) ->
    let id = find_fmla_left (kspine,Conj(f1,f2)) (jspine,gls)
    in ((tclTHEN (dAnd (Some id)) ((trad kspine jspine pf)))) gls

  | Proof2(_,_,LDisj2(f1,pf1,f2,pf2)) ->
    let id = find_fmla_left (kspine,Disj(f1,f2)) (jspine,gls)
    in (match pf1 with 
          Proof2(_,[],_) -> ((tclTHENS (orE id) 
                            ([ ((tclTHEN (elimTypeFalse) (trad kspine jspine pf1)));
                              trad kspine jspine pf2]))) gls
        | _              -> ((tclTHENS (orE id) 
                            ([ trad kspine jspine pf1;
                              ((tclTHEN (elimTypeFalse) (trad kspine jspine pf2)))]))) gls
       )

  | Proof2(_,_,RImp2(f1,f2,pf)) ->
    ((tclTHEN (dImp None) ((trad kspine jspine pf)))) gls

  | Proof2(_,_,LImp2(f1,pf1,f2,pf2)) ->
    let id = find_fmla_left (kspine,Imp(f1,f2)) (jspine,gls)
    in ((tclTHENS (dImp (Some id))
	   ([trad kspine jspine pf2;
             trad kspine jspine pf1]))) gls

  | Proof2(_,_,RForAll2(kid,f,pf)) ->
    ((tclTHEN (forAllI)
     ((onLastHyp (fun (Some jid) ->
                     trad (kid::kspine) (jid::jspine) pf))))) gls

  | Proof2(_,_,LForAll2(kid,kterm,f,pf)) ->
    let jterm = cci_of_dpc_term (kspine,jspine) (pf_untyped_hyps gls) kterm in
    let id = find_fmla_left (kspine,ForAll(kid,f)) (jspine,gls)
    in ((tclTHEN (forAllE id jterm)
        (trad kspine jspine pf))) gls

  | Proof2(_,_,LExists2(kid,f,pf)) ->
    let id = find_fmla_left (kspine,Exists(kid,f)) (jspine,gls)
    in ((tclTHEN (existE id)
        ((onNthClause (fun (Some jid) ->
                          trad (kid::kspine) (jid::jspine) pf)
         (-2))))) gls

  | Proof2(_,_,RExists2(kid,kterm,f,pf)) ->
    let jterm = cci_of_dpc_term (kspine,jspine) (pf_untyped_hyps gls) kterm
    in ((tclTHEN (existI jterm) (tradpf kspine jspine pf))) gls

  | Proof2(_,_,RDisj2(f1,f2,Proof2(_,_,RWeakening2(f3,pf)))) ->
    if alpha_eq (kspine,f1) (kspine,f3) then
        ((tclTHEN (right []) (tradpf kspine jspine pf))) gls
    else if alpha_eq (kspine,f2) (kspine,f3) then
        ((tclTHEN (left []) (tradpf kspine jspine pf))) gls
    else error "Not Intuitionistic, eh?"

  | Proof2(_,_,RNeg2(f,pf)) -> 
      ((tclTHEN ((tclTHEN (red_in_concl) (Tactics.intro))) (tradpf kspine jspine pf))) gls

  | Proof2(_,_,LNeg2(f,pf)) -> 
     let id = find_fmla_left (kspine,Neg(f)) (jspine,gls)
     in ((tclTHEN (negE id) (tradpf kspine jspine pf))) gls

  | _ -> error "tradpf : Not an intuitionistic proof !"

and trad kspine jspine dpcpf gls =
    tradpf kspine jspine dpcpf gls
;;

(* $Id: ccidpc.ml,v 1.13 1999/08/06 20:49:21 herbelin Exp $ *)
