(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                              extraction.ml                               *)
(****************************************************************************)

open Std;;
open Pp;;
open Names;;
open Impuniv;;
open Constrtypes;;
open Environ;;
open Generic;;
open Evd;;
open Term;;
open Reduction;;
open Typing;;
open Termenv;;
open Vectops;;

let mt_evd = Evd.mt_evd();;

let whd_all = whd_betadeltaiota mt_evd;;

(* The constructive engine can be used for two purposes :
    - checking that a "logical" proof is well-formed.
    - writing a "constructive" proof and extracting its computational meaning
                (an Fomega + Ind + Fix + Implicit program)
Each term c of the Calculus of Construction contains a computational 
information that is :
    Inf(Judge(c',t,k))if c is already typed in Fw and if c' is its translation 
                      as an fterm.
    Logic             if c is a "logical" term (without computational meaning).

The information is initialized at the level of propositional variables :
    A : Prop(Null)   A is a logical propositional variable.
    A : Prop(Pos)   A is a constructive propositional variable.

This file contains the rule for computing the information during an 
application, generalization or abstraction.

When processing a term c equal to  (M N) or (x:N)M or [x:N]M,
three cases are distinguished :
  If M is a logical term then c is also a logical term.
  If M is a constructive term then depending of the level of N and of its
            information, the information of c will be the same as the 
            information of M or computed using the fterm extracted from N.

*)

(* a special constant to solve the problem of extracting informations 
where it is not compatible with the current type checking algorithm *)

let informative_construction = is_info_cast_type mt_evd;;

let informative_reference = function
    (DOPN(Const sp,_) as k) ->
    (match kind_of_path sp with
     FW -> true
   | CCI ->
     let (_,cb) = const_of_path sp in
     let {body=t;typ=s} = cb.cONSTTYPE in
       is_info_judge mt_evd {_VAL=k;_TYPE=t;_KIND=DOP0 (Sort s)}
   | OBJ ->
     anomalylabstrm "extraction__informative_reference"
     [< 'sTR"Found a constant with a ill-formed path-kind" >])

  | (DOPN(MutInd (sp,tyi),_)) ->
    (match kind_of_path sp with
     FW -> true
   | CCI ->
     let (_,mib) = mind_of_path sp in
     let ar = mib.mINDPACKETS.(tyi).mINDARITY
     in informative_construction (DOP2 (Cast, ar.Term.body, DOP0 (Sort (ar.typ))))
   | OBJ ->
     anomalylabstrm "extraction__informative_reference"
     [< 'sTR"Found a constant with a ill-formed path-kind" >])
   | x -> errorlabstrm "extraction__informative_reference"
          [< 'sTR"Malformed reference" >]
;;

let rec is_implicit = function 
    DOP0(Implicit) -> true 
  | DOPN(AppL,cl) -> is_implicit cl.(0)
  | DOPN(MutCase _,_) as mc -> let (_,_,c,_) = destCase mc in is_implicit c
  | DOP2(Cast,c,_) -> is_implicit c
  | DOP2(Prod,_,c) -> is_implicit c
  | DOP2(Lambda,_,c) -> is_implicit c
  | DLAM(_,c) -> is_implicit c
  | DOPN(Fix(_,i),cl) -> is_implicit_proj i (last_vect cl)
  | DOPN(CoFix(i),cl) -> is_implicit_proj i (last_vect cl)
  | _ -> false
and is_implicit_proj i = function 
    DLAM(_,c) -> is_implicit_proj i c
  | DLAMV(_,c) -> is_implicit c.(i)
  | _ -> anomaly "is_implicit_proj"
;;
   
let contains_implicit sigma = 
 let rec srec c = match whd_betadeltaiota_stack sigma c [] with 
    (DOP0(Implicit),l) -> true 
  | DOPN(_,cl),l -> exists_vect srec cl or List.exists srec l
  | DOP2(_,c1,DLAM(_,c2)),l -> srec c1 or srec c2 or List.exists srec l
  | _,l -> List.exists srec l
 in srec 
 ;;

let kind_of_inf = function
  | (Inf j) -> j._KIND
  |  _ -> anomaly "kind_of_inf"
;;

let inf_var env cts n =
    match cts with
    Null -> Logic
  | _ ->
    let {body=typ;typ=s} = snd(lookup_glob n env) in
      Inf{_VAL=VAR n;
          _TYPE= typ;
          _KIND = DOP0(Sort s)}
;;

let inf_sort = function
    Prop Null -> Logic 
  | (Prop Pos as t)  ->
    Inf{_VAL= DOP0(Sort t);
        _TYPE= DOP0(Sort(Type(prop_univ)));
        _KIND = DOP0(Sort(Type(prop_univ)))}
  | Type(u) as t -> 
    Inf{_VAL= DOP0(Sort t);
        _TYPE= DOP0(Sort t);
        _KIND = DOP0(Sort t)}
;;

let inf_apply p_0 p_1 =
  match p_0,p_1 with
    (Logic, _) -> Logic
  | ((Inf c1 as funinf), Logic) -> 
          let typ = whd_all c1._TYPE in  
               (match typ with DOP2(Prod,b,c) -> 
                     if is_type_arity mt_evd b then 
              Inf{_VAL= mkImplicit ; _TYPE= mkImplicit; _KIND = mkImplicit}
                        else funinf
                            | _ -> funinf)
  | ((Inf c1 as funinf), (Inf c2)) ->
    if le_kind_implicit c1._KIND c2._KIND then
        let app_val = applist(c1._VAL,[c2._VAL])
        and app_ty = let typ = whd_all c1._TYPE 
                     in  match typ with DOP2(Prod,_,_) ->
                        hnf_prod_app mt_evd "inf_apply" typ c2._VAL
                          |_ -> if is_implicit typ then mkImplicit 
                        else 
                     (* found a non-product type which was not implicit *)
                         hnf_prod_app mt_evd "inf_apply" typ c2._VAL
        in Inf{_VAL= app_val ;
               _TYPE= app_ty;
               _KIND = c1._KIND}
    else funinf
;;

(* No type cumulativity in extracted programs *)
let type_of_product domtyp rangtyp = rangtyp;;

(* In extracted programs, Type:Type *)
let type_of_type t =
  match t with
      DOP0(Sort(Type _)) -> t
    | DOP0(Sort(Prop cts)) -> mkType prop_univ
    | DOP0(Implicit) -> mkImplicit
    | _ -> invalid_arg "type_of_type called with something which wasn't Type"
;;

let j_val j = mkCast j._VAL j._TYPE;;

let rec sort_of_arity sigma c =
     match whd_betadeltaiota sigma c with
     DOP0(Sort(Prop(_))) as c'      -> c'
   | DOP0(Sort(Type _)) as c'      -> c'
   | DOP2(Prod,_,DLAM(_,c2)) -> sort_of_arity sigma c2
   | DOP0(Implicit) -> mkImplicit
   | _            -> error "sort_of_arity: Not an arity";;

let inf_gen_rel name p_0 p_1 =
  match p_0,p_1 with
    (Logic, _) -> Logic
  | ((Inf c), Logic) -> Inf(j_pop c)
  | ((Inf c), (Inf tv)) ->
    if le_kind_implicit c._TYPE tv._TYPE then
        let res_type = type_of_product (whd_all tv._TYPE) (whd_all c._TYPE)
        in Inf{_VAL = DOP2(Prod,tv._VAL,DLAM(name,c._VAL));
               _TYPE = res_type;
               _KIND = type_of_type res_type}
    else Inf(j_pop c)
;;

let abs_compat p_0 p_1 =
  match p_0,p_1 with
    (Logic, _) -> anomaly "abs_compat"
  | (_, Logic) -> false
  | ((Inf main), (Inf var)) -> le_kind_implicit main._KIND var._TYPE
;;

let inf_abs_rel name p_0 p_1 =
  match p_0,p_1 with
    (Logic, _) -> Logic
  | ((Inf c), Logic) -> Inf(j_pop c)
  | ((Inf c), (Inf tv)) ->
    if le_kind_implicit c._KIND tv._TYPE then
        let abs_val = (DOP2(Lambda,tv._VAL,DLAM(name,c._VAL)))
        and abs_ty = DOP2(Prod,tv._VAL,DLAM(name,c._TYPE))
        and abs_tyty = type_of_product (whd_all tv._TYPE) (whd_all c._KIND)
        in Inf{_VAL= abs_val;
               _TYPE= abs_ty;
               _KIND = abs_tyty}
    else Inf(j_pop c)
;;

let inf_abs_rel_count name p_0 p_1 =
  match p_0,p_1 with
    (Logic, _) -> Logic,ERASE
  | ((Inf c), Logic) -> Inf(j_pop c),ERASE
  | ((Inf c), (Inf tv)) ->
    if le_kind_implicit c._KIND tv._TYPE then
        let abs_val = (DOP2(Lambda,tv._VAL,DLAM(name,c._VAL)))
        and abs_ty = DOP2(Prod,tv._VAL,DLAM(name,c._TYPE))
        and abs_tyty = type_of_product (whd_all tv._TYPE) (whd_all c._KIND)
        in Inf{_VAL= abs_val;
               _TYPE= abs_ty;
               _KIND = abs_tyty},ABSTRACT
    else Inf(j_pop c),ERASE
;;

(* Inutile
let inf_abs_var name varty inf =
let varinf = (Inf{_VAL=varty;_TYPE=cast_type varty;
                  _KIND = type_of_type (cast_type varty)})
in (inf_abs_rel (Name name) (inf_app (j_subst_var name) inf) varinf,
    if abs_compat inf varinf then
        ABSTRACT
    else ERASE)
;;
*)

let inf_rel env cts d =
    match cts with
    Null -> Logic
  | _ ->
    let {body=typ;typ=s} = snd(lookup_rel d env) in
      Inf{_VAL = Rel d;
          _TYPE = lift d typ;
          _KIND = DOP0 (Sort s)}
;;

(* Inductive types in Fw *)

let extract = function Inf(c) -> c | _ -> anomaly "not informative";;
let extract_val = function Inf(c) -> c._VAL | _ -> anomaly "not informative";;
let extract_typ = function Inf(c) -> c._TYPE | _ -> anomaly "not informative";;
let extract_kind = function Inf(c) -> c._KIND | _ -> anomaly "not informative";;

let inf_mutcase env sigma pj ci cinf lfinf = 
    match cinf with Inf cj -> 
    (try if Array.length lfinf = 1 then 
    (* check if it is a singleton *)
       match ci with 
	   None -> anomaly "Extraction.inf_mutcase" 
         | Some (sp,_) -> if is_singl sp then 
            let f = extract_val (hd_vect lfinf) 
            in  Inf{_VAL= mkAppL [| f; cj._VAL |];
                    _TYPE= pj._VAL;
                    _KIND = pj._TYPE}
	   else raise Singleton
       else raise Singleton
     with Singleton -> 
     let (rec_ty,kind) = 
       if contains_implicit sigma pj._TYPE or contains_implicit sigma cj._VAL 
       then (mkImplicit,mkImplicit)
       else let cit = whd_betadeltaiota sigma cj._TYPE in
         if is_implicit cit then (mkImplicit,mkImplicit)
         else (type_case env sigma cit pj._TYPE pj._VAL cj._VAL,
               sort_of_arity mt_evd pj._TYPE) in
     let val_0 =
       if is_Type cj._KIND or is_Type kind or is_implicit cj._TYPE then mkImplicit
       else (try
        let (mind,_) = find_mrectype sigma cj._TYPE in
	let lfext = Array.map extract_val lfinf in
	mkMutCaseA (ci_of_mind mind) pj._VAL cj._VAL lfext
        with Induc -> mkImplicit)
     in Inf{_VAL=val_0; _TYPE = rec_ty; _KIND = kind})
   | _ -> if Array.length lfinf = 1 
          then let jf = lfinf.(0) in 
              if conv_x mt_evd pj._VAL (extract_typ jf) then jf 
              else anomaly "The elimination should be on an informative term"
          else anomaly "The elimination should be on an informative term";;

let inf_fix sigma i aityp lfi inflars infdefs_ind = 
(* collect the informative components, contains at least one of them Ai *)
let rec collect  k (lind,lf,larv,ldefj) = 
    function [] -> (lind,List.rev lf,larv,ldefj)
         | (fi::lfrest) -> (match inflars.(k) with 
              Inf ark -> let (infdef,indk) = infdefs_ind.(k) 
                in collect (k-1)
               (indk::lind,fi::lf,ark._VAL::larv,(extract infdef)::ldefj) lfrest
             | Logic   -> collect (k-1) (lind,lf,larv,ldefj) lfrest)

in let (lind,lf,larv,ldefj) = collect (List.length lfi -1) ([],[],[],[]) lfi
in let vdefj = Array.of_list ldefj
in let newdef = put_DLAMSV lf (Array.map j_val vdefj) 
in let fix = DOPN(Fix(Array.of_list lind,i),
                  Array.append (Array.of_list larv) [|newdef|])     
   in Inf{_VAL=fix;
          _TYPE=List.nth larv i;
          _KIND=aityp};;
   

let inf_cofix sigma i aityp lfi inflars infdefs  = 
let rec collect  k (lf,larv,ldefj) = 
    function []           -> (List.rev lf,larv,ldefj)
             | (fi::lfrest) -> 
                    match inflars.(k) with 
                          Inf ark ->  collect (k-1) 
                                              (fi::lf,ark._VAL::larv,
                                                 (extract infdefs.(k))::ldefj) 
                                              lfrest
                       | Logic   -> collect (k-1) (lf,larv,ldefj) lfrest

in let (lf,larv,ldefj) = collect (List.length lfi -1) ([],[],[]) lfi
in let vdefj = Array.of_list ldefj
in let newdef = put_DLAMSV lf (Array.map j_val vdefj) 
in let cofix = DOPN(CoFix i,Array.append (Array.of_list larv) [|newdef|])     
   in Inf{_VAL=cofix;
          _TYPE=List.nth larv i;
          _KIND=aityp};;

(* Reductions during extraction *)
let extraction_expanded = ref false;;
let is_extraction_expanded () = !extraction_expanded;;
let make_extraction_expanded b = extraction_expanded :=b;;

Options.declare_async_bool_option
  {Options.optasyncname = "expansion during extraction";
   Options.optasynckey   = Options.PrimaryTable ("Extraction");
   Options.optasyncread = is_extraction_expanded;
   Options.optasyncwrite = make_extraction_expanded};;


(* $Id: extraction.ml,v 1.19 1999/08/06 20:49:14 herbelin Exp $ *)
