(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 lib.mli                                  *)
(****************************************************************************)

open Names;;
open Libobject;;

type node =
    LEAF of obj
  | OpenDir of string * open_dir_contents
  | ClosedDir of
      	  string * open_dir_contents * closed_dir_contents * library_segment
  | Import of string * section_path * bool

and open_dir_contents = {module_p : bool;
                         imports : section_path list}

and closed_dir_contents = {exports : section_path list;
                           mutable export_fn : string option}

and library_segment = (section_path * node) list;;
type library_entry = section_path * node;;
type frozen_t = library_segment;;

val init : unit -> unit;;

val add_entry : (section_path * node)  -> (section_path * node);;
val add_leaf_entry : (section_path * obj) -> (section_path * node);;
val add : (identifier * path_kind) * node -> (section_path * node);;
val add_leaf : (identifier * path_kind) * obj -> (section_path * node);;
val add_module : section_path * node -> unit;;
(* Adds an object to the library.  If the identifier is provided, then
 * the object is added to the library with the section_path gotten by
 * taking the current path-prefix, and adding the identifier and path_kind.
 *)

val reset_to : section_path -> unit;;
val reset_keeping : section_path -> unit;;
val reset_to_P : (section_path * node -> bool) -> unit;;
val reset_keeping_P : (section_path * node -> bool) -> unit;;
(* Resets the library to everything before the specified object.
 * This object cannot be in a closed directory - that is an error.
 *)

val app : ((section_path * node) -> unit) -> unit;;

val leaf_object_tag : section_path -> string;;
val map : section_path -> node;;
val map_leaf : section_path -> obj;;
(* Produces the library-object referenced by the supplied section_path *)

val find_entry_P : ((section_path * node) -> bool) -> section_path;;
val find_dir : string option -> section_path;;
(* finds the section_path of the named directory, which must be open.
 * if there is no name, then it finds the nearest still-open directory.
 *)

val open_dir : string -> open_dir_contents -> (section_path * node);;
(* opens a directory with the desired name. *)

val close_dir : section_path -> closed_dir_contents -> unit;;
(* Closes the named directory, verifying that there are no intervening
 * open directories.
 *)

val contents_after : section_path option -> library_segment;;
(* Given a section_path, returns the list of objects which follow it. *)

val make_path : path_kind -> identifier -> section_path;;
val cwd : unit -> string list;;

val freeze : unit -> frozen_t;;
val unfreeze : frozen_t -> unit;;

(* The operation of section closing consists of (1) discovering the
   section_path corresponding to the nearest open directory, (2)
   getting the contents which follows this OpenDir, (3) closing the
   directory, (4) re-installing the entries which remain.

   The operation of resetting to a section or an entry is just the
   reset_to command.

 *)

val node_name : (section_path * node) -> identifier;;

val next_anonymous_id : unit -> identifier;;

val last_entry : unit -> section_path option;;

(* $Id: lib.mli,v 1.7 1999/06/29 07:47:26 loiseleu Exp $ *)
