(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                impuniv.ml                                *)
(****************************************************************************)

(* Universes are stratified by a partial ordering >=.
   Let ~ be the associated equivalence. We also have a strict ordering
   > between equivalence classes, and we maintain that > is acyclic,
   and contained in >= in the sense that [U]>[V] implies U>=V.

   At every moment, we have a finite number of universes, and we
   maintain the ordering in the presence of assertions U>V and U>=V.

   The equivalence ~ is represented by a tree structure, as in the
   union-find algorithm. The assertions > and >= are represented by
   adjacency lists *)

open Std;;
open Pp;;
open More_util;;
open Names;;

type universe = {u_sp : section_path ; u_num : int};;

let pr_uni u = [< 'sTR(string_of_path u.u_sp) ; 'sTR"." ; 'iNT u.u_num >];;

type relation = 
   Greater of bool * universe * relation (* if bool then > else >= *)
 | Equiv of universe
 | Terminal;;


type arc = Arc of universe * relation
and universes = (universe,arc) Fmavm.frozen_t;;

let uni_ord u1 u2 =
    let sp_bit = sp_ord u1.u_sp u2.u_sp
    in if sp_bit < 0 then -1
       else if sp_bit > 0 then 1
       else u1.u_num - u2.u_num
;;

(* in Arc(u,Greater(b,v,r))::arcs, we have u>v if b, and u>=v if not b, 
   and r is the next relation pertaining to u; this relation may be
   Greater or Terminal. *)


(* points to the first arc pertaining to u 

let arc_of u = arc_of_u where rec arc_of_u = function
    [] -> anomaly "Wrong universe structure"
  | (Arc(v,_) as arc)::l -> if u=v then arc else arc_of_u l;;

A completely applicative structure would search the graph structure
with (arc_of u graph). We prefer, for efficiency, give a direct access
as ARC.(Int i), where (Universe i) = u. *)


(* The latest universe *)
let cnt_uni = ref 0;;
let curr_sp = ref (None : section_path option);;

Summary.declare_summary "Universes_numbering"
  {Summary.freeze_function = (fun () -> !cnt_uni);
   Summary.unfreeze_function = (fun x -> cnt_uni:=x);
   Summary.init_function = (fun () -> cnt_uni:=0)}
;;

let mk_univ () =
  incr cnt_uni;
  {u_sp = outSOME !curr_sp; u_num = !cnt_uni };;

let dummy_sp = make_path ["univ"] (id_of_string "dummy") OBJ;;
let dummy_univ = {u_sp = dummy_sp; u_num = 0};; (* for prover terms *)


(* Fast access to first arc pertaining to given universe *)
let local_graph = (Fmavm.create (uni_ord,17) : (universe,arc) Fmavm.t);;
let set_uniarc universes = Fmavm.unfreeze universes local_graph;;
let get_uniarc ()        = Fmavm.freeze local_graph;;
let empty_universes = get_uniarc();;

let enter_arc a =
  let Arc(i,_) =a in
  if Fmavm.in_dom local_graph i then Fmavm.remap local_graph i a
  else Fmavm.add local_graph (i,a)
;;

let declare_univ u =
  if not (Fmavm.in_dom local_graph u)
  then Fmavm.add local_graph (u,Arc(u,Terminal));;


(* Creates an unconstrained new universe. *)
let new_univ () = 
  if !curr_sp = None then dummy_univ
  else
    let v = mk_univ() in enter_arc(Arc(v,Terminal)); v



(* Operations with universe backtracking *)

let uni_pred p a =
  let u = get_uniarc() in
    try
      if p a then true
      else (set_uniarc u;false)
    with reraise -> (set_uniarc u; raise reraise)
;;

let uni_or p q =
  let u = get_uniarc() in
    try
      if p() then true
      else
        (set_uniarc u;
         if q() then true
         else (set_uniarc u;false))
    with reraise -> (set_uniarc u; raise reraise)
;;

(* If we adjust universes (flag = true), we have to use uni_or for disjunction
 * If not, the logical or is sufficient
 *)
let uni_or_if flag f1 f2 =
  if flag then uni_or f1 f2 else (f1() or f2());;



let with_universes f (sp,u,a) =
  if !curr_sp <> None then anomaly "already inside a with_universes"
  else
    try (curr_sp := Some sp;
         set_uniarc u;
         let r = f a in
         let u' = get_uniarc ()
         in (curr_sp := None;
             (u',r)))
    with reraise -> (curr_sp:=None; raise reraise)
;;

(* The universes of Prop and Set: Type_0, Type_1 and Type_2, and the
   resulting graph. *)
let (initial_universes,(prop_univ,prop_univ_univ,prop_univ_univ_univ)) =
  let prop_sp = make_path ["univ"] (id_of_string "prop_univ") OBJ in
    with_universes
      (fun () ->
         let u = new_univ() in
         let su = new_univ() in
         let ssu = new_univ() in
           enter_arc(Arc(su,Greater(true,u,Terminal)));
           enter_arc(Arc(ssu,Greater(true,su,Terminal)));
           (u,su,ssu))
      (prop_sp, empty_universes, ())
;;


(* Every universe has a unique canonical arc representative *)

(* repr : universe -> arc *)
(* canonical representative : we follow the Equiv links *)
let rec repr u = 
  let arc =
    try Fmavm.map local_graph u
    with Not_found -> anomalylabstrm "Impuniv.repr"
	[< 'sTR"Universe "; pr_uni u; 'sTR" undefined" >] in
  match arc with 
    Arc(_,Equiv(v)) -> repr v
  | _ -> arc
;;

let can = List.map repr;;

(* transitive closure : we follow the Greater links *)
(* close : relation -> universe list * universe list *)
let close = 
 let rec closerec ((u,v) as pair) = function
    Terminal           -> pair
  | Greater(true,v_0,r)  -> closerec (v_0::u,v) r
  | Greater(false,v_0,r) -> closerec (u,v_0::v) r
  | _ -> anomaly "Wrong universe structure"
 in closerec ([],[]) 
;;

(* reprgeq : arc -> arc list *)
(* All canonical arcv such that arcu>=arcc with arcv#arcu *)
let reprgeq (Arc(_,ru) as arcu) =
  let (_,v) = close ru in
  let rec searchrec w = function
      []       -> w
    | v_0 :: v ->
	let arcv = repr v_0 in
        if List.memq arcv w or arcu=arcv then searchrec w v
        else searchrec (arcv :: w) v
  in searchrec [] v
;;

(* collect : arc -> arc list * arc list *)
(* collect u = (V,W) iff V={v canonical | u>v} W={w canonical | u>=w}-V *)
(* i.e. collect does the transitive closure of what is known about u *)
let collect u = 
 let rec coll_rec v w = function
    [],[] -> (v,subtractq w v)
  | (Arc(_,rv) as arcv)::v',w' ->
      if List.memq arcv v then coll_rec v w (v',w')
      else
        let (gt,geq) = close rv
        in coll_rec (arcv::v) w (can(gt@geq)@v',w')
  | [],(Arc(_,rw) as arcw)::w' -> 
      if (List.memq arcw v) or (List.memq arcw w) then coll_rec v w ([],w')
      else
        let (gt,geq) = close rw 
        in coll_rec v (arcw::w) (can(gt),can(geq)@w')
 in coll_rec [] [] ([],[u])
;;

type order = EQ | GT | GE | NGE;;

(* compare : universe -> universe -> order *)
let compare(u,v) = 
    let arcu = repr(u) and arcv = repr(v)
    in if arcu=arcv then EQ
       else let (v,w) = collect(arcu)
            in if List.memq arcv v then GT
               else if List.memq arcv w then GE
               else NGE;;

(* Invariants : compare(u,v) = EQ <=> compare(v,u) = EQ
                compare(u,v) = GT or GE => compare(v,u) = NGE
                compare(u,v) = NGE => compare(v,u) = NGE or GE or GT

   Adding u>=v is consistent iff compare(v,u) # GT 
    and then it is redundant iff compare(u,v) # NGE
   Adding u>v is consistent iff compare(v,u) = NGE 
    and then it is redundant iff compare(u,v) = GT *)


(* between : universe -> arc -> arc list *)
(* we assume  compare(u,v) = GE with v canonical    *)
(* between u v = {w|u>=w>=v, w canonical}          *)     
let between u arcv = 
  let rec explore (memo,l) arcu = 
    try (memo,unionq (List.assq arcu memo) l) (* when memq arcu memo *)
    with Not_found ->
      let w = reprgeq arcu in
      let (memo',sols) = List.fold_left explore (memo,[]) w in
      let sols' = if sols=[] then [] else arcu::sols in
      ((arcu,sols')::memo', unionq sols' l) in
  snd (explore ([(arcv,[arcv])],[]) (repr u))
;;
(* Note: hd(between u v) = repr u  *)
(* between is the most costly operation *)

(* setgt : universe -> universe -> unit *)
(* forces u > v *)
let setgt u v =
  let Arc(u',ru) = repr u in
  enter_arc(Arc(u',Greater(true,v,ru))); ()
;;

(* checks that non-redondant *)
let setgt_if u v = match compare(u,v) with
    GT -> ()
  | _ -> setgt u v;;

(* setgeq : universe -> universe -> unit *)
(* forces u >= v *)
let setgeq u v =
  let Arc(u',ru) = repr u in
  enter_arc(Arc(u',Greater(false,v,ru))); ()
;;

(* checks that non-redondant *)
let setgeq_if u v = match compare(u,v) with
    NGE -> setgeq u v
  | _ -> ();;

(* merge : universe -> universe -> unit *)
(* we assume  compare(u,v) = GE *)
(* merge u v  forces u ~ v with repr u as canonical repr *)
let merge u v =
  match between u (repr v) with
    Arc(u',_)::v ->
      let redirect (w,w') (Arc(v',rv)) =
       	let v,v'_0 = close rv in
 	enter_arc (Arc(v',Equiv(u')));
 	(unionq v w,v'_0@w') in
      let (w,w') = List.fold_left redirect ([],[]) v in
      List.iter (setgt_if u') w;
      List.iter (setgeq_if u') w'
  | [] -> anomaly "between"
;;

(* merge_disc : universe -> universe -> unit *)
(* we assume  compare(u,v) = compare(v,u) = NGE *)
(* merge_disc u v  forces u ~ v with repr u as canonical repr *)
let merge_disc u v =
  let (Arc(u',_), Arc(v',rv)) = (repr u, repr v) in
  let v,v'_0 = close rv in
  enter_arc (Arc(v',Equiv(u')));
  List.iter (setgt_if u') v;
  List.iter (setgeq_if u') v'_0;
;;


(* Universe inconsistency: error raised when trying to enforce a relation
   that would create a cycle in the graph of universes. *)
let error_inconsistency () =
  errorlabstrm "Impuniv.error_inconstency" [< 'sTR"Universe Inconsistency" >]

(* enforcegeq : universe -> universe -> unit *)
(* enforcegeq u v will force u>=v if possible, will fail otherwise *)
let enforcegeq u v =
  if !curr_sp = None then () else
  (declare_univ u; declare_univ v;
   match compare(u,v) with
        NGE -> (match compare(v,u) with
                  GT -> error_inconsistency()
                | GE -> merge v u
                | NGE -> setgeq u v
                | EQ -> anomaly "compare")
      | _ -> ());;

(* enforceq : universe -> universe -> unit *)
(* enforceq u v will force u=v if possible, will fail otherwise *)
let enforceq u v =
  if !curr_sp = None then () else
  (declare_univ u; declare_univ v;
   match compare(u,v) with
        EQ -> ()
      | GT -> error_inconsistency()
      | GE -> merge u v
      | NGE -> match compare(v,u) with
                    GT -> error_inconsistency()
                  | GE -> merge v u
                  | NGE -> merge_disc u v
                  | EQ -> anomaly "compare");;

(* enforcegt u v will force u>v if possible, will fail otherwise *)
let enforcegt u v =
  if !curr_sp = None then () else
  (declare_univ u; declare_univ v;
   match compare(u,v) with
        GT -> ()
      | GE -> setgt u v
      | EQ -> error_inconsistency()
      | NGE -> (match compare(v,u) with
                     NGE -> setgt u v
                   | _ -> error_inconsistency()));;


let enforce_relation u = 
 let rec enfrec = function
    Terminal -> ()
  | Equiv v -> enforceq u v
  | Greater(false,v,r) -> (enforcegeq u v; enfrec r)
  | Greater(true,v,r) -> (enforcegt u v; enfrec r)
 in enfrec
    
;;

(* Merging 2 universe graphs *)
let merge_universes sp u1 u2 =
  let ouni = get_uniarc() in
  let am = Avm.unfreeze u2 (Avm.create uni_ord) in
  fst(with_universes
        (fun () -> Avm.app (function (_,Arc(u,r)) -> enforce_relation u r) am)
        (sp,u1,()))
;;


(* Returns the least upper bound of universes u and v. If they are not
   constrained, then a new universe is created.
   Used to type the products. *)
let sup(u,v) = 
  if !curr_sp = None then dummy_univ
  else
    (declare_univ u; declare_univ v;
     match compare(u,v) with
         NGE -> (match compare(v,u) with
                     NGE -> let w = new_univ() in (setgeq w u; setgeq w v; w)
                   | _   -> v)
       | _   -> u);;

(* Returns a fresh universe, juste above u. Does not create new universes
   for Type_0 (the sort of Prop and Set).
   Used to type the sort u. *)
let super u = 
  if !curr_sp = None then dummy_univ
  else if u = prop_univ then prop_univ_univ 
  else if u = prop_univ_univ then prop_univ_univ_univ
  else
    (declare_univ u;
     let v = mk_univ() in
     enter_arc(Arc(v,Greater(true,u,Terminal))); v);;




(* Pretty-printing *)
let num_universes u =
  let am = Avm.unfreeze u (Avm.create uni_ord)
  in List.length (Avm.dom am)
;;

let num_edges u =
  let am = Avm.unfreeze u (Avm.create uni_ord) in
  let reln_len = 
    let rec lenrec n = function
        Terminal -> n
      | Equiv _ -> n+1
      | Greater(_,_,r) -> lenrec (n+1) r
    in lenrec 0 in
  let edges = ref 0 in
  Avm.app (function (_,Arc(_,r)) -> edges := !edges + (reln_len r)) am;
  !edges
;;

let pr_reln u r = 
 let rec prec = function
    Greater(true,v,r) -> [< pr_uni u ; 'sTR">" ; pr_uni v ; 'fNL ; prec r >]
  | Greater(false,v,r) -> [< pr_uni u ; 'sTR">=" ; pr_uni v ; 'fNL ; prec r >]
  | Equiv v -> [< pr_uni u ; 'sTR"=" ; pr_uni v >]
  | Terminal -> [< >]
 in prec r
    
;;

let pr_universes u =
  let am = Avm.create uni_ord in
  let am' = Avm.unfreeze u am in
  let graph = Avm.toList am in
  prlist_with_sep pr_fnl (function (_,Arc(u,r)) -> pr_reln u r) graph
;;


(* $Id: impuniv.ml,v 1.16 1999/11/07 17:57:13 barras Exp $ *)
