/*
 * Author:	Peter Holst Andersen
 * Last change:	August, 1994
 * Contents:	Constants.
 *		Declaration of types, external variables, and functions. 
 */

#ifndef __RAY_H
#define __RAY_H

#define FILE_RAY_H __FILE__

#define TRUE	1
#define FALSE	0

#define DEFAULT_SCENE		1               /* Default scene number      */
#define DEFAULT_HEIGHT		128		/* Default image height	     */
#define DEFAULT_WIDTH		128		/* Default image width	     */
#define MAXWIDTH                1024            /* Maximum image width       */

#define MAXLEVEL		5		/* Default maximum depth of
						 * the trace tree	     */
#define MINWEIGHT		0.002		/* Default minimum weight of
						 * the trace tree	     */

#define MIN_DISTANCE		0.001		/* Minimum distance for
						 * intersection tests	     */

#define BACKGROUND_RED			0.0	/* Background color	     */
#define BACKGROUND_GREEN		0.3
#define BACKGROUND_BLUE			0.3

#define MAX_ARITY_OBJECT_TREE		50
#define MAX_OBJECTS_SCENE		40
#define MAX_SURFACES_SCENE		7
#define MAX_LIGHTS_SCENE		7

/* object tags */

#define NONE			0
#define BOUNDING_SPHERE		1
#define SPHERE			2
#define DISC			3
#define SQUARE			4

/* surface tags */

#define SIMPLE		1
#define CHECKED		2

#define EPSILON         1.0e-12
#define fzero(a)        (fabs(a) < EPSILON)
#ifndef PI
#  define PI		3.14159265
#endif
#define deg2rad(x)	(x * PI / 180.0)

#define myfloat double
typedef unsigned short ush;
typedef struct pointType { myfloat x, y, z; } pointType;
typedef struct colorType { myfloat red, green, blue; } colorType;
typedef pointType vectorType;
typedef struct rayType { pointType p; vectorType v; } rayType;

typedef struct sphereType{/* A sphere centered in (c.x, c.y, c.z) 	*/
  pointType c;
  vectorType n, v1, v2;   /* with north pole n, equator v1,             */
  myfloat r2, r;	  /* radius r. r2 = r*r	       	                */
} sphereType;             /* v2 normal to n and v1                      */

typedef struct squareType{/* The square lies in the plane described by  */
                          /* n.x * x + n.y * y + n.z * z = d            */
  pointType c;	          /* The square is centered in (c.x, c.y, c.z)  */
  vectorType n, v1, v2;   /* (v1, v2) is the orientation of the square  */
  myfloat r2, d;	  /* If s is the sidelength, then r2 is (s/2)^2 */
} squareType;		  

typedef struct discType{  /* The disc lies in the plane described by    */
  pointType c;	          /* n.x * x + n.y * y + n.z * z = d		*/
  vectorType n, v1, v2;   /* The disc is centered in (c.x, c.y, c.z)	*/
  myfloat r2, d, r;	  /* r is the radius of the disc, and r2 = r*r	*/
} discType;

typedef struct surfaceInfoType {
  colorType ambient;	  /* Ambient reflectance constant 		*/
  colorType diffuse;	  /* Diffuse reflectance constant		*/
  myfloat reflectivity;	  /* Specular reflection constant		*/
  myfloat transparency;	  /* Transmission constant			*/
  myfloat refrindex;	  /* Refraction index				*/
  int specpow;		  /* Specular exponent (= Ns/2)			*/
} surfaceInfoType;

typedef struct lightType {/* Positional light source in p with		*/
  pointType p;		  /* intensity c				*/
  colorType c;
} lightType;

typedef struct checkedType{/*For checked surfaces			*/
  int s1, s2;		  /* Surface for the "squares"			*/
  myfloat checksize;	  /* Checksize					*/
  myfloat dbl_checksize;  /* Twice the checksize			*/
} checkedType;

typedef union surfaceSubType {
  surfaceInfoType sinfo;  /* For simple surfaces			*/
  checkedType checked;    /* For checked surfaces			*/
} surfaceSubType;

typedef struct surfaceType {
  int tag;		  /* SIMPLE or CHECKED				*/
  surfaceSubType u;
} surfaceType;

union propertyType {
  int skip_on_miss;  	  /* For BOUNDING_SPHERES			*/
  int surface;	          /* For SPHERE, DISC, and SQAURE		*/
};

union objectSubType {
  sphereType sphere;
  discType   disc;
  squareType square;
};

typedef struct objectType {
  int tag;	          /* NONE/BOUNDING_SPHERE/SPHERE/DISC/SQUARE    */
  union propertyType prop;
  union objectSubType obj;
} objectType;

typedef struct viewType {
  pointType pos;	  /* Eyeposition				*/
  pointType lookp;	  /* The point we are looking at		*/
  vectorType up;	  /* View up vector			 	*/
  vectorType dir;	  /* dir = (lookp-pos)/|lookp-pos|	 	*/
  myfloat lookdist;	  /* lookdist = |lookp-pos|		 	*/
  myfloat hfov, vfov;	  /* Horisontal and vectical field of view 	*/
} viewType;

typedef struct screenType {
  vectorType firstv;	  /* Direction of the first primary ray	 	*/
  vectorType scrnx, scrny;/* Increment for the other primary rays  	*/
  vectorType scrni;	  /* scrni = norm(view.dir x view.up)	 	*/
  vectorType scrnj;	  /* scrnj = norm(scrni x view.dir)	 	*/
  int height, width;	  /* Height and width of the screen	 	*/
} screenType;    

typedef struct intersectionType {
  myfloat t;		  /* distance from the ray's anchor point       */
  int enter;		  /* entering: enter=TRUE, exiting: enter=FALSE */
  int object;		  /* Intersected object				*/
  pointType p;            /* Intersection point				*/
} intersectionType;

#ifndef INLINE_EXTERN
#define INLINE_EXTERN
#endif

/*void init_srgp(char *, int, int, int);
  void plot(int, int, myfloat, myfloat, myfloat);
*/

myfloat mypower(myfloat, int);
#pragma cmix pure: mypower()
myfloat copysign(myfloat, myfloat);
#pragma cmix pure: copysign()


typedef int surfaceT;
typedef objectType* objectT;
typedef lightType* lightT;

struct IdxType { int last, max; char* name, * maxname; };
typedef struct IdxType IdxType;

extern IdxType surfaceIdx;
extern IdxType objectIdx;
extern IdxType lightIdx;
#ifndef EXTERNS_DEFINED
extern int root_object;		/* root_object is always 0		*/
extern objectType scene[];	/* objects in the scene		 	*/
extern surfaceType surface[];	/* surface descriptions			*/
extern lightType light[];	/* light sources			*/
extern viewType defaultview;    /* view specification			*/
extern screenType screen;	/* screen specification			*/
extern colorType background;	/* background color			*/
extern int to_screen;		/* write to screen if to_screen = TRUE	*/
extern int to_file;		/* write to file if to_file = TRUE	*/
extern int maxlevel;		/* Maximum depth of the trace tree	*/
extern myfloat minweight;	/* Minimum weight in the trace tree	*/
extern int stat_intersections;	/* Total number of intersection tests   */
extern int stat_eye;		/* Total number of eye rays		*/
extern int stat_shadow;		/* Total number of shadow rays		*/
extern int stat_reflected;	/* Total number of reflected rays	*/
extern int stat_refracted;	/* Total number of refracted rays	*/
#endif

#endif
