#include <string.h>
#include <sys/types.h>
#include <asm/page.h>
#include <sys/swap.h>
#include <sys/mount.h>
#include <stdio.h>
#include "dbootstrap.h"
#include "lang.h"
#include <syslog.h> 
#include "util.h"

int check_for_native_partition (struct fdisk_disk **d, int first);

static struct fdisk_disk *
select_drive(void) {
  struct fdisk_disk *disk,**dlist;
  struct d_choices *choices;
  int rs, items=0;

  choices=malloc(5*sizeof(struct d_choices));
  dlist=malloc(5*sizeof(struct fdisk_disk *));

  /* shell version used try_open here. Why ? */
  
  if (NULL == fdisk_disks) {
    problemBox(_("No hard disk drives could be found. Make sure they are cabled correctly and are turned on before the system is started. You may have to change driver settings when you start the system with a command at the \"boot:\" prompt, or you may have to load a driver that is in a loadable module to solve this problem."), _("No hard disk!"));
    return(NULL);
  }
  
  disk = fdisk_disks;
  while (disk) {
    choices[items].string=disk->name;
    choices[items].tag=NULL;
    choices[items].state=0;
    dlist[items]=disk;
    items++;
    if ((items%5)==0) {
      choices = realloc(choices,(items+5)*sizeof(struct d_choices));
      dlist = realloc(dlist,(items+5)*sizeof(struct fdisk_disk *));
    }
    disk = disk->next;
  }
  /* #### FixMe: ID CD's. */
  rs = menuBox(_("Select the drive to partition. SCSI drives are listed in disk ID number order. Only drives that were connected and operating when the system was started will show up in this display. CD-ROM drives may be mis-identified as writable disk drives by this menu."),
	       _("Select Disk Drive"), choices, items, 1);
  if (rs != -1)
    disk = dlist[rs];
  else
    disk = NULL;

  free(dlist);
  free(choices);
  return(disk);
}

int partition_disk (void) {
  struct fdisk_disk *d;
  struct fdisk_partition *p;
  char *mounts, *swaps, *tmpbuf; 
  int root_on_disk=0, status=0;
  struct stat statbuf;

  sync();
  mounts = swaps = NULL;

  if (! (d = select_drive())) {
    return(1);
  }

  p = d->partitions;

  while (p) {
    if (p->in_use) {
      if (fdisk_fstype_of(p->type) == FSTYPE_SWAP) {
	if (swaps) {
	  addtolist(&swaps,p->name);
        } else {
	  swaps = strdup(p->name);
        }
      } else {
	if (mounts) {
	  addtolist(&mounts,p->name);
        } else {
          mounts = strdup(p->name);
        }
      }  
    }
    p=p->next_by_disk;
  }
  prtbuf[0]='\0';
  if (mounts)
    sprintf(prtbuf,_("You have mounted %s from disk %s . If you choose to go ahead and re-partition %s , the filesystem(s) will be un-mounted, and you can re-mount when you have finished partitioning the disk. If you change a previously existing filesystem, you will have to re-initialize that filesystem before you mount again, and any data you installed on it will be erased.\n")
	    ,mounts,d->name,d->name);
  if (swaps) {
    tmpbuf=(char *)malloc(strlen(swaps)+2*strlen(d->name)+320);
    sprintf(tmpbuf, _("You have activated swap partition(s) on '%s'. If you choose to go ahead and re-partition '%s', your swap partition(s) will be de-activated, and you will have to re-initialize and/or re-activate the swap partition(s) after you have re-partitioned '%s'.\n"),
	    swaps, d->name, d->name);
    strcat(prtbuf, tmpbuf);
    free(tmpbuf);
  }
  if ((mounts) || (swaps)) {
    strcat(prtbuf, _("Re-partition the disk?"));
    if ( yesNoBox(prtbuf, _("Re-partition the disk?")) == DLG_NO )
      return (1);
  }
  if (mounts) {
    p = mounted_partitions;
    while (p) {
      if (strstr(mounts, p->name)) {
        INFOMSG("unmounting partition %s so we can re-partition", p->name);
	sprintf(prtbuf, "umount %s", p->name);
	if (execlog(prtbuf, LOG_INFO) != 0) {
	  sprintf(prtbuf, _("Attempting to unmount %s failed. This could mean you have another process running in a directory on %s, or you have another filesystem mounted on a mount point on %s ."),
		  p->name, p->name, p->name);
	  problemBox(prtbuf,_("Unmount failed"));
	  root_on_disk=1;
	}
      }
      p=p->next_in_use;
    }
    free (mounts);
  }
  if (swaps) {
    p = swapon_partitions;
    while (p) {
      if (strstr(swaps, p->name)) {
        INFOMSG("turning off swap on %s so we can re-partition", p->name);
	swapoff(p->name);
      }

      p = p->next_in_use;
    }
    free (swaps);
  }
#if #cpu(i386)
  if (! bootargs.isquiet)
    wideMessageBox(_(
"Lilo, the LInux LOader, when installed in the Master Boot Record,\n"
"(MBR) and the alternative `mbr' program both have support for large\n"
"disks if you have a modern (newer than early 1994-98) BIOS that\n"
"supports LBA and the \"Enhanced Disk Drive Support Specification\".\n"
"Therefore, if you are certain that your BIOS supports the int 0x13\n"
"large disk access extensions, you may partition your drive however you\n"
"like.\n"
"\n"
"However, if you have an older BIOS, neither Lilo's MBR nor the\n"
"alternative `mbr' will be able to load any block of the kernel from a\n"
"disk cylinder numbered higher than 1023.  This is often a problem with\n"
"disks larger than 528MB, coupled with an older (pre 1995-98, depending\n"
"on the manufacturer) BIOS that does not support the int 0x13 large\n"
"disk access extensions.  If your BIOS supports LBA or CHS Translation\n"
"(aka \"Large\"), but does not have the large disk extensions, the MBR\n"
"will be able to address anything below the translated 1024th cylinder,\n"
"usually around 8GB.\n"
"\n"
"One way to solve the problem is to make a small (5-10Mb) partition at\n"
"the beginning of the disk, so that the entire partition is below the\n"
"1023rd cylinder.  For example, make a 5-10Mb primary partition in the\n"
"low cylinder range, marked bootable, another for swap space, one for\n"
"\"/\", and whatever other partitions you like.  Before you install the\n"
"operating system and modules, after you have mounted your \"/\"\n"
"partition, initialize and mount that small partition on \"/boot\".\n"
"This is where the OS kernel will be stored, and thus, Lilo will find\n"
"that all blocks of the kernel reside below the 1024 cylinder boundary.\n"
"\n"
"Please refer to the partitioning section in the Installation Manual\n"
"for slightly more detailed information and references to relevant\n"
"documentation concerning the BIOS large disk access extensions."
	), _("LILO Limitations"));
#endif

  if ( NAME_ISEXE( "/sbin/cfdisk", &statbuf )
#if #cpu(powerpc)
       && ( strcmp(Arch2, "PowerMac") != 0 )
#elif #cpu(alpha)
       && ! srm_boot
#endif
       ) {

#if #cpu(alpha)
    wideMessageBox(_(
"You appear to be booting from MILO or APB.  In order to\n"
"make Linux bootable from the hard drive, you will need to\n"
"install your bootloader (and, in the case of APB, also your\n"
"kernel) on a DOS FAT partition.  If you do not have any FAT\n"
"partitions already, you should create a small one (5 megabytes\n"
"is more than enough) to hold these files.\n"),
		   _("Note on MILO/APB installation"));
#endif

    INFOMSG("running cfdisk to partition %s", d->name);
    sprintf(prtbuf, "cfdisk %s", d->name);
    status = fullscreen_execlog(prtbuf);
#ifdef pc_use_cfdisk_err_ret_switch
    switch (status) {
    case 0:
    default:
	break;
    case 3:
	/* Cannot get geometry -- punt!  Will using fdisk on vc2 do
           any good?  I'm not sure what to do in this case. - karlheg */
        ERRMSG("could not determine the drive geometry of %s, cannot partition", d->name);
	problemBox(_("cfdisk has failed while trying to repartition your disk. It failed to divine the geometry of the drive. Maybe you can switch to the second virtual console with Alt-F2 and investigate the problem using fdisk."), 
                   _("Cannot Determine Drive Geometry."));
	return (1);
	break;
    case 4:
	if (yesNoBox(_("cfdisk has failed while trying to repartition your disk. That may mean your disk's partition table is corrupt, or your disk is `factory clean'.\nIt often helps to wipe out your disk's current partition table and run cfdisk again.\nWARNING: You will lose any data currently on that disk.\nAre you sure you want me to do this?"),
		     _("Invalid Partition Table")) == DLG_YES) {
            INFOMSG("re-running cfdisk to ignore existing partition table");
	    sprintf(prtbuf, "cfdisk -z %s", d->name);
	    status = fullscreen_execlog(prtbuf);
	}
	break;
    }
#else
    if (status != 0) {
	if (yesNoBox(_("cfdisk has failed while trying to repartition your disk. That may mean your disk's partition table is corrupt, or your disk is `factory clean'.\nIt often helps to wipe out your disk's current partition table and run cfdisk again.\nWARNING: You will lose any data currently on that disk.\nAre you sure you want me to do this?"),
		     _("Invalid Partition Table")) == DLG_YES) {
            INFOMSG("re-running cfdisk to ignore existing partition table");
	    sprintf(prtbuf, "cfdisk -z %s", d->name);
	    status = fullscreen_execlog(prtbuf);
	}
    }
#endif /* pc_use_cfdisk_err_ret_switch */
  } else {
    int first = 1;

    /* FIXME: this should probably be split off into a function that
     gives fdisk instructions for all architectures that use it, so
     that they can be read multiple times (i.e. if the user doesn't do
     the right thing, and can't remember how) */
#if #cpu(alpha)
    /* Note that we only get here when using fdisk, i.e. on SRM */
    wideMessageBox(_(
"You appear to be booting from the SRM console.  In order to\n"
"make Linux bootable from the hard drive, you will need to\n"
"create an OSF/1 style disklabel rather than a DOS partition\n"
"table.  Since the 'cfdisk' program is unable to create OSF/1\n"
"disklabels, I will start 'fdisk' instead.  You must use the\n"
"'b' command to enter disklabel mode.\n\n"
"As well, because the 'aboot' second-stage bootloader must\n"
"occupy the first sector of the disk, you will need to leave\n"
"about a megabyte of empty space before the first partition.\n"
"On most drives, it should suffice to start the 'a' partition\n"
"at cylinder 2, or the equivalent in sectors (use the 'u'\n"
"command to switch units, and the 'm' command for help.)"),
		     _("Using fdisk on SRM"));
#endif

    while (check_for_native_partition(&d, first) != 0) {
      INFOMSG("running fdisk to partition %s", d->name);
      sprintf(prtbuf, "fdisk %s", d->name);
      status = fullscreen_execlog(prtbuf);
      first = 0;
    }
  }
  INFOMSG("partition program returned %d", status);
  if (root_on_disk && 
      yesNoBox(_("You changed the partition scheme on a disk which is currently mounted.  Often, in order to work with the new partitions, you may need to reboot the system.  If you reboot, you will be able to carry on with installation from where you left off.\n\nReboot now?"),
	       _("Reboot system?")) == DLG_YES) {
    INFOMSG("partition changed on mounted disk, rebooting");
    reboot_system();
  }
  return 0;
}

static int
mount_partition (struct fdisk_partition *partition)
{
  char             *mount_point, *real_mount_point, *type;
  int              status;
  struct stat      statbuf;

  if (!(type = fdisk_fstype_name_of (partition->type)))
  {
    sprintf (prtbuf, _("Don't know which filesystem type to use for partition %s of type '%s'."),
	     partition->name, fdisk_sysname (partition->type));
    problemBox (prtbuf, _("Invalid filesystem"));
    return 1;
  }
  if (! Root) {
    sprintf(prtbuf,_("You must mount your root filesystem (\"/\") before you can mount any other filesystems. Would you like to mount %s as the root filesystem?"),
	    partition->name); 
    if (! bootargs.isquiet &&
	yesNoBox(prtbuf, _("Mount as the Root Filesystem?")) != DLG_YES)
      return 1;
    mount_point = strdup("/");
    real_mount_point=strdup("/target");
  }
  else
  {
    struct d_choices mount_points[5];
    int i = 0;

    if (fdisk_find_partition_by_mntpoint("/target/boot") == NULL)
    {
      mount_points[i  ].tag    = "/boot";
      mount_points[i++].string = _("A small (5-10Mb) partition at the start of the disk.");
    }
    if (fdisk_find_partition_by_mntpoint("/target/usr") == NULL)
    {
      mount_points[i  ].tag    = "/usr";
      mount_points[i++].string = _("A large partition where most software resides.");
    }
    if (fdisk_find_partition_by_mntpoint("/target/var") == NULL)
    {
      mount_points[i  ].tag    = "/var";
      mount_points[i++].string = _("Variable data such as mail spools and databases.");
    }
    if (fdisk_find_partition_by_mntpoint("/target/home") == NULL)
    {
      mount_points[i  ].tag    = "/home";
      mount_points[i++].string = _("User home directories.");
    }
    if (i > 0)
    {
      mount_points[i].tag    = _("Other");
      mount_points[i].string = _("Enter name of a different mount point.");
    }
 
    sprintf(prtbuf, _("Select the mount point for %s"), partition->name);
    status = -1;
    if (i > 0) {
	if ((status = menuBox(prtbuf, _("Select Mount Point"), mount_points, i + 1, 1)) < 0)
	    return 1;
    }
    if (status == i || status == -1) { /* user selected "Other" */
      mount_point = inputBox(prtbuf, _("Select Mount Point"), "/");
      if ( strcmp(mount_point, "/etc") == 0 ) {
        problemBox(_("You cannot set /etc as a mount point.  This directory is required to boot."),
                   _("Problem"));
        return 1;
      }
    }
    else {
      mount_point = strdup(mount_points[status].tag);
    }
    if (NULL == mount_point) return 1;
    real_mount_point = (char *)malloc(9 + strlen(mount_point));
    strcpy(real_mount_point, "/target");
    if (mount_point[0] != '/')
	strcat(real_mount_point, "/");
    strcat(real_mount_point, mount_point);
  }

  if (! NAME_ISDIR( real_mount_point, &statbuf ) ) {
    int WithMode;
    char *p = strchr(real_mount_point, '/');
    p = strchr(p + 1, '/');
    while (1) {
	if ((p = strchr(p + 1, '/')) != NULL)
	    *p = '\0';
	WithMode = strcmp(real_mount_point,"/target/tmp") ? 0755 : 01777;
	DEBUGMSG("making mount point %s", real_mount_point);
	if (! mkdir(real_mount_point, WithMode)) {
	    chmod(real_mount_point, WithMode);
	    chown(real_mount_point, 0, 3); /* root.sys, may be overriden when base is installed */
	} else if (errno != EEXIST){
	    snprintf(prtbuf, sizeof(prtbuf), _("Failed to create directory, `%s'"), real_mount_point);
	    perrorBox(prtbuf);
	    free(real_mount_point);
	    free(mount_point);
	    return 1;
	}
	if (p)
	    *p = '/';
	else
	    break;
    }
  }
  INFOMSG("mounting %s at %s", partition->name, real_mount_point);
  sprintf(prtbuf, "mount -t %s %s %s",
          type, partition->name, real_mount_point);
  status = execlog(prtbuf, LOG_INFO);
  if (status) {
    perrorBox(_("Mount failed"));
    free(real_mount_point);
    free(mount_point);
    return 1;
  }

  if (! strcmp(mount_point, "/")) {
    /* we just mounted root, so have some other config tasks to perform */
    Boot = Root = partition;
    check_pending_config();	/* Must NEVER write to "/target/boot". */
  }
  if (! strcmp(mount_point, "/boot"))
    Boot = partition;

  free(mount_point);
  return 0;
}

/* returns non-zero for "yes, we want cflag" */
static int get_cflag (const char *name) {
  struct stat statbuf;

  if (! NAME_ISEXE("/sbin/badblocks", &statbuf) ||
      bootargs.isquiet ) {
    return 0;
  }

  sprintf(prtbuf, _("The system can scan the entire partition for un-readable disk blocks and will mark any such bad blocks it finds so that they will not be used. This requires that every block be read, and thus could take a long time, but may save you trouble later.  Modern disk controllers generally do not need this, since they can identify and deal with bad blocks automatically, so the default is not to perform this check.\n\nShould the bad-block scan be skipped on '%s'?"), name);
  return ( yesNoBox(prtbuf, _("Skip the Scan for Bad Blocks?")) == DLG_NO );
}

int init_linux (void) {
  struct fdisk_partition *p;
  int Oflag, cflag, status;

  sync();
  p = select_not_mounted(_("Please select the partition to initialize as a Linux \"ext2\" filesystem.") ,_("Select Partition"),1,FSTYPE_EXT2);
  if (! p)
    return 1;

#if #cpu(alpha)
  if (srm_boot || strcmp(Arch2, "nautilus") == 0)
#endif
  if (! bootargs.isquiet) {
#if #cpu(arm)
    Oflag = yesNoBox(_("This 2.2 version of the Linux kernel has new \"ext2\" filesystem features not present in earlier kernel versions. Using these features, however, means that you will not be able to use this filesystem with earlier kernels, such as Linux 2.0 or with the Netwinder firmware.\n\nDo you want to retain Linux kernel 2.0 compatibility?"),
#else
    Oflag = yesNoBox(_("This 2.2 version of the Linux kernel has new \"ext2\" filesystem features not present in earlier kernel versions. Using these features, however, means that you will not be able to use this filesystem with earlier kernels, such as Linux 2.0.\n\nDo you want to retain Linux kernel 2.0 compatibility?"),
#endif
		   _("Pre-2.2 Linux Kernel Compatibility?"));
  } else {
    Oflag = DLG_YES;
  }

  cflag = get_cflag(p->name);

  if (! bootargs.isquiet) {
    sprintf(prtbuf,_("You have chosen to initialize %s as a Linux \"ext2\" filesystem. This will permanently erase any data on this partition. Are you sure you want to do this?"),
	    p->name);
    if (yesNoBox(prtbuf, _("Are You Sure?")) == DLG_NO)
      return 1;
  }

  INFOMSG("formatting partition %s as ext2 for kernel %s or better", 
          p->name, 
          ( Oflag == DLG_NO ? "2.2" : "2.0"));
#if #cpu(alpha)
  /* Alpha-specific workaround for MILO's failure to support 4k blocks */
  if (!(srm_boot || strcmp(Arch2, "nautilus") == 0))
    sprintf(prtbuf, "mkfs.ext2 -O none -b 1024 %s %s",
	    ( cflag ? "-c" : ""), p->name);
  else
#endif
  sprintf(prtbuf, "mkfs.ext2 %s %s %s", ( Oflag == DLG_NO ? "" : "-O none"),
	  ( cflag ? "-c" : ""), p->name);
  boxSuspend();
  printf(CLEAR);
  if ( Oflag == DLG_YES )
    printf(_("Creating filesystem (with 2.0 kernel compatibility)...\n"));
  else
    printf(_("Creating filesystem (for 2.2 kernels only)...\n"));
  status = system(prtbuf);
  boxResume();
  if (status) {
    ERRMSG("make filesystem with cmd '%s' failed (%d)", prtbuf, status);
    problemBox(_("The filesystem was not created."), _("Problem"));
    return 1;
  }
  return mount_partition(p);
}

int init_swap (void) {
  struct fdisk_partition *p;
  int cflag, status;

  sync();
  p = select_not_mounted(_("Please select the partition to initialize as a swap device."),
			 _("Select Swap Partition"), 1, FSTYPE_SWAP);
  if (! p)
    return 1;
  cflag = get_cflag(p->name);
  if ( ! bootargs.isquiet ) {
    sprintf(prtbuf, _("You have chosen to initialize %s as a swap device. This will permanently erase any data on this partition. Are you sure you want to do this?"),
	    p->name);
    if ( yesNoBox(prtbuf, _("Are You Sure?")) == DLG_NO )
      return 1;
  }
#if #cpu(sparc)
  sprintf(prtbuf,"mkswap -v1 %s %s",( cflag ? "-c" : ""),p->name);
#else
  sprintf(prtbuf,"mkswap %s %s",( cflag ? "-c" : ""),p->name);
#endif
  boxSuspend();
  printf(CLEAR);
  printf(_("Initializing swap partition...\n"));
  status=system(prtbuf);
  boxResume();
  if (status) {
    problemBox(_("The swap partition could not be initialized."),_("Problem"));
    return 1;
  }
  if (swapon(p->name, 0)) {
    sprintf(prtbuf,_("The swap partition %s could not be activated"),p->name);
    perrorBox(prtbuf);
    return 1;
  }
  return 0;
}

int no_swap (void) {
  struct fdisk_partition *p;
  
  if (bootargs.isquiet || 
      yesNoBox( _("A swap partition is a good idea for all systems, even ones with lots of memory. It's strongly encouraged that you create one.\n\nIf you have less than 8 megabytes of RAM in your system, you will need to create a swap partition simply so that there is enough virtual memory to finish the installation.\n\nIf you answer \"Yes\" to the following question, the system will not require you to create a swap partition. It's strongly advised that you answer \"No\".\n\nWould you like to do without a swap partition?"),
		_("Do Without a Swap Partition")) == DLG_YES ) {
    INFOMSG("doing without a swap partition");
    if (swapon_partitions) {
      p = swapon_partitions;
      while (p) {
        swapoff(p->name);
        p = p->next_in_use;
      }
    }
    noSwap = 1;
  }
  return 0;
}

int mount_any (void) {
  struct fdisk_partition *p;

  int ntypes = 6;
#ifdef NFSROOT
  /* We allow this for NFSROOT, and check for network config there */
  ntypes++;
#endif
  p=select_not_mounted( _("Please select the partition to mount."), _("Select Partition"), ntypes,
		FSTYPE_EXT2,FSTYPE_MINIX,FSTYPE_MSDOS,FSTYPE_AFFS,FSTYPE_HFS,FSTYPE_ADFS, FSTYPE_NFS);
  if (!p) return 1;
  return mount_partition(p);
}

static int unmount_partition (struct fdisk_partition *p) {
  INFOMSG("unmounting partition mounted at %s", p->mount_point); 
  sprintf(prtbuf, "umount %s", p->mount_point);
  if (execlog(prtbuf, LOG_INFO) != 0) {
    sprintf(prtbuf, _("Attempting to unmount the partition mounted on the directory '%s' failed. This could mean you have another process running in a directory under '%s', or you have another filesystem mounted on a mount point under '%s'."), 
            p->name, p->mount_point, p->mount_point);
    problemBox(prtbuf, _("Unmount Failed"));
    ERRMSG("attempt to unmount the partition %s mounted on %s failed", p->name, p->mount_point);
    return 1;
  }
  if ( p == Root )
    Root = NULL;
  return 0;
}

int unmount_any (void) {
  struct fdisk_partition *p, **plist;
  struct d_choices *choices;
  int rs, items=0;

  choices=malloc(5*sizeof(struct d_choices));
  plist=malloc(5*sizeof(struct fdisk_partition *));

  p=mounted_partitions;
  if ((p == NULL) || 
      ((! strcmp(p->name,InstallationRootDevice)) && (p->next_in_use==NULL))) {
    problemBox(_("No mounted partitions were detected."),_("Problem"));
    return 1;
  }

  p = mounted_partitions;
  while (p) {
    if (strcmp(p->name,InstallationRootDevice)) {
      choices[items].string=p->name;
      choices[items].tag=NULL;
      choices[items].state=0;
      plist[items]=p;
      items++;
      if ((items%5)==0) {
        choices = realloc(choices,(items+5)*sizeof(struct d_choices));
        plist = realloc(plist,(items+5)*sizeof(struct fdisk_partition *));
      }
    }
    p = p->next_in_use;
  }

  rs = menuBox(_("Select the partition to unmount"), _("Select Partition"),
	choices, items, 1);
  if (rs != -1)
    p = plist[rs];
  else
    p = NULL;

  free(plist);
  free(choices);
  if (p == NULL) return 1;
  return unmount_partition(p);
}

int activate_swap (void) {
  struct fdisk_partition *p;

  p = select_not_mounted(_("Please select the partition to activate as a swap device."),
			 _("Select Swap Partition"), 1, FSTYPE_SWAP);
  if (! p)
    return 1;

  if ( ! bootargs.isquiet ) {
    sprintf(prtbuf, _("You have chosen to activate %s as a swap device. This will permanently erase any data on this partition. Are you sure you want to do this?"),
	    p->name);
    if ( yesNoBox(prtbuf, _("Are You Sure?")) == DLG_NO )
      return 1;
  }
  if (swapon(p->name, 0)) {
    sprintf(prtbuf, _("The swap partition %s could not be activated"),
	    p->name);
    perrorBox(prtbuf);
    return 1;
  }
  return 0;
}

int view_partitions (void) {
  struct fdisk_disk *d;
  struct fdisk_partition *p;

  d = fdisk_disks;
  if (d) {
    /*
     *  TRANS: This allows nice formatting for partition table.
     *
     *  Formatting chart:
     *
     * |         1         2         3         4         5         6         7         8
     * |12345678901234567890123456789012345678901234567890123456789012345678901234567890
     * |                                                                     |
     * |      123456789          12345678901234567                           |
     * |Device         Mounted on                 Type                       |
     * |-------------  -------------------------  -------------------------  |
     * |/dev/hda1      /target/boot               Linux native               |
     * |/dev/hda2      swap                       Linux swap                 |
     * |/dev/hda3      /target                    Linux native               |
     * |/dev/hda4       -- N/A --                 DOS Extended               |
     * |/dev/hda5      /target/usr                Linux native               |
     * |/dev/hda6      /target/var                Linux native               |
     * |/dev/hda7      /target/home               Linux native               |
     * |/dev/hda8      /target/usr/local          Linux native               |
     * |/dev/hda9       -- Not mounted --         Linux native               |
     * |1234567890123  1234567890123456789012345  1234567890123456789012345  |
     * |             12                         12                           |
     *
     * karlheg -> Longest device name I know about is: /dev/ida/c0d0 == 13 chars.
     */
    strcpy(prtbuf, _("The following partitions have been detected:\n"
		     "\n"
		     "Device         Mounted on                 Type\n"
		     "-------------  -------------------------  -------------------------"));
    strcat(prtbuf, "\n");
    while (d) {
      p = d->partitions;
      if (p) {
        while (p) {
	  char *mp;
	  /* Use fstype to reduce the number of compares needed in this IF statement */
	  int fstype = fdisk_fstype_of(p->type);
	  if (fstype == FSTYPE_SWAP) {
	    mp = (p->in_use) ? _("swap") : _(" -- Not in use --");
	  }
	  else if ((fstype == FSTYPE_EXTPART) || (fstype == FSTYPE_UNKNOWN))
	  {
	    mp = _(" -- Not available --");
	  }
	  else {
	    mp = (p->mount_point) ? p->mount_point : _(" -- Not mounted --");
	  }
          snprintf(prtbuf + strlen(prtbuf), (size_t)71, /* No more than 70 characters per line. */
		   _("%-13s  %-25s  %-25s\n"), p->name, mp, fdisk_sysname(p->type));
          p = p->next_by_disk;
        }
      } else {
        sprintf(prtbuf + strlen(prtbuf),
		_("Disk %s has no partitions defined!"), d->name);
	strcat(prtbuf, "\n");
      }
      d = d->next;
      strcat(prtbuf, "\n");
    }
    wideMessageBox(prtbuf, _("Partitions List"));
    return 0;
  } else {
    problemBox(_("No hard disk drives could be found. Make sure they are cabled correctly and are turned on before the system is started. You may have to change driver settings when you start the system with a command at the \"boot:\" prompt, or you may have to load a driver that is in a loadable module to solve this problem."), _("No hard disk!"));
    return 1;
  }
}

/*
 * Check for a partition table that could be understood by the boot monitor
 * (eg. SMD disk label on sparc, MS-DOS partition table on x86, ...)
 *
 * This function should return 1 to continue running fdisk again and again.
 * It is called the first time just before running fdisk (with first=1)
 * therefore it should return 1 at least once.
 */
int check_for_native_partition( struct fdisk_disk **d, int first ) {
  /* check for non native disk label before running fdisk */
  if (! first) {
    /* before checking for some kind of partition table format, the disk
     * should be reread because the partition table could have changed.
     * Note1: don't have to be done the first time.
     * Note2: since d will not be valid past fdisk_reread(), we need to
     *        reassign it after the call. */
    struct fdisk_disk *disk;
    char *diskname = strdup( (*d)->name );
    fdisk_reread();
    disk = fdisk_disks;
    while (disk) {
      if (0 == strcmp( diskname, disk->name ))
	break;
      disk = disk->next;
    }
    free( diskname );
    *d = disk;
  }
#if #cpu(sparc)
  if ( (*d) &&
       (*d)->partition_format &&
       0 != strcmp((*d)->partition_format, "sun") ) {
    sprintf(prtbuf,_("%s appears not to have a Sun disk label. This means you will not be able to boot directly from this disk. You can create a Sun disk label with the option 's' in fdisk.\n"), (*d)->name);
    if (first)
      strcat(prtbuf, _("Do you still want to partition this disk?"));
    else
      strcat(prtbuf, _("Are you sure you want to keep this partition table?"));
    /* returns 1 if first & YES or !first & NO, 0 otherwise */
    return (yesNoBox(prtbuf, _("Unknown disk label")) == DLG_NO) ^ first;
  }
#elif #cpu(alpha)
  if (srm_boot) {
    if ((*d) && (*d)->partition_format
	&& 0 != strcmp((*d)->partition_format, "osf")) {
      snprintf(prtbuf, sizeof(prtbuf),
	       _("%s appears not to have an OSF/1 disk label.  "
		 "This means you will not be able to boot directly "
		 "from this disk.  You can create an OSF/1 disk label "
		 "with the option 'b' in fdisk.\n"), (*d)->name);
      if (first) {
	strcat(prtbuf, _("Do you still want to partition this disk?"));
	return (yesNoBox(prtbuf, _("Unknown disk label")) == DLG_YES);
      } else {
	strcat(prtbuf,
	       _("Are you sure you want to keep this partition table?"));
	return (yesNoBox(prtbuf, _("Unknown disk label")) == DLG_NO);
      }
    }
  }
#else /* neither sparc nor alpha */
    /* XXX: Check for MS-DOG partition table? */
#endif
  /* returns 1 if first, 0 otherwise to run fdisk only once */
  return first;
}

#ifdef _TESTING_

void usage(char * name)
{
  fprintf(stderr, "Usage: %s (mount | swap | partition) [subarch]\n", name);
}

int main(int argc, char* argv[])
{
  int test_mount = 0, test_swap = 0, test_partition = 0;
  LOAD_TRMFILE("test.trm");
  get_kver();
  if (argc < 2) {
    usage(argv[0]);
    return 1;
  }

  /* totally losing, but I don't care */
  if (strcmp(argv[1], "mount") == 0) {
    test_mount = 1;
  } else if (strcmp(argv[1], "swap") == 0) {
    test_swap = 1;
  } else if (strcmp(argv[1], "partition") == 0) {
    test_partition = 1;
  } else {
    usage(argv[0]);
    return 1;
  }

  if (argc == 3)
    Arch2 = argv[2];
  else
    Arch2 = "";

  fdisk_reread();
  boxInit();

  if (test_mount) {
    while (mount_any() == 0)
        fdisk_reread();
  } else if (test_swap)
    init_swap();
  else
    partition_disk();

  boxFinished();
  return 0;
}

#endif
