#! /bin/sh
# Boot Disk maker. Bruce Perens, July-September 1995
# Eric Delaunay, 1998 [silo boot]
# Enrique Zanardi, 1998
# This is free software under the GNU General Public License.

. ./common.sh

#-----------------------------------------------
# Write (or don't write) the install.sh script
#
write_installsh () {
	if [ "x$noinstall" = x ]; then
		m4 -D__ERROR_MESS__=$scripts/messages/$language/install.sh.txt \
		   -D__version__=$KERNEL_VERSION -D__arch__=$arch \
		   <$scripts/install.sh >$mnt/install.sh
		chmod 755 $mnt/install.sh
	fi
}

#-----------------------------------------------
# write SILO boot infos
#
write_silo () {
	cat > $mnt/etc/silo.conf << EOF
partition=1
message=/debian.txt
root=/dev/fd0
timeout=600
image=/linux
label=linux
append="load_ramdisk=1 prompt_ramdisk=1"
read-write
EOF
	write_installsh
}


#-----------------------------------------------
# write PowerPC boot infos
#
# For the moment only with quik
#
write_powerpc () {
	mkdir $mnt/etc
        mkdir $mnt/boot
	cat > $mnt/etc/quik.conf << EOF
partition=1
message=/debian.txt
root=/dev/fd0
timeout=600
image=/linux
label=Linux
append="load_ramdisk=1 prompt_ramdisk=1"
read-write
EOF
	write_installsh
}

#-----------------------------------------------
# write SysLinux boot infos
#
write_syslinux () {
	local flavor
	relax
	
	flavor=${revext:+"flavor=$revext"}
	[ -f "$sfont" ] && cp $sfont $mnt
	# Question: shouldn't 1440 be '$blocks'
	# font.psf is copied to the disk by the user (see scripts/dos/setlang.bat)
	disksize=`echo 1440 | sed -e 's/\([0-9]\)\([0-9][0-9]\)[0-9]/\1.\2/'`
	{ [ -f "$sfont" ] && echo FONT $sfont || echo FONT font.psf; cat $scripts/syslinux.cfg; } |\
		m4 -D__size${blocks}__ -D__disksize__="disksize=${disksize}" \
		   -D__flavor__="${flavor}" | $nlfmt >$mnt/syslinux.cfg
	if [ "$language" = "ja" ]; then
		sed -e 's/vga=normal/vga=normal video=vga16:on/' $mnt/syslinux.cfg > ./syslinux.cfg.ja
		mv ./syslinux.cfg.ja $mnt/syslinux.cfg
	fi
	sed -e s/__arch__/$arch/ \
		<$scripts/rdev.sh >$mnt/rdev.sh
	chmod 777 $mnt/rdev.sh
	write_installsh
}

#-----------------------------------------------
# write m68k info
#
write_m68kinfo () {
	write_installsh
}

#-----------------------------------------------

# write milo boot infos
#
write_milo () {
  	if test -n "$APB"; then
	    # APB is obtained from
	    # http://www.alpha-processor.com/products/downloads/customer_support/UP1000/up1000.zip
	    cp apb.exe $palcode $mnt/
	    cp apb_install.txt $mnt/apb.txt
	    cp apb_README.txt $mnt/up1000.txt
	    cat > $mnt/apb.cfg <<EOF
debian_install|bootl $palcode linux root=/dev/fd0 load_ramdisk=1
EOF
	else
	   # Generic disk, put both milo and aboot config on it
           umount $mnt
#           e2writeboot $floppy ../alpha-kernel-config/aboot.bootlx
           e2writeboot $floppy /boot/bootlx
           mount -o loop $fsopts -t $fstype $floppy $mnt
	   #
	   # for aboot a compressed kernel needs the suffix gz
	   # (not strictly true, but it's useful)
	   #
	   mv $mnt/linux $mnt/linux.gz
	   ln -s linux.gz $mnt/linux
	   mkdir $mnt/etc
	   cat  > $mnt/etc/aboot.conf << EOF
#
# Bootparameters for Debian-Installation 
#
# Installation-Boot with:  boot dva0 -fl 0
# Rescue-Boot with:        boot dva0 -fi linux.gz -fl root=/dev/...
#
# from SRM-Console
#
0:linux.gz ro root=/dev/fd0 load_ramdisk=1
EOF
	   cat > $mnt/milo.cfg << EOF
#
# MILO-Configuration for Debian-Boot
#
# Boot with:  boot fd0:
#
timeout=100
 
image=fd0:linux
label=install
root=/dev/fd0
append="load_ramdisk=1 prompt_ramdisk=1"
EOF

	    # In the (nonexistent) case where we would be writing MILO
	    # Milos, miloctl, srmbootfat are taken from: 
	    # ftp://genie.ucd.ie/pub/alpha/milo/ (see the config file)
	    if [ $WRITE_MILO = YES ]; then
		cp milo_$SUBARCH $mnt/milo
		miloctl 'cmd=boot fd0:' $mnt/milo
	    fi
        fi
	write_installsh
}



#-----------------------------------------------
# write VMELILO boot infos
#
write_vmelilo () {
	mkdir $mnt/etc
	mkdir $mnt/boot

	# determine which loop device the image is mounted on
	loopdev=$(mount | perl -ne 'm|$floppy.*loop=(/dev/loop[0-9]+)| && print $1;')

	if [ "$blocks" = "1440" ]; then
		cat > $mnt/etc/vmelilo.conf << EOF
prompt  = true
default = install
message = /debian.txt
root    = /dev/sda
image   = /linux
read-write

[boot]
label   = Install
append  = load_ramdisk=1 prompt_ramdisk=1

[boot]
label   = Rescue
append  = load_ramdisk=1 prompt_ramdisk=1 single
EOF
	else
		cat > $mnt/etc/vmelilo.conf << EOF
prompt  = true
default = install
message = /debian.txt
root    = /dev/ram0
image   = /linux
read-write

[boot]
label   = Install
ramdisk = /root.bin

[boot]
label   = Rescue
ramdisk = /root.bin
append  = single
EOF
	fi
	case $system in
		bvme6000)         vmearch=bvme ;;
		mvme16x|mvme147)  vmearch=mvme ;;
	esac
	vmelilo -f -a $vmearch -b /dev/null -w $mnt -d $loopdev
	write_installsh
}

# Print a usage message and exit if the argument count is wrong.
if [ $# -lt 7 ]; then
echo "Usage: "$0" revision-extension kernel-package root-image blocks debian-version system language sfont" 1>&2
	cat 1>&2 << EOF

	revision-extension: the revision extension of the kernel.
	kernel-package: the Debian package containing the kernel.
	root-image: a compressed disk image to load in ramdisk and mount as root.
	blocks: the size of the floppy in 1K blocks, use 720, 1200, 1440, or 2880.
	debian-version: version of the Debian release.
	system: name of the target system.
	language: language for the boot messages and help files.
	sfont: name of the font loaded by syslinux, if any
	noinstall: make this non-blank to make this a boot disk only
EOF

	exit -1
fi

revext=$1

# Set this to the location of the package archive.
archive=$2

# Set this to the location of the root filesystem image
rootimage=$3

# Set to the number of blocks per floppy.
blocks=$4

# Set this to the Debian version
debianversion=$5

# Set this to the system
system=$6

# Set this to the language for the boot messages and help files.
language=$7

# Set this to the font loaded by syslinux, if any.
sfont=$8

# Set this to non blank if you want this to be for booting only
noinstall=$9

# Set this to the charset for the boot messages and help files.
# FIXME: this should be a runtime parameter.
charset=latin1

testroot

# Set this to the block device for the floppy disk.
floppy=$(tempfile -p fimg -m 644)

mnt=${tmpdir}/boot-floppies/mnt.rescue

# The home of the scripts used by this script.
scripts=`pwd`/scripts/rescue

# Make sure the kernel package is available.
if [ ! -f $archive ]; then
	error "can't find $archive"
fi

# Try to unmount anything that should not be mounted. Aborted runs of this
# script may have left things mounted.
(umount /mnt; umount $mnt; umount $floppy; true)2>/dev/null >/dev/null

make_tmpdir $mnt

# Zero the entire disk, so that when I compress the raw disk image,
# unused blocks will compress well.
if [ "$arch" != sparc ]; then
	info "making loop filesystem in $floppy, size ${blocks}k"
	dd if=/dev/zero of=$floppy bs=1k count=$blocks
fi

# Get the kernel version from its package name. [KOP: - for Subarchs]
if [ "$arch" = powerpc -o "$arch" = m68k ]; then
   KERNEL_VERSION=`basename $archive | sed -e 's/kernel-image-//' -e 's/-.*//'`
else
   KERNEL_VERSION=`basename $archive | sed -e 's/kernel-image-//' -e 's/_.*//'`
fi
DEBIAN_KERNEL_IMAGE=`basename $archive .deb | sed -e s/_$arch\$//`
case $arch in
    alpha) SUBARCH=`expr $revext : 'alpha_\(.*\)'`;;
esac

umount $floppy >/dev/null 2>&1 || true

info "formatting and conditioning the new image"
case "$arch" in
    alpha)
          case $SUBARCH in
	  nautilus)
                  mkdosfs -r 112 -F 12 $floppy $blocks
		  APB=1
		  palcode=up1000.pal
		  fstype=msdos
		  fsopts="-o fat=12"
		  nlfmt="cat -"
	      ;;
              *)
                  mke2fs -O none -F -q $floppy $blocks
                  fstype=ext2
                  nlfmt="cat -"       #  UNIX newline convention
              ;;
          esac
    ;;
    arm)
        # netwinder doesn't use this, but we need it to install kernel
        # other sustems will need more help
        mke2fs -F -q $floppy $blocks
        fstype=ext2
        nlfmt='cat -'       # UNIX newline convention
    ;;
    i386)
        if [ -f /usr/lib/syslinux/img$blocks""k.gz ]; then
                gunzip < /usr/lib/syslinux/img$blocks""k.gz > $floppy
        elif [ -f $scripts/img$blocks""k.gz ]; then
                gunzip < $scripts/img$blocks""k.gz > $floppy
        else
                error "can't find $blocks KB floppy image"
        fi
        syslinux $floppy    # Use the "fast" loader.
        fstype=msdos
        if [ -f "$sfont" ]; then   # MSDOS newline convention
		nlfmt=sed\ s/$/$'\r'/
        else
		nlfmt="recode $charset..ibmpc"
        fi
        fsopts="-o fat=12"
    ;;
    m68k)
          case "$system" in
              amiga|atari)
                  mkdosfs -r 112 -F 12 $floppy $blocks
                  fstype=msdos
                  fsopts="-o fat=12"
                  nlfmt="recode $charset..ibmpc"       # MSDOS newline convention
              ;;
	      mac)
	          # dhd: We are not actually going to use this to boot, and I can't
	          # seem to get HFS or msdos images to transfer reliably over NFS,
		  # so we will use ext2 here.  This is still suboptimal, dbootstrap
	          # should be able to install a "loose" kernel... work in progress
                  mke2fs -F -q $floppy $blocks
		  fstype=ext2
		  nlfmt="recode $charset..applemac"    # Does not matter really
	      ;;
              bvme6000|mvme16x|mvme147)
                  # uses ext2 formatted floppies & VMELILO boot loader
                  mke2fs -F -q $floppy $blocks
                  fstype=ext2
                  nlfmt='cat -'       # UNIX newline convention
              ;;
          esac
    ;;
    mips)
	error "no support for MIPS"
    ;;
    powerpc)
            case "$system" in
              apus|chrp)
                  mkdosfs -r 112 -F 12 $floppy $blocks
                  fstype=msdos
                  fsopts="-o fat=12"
                  nlfmt="recode $charset..ibmpc"       # MSDOS newline convention
              ;;
# CHRP can't boot off an ext2 floppy...
#              chrp)
#                  mke2fs -F -q $floppy $blocks
#                  fstype=ext2
#                  nlfmt="recode $charset..ibmpc"        # MSDOS newline convention
#              ;;
              mbx)
                  error "don't know what to do for MBX"
              ;;
              pmac)
                  mke2fs -F -q $floppy $blocks		# just for resc1440; 
                  fstype=ext2				# really want HFS for boot floppy
                  nlfmt="recode $charset..applemac"	# Mac newline convention
              ;;
	      prep)
		  # use ext2 formatted floppies for now, a port of milo or
		  # quik is not far off. 
		  mke2fs -F -q $floppy $blocks
		  fstype=ext2
		  nlfmt='cat -'       			# UNIX newline convention
              esac
    ;;
    sparc)
        # [ED] sparc uses ext2 formatted floppies & SILO boot loader. The
	# silo images are floppies with a Sun Disklabel & SILO installed
	#
	#   silo1440k-sun4cdm
	#   silo1440k-sun4u
	#
	if [ ${revext} = "-sun4dm-pci" ]; then
	  silokver="-sun4cdm"
	else
	  silokver="${revext}"
	fi
	gunzip < silo${blocks}k${silokver}.gz > $floppy
	fstype=ext2
	nlfmt='cat -'	# UNIX newline convention
    ;;
esac

sync
relax
sync

# HACK - for kernels with broken loop device handling
if [ "$arch" = "powerpc" ]; then
  case "$system" in
	prep | pmac)
		losetup /dev/loop7 $floppy
		mke2fs -F -q /dev/loop7 $blocks
		sync
		losetup -d /dev/loop7
		;;
  esac
fi 

# mount the disk image to install additional files
info "mounting new disk image at $mnt"
mount -o loop $fsopts -t $fstype $floppy $mnt

# for atari, copy bootloader and linux on disk
if [ "$system" = atari ]; then
	info "installing atari bootloader"
	mkdir $mnt/auto
	# hardcoded bootargs; kernel is in \, bootstrap in \auto
	cat <<EOF > $mnt/auto/bootargs
-s -k \linux -r \root.bin root=/dev/ram load_ramdisk=1 video=keep
EOF
	cp m68k-specials/bootstra.ttp $mnt/auto/bootstra.prg
	# both bootstrap and kernel/ramdisk in \
	cat <<EOF > $mnt/bootargs
-s -k linux -r root.bin root=/dev/ram load_ramdisk=1 video=keep
EOF
	cp m68k-specials/bootstra.ttp $mnt/bootstra.ttp
fi

##
## install kernel
##

relax

if [ "$arch" = powerpc -o "$arch" = m68k ] && [ "$revext" = -graphical ]; then
	# powerpc and m68k doesn't need an extra kernel-image for framebuffer support
	info "installing kernel 'linux$system'"
	cp linux$system $mnt/linux
elif [ "$arch" = powerpc -a "$revext" = pmac ]; then
	info "compressing and installing kernel 'linux$revext'"
 	gzip -9f linux$revext
	cp linux$revext.gz $mnt/linux.gz
else
	info "installing kernel 'linux$revext'"
	cp linux$revext $mnt/linux
fi

relax

# only i386 can use rdev
if [ "$arch" = i386 ]; then
	debug "running rdev on kernel"
	/usr/sbin/rdev -R $mnt/linux 1
	if [ "$blocks" = "2880" ]; then
		/usr/sbin/rdev -r $mnt/linux 0
	else
		/usr/sbin/rdev -r $mnt/linux 49152
	fi
       	/usr/sbin/rdev -v $mnt/linux -1
       	/usr/sbin/rdev $mnt/linux /dev/ram0
fi

# copy root.bin to 2880k floppies
if [ \( "$arch" = i386 -a "$blocks" = "2880" \) -o \
     \( "$arch/$system" = "m68k/bvme6000" -a "$blocks" = "2880" \) -o \
     \( "$arch" = powerpc -a "$revext" != -graphical -a "$blocks" != "1440" \) ]; then
	info "copying root filesystem ($rootimage) into filesystem"
	cp $rootimage $mnt/root.bin
else
	info "rescue image too small to include root.bin"
fi	

# copy system.map to the floppy
if [ "$revext" != lowmem ]; then
	cp sys_map${revext}.gz $mnt/sys_map.gz
	cp config${revext}.gz $mnt/config.gz
fi

relax

info "copying help messages"

suffix=".$arch"                 # can be overriden below

case "$arch/$system" in
    */bvme6000|*/mvme16x|*/mvme147)
	suffix=.vme
	files=debian.txt
	;;
    powerpc/*|sparc/*|alpha/*)
	files="debian.txt readme.txt"
	;;
    *)
	files="debian.txt readme.txt f1.txt f2.txt f3.txt f4.txt f5.txt f6.txt f7.txt f8.txt f9.txt f10.txt" 
	;;
esac

for i in $files; do
	file=$scripts/messages/$language/$i$suffix
	if [ ! -f $file ]; then
		file=$scripts/messages/$language/$i
	fi
	debug "m4 processing $file"
	m4 -EP -D__kernel_version__=$KERNEL_VERSION \
	      -D__kernel_image__=$DEBIAN_KERNEL_IMAGE \
	      -D__size${blocks}__ \
	      -D__debianversion__=$debianversion \
	      -D__arch__=$arch $file \
		| $nlfmt > $mnt/$i
	testexit
done

info "installing boot loader"
case "$arch" in
    sparc)
	write_silo
    ;;
    alpha)
        write_milo
    ;;
    powerpc)
	case $system in
	    apus|bbox|chrp|mbx|prep)
		# no bootloader yet, write out install.sh like m68k
		write_m68kinfo
	    ;;
	    pmac)
		# boot from vmlinux.coff installer for now.
		# write_powerpc
		write_m68kinfo
	    ;;
	esac
    ;;
    m68k)
	case $system in
	    bvme6000|mvme16x|mvme147)
		write_vmelilo
	    ;;
	    atari|amiga|mac)
		# these LILOs not implemented yet, but need to write install.sh
		echo "Not writing syslinux for $system !"
		write_m68kinfo
	    ;;
	esac
    ;;
    i386)
	write_syslinux
    ;;
    arm)
	write_m68kinfo
    ;;
esac

if [ "x$noinstall" = x ]; then
	echo rescue >$mnt/type.txt
else
	echo boot >$mnt/type.txt
fi

# do a few tests
[ -f $mnt/linux -o $mnt/linux.gz ] || error "can't fine kernel"
[ -f $mnt/install.sh -o "x$noinstall" != x ] || error "can't find install.sh"
[ -x $mnt/install.sh -o "x$noinstall" != x ] || error "install.sh is not excutable"

relax

info "unmounting, cleaning up"
umount $mnt
rmdir $mnt
rm -fr /var/tmp/extract-tmp-$$ resc${blocks}${revext}.bin
mv $floppy resc${blocks}${revext}.bin

info "rescue image complete:" `ls -l resc${blocks}${revext}.bin`

# make root disk for i386, 1200k or 1440k
if [ "$arch" = i386 -a $blocks -le 2800 ]; then
	info "making detached root disk root${blocks}${revext}.bin from $rootimage"
	dd if=/dev/zero of=root${blocks}${revext}.bin bs=1k count=$blocks
ls -l $rootimage
	dd if=$rootimage of=root${blocks}${revext}.bin bs=1k count=$blocks conv=notrunc
elif  [ "$arch" = powerpc -a $blocks = 1440 ]; then
	info "making detached root disk root${blocks}${revext}.bin from $rootimage"
	dd if=/dev/zero of=root${blocks}${revext}.bin bs=1k count=$blocks
	dd if=$rootimage of=root${blocks}${revext}.bin bs=1k count=$blocks conv=notrunc
elif  [ "$arch" = m68k -a $blocks = 1440 ]; then
	info "making detached root disk root${blocks}${revext}.bin from $rootimage"
	# we seem to have dropped 720, rootmac.bin doesn't quite fit)
	dd if=/dev/zero of=root${blocks}${revext}.bin bs=1k count=$blocks
	dd if=$rootimage of=root${blocks}${revext}.bin bs=1k count=$blocks conv=notrunc
fi

exit 0
