/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Platform.h"

#include "Hordes.h"						// Hordes::EventCounter
#include "Miscellaneous.h"				// StMemory
#include "PreferenceMgr.h"
#include "ResStrings.h"
#include "SessionFile.h"
#include "Startup.h"					// Startup::ScheduleQuit

#include <unistd.h>
#include <sys/time.h>
#include <sys/stat.h>					// mkdir
#include <time.h>
#include <ctype.h>

#include "omnithread.h"
#include "Document.h"					// getCPU, saveRAMImage
#include "fltk_Dialog.h"				// showNewEmulatorWindow
#include "fltk_MainWindow.h"			// MainWindow::getMainWindow


// ===========================================================================
//		 Globals
// ===========================================================================
static void CommRead (void*);
static void CommWrite (void*);

unsigned long gPseudoTickcount;
static long long gStartTime;
static Bool gForceNewDirectory;
bool gShowingDialog;
//static FILE *		gCommHandle;
static int		gCommHandle;

// ===========================================================================
//		 TByteQueuePrivate
// ===========================================================================

// Queue used for keyboard and serial input/output.	 This is the
// private, platform-dependent implementation used by TByteQueue.

class TByteQueuePrivate
{
public:
	TByteQueuePrivate();
	~TByteQueuePrivate();

	void Put( uae_u8 b );
	uae_u8 Get( void );
	int GetUsed( void ) { return _usedCount; }
	int GetFree( void ) { return _freeCount; }

	Bool WaitForDataAvailable( int timeoutms );

private:

	enum { QUEUE_SIZE = 8 };

	long _usedCount;
	long _freeCount;
	uae_u8 _buffer[QUEUE_SIZE];
	int _head;
	int _tail;

	omni_mutex _mutex;
	omni_condition _available;
};

long long PrvGetMicroseconds (void)
{
	struct timeval tv;
	gettimeofday (&tv, NULL);

	long long usecs = ((long long) tv.tv_sec) * 1000000ULL + tv.tv_usec;

	return usecs;
}

// ===========================================================================
//		 Platform
// ===========================================================================

#ifndef __QNXNTO__
// Compare lexigraphically two strings

int _stricmp( const char *s1, const char *s2 )
{
	return strcasecmp( s1, s2 );
}

int _strnicmp( const char *s1, const char *s2, int n )
{
	return strncasecmp( s1, s2, n );
}

char* _strdup( const char *s )
{
	return strdup( s );
}
#endif
// ---------------------------------------------------------------------------
//		 Platform::Initialize
// ---------------------------------------------------------------------------
// Initializes platform-dependent stuff.

void Platform::Initialize( void )
{
	gStartTime = ::PrvGetMicroseconds();
}


// ---------------------------------------------------------------------------
//		 Platform::Reset
// ---------------------------------------------------------------------------

void Platform::Reset( void )
{
	gStartTime = ::PrvGetMicroseconds();
}


// ---------------------------------------------------------------------------
//		 Platform::Save
// ---------------------------------------------------------------------------

void Platform::Save(SessionFile&)
{
}


// ---------------------------------------------------------------------------
//		 Platform::Load
// ---------------------------------------------------------------------------

void Platform::Load(SessionFile&)
{
}


// ---------------------------------------------------------------------------
//		 Platform::Dispose
// ---------------------------------------------------------------------------

void Platform::Dispose( void )
{
}


// ---------------------------------------------------------------------------
//		 Platform::CommonDialog
// ---------------------------------------------------------------------------

int Platform::CommonDialog( const char* title, const char* msg, int flags )
{
	gShowingDialog = true;
	int result = fltkDialog( title, msg, flags );
	gShowingDialog = false;
	return result;
}


// ---------------------------------------------------------------------------
//		 Platform::GetString
// ---------------------------------------------------------------------------

string Platform::GetString( int id )
{
	return _ResGetString( id );
}


// ---------------------------------------------------------------------------
//		 Platform::GetIDForError
// ---------------------------------------------------------------------------

int Platform::GetIDForError( ErrCode error )
{
#if 0
	return kStr_GenericError;
#endif
	return 0;
}


// ---------------------------------------------------------------------------
//		 Platform::GetIDForRecovery
// ---------------------------------------------------------------------------

int Platform::GetIDForRecovery( ErrCode error )
{
	return 0;
}


// ---------------------------------------------------------------------------
//		 Platform::GetShortVersionString
// ---------------------------------------------------------------------------
// Returns a short version string.	The format of the string is:
//
//		#.# (.#) ([dab]#)
//
//		# = one or more numeric digits
//		. = literal '.'
//		Patterns in parentheses are optional
//		Patterns in brackets mean "one of these characters"
//		Spaces are shown above for clarity; they do not appear in the string
//
// Valid strings would be: 2.1d7, 2.1.1b14, 2.99, 2.1.1

string Platform::GetShortVersionString( void )
{
	return string("Platform::GetShortVersionString not implemented");
}


// ---------------------------------------------------------------------------
//		 Platform::ROMResourcePresent
// ---------------------------------------------------------------------------
Bool Platform::ROMResourcePresent (void)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::GetROMResource
// ---------------------------------------------------------------------------

Bool Platform::GetROMResource (void*& rom, uae_s32& romSize)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::PSFResourcePresent
// ---------------------------------------------------------------------------

Bool Platform::PSFResourcePresent (void)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::GetPSFResource
// ---------------------------------------------------------------------------

Bool Platform::GetPSFResource (void*& psf, uae_s32& psfSize)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::ConfigResourcePresent
// ---------------------------------------------------------------------------

Bool Platform::ConfigResourcePresent (void)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::GetConfigResource
// ---------------------------------------------------------------------------

Bool Platform::GetConfigResource (void*& config, uae_s32& configSize)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::GetEmbeddedDeviceType
// ---------------------------------------------------------------------------

Bool Platform::GetEmbeddedDeviceType (DeviceType& device)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::GetEmbeddedRAMSize
// ---------------------------------------------------------------------------

Bool Platform::GetEmbeddedRAMSize (uae_s32& ramSize)
{
	return false;
}


/***********************************************************************
 *
 * FUNCTION:	Platform::PinToScreen
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	None
 *
 * RETURNED:	True if the rectangle was changed.
 *
 ***********************************************************************/

Bool Platform::PinToScreen (AbsRectType& r)
{
	// !!! TBD
	return false;
}


/***********************************************************************
 *
 * FUNCTION:	QueryNewDocument
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

Bool Platform::QueryNewDocument (Configuration& cfg)
{
	return showNewEmulatorWindow(cfg);
}


/***********************************************************************
 *
 * FUNCTION:    Platform::GetCPU
 *
 * DESCRIPTION: Return a pointer to the currently instantiated CPU
 *				object.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    The pointer to the global CPU object, or NULL if one
 *				has not been created (or has been deleted).
 *
 ***********************************************************************/

CPU* Platform::GetCPU (void)
{
	CPU* cpu = NULL;

	MainWindow* w = MainWindow::getMainWindow();
	if (w)
	{
		Document* d = w->getDocument();
		if (d)
		{
			cpu = d->getCPU();
		}
	}

	return cpu;
}


/***********************************************************************
 *
 * FUNCTION:	ToHostEOL
 *
 * DESCRIPTION:	Converts a string of characters into another string
 *				where the EOL sequence is consistant for files on the
 *				current platform.
 *
 * PARAMETERS:	dest - receives the result.	 The buffer is sized by
 *					this function, so the caller doesn't have to
 *					allocate any space itself.
 *
 *				destLen - receives the length of the resulting string.
 *
 *				src - pointer to input characters.
 *
 *				srcLen - number of characters pointed to by src.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Platform::ToHostEOL (	StMemory& dest, long& destLen,
				const char* src, long srcLen)
{
	char*	d = (char*) Platform::AllocateMemory (srcLen);
	char*	p = d;
	Bool	previousWas0x0D = false;

	for (long ii = 0; ii < srcLen; ++ii)
	{
		char	ch = src[ii];

		// Convert 0x0D to 0x0A.
		
		if (ch == 0x0D)
		{
			*p++ = 0x0A;
		}

		// If we're looking at a 0x0A that's part of
		// a 0x0D/0x0A, just swallow it.

		else if (ch == 0x0A && previousWas0x0D)
		{
			// Nothing
		}

		// Copy all other characters straight through.

		else
		{
			*p++ = ch;
		}

		previousWas0x0D = ch == 0x0D;
	}

	destLen = p - d;
	d = (char*) Platform::ReallocMemory (d, destLen);
	dest.Adopt (d);
}


/***********************************************************************
 *
 * FUNCTION:	Platform::CreateReferenceInEmulatorPath
 *
 * DESCRIPTION:	Create a file reference to a file with the given name
 *				and that resides in the same directory as the emulator.
 *
 * PARAMETERS:	fileName - Name of the file the caller wants to create.
 *
 * RETURNED:	The desired FileReference.	This object could be
 *				invalid if an error occured.
 *
 ***********************************************************************/

FileReference Platform::CreateReferenceInEmulatorPath (const char* fileName)
{
	char*	home = getenv ("HOME");
	char*	path = (char*) malloc (strlen (home) + 1 + strlen (fileName) + 1);

	strcpy (path, home);
	strcat (path, "/");
	strcat (path, fileName);

	FileReference result (path);

	free (path);

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	Platform::CreateReferenceInGremlinStatePath
 *
 * DESCRIPTION:	Create a file reference to a file with the given name
 *				and that resides in the same directory designated to
 *				hold auto-saved Gremlin states.
 *
 *				If gForceNewDirectory is true, a directory is created
 *				and used.
 *
 * PARAMETERS:	fileName - Name of the file the caller wants to create.
 *
 * RETURNED:	The desired FileReference.	This object could be
 *				invalid if an error occured.
 *
 ***********************************************************************/

FileReference Platform::CreateReferenceInGremlinStatePath (const char* fileName)
{
	FileReference result;
	
	// If requested, create the directory to use.

	static char	gGremlinStatePath[PATH_MAX];

	if (gForceNewDirectory)
	{
		gForceNewDirectory = false;
		
		sprintf (gGremlinStatePath, "%s/GremlinStates_%ld",
				 getenv ("HOME"),
				 Platform::GetMilliseconds ());

		if (mkdir (gGremlinStatePath, 0777) != 0)
		{
			// !!! Put up some error message.
			
			gGremlinStatePath[0] = 0;
		}
	}

	// If we don't have a directory, blow out.

	if (gGremlinStatePath[0] == 0)
	{
		return result;
	}

	// Create the file reference from the full path.

	char		fullPath[PATH_MAX];
	sprintf (fullPath, "%s/%s", gGremlinStatePath, fileName);

	result = FileReference (fullPath);

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	Platform::GetPreferenceFile
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	.
 *
 * RETURNED:	The desired FileReference.	This object could be
 *				invalid if an error occured.
 *
 ***********************************************************************/

FileReference Platform::GetPreferenceFile (void)
{
	return Platform::CreateReferenceInEmulatorPath (".poserrc");
}


// ---------------------------------------------------------------------------
//		 Platform::GetLoadableFileList
// ---------------------------------------------------------------------------

void Platform::GetLoadableFileList( string directoryName, FileRefList& fileList )
{
#if 0
	// Get the application's directory.

	char	path[_MAX_PATH], drive[_MAX_DRIVE], dir[_MAX_DIR];

	::GetModuleFileName (gInstance, path, _MAX_PATH);
	_splitpath (path, drive, dir, NULL, NULL);
	_makepath (path, drive, dir, NULL, NULL);

	string	pathName (path + directoryName + "\\");
	string	searchPattern (pathName + "*");

	// Start iterating over it.	 We'll iterate over all the contents
	// and filter by filetype by hand.

	WIN32_FIND_DATA	fileData;
	HANDLE			hSearch = ::FindFirstFile (searchPattern.c_str (), &fileData);
	if (hSearch == INVALID_HANDLE_VALUE)
	{
		// If an error occurs (probably no such directory), just return.

		return;
	}


	BOOL	finished = FALSE;
	while (!finished)
	{
		if (::EndsWith (fileData.cFileName, ".prc") ||
			::EndsWith (fileData.cFileName, ".pdb") ||
			::EndsWith (fileData.cFileName, ".pqa"))
		{
			string	fullPath (pathName + fileData.cFileName);
			fileList.push_back (LoadFilePB (FileReference (fullPath)));
		}

		if (!::FindNextFile (hSearch, &fileData))
		{
			DWORD	lastError = GetLastError ();
			if (lastError != ERROR_SUCCESS)
			{
				// Blow out of here on *any* error, not just ERROR_NO_MORE_FILES.
				finished = TRUE;
			}
		}
	}

	::FindClose (hSearch);
#endif
}

// -----------------------------------------------------------------------------
// find the ROM file path embedded in the saved ram image
Bool Platform::ReadROMFileReference (ChunkFile& docFile, FileReference& f)
{
	// First look for a ROM file path.

	string	path;
	if (docFile.ReadString (SessionFile::kROMUnixPathTag, path))
	{
		f = FileReference (path);
		return true;
	}

	// If a path can't be found, look for a simple ROM name.

	string	name;
	if (docFile.ReadString (SessionFile::kROMNameTag, name))
	{
		// !!! TBD
	}

	return false;
}

void Platform::WriteROMFileReference (ChunkFile& docFile, const FileReference& f)
{
	docFile.WriteString (SessionFile::kROMUnixPathTag, f.GetFilePath ());
}


// ---------------------------------------------------------------------------
//		 Platform::UseNewAutoSaveDirectory
// ---------------------------------------------------------------------------

void Platform::UseNewAutoSaveDirectory( void )
{
	gForceNewDirectory = true;
}


// ---------------------------------------------------------------------------
//		 Platform::AutoSaveState
// ---------------------------------------------------------------------------

void Platform::AutoSaveState(const FileReference& fileRef)
{
	MainWindow::getMainWindow()->getDocument()->saveRAMImage (fileRef, true);
}


// ---------------------------------------------------------------------------
//		 Platform::Delay
// ---------------------------------------------------------------------------
// Delay for a time period appropriate for a sleeping 68328.

void Platform::Delay( void )
{
	// Delay 10 msecs.	Delaying by this amount pauses us 1/100 of a second,
	// which is the rate at which the device's tickcount counter increments.
	//
	// Wait on an event instead of just calling Sleep(10) so that another
	// thread can wake us up before our time.

	omni_thread::sleep( 0, 10000 ); // 10k nanoseconds = 1/100 sec
}


// ---------------------------------------------------------------------------
//		 Platform::Cycle
// ---------------------------------------------------------------------------

void Platform::Cycle( void )
{
	// Nothing to do in Unix.
}


// ---------------------------------------------------------------------------
//		 Platform::RealAllocateMemory
// ---------------------------------------------------------------------------

void* Platform::RealAllocateMemory (size_t size, Bool clear, const char*, int)
{
	void*	result;

	if (clear)
		result = calloc (size, 1);
	else
		result = malloc (size);

	Errors::ThrowIfNULL (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 Platform::RealReallocMemory
// ---------------------------------------------------------------------------

void* Platform::RealReallocMemory (void* p, size_t size, const char*, int)
{
	void*	result = realloc (p, size);

	Errors::ThrowIfNULL (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 Platform::RealDisposeMemory
// ---------------------------------------------------------------------------

void Platform::RealDisposeMemory (void* p)
{
	if (p)
	{
		free (p);
	}
}


/***********************************************************************
 *
 * FUNCTION:	Platform::ForceStartupScreen
 *
 * DESCRIPTION:	See if the user has requested that the Startup dialog
 *				be presented instead of attempting to use the latest
 *				session file or creation settings.
 *
 *				The current signal is to toggle the CAPSLOCK key.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	True if the user has signalled that the dialog should
 *				be presented.
 *
 ***********************************************************************/

Bool Platform::ForceStartupScreen (void)
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::StopOnResetKeyDown
// ---------------------------------------------------------------------------

Bool Platform::StopOnResetKeyDown( void )
{
	// Comment this out for now.  It seems that Windows doesn't always return
	// the expected result.	 That is, this function often returns TRUE even
	// though the Control is not down.
	//
	// Since this functionality is really only required by Palm OS engineers,
	// and since they're working on Macs, we don't really need this feature
	// in the Windows version now anyway.

//	return ::GetAsyncKeyState (VK_CONTROL) != 0;
	return false;
}


// ---------------------------------------------------------------------------
//		 Platform::GetMilliseconds
// ---------------------------------------------------------------------------

uae_u32 Platform::GetMilliseconds( void )
{
	long long usecs = ::PrvGetMicroseconds (); // - gStartTime;
	uae_u32   millis = (uae_u32) (usecs / 1000);

	return millis;
}



// ---------------------------------------------------------------------------
//		 Platform::HeadPatch
// ---------------------------------------------------------------------------

#if 0
int Platform::HeadPatch( uae_u16 iATrap )
{
	switch (iATrap)
	{
	case sysTrapSndDoCmd:
	{
		SndCommandType*	sc = (SndCommandType*) get_real_address(get_long(m68k_areg(regs, 7) + 4));
		if (do_get_mem_word(&sc->cmd) == 1)
		{
			if (gEnableSounds)
			{
				Beep(do_get_mem_word(&sc->param1lo), do_get_mem_word(&sc->param2));
			}

			return true;
		}

		break;
	}
	}

	return false;
}
#endif


// ---------------------------------------------------------------------------
//		 Platform::ConnectToDebugger
// ---------------------------------------------------------------------------

CSocket* Platform::CreateDebuggerSocket (void)
{
	return NULL;
}


// ---------------------------------------------------------------------------
//		 Platform::ExitDebugger
// ---------------------------------------------------------------------------
//	Perform platform-specific operations when debug mode is exited.

void Platform::ExitDebugger( void )
{
}


/***********************************************************************
 *
 * FUNCTION:    Platform::SessionRunning
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

Bool Platform::SessionRunning (void)
{
	return GetCPU() != NULL;
}



ErrCode Platform::CreateSession (void)
{
	MainWindow* mw = MainWindow::getMainWindow();
	if (mw)
	{
		mw->doEmulatorStart();
	}

	return errNone;
}

ErrCode Platform::OpenSession (void)
{
	MainWindow* mw = MainWindow::getMainWindow();
	if (mw)
	{
		mw->doOpenMemory();
	}

	return errNone;
}


// ---------------------------------------------------------------------------
//		 Platform::CreatePortNameCache
// ---------------------------------------------------------------------------

void Platform::CreatePortNameCache (StringList& results)
{
#ifdef __QNXNTO__
	results.push_back ("/dev/ser1");
	results.push_back ("/dev/ser2");
#else
	results.push_back ("/dev/ttyS0");
	results.push_back ("/dev/ttyS1");
	results.push_back ("/dev/ttyS2");
	results.push_back ("/dev/ttyS3");
#endif
}


// ---------------------------------------------------------------------------
//		 Platform::ViewDrawLine
// ---------------------------------------------------------------------------

void Platform::ViewDrawLine( int xStart, int yStart, int xEnd, int yEnd )
{
}


// ---------------------------------------------------------------------------
//		 Platform::ViewDrawPixel
// ---------------------------------------------------------------------------

void Platform::ViewDrawPixel( int xPos, int yPos )
{
}

// ---------------------------------------------------------------------------
// from windows
//#define PRINTF	if (!LogSerial ()) ; else LogAppendMsg
//#define PRINTF	printf
#define PRINTF		if (0) printf 

/***********************************************************************
 *
 * FUNCTION:	Platform::OpenSerialPort
 *
 * DESCRIPTION:	Open the serial port with default settings.  A call to
 *				this function  should be followed by a call to
 *				SetSerialPortSettings.
 *
 *				This function will close the serial port before trying
 *				to open it.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	Error code for any errors that occur.
 *
 ***********************************************************************/
ErrCode Platform::OpenSerialPort (void)
{
	PRINTF ("Serial Emulation (Unix): Attempting to open serial port...\n");

	ErrCode	err = kError_NoError;

	// Make sure the serial port is closed.

	CloseSerialPort ();

	// Create the comm handle used for serial communications.

	Preference<string>	pref (kPrefKeyCommPort);
	string	commPort = *pref;

	if (commPort.length() > 0 )
	{
		PRINTF ("Serial Emulation (Unix): Opening serial port...:%s\n",commPort.c_str());

		gCommHandle = open(commPort.c_str(),O_RDWR);

		if (gCommHandle <= 0)
		{
			err = errno;
			gCommHandle = 0;

			PRINTF ("Serial Emulation (Unix): Error %ld serial port...\n", err);

			return err;
		}
	}
	else
	{
		PRINTF ("Serial Emulation (Unix): No port selected in the Properties dialog box...\n");
	}

	if (gCommHandle)
	{
		PRINTF ("Serial Emulation (Unix): Creating serial port handler threads...\n");

		// Create the threads that will interact with the port.  Make
		// them high priority threads.

		omni_thread *ReadThread = omni_thread::create( CommRead,  NULL, omni_thread::PRIORITY_NORMAL );
		omni_thread *WriteThread= omni_thread::create( CommWrite, NULL, omni_thread::PRIORITY_NORMAL );		

        PRINTF("OpenSerialPort: ReadThread=<%p> and WriteThread=<%p>\n", ReadThread, WriteThread);
	}

	return err;
}


/***********************************************************************
 *
 * FUNCTION:	Platform::SetSerialPortSettings
 *
 * DESCRIPTION:	Establish the serial port settings.  This function
 *				should be called after OpenSerialPort.
 *
 * PARAMETERS:	parity - desired parity
 *
 *				stopBits - number of stop bits: 1, or 2
 *
 *				dataBits - number of data bits: 7 or 8
 *
 *				baud - desired baud (e.g., 57600)
 *
 *				hwHandshaking - true for hardware hardshaking
 *
 * RETURNED:	Error code for any errors that occur.
 *
 ***********************************************************************/

ErrCode Platform::SetSerialPortSettings (	UART::Parity	parity,
						int	stopBits,
						int	dataBits,
						uae_u32	baud,
						Bool	hwHandshaking)
{
	PRINTF ("Serial Emulation (Unix): Setting settings...\n");

	if (Platform::SerialPortOpen ())
	{
		PRINTF ("Serial Emulation (Unix): Serial port open, so plowing ahead...\n");
		PRINTF ("	parity			= %ld\n", (long) parity);
		PRINTF ("	stopBits		= %ld\n", (long) stopBits);
		PRINTF ("	dataBits		= %ld\n", (long) dataBits);
		PRINTF ("	baud			= %ld\n", (long) baud);
		PRINTF ("	hwHandshaking	= %ld\n", (long) hwHandshaking);

		char buf[100];
//kedl, need to handle with ioctls...
        Preference<string>	pref (kPrefKeyCommPort);
        string	commPort = *pref;
		sprintf (buf,"stty baud=%ld < %s",baud,commPort.c_str());
		system(buf);
	}
	else
	{
		PRINTF ("Serial Emulation (Unix): Serial port closed, so not setting settings...\n");
	}
return errNone;

	return kError_CommNotOpen;
}

/***********************************************************************
 *
 * FUNCTION:	Platform::GetSerialPortSettings
 *
 * DESCRIPTION:	Return the serial port settings.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Platform::GetSerialPortSettings (	UART::Parity&	parity,
					int&			stopBits,
					int&			dataBits,
					uae_u32&		baud,
					Bool&			hwHandshaking)
{
PRINTF ("getting serial port settings\n");
/*
	parity			= UART::GetLastParity ();
	stopBits		= UART::GetLastStopBits ();
	dataBits		= UART::GetLastDataBits ();
	baud			= UART::GetLastBaud ();
	hwHandshaking	= UART::GetLastHwHandshaking ();
*/
}


/***********************************************************************
 *
 * FUNCTION:	Platform::SerialPortOpen
 *
 * DESCRIPTION:	Returns whether or not the serial port is open for
 *				business.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool Platform::SerialPortOpen (void)
{
	return gCommHandle != 0;
}


/***********************************************************************
 *
 * FUNCTION:	Platform::CloseSerialPort
 *
 * DESCRIPTION:	Close the serial port.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Platform::CloseSerialPort (void)
{
	PRINTF ("Serial Emulation (Unix Attempting to close serial port...\n");

	if (Platform::SerialPortOpen ())
	{
		PRINTF ("Serial Emulation (Unix): Closing serial port...\n");

		// Set gCommHandle to 0 now. That will cause the CommRead
		// thread to quit after we close the handle.

//		FILE *	commHandle = gCommHandle;
		int	commHandle = gCommHandle;
		gCommHandle = 0;

//		::fclose (commHandle);
		::close (commHandle);
	}
	else
	{
		PRINTF ("Serial Emulation (Unix): Serial port already closed...\n");
	}
}


/***********************************************************************
 *
 * FUNCTION:	Platform::TransmitTxFIFO
 *
 * DESCRIPTION:	Transmit any bytes in the Tx FIFO out the serial port.
 *				Assumes that the serial port is open.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Platform::TransmitTxFIFO (void)
{
	// Nothing to do on Windows; the Comm threads will pick up the new
	// data automatically.
}


/***********************************************************************
 *
 * FUNCTION:	Platform::ReceiveRxFIFO
 *
 * DESCRIPTION:	Fills up the Rx FIFO with as many bytes as it can from
 *				the host serial port.  Assumes that the serial port is
 *				open.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Platform::ReceiveRxFIFO (void)
{
	// Nothing to do on Windows; the Comm threads will pick up the new
	// data automatically.
}


/***********************************************************************
 *
 * FUNCTION:	CommRead
 *
 * DESCRIPTION:	This function sits in its own thread, waiting for data
 *				to show up in the serial port.  If data arrives, this
 *				function plucks it out and stores it in a thread-safe
 *				queue.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void CommRead (void*)
{
	PRINTF ("CommRead starting.\n");
	while (gCommHandle)
	{
		unsigned char	buf[16];
		long			n = UART::GetRxQueue ().GetFree ();
		int				status;

		if (n == 0)
		{
			// No space for new data, so don't read any.

			PRINTF ("no space\n");
#if 1
			omni_thread* self = omni_thread::self();
			if (self)
				self->yield();
#else
			omni_thread::sleep( 0, 50000 ); // 10k nanoseconds = 1/100 sec
#endif
			continue;
		}

		struct timeval	timeout;
		int				fd = gCommHandle;
		fd_set			read_fds;

		FD_ZERO(&read_fds);
		FD_SET(fd, &read_fds);
//		PRINTF ("doing select: %d\n", fd);

		timeout.tv_sec = 10;
		timeout.tv_usec = 0;
		status = select(10, &read_fds, 0, 0, &timeout);
//		PRINTF ("read status: %d\n", status);
//		if (status < 0)
//		    delay(30);	// probably port closed

		if (status >= 1)	// data available
		{
			if (FD_ISSET(fd, &read_fds))
			{
				if ((n = read(gCommHandle, buf, n)) != 0)
				{
					if (n == 0)
						break; // port closed

					if (n > 0)
					{
//						if (LogSerialData ())
//							LogAppendData (buf, n, "Serial Emulation (Unix): Received data:");
//						else
							PRINTF ("Serial Emulation (Unix): Received %ld serial bytes.\n", n);

						for (int ii = 0; ii < n; ++ii)
						{
							UART::GetRxQueue ().Put (buf[ii]);
						}
					}
				}
			}
		}
	}

	PRINTF ("CommRead exitting.\n");
    omni_thread* self = omni_thread::self();
    if (self)
		self->exit(NULL);

	abort();  /* should never get here! */
}


/***********************************************************************
 *
 * FUNCTION:	CommWrite
 *
 * DESCRIPTION:	This function sits in its own thread, waiting for data
 *				to show up in the write queue.  If data arrives, this
 *				function plucks it out and sends it out to the port.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void CommWrite (void*)
{
	PRINTF ("CommWrite starting.\n");

	while (gCommHandle)
	{
		unsigned char	buf[8];
		long			n = 0;

		while (n < sizeof (buf) && UART::GetTxQueue ().GetUsed () > 0)
		{
			buf[n++] = UART::GetTxQueue ().Get ();
		}

		if (n == 0)
		{
			// Never mind, there really wasn't any data in the buffer
			// (see note in TByteQueuePrivate::Put).

			continue;
		}

//		if (LogSerialData ())
//			LogAppendData (buf, n, "Serial Emulation (Unix): Transmitted data:");
//		else
		    PRINTF ("Serial Emulation (Unix): Transmitted %ld serial bytes.\n", n);

		for (int i = 0; i < n; ++i)
			PRINTF ("%x ", buf[i]);
		PRINTF ("\n");

		write(gCommHandle, buf, n);
	}

	PRINTF ("CommWrite exitting.\n");

    omni_thread* self = omni_thread::self();
    if (self)
		self->exit(NULL);

	abort();  /* should never get here! */
}

// ---------------------------------------------------------------------------
//		 Load File Progress Window
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
//		 Platform::LFPW_Open
// ---------------------------------------------------------------------------

void Platform::LFPW_Open( void )
{
}


// ---------------------------------------------------------------------------
//		 Platform::LFPW_SetNumFiles
// ---------------------------------------------------------------------------

void Platform::LFPW_SetNumFiles( int numFiles )
{
}


// ---------------------------------------------------------------------------
//		 Platform::LFPS_StartNewFile
// ---------------------------------------------------------------------------

void Platform::LFPW_StartNewFile( const char* path )
{
}


// ---------------------------------------------------------------------------
//		 Platform::LFPW_SetFileMax
// ---------------------------------------------------------------------------

void Platform::LFPW_SetFileMax( int max )
{
}


// ---------------------------------------------------------------------------
//		 Platform::LFPW_SetFileValue
// ---------------------------------------------------------------------------

void Platform::LFPW_SetFileValue( int value )
{
}


// ---------------------------------------------------------------------------
//		 Platform::LFPW_Close
// ---------------------------------------------------------------------------

void Platform::LFPW_Close( void )
{
}


// ---------------------------------------------------------------------------
//		 Platform::LFPW_UserAborts
// ---------------------------------------------------------------------------

Bool Platform::LFPW_UserAborts( void )
{
	return false;
}


// ---------------------------------------------------------------------------
//		 Gremlin Control Window
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
//		 Platform::GCW_Open
// ---------------------------------------------------------------------------

void Platform::GCW_Open( void )
{
}


// ---------------------------------------------------------------------------
//		 Platform::GCW_Close
// ---------------------------------------------------------------------------

void Platform::GCW_Close( void )
{
}


// ===========================================================================
//		 TByteQueue
// ===========================================================================

// ---------------------------------------------------------------------------
//		 TByteQueue::TByteQueue
// ---------------------------------------------------------------------------

TByteQueue::TByteQueue():
		fImplementation( new TByteQueuePrivate )
{
}


// ---------------------------------------------------------------------------
//		 TByteQueue::~TByteQueue
// ---------------------------------------------------------------------------

TByteQueue::~TByteQueue()
{
	delete fImplementation;
}


// ---------------------------------------------------------------------------
//		 TByteQueue::Put
// ---------------------------------------------------------------------------

void TByteQueue::Put( uae_u8 b )
{
	fImplementation->Put( b );
}


// ---------------------------------------------------------------------------
//		 TByteQueue::Get
// ---------------------------------------------------------------------------

uae_u8 TByteQueue::Get( void )
{
	return fImplementation->Get();
}


// ---------------------------------------------------------------------------
//		 TByteQueue::GetUsed
// ---------------------------------------------------------------------------

int TByteQueue::GetUsed( void )
{
	return fImplementation->GetUsed();
}


// ---------------------------------------------------------------------------
//		 TByteQueue::GetFree
// ---------------------------------------------------------------------------

int TByteQueue::GetFree( void )
{
	return fImplementation->GetFree();
}


// ---------------------------------------------------------------------------
//		 TByteQueue::WaitForDataAvailable
// ---------------------------------------------------------------------------

Bool TByteQueue::WaitForDataAvailable( int timeoutms )
{
	return fImplementation->WaitForDataAvailable( timeoutms );
}

// ===========================================================================
//		 TByteQueuePrivate
// ===========================================================================

// ---------------------------------------------------------------------------
//		 TByteQueuePrivate::TByteQueuePrivate
// ---------------------------------------------------------------------------

TByteQueuePrivate::TByteQueuePrivate() :
	_available( &_mutex )
{
	_usedCount = 0;
	_freeCount = QUEUE_SIZE;
  
	_head = 0;
	_tail = 0;
}


// ---------------------------------------------------------------------------
//		 TByteQueuePrivate::~TByteQueuePrivate
// ---------------------------------------------------------------------------

TByteQueuePrivate::~TByteQueuePrivate()
{
}


// ---------------------------------------------------------------------------
//		 TByteQueue::Put
// ---------------------------------------------------------------------------

void TByteQueuePrivate::Put( uae_u8 b )
{
	omni_mutex_lock l(_mutex);

	// If the queue is full, just discard the data byte.  
	if (_usedCount >= QUEUE_SIZE)
	{
		return;
	}

	_buffer[_head] = b;
	_head = (_head + 1) % QUEUE_SIZE;
  
	_usedCount++;
	_freeCount--;
  
	// Tell clients that there may be new data in the buffer.
	_available.signal();
}


// ---------------------------------------------------------------------------
//		 TByteQueue::Get
// ---------------------------------------------------------------------------

uae_u8 TByteQueuePrivate::Get()
{
	omni_mutex_lock l(_mutex);

	// If the queue is empty, return a null byte (this shouldn't happen,
	// because the caller should always call GetUsed before Get).
  
	if (_usedCount == 0)
	{
		return 0;
	}
  
	uae_u8 r = _buffer[_tail];
	_tail = (_tail + 1) % QUEUE_SIZE;
  
	_freeCount++;
	_usedCount--;
  
	return r;
}


// ---------------------------------------------------------------------------
//		 TByteQueue::WaitForDataAvailable
// ---------------------------------------------------------------------------

Bool TByteQueuePrivate::WaitForDataAvailable( int timeoutms )
{
	// calc absolute time to wait until:
	unsigned long sec = 0, nanosec = 0;
	omni_thread::get_time( &sec, &nanosec, 0, timeoutms * 1000 );

	return _available.timedwait( sec, nanosec ) == 1;
}

// ---------------------------------------------------------------------------
//		 Others...
// ---------------------------------------------------------------------------

#if 0
static void InsertStr( char* dest, int insertBefore, const char* src )
{
}


static void FormatInteger( char* dest, uae_u32 integer )
{
}
#endif

