/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include <stdio.h>
#include <string.h>

#include "MapFile.h"


// -----------------------------------------------------------------------------
// private types
// -----------------------------------------------------------------------------

class MapEntry
{
public:
	char *_key;
	char *_value;
	struct MapEntry* _next;
	struct MapEntry* _prev;
	
	MapEntry() : _key(NULL), _value(NULL), _next(NULL), _prev(NULL) {}
	~MapEntry()
	{
		if (_key!=NULL) delete _key;
		if (_value!=NULL) delete _value;
	}
};

// -----------------------------------------------------------------------------
// constructor / destructor
// -----------------------------------------------------------------------------

MapFile::MapFile( int buckets )
: _numBuckets( buckets )
{
	initialize();
}

MapFile::MapFile( const char* filename, int buckets )
: _numBuckets( buckets )
{
	initialize();
	parseFile( filename );
}

MapFile::~MapFile( void )
{
	for (int i=0; i<_numBuckets; ++i)
	{
		while ( _buckets[i] != NULL )
		{
			MapEntry *e = _buckets[i];
			removeEntry( e );
			delete e;
		}
	}
}

// -----------------------------------------------------------------------------
// public methods
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// add an entry to the Map
void MapFile::add( const char* key, const char* value )
{
	remove( key );
	
	MapEntry *entry = new MapEntry;
	entry->_key = strdup( key );
	entry->_value = strdup( value );
	insertEntry( entry );
}

bool MapFile::remove( const char* key )
{
	MapEntry *e = lookup( key );
	if ( e != NULL )
	{
		removeEntry( e );
		delete e;
		return true;
	}
	return false;
}

char* MapFile::find( const char* key )
{
	MapEntry *e = lookup( key );
	if ( e != NULL )
	{
		return e->_value;
	}
	return NULL;
}

// -----------------------------------------------------------------------------
// parse a file
void MapFile::parseFile( const char* fname )
{
	FILE *f = fopen( fname, "r" );
	if ( f == NULL )
		return;
	
	const int BUFFSIZE = 256;
	
	char buffer[BUFFSIZE];
	while ( fgets( buffer, BUFFSIZE, f ) != NULL )
	{
		scanLine( buffer );
	}
	fclose( f );
}

// -----------------------------------------------------------------------------
// private methods
// -----------------------------------------------------------------------------

void MapFile::initialize()
{
	_useCase = false;
	_count = 0;
	_buckets = new MapEntry*[ _numBuckets ];
	for (int i=0; i<_numBuckets; ++i)
	{
		_buckets[i] = NULL;
	}
}

// -----------------------------------------------------------------------------
// delete a MapEntry from the Map
void MapFile::removeEntry( MapEntry* entry )
{
	if ( entry->_prev != NULL )
	{
		entry->_prev->_next = entry->_next;
	}
	else
	{
		_buckets[ hash( entry->_key ) ] = entry->_next;
	}
	if ( entry->_next != NULL )
	{
		entry->_next->_prev = entry->_prev;
	}
	_count--;
}

// -----------------------------------------------------------------------------
// insert a MapEntry to the correct bucket
void MapFile::insertEntry( MapEntry* entry )
{
	int idx = hash( entry->_key );
	
	if ( _buckets[idx] == NULL )
	{
		_buckets[idx] = entry;
	}
	else
	{
		entry->_next = _buckets[idx];
		_buckets[idx]->_prev = entry;
		_buckets[idx] = entry;
	}
	_count++;
}

// -----------------------------------------------------------------------------
// find the map entry by key
MapEntry* MapFile::lookup( const char* key )
{
	MapEntry *e = _buckets[ hash(key) ];
	while ( e != NULL )
	{
		if ( compare( key, e->_key ) == 0 )
		{
			return e;
		}
		
		e = e->_next;
	}
	return NULL;
}

// -----------------------------------------------------------------------------
// hash the key into the bucket space
int MapFile::hash( const char* key )
{
	return strlen(key) % _numBuckets;
}

// -----------------------------------------------------------------------------
// scan a line. determine if it's real data, and if so, add it to the Map
void MapFile::scanLine( char* line )
{
	int len = strlen( line );
	if ( len < 3 )
		return;
	
	// skip leading WS
	while (( *line == ' ' ) || ( *line == '\t' )) line++;
	
	// skip comment and empty lines
	if (( *line == ';' ) || ( *line == '#' ) || (*line == '\0')) return;
	
	// everything else is data
	parseLine( line );
}

// -----------------------------------------------------------------------------
// parse the line into a "key = value" pair. Trim each, then insert into
// the map
void MapFile::parseLine( char* line )
{
	char *p = strchr( line, '=' );
	if ( p == NULL )
		return;
	*p++ = '\0';
	
	MapEntry *e = new MapEntry;
	e->_key = strdup( trim( line ) );
	e->_value = strdup( trim( p ) );
	insertEntry( e );
}

// -----------------------------------------------------------------------------
// trim WS from front & back, and nl from back.
// If first & last are then "", remove those too
char* MapFile::trim( char *s )
{
	while ( (*s==' ') || (*s=='\t') ) s++;
	
	char *e =  &s[strlen(s)] -1;
	while ( (*e==' ') || (*e=='\t') || (*e=='\n') ) *e-- = '\0';
	
	if (( *s == '"' ) && ( *e == '"' ))
	{
		s++;
		*s = '\0';
	}
	
	return s;
}

// -----------------------------------------------------------------------------
// compare using case flag
int MapFile::compare( const char *s1, const char *s2 )
{
	return _useCase? strcmp( s1, s2) : strcasecmp( s1, s2 );
}

// -----------------------------------------------------------------------------
// dump the data to the given stream
void MapFile::dump( FILE *f )
{
	for (int i=0; i<_numBuckets; ++i)
	{
		MapEntry *e = _buckets[i];
		while ( e != NULL )
		{
			fprintf( f, "%s = %s\n", e->_key, e->_value );
			e = e->_next;
		}
	}
}

// -----------------------------------------------------------------------------
// SIMPLE TEST
// -----------------------------------------------------------------------------

#if 0
int main( int argc, char *argv[] )
{
	if (argc != 2)
	{
		fprintf( stderr, "usage: %s <file> ", argv[0] );
	}
	
	MapFile m( argv[1] );
	m.dump( stderr );
}
#endif
